	NAME	spawno
	TITLE	Disk/EMS/XMS-swapping overlaying spawn() by Ralf Brown
	PAGE	60,132

;-----------------------------------
; (c) Copyright 1990, 1993 Ralf Brown  All Rights Reserved
; This code may be redistributed provided that:
;	1) this copyright notice remains intact and any changes are clearly
;	   marked
;	2) no fee of any kind is charged
;-----------------------------------
;  SPAWNO
;	Overlaying spawn()
;
;	Swaps all of the current program but a small stub to disk, XMS, or EMS
;	and then executes another program, restoring the current program on
;	return.	 The stub requires only 448 bytes when swapping to disk or XMS,
;	464 bytes when swapping to EMS.
;-----------------------------------
;  assembler defines:
;   __TINY__, __SMALL__, __COMPACT__, __MEDIUM__, __LARGE__, __HUGE__
;	exactly one of the above must be defined to specify the memory
;	model in use
;-----------------------------------

	INCLUDE RULES.ASI		; define the various standard macros

	Header@				; set up segment and group definitions

;-----------------------------------------------------------
; define mnemonic labels

SWAP_DISK equ 0
SWAP_XMS  equ 1
SWAP_EMS  equ 2

EWRITE_FAULT equ 29

;-----------------------------------------------------------
; describe the parameter block we need to pass to the DOS EXEC function

EXEC_data_block STRUC
   envseg      DW ?
   cmdline     DD ?
   firstfcb    DD ?
   secondfcb   DD ?
EXEC_data_block ENDS

;-----------------------------------------------------------
; describe the parameter block we need to pass to the XMS COPY function

_XMS_command struc
  ?XMSbytes   DD ?
  ?XMSsrc     DW ?
  ?XMSsoffset DD ?
  ?XMSdest    DW ?
  ?XMSdoffset DD ?
_XMS_command ends

;-----------------------------------------------------------
; define useful macros

ifdef __TINY__
    MOV_ES_PSP macro
	    push    cs
	    pop	    es
	endm
else
    MOV_ES_PSP macro
	     mov    es,_psp@
	 endm
endif ;__TINY__

;-----------------------------------------------------------
; describe the fields of the PSP we will be using, as well as the
; additional info that will be stuffed into it

PSP_SEG SEGMENT 'PSP' AT 0
	ORG 0Eh
OrigINT23h  LABEL DWORD
	ORG 12h
OrigINT24h  LABEL DWORD

; data is located as follows:
;  PSP+040h  14 bytes for exec parameter block
;  PSP+04Eh  128 bytes for commandline
;  PSP+0CEh  12 bytes for FCB1
;  PSP+0DAh  12 bytes for FCB1
;  PSP+0E6h  120 bytes for stack (16 bytes used by SPAWNO)
;  PSP+15Eh  98 bytes for disk swap-in code, 90 bytes for XMS swap-in code
;  PSP+1C0h  end of resident code (disk/XMS)
;  PSP+1D0h  end of resident code (EMS)

	ORG 40h
PSPstart label byte			; first byte of PSP we actually use
exec_block  EXEC_data_block <>		; 14 bytes long

CMD_LEN	    DB ?
cmd_line    DB 127 dup (?)
end_cmdline label byte

FCB1	    DB 12 dup (?)
FCB2	    DB 12 dup (?)

; reserve space for the temporary stack
	    DB 120 dup (?)
StackTop    label byte
PSP_data_size equ StackTop-PSPstart

relocated_swap_in_code label byte
temp = $ - PSPstart
relocated_swap_in_code_addr equ 15Eh	; unfortunately, TASM requires an
					;  absolute value in some places
PSP_SEG ENDS

IF 40h+temp ne relocated_swap_in_code_addr
  %out Update relocated_swap_in_code_addr
  .err
ENDIF

;-----------------------------------------------------------
; initialized data

DSeg@
ExtSym@ _psp,  WORD, __CDECL__
ExtSym@ errno, WORD, __CDECL__
ExtSym@ _doserrno, WORD, __CDECL__
ExtSym@ _osmajor,BYTE,__CDECL__

reload_error_msg DB 13,10
		 DB 9,"Error encountered while reloading program.  Your",13,10
		 DB 9,"system is probably in an unstable state and should",13,10
		 DB 9,"be rebooted as soon as possible.",7,13,10,"$"

emm_name    DB "EMMXXXX0"

PubSym@ __spawn_xms,<DB 1>,__CDECL__
PubSym@ __spawn_ems,<DB 1>,__CDECL__
PubSym@ __spawn_swap_UMA,<DB 1>,__CDECL__
PubSym@ __spawn_resident,<DW 0>,__CDECL__

DSegEnd@

;-----------------------------------------------------------
; uninitialized data storage area

BSeg@
OverlayFile DB 80 DUP (?)
XMSswapout_cmd _XMS_command <>
page_frame_seg DW ?

swapstack_buffer db PSP_data_size dup (?)
OldINT23h   DD ?
OldINT24h   DD ?

OverlayFD   DW ?
SwapParas   DW ?			; number of paragraphs to be swapped
XMS_size    DW ?
EMS_pages   DW ?			; number of 16K pages required

SwapEMSPages     dw ?			; temp storage for EMS address of
SwapEMSExtraPara dw ?			; additional memory blocks in swapspace

UMBs_linked DB ?			; were UMBs in the DOS memory chain?

swap_type   DB ?			; where are we swapping to?

extra_segs	 DW ?
ems_page_num	 DW ?
ems_page_offset	 DW ?
arena_header_buf DB 32 dup (?)		; large enough to get para-aligned buf
arena_buf_seg	 DW ?			; normalized segment of above buffer

BSegEnd@

;-----------------------------------------------------------

CSeg@
	ASSUME	CS:_TEXT

extrn __SPAWN_ERRNO:near

public __SPAWNV_start
__SPAWNV_start label byte

;----------------------------------------------------------------
; entry: DI = offset of XMS command block to clear
;----------------------------------------------------------------

clear_XMS_cmd_CS proc near
	push	es
	push	cs
	jmp short clear_XMS_cmd_body
clear_XMS_cmd_CS endp

clear_XMS_cmd proc near
	ASSUME	DS:DGROUP,ES:NOTHING
	push	es
	push	ds
clear_XMS_cmd_body:
	pop	es
	push	ax
	push	cx
	mov	cx,size _XMS_command
	cld
	xor	ax,ax
	rep	stosb
	pop	cx
	pop	ax
	pop	es
	ASSUME	ES:NOTHING
	ret
clear_XMS_cmd endp

;----------------------------------------------------------------
; entry: ES = starting segment
;	 DX = number of paragraphs
; return: CX = 0 failed, = 1 success
;	  AX destroyed
;----------------------------------------------------------------
write_block_disk proc near
	mov	ah,40h
read_write_disk:
	ASSUME	DS:DGROUP
	push	ds
	push	dx
	mov	cx,dx
	and	cx,7FFh			; modulo 32K
	shl	cx,1
	shl	cx,1
	shl	cx,1
	shl	cx,1			; para to bytes
	mov	bx,OverlayFD
	push	es
	pop	ds
	ASSUME	DS:NOTHING		; DS will differ every time through loop
	xor	dx,dx
	jcxz	rwd_no_partial
	push	ax
	int	21h
	jc	rwd_err_pop
	cmp	ax,cx
	pop	ax
	jne	rwd_err
	mov	dx,cx			; offset on later moves = size of first
rwd_no_partial:
	pop	cx			; get back number of paragraphs
	mov	cl,ch
	xor	ch,ch
	shr	cx,1
	shr	cx,1
	shr	cx,1			; number of 32K blocks to move
	jcxz	rwd_done
rwd_loop:
	push	cx
	push	ax
	mov	cx,8000h		; move 32K at a time
	int	21h			; no need to check CF, no error 8000h
	cmp	ax,cx
	jne	rwd_err_pop
	mov	ax,ds
	add	ax,800h			; move 32K higher in memory
	mov	ds,ax
	ASSUME	DS:NOTHING
	pop	ax
	pop	cx
	loop	rwd_loop
rwd_done:
	mov	cx,1			; say we were successful
	pop	ds
	ASSUME	DS:DGROUP
	ret

rwd_err_pop:
	ASSUME	DS:NOTHING
	pop	ax
	pop	cx
rwd_err:
	xor	cx,cx
	pop	ds
	ASSUME	DS:DGROUP
	ret
write_block_disk endp

;----------------------------------------------------------------
; entry: ES = starting segment
;	 DX = number of paragraphs
; return: CX = 0 failed, = 1 success
;	  ES, AX destroyed
;----------------------------------------------------------------
write_block_xms proc near
	ASSUME	DS:DGROUP
	push	si
	; start by adjusting offset in XMS to reflect last write
	mov	ax,word ptr XMSswapout_cmd.?XMSbytes
	mov	cx,word ptr XMSswapout_cmd.?XMSbytes+2
	add	word ptr XMSswapout_cmd.?XMSdoffset,ax
	adc	word ptr XMSswapout_cmd.?XMSdoffset+2,cx
	; set address at which to start copying
	mov	word ptr XMSswapout_cmd.?XMSsoffset+2,es
	mov	ax,16
	mul	dx			; bytes to paragraphs
	mov	word ptr XMSswapout_cmd.?XMSbytes,ax
	mov	word ptr XMSswapout_cmd.?XMSbytes+2,dx
	mov	si,offset DGROUP:XMSswapout_cmd
	mov	ah,0Bh
	call	dword ptr _TEXT:XMSentry
	mov	cx,ax			; XMS returns 0 on failure, 1 on success
	pop	si
	ret
write_block_xms endp

;----------------------------------------------------------------
; entry: ES = starting segment
;	 DX = number of paragraphs
; return: CX = 0 failed, = 1 success
;	  ES destroyed, possibly others
;----------------------------------------------------------------
write_mem_block proc near
	cmp	swap_type,SWAP_XMS	; find out where we aer swapping to
	je	write_block_xms		; type 1 = XMS
	ja	write_block_ems		; type 2 = EMS
	jmp	write_block_disk	; type 0 = disk
write_mem_block endp

;----------------------------------------------------------------
; entry: ES = starting segment
;	 DX = number of paragraphs
; return: CX = 0 failed, = 1 success
;	  ES destroyed
;----------------------------------------------------------------
write_block_ems proc near
	ASSUME	DS:DGROUP
	push	di
	push	si
	mov	cx,dx
write_ems_loop:
	mov	ax,4400h
	mov	bx,ems_page_num
	mov	dx,OverlayFD
	int	67h			; map in the appropriate EMS page
	or	ah,ah
	jnz	write_ems_failed
	mov	bx,ems_page_offset
	mov	dx,bx
	sub	bx,400h			; 1024 paragraphs in a 16K EMS page
	neg	bx			; adjust for doing sub the wrong way
	cmp	bx,cx
	jbe	write_ems_fullpage
	mov	bx,cx
	add	ems_page_offset,bx
	jmp short write_ems_do_it
write_ems_fullpage:
	mov	ems_page_offset,0
	inc	ems_page_num
write_ems_do_it:
	push	cx
	push	bx
	push	ds
	mov	ax,page_frame_seg
	add	ax,dx			; copy seg = page_fr + offset
	mov	dx,es
	mov	ds,dx
	mov	es,ax
	ASSUME	DS:NOTHING,ES:NOTHING
	xor	si,si
	xor	di,di
	mov	cx,bx
	shl	cx,1
	shl	cx,1
	shl	cx,1			; paragraphs to words
	cld
	rep	movsw			; copy as much memory as we can
	add	dx,bx
	mov	es,dx			; move higher in mem by amount written
	pop	ds
	ASSUME	DS:DGROUP
	pop	bx
	pop	cx
	sub	cx,bx
	jnz	write_ems_loop
	mov	cx,1			; tell caller we succeeded
	pop	si
	pop	di
	ret

write_ems_failed:
	xor	cx,cx			; tell caller we failed
	pop	si
	pop	di
	ret
write_block_ems endp

;----------------------------------------------------------------
; write memory blocks other than PSP block and environment to swap device
;----------------------------------------------------------------
write_other_blocks proc near
	ASSUME	DS:DGROUP,ES:NOTHING
	push	es
	push	dx
	push	cx
	mov	ah,52h
	int	21h			; get list of lists
	mov	es,es:[bx-2]		; get address of first MCB into ES
	mov	cx,extra_segs
write_extra_loop:
	mov	dx,1			; assume only arena header written
	mov	ax,es
	inc	ax			; memory block follows MCB in next para
	cmp	ax,_psp@		; is this our PSP block?
	je	write_extra_block	; if yes, write only arena header
	mov	ax,es:[1]		; get block's owner
	cmp	ax,_psp@		; block owned by someone else?
	jne	write_extra_block	; if yes, write only arena header
	add	dx,es:[3]		; otherwise, swap out entire block
write_extra_block:
	push	cx
	push	es
	call	write_mem_block
	pop	es
	mov	ax,es
	add	ax,es:[3]
	inc	ax			; point ES at next MCB
	mov	es,ax
	or	cx,cx			; check whether the write succeeded
	pop	cx
	loopne	write_extra_loop	; until done or write failed
	jcxz	write_others_done	; did we write all segments?
	pop	cx
	pop	dx
	pop	es
	inc	sp
	inc	sp			; discard return address
	jmp	write_err		; error while writing
write_others_done:
	mov	cx,extra_segs
	mov	ah,52h
	int	21h			; get list of lists
	mov	dx,es:[bx-2]		; get address of first MCB into DX
	mov	es,dx
	ASSUME	ES:NOTHING
	add	dx,es:[3]
	inc	dx
free_extra_loop:
	mov	es,dx			; point ES at current MCB
	mov	ax,dx			; remember current MCB address
	add	dx,es:[3]
	inc	dx			; point DX at next MCB
	inc	ax
	cmp	ax,_psp@		; are we looking at our PSP block?
	je	extra_block_freed	; if yes, don't free it
	mov	ax,es:[1]		; get block's owner
	cmp	ax,_psp@		; someone else's block?
	jne	extra_block_freed	; if yes, don't free it
	mov	ax,es
	inc	ax			; mem block is at paragraph after MCB
	mov	es,ax
	mov	ah,49h
	int	21h			; free the memory block
extra_block_freed:
	loop	free_extra_loop
	mov	ah,48h
	mov	bx,0FFFFh		; force DOS to coalesce free blocks
	int	21h
	pop	cx
	pop	dx
	pop	es
	ret
write_other_blocks endp

;----------------------------------------------------------------
; entry: AX=starting segment
;	 DX=number of paragraphs
; exit:	 CX=0 failed, 1=success
;	 AX destroyed
;----------------------------------------------------------------
read_memory_disk proc near
	push	es
	mov	es,ax
	mov	ah,3Fh
	call	read_write_disk
	pop	es
	ret
read_memory_disk endp

;----------------------------------------------------------------
; entry: AX=starting segment
;	 DX=number of paragraphs
;----------------------------------------------------------------
read_memory_block proc near
	ASSUME	DS:DGROUP
	cmp	swap_type,SWAP_XMS
	jb	read_memory_disk	; swap type 0 is disk
	ja	read_memory_ems		; swap type 2 is EMS
	;; fall through to read_memory_xms
read_memory_block endp

;----------------------------------------------------------------
; entry: AX=segment
;	 DX=number of paragraphs
; exit:	 CX=0 failed, =1 success
;	 AX,DX destroyed
;----------------------------------------------------------------
read_memory_xms proc near
	ASSUME	DS:DGROUP
	push	si
	push	ax
	push	ds
	push	cs
	pop	ds
	ASSUME	DS:_TEXT
	; set address at which to start copying
	mov	word ptr XMSswapin_cmd.?XMSdoffset+2,ax
	; adjust offset in XMS to reflect last read
	mov	ax,word ptr XMSswapin_cmd.?XMSbytes
	mov	cx,word ptr XMSswapin_cmd.?XMSbytes+2
	add	word ptr XMSswapin_cmd.?XMSsoffset,ax
	adc	word ptr XMSswapin_cmd.?XMSsoffset+2,cx
	mov	ax,16
	mul	dx			; bytes to paragraphs
	mov	word ptr XMSswapin_cmd.?XMSbytes,ax
	mov	word ptr XMSswapin_cmd.?XMSbytes+2,dx
	mov	si,offset _TEXT:XMSswapin_cmd
	mov	ah,0Bh
	call	dword ptr _TEXT:XMSentry
	pop	ds
	ASSUME	DS:DGROUP
	mov	cx,ax			; XMS returns 0 on failure, 1 on success
	pop	ax
	pop	si
	ret
read_memory_xms endp


;----------------------------------------------------------------
; enter: AX=starting segment
;	 DX=num paras
;----------------------------------------------------------------
read_memory_ems proc near
	ASSUME	DS:DGROUP
	push	es
	mov	es,ax
	ASSUME	ES:NOTHING
	push	di
	push	si
	mov	cx,dx
read_ems_loop:
	mov	ax,4400h
	mov	bx,ems_page_num
	mov	dx,OverlayFD
	int	67h			; map in the appropriate EMS page
	or	ah,ah
	jnz	read_ems_failed
	mov	bx,ems_page_offset
	mov	dx,bx
	sub	bx,400h			; 1024 paragraphs in a 16K EMS page
	neg	bx			; adjust for doing sub the wrong way
	cmp	bx,cx
	jbe	read_ems_fullpage
	mov	bx,cx
	add	ems_page_offset,bx
	jmp short read_ems_do_it
read_ems_fullpage:
	mov	ems_page_offset,0
	inc	ems_page_num
read_ems_do_it:
	push	cx
	push	bx
	push	ds
	mov	ax,page_frame_seg
	add	ax,dx			; copy seg = page_fr + offset
	mov	ds,ax
	ASSUME	DS:NOTHING
	xor	si,si
	xor	di,di
	mov	cx,bx
	shl	cx,1
	shl	cx,1
	shl	cx,1			; paragraphs to words
	cld
	rep	movsw			; copy as much memory as we can
	mov	dx,es
	add	dx,bx
	mov	es,dx			; move higher in mem by amount written
	pop	ds
	ASSUME	DS:DGROUP
	pop	bx
	pop	cx
	sub	cx,bx
	jnz	read_ems_loop
	mov	cx,1			; tell caller we succeeded
read_ems_done:
	pop	si
	pop	di
	pop	es
	ret

read_ems_failed:
	xor	cx,cx			; tell caller we failed
	jmp	read_ems_done
read_memory_ems endp

;----------------------------------------------------------------
; copy memory blocks other than PSP block and environment back in from swap
; return: AX, DX destroyed
;----------------------------------------------------------------
read_other_blocks proc near
	ASSUME	DS:DGROUP
	push	si
	push	di
	push	es
	push	cx
	mov	ax,ds
	mov	dx,offset DGROUP:arena_header_buf+15
	mov	cl,4
	shr	dx,cl
	add	ax,dx
	mov	arena_buf_seg,ax	; store segment address of MCB buffer
	mov	ah,52h
	int	21h			; get list of lists
	mov	ax,es:[bx-2]		; get address of first MCB into AX
	mov	cx,extra_segs
read_memory_loop:
	push	cx			; save loop counter
	push	ax			; remember current MCB address
	mov	ax,arena_buf_seg	; read next MCB from swap into temp
	mov	es,ax
	ASSUME	ES:NOTHING
	mov	dx,1
	call	read_memory_block	; read DX paragraphs into AX:0000h
	pop	ax			; retrieve current MCB address
	jcxz	abort_reload
	mov	cx,ax
	inc	cx			; memory block is para after MCB
	cmp	cx,_psp@		; are we looking at our PSP block?
	je	read_memory_loopend	; if yes, nothing to restore
	mov	cx,es:[1]		; get block's owner
	mov	dx,es:[3]		;  and size, in case we need to read it
	mov	es,ax			; point ES at MCB to be restored
	jcxz	restore_MCB		; restore MCBs only for free memory
	cmp	cx,_psp@		; don't restore block at all if not
	jne	read_memory_loopend	;   ours and not free
	inc	ax			; memory block is paragraph after MCB
	call	read_memory_block	; read DX paragraphs into AX:0000h
	jcxz	abort_reload		; quit if error while reading
restore_MCB:
	push	ds
	mov	ds,arena_buf_seg	; source of copy is temporary buffer
	ASSUME	DS:NOTHING
	xor	si,si			; MCBs are always at offset 0
	xor	di,di
	mov	cx,8			; eight words in paragraph
	cld
	rep	movsw			; copy MCB from temp to final location
	pop	ds
	ASSUME	DS:DGROUP
	mov	ax,es			; restore AX to MCB address
read_memory_loopend:
	add	ax,es:[3]
	inc	ax			; move to next MCB
	pop	cx
	loop	read_memory_loop	; repeat until all blocks processed
read_others_done:
	pop	cx
	pop	es
	pop	di
	pop	si
	ret
read_other_blocks endp

abort_reload proc near
	mov	dx,offset DGROUP:reload_error_msg
	mov	ah,9
	int	21h
	mov	ax,4CFFh
	int	21h
abort_reload endp

;----------------------------------------------------------------
; the code which is copied down into the resident portion starts here
;

;----------------------------------------------------------------
; entry: CS = PSP
;	 AX = 4B00h
;	 ES:BX -> filled exec data block
;	 DS:DX -> program name
;	 STACK: WORD	PSP segment
;		WORD	SwapCount
;		WORD	SwapSize
;		WORD	OrigSize
;		WORD	OverlayFD
;		WORD	LoadSeg
;		DWORD	return address
; exit:	 CF clear if spawn successful
;	 CF set if spawn failed
;	     AX = DOS error code
;	 all other registers except SS, SP, and BP destroyed
;
$overlay proc far
	int	21h			; go EXEC the child program
	ASSUME	DS:NOTHING,ES:NOTHING
	cli				; don't interrupt while switching
	mov	dx,cs			; restore stack that DOS 2.x
	mov	ss,dx			;   clobbered
	mov	sp,offset PSP_SEG:StackTop-16
	sti				; OK to interrupt now
	pop	es			; get value of PSP segment
	ASSUME	ES:PSP_SEG
	pop	dx			; get value of SwapCount
	pop	cx			; get value of SwapSize
	pop	bx			; get value of OrigSize
	pop	si			; get value of OverlayFD
	pop	ds			; get value of LoadSeg
	pushf				; save carry flag
	push	ax			; and squirrel away return/error code
; expand the memory block back to its original size
	mov	ah,4Ah			; BX = OrigSize
	int	21h
	jnc	reload_memory		; did memory block get expanded?
abort_overlay:
	push	cs
	pop	ds
	ASSUME	DS:_TEXT
	mov	dx,offset relocated_swap_in_code + (ReloadErrMsg - $overlay)
	mov	ah,9			; display error message
	int	21h
	mov	ax,4CFFh		; and terminate with ERRORLEVEL 255
	int	21h

ReloadErrMsg DB "SWAP ERROR",7,"$"

reload_memory:
$overlay endp
;
; at this point, the following register values are available:
;    BX = OrigSize
;    CX = SwapSize
;    DX = SwapCount
;    SI = OverlayFD
;    DS = LoadSeg
;    ES = PSPseg

overlay_size equ $ - $overlay

; DON'T PUT ANY CODE IN BETWEEN $OVERLAY AND $DISK_OVERLAY

$disk_overlay proc far
; reload from temp file
	ASSUME	DS:NOTHING
	push	dx			; save SwapCount for later
	mov	bx,si			; BX = OverlayFD, CX already = SwapSize
	xor	dx,dx			; DS already has LoadSeg
	mov	ah,3Fh
	jcxz	read_ok			; skip read if swap is multiple of 16K
	int	21h			; read in CX bytes
	jc	abort_overlay		; was read successful?
	cmp	ax,cx			; did we get all the data we asked for?
	jne	abort_overlay
    ; code to swap in >16K in 16K chunks (remainder already loaded by prev
    ; section)
read_ok:
	pop	cx			; get back number of 16K chunks to read
	jcxz	reload_done		; skip loop if no 16K chunks to read
	mov	dx,ax			; offset reads by amount of first read
reload_loop:
	push	cx
	mov	cx,4000h		; read 16K, BX is still OverlayFD
	mov	ah,3Fh			; DS:DX -> next chunk to restore
	int	21h			; read in CX bytes
					; no need to check CF, since error 4000h
					;    doesn't exist
	cmp	ax,cx			; did we get everything we asked for?
	jne	abort_overlay
	mov	ax,ds			; move up 16K in memory
	add	ax,400h
	mov	ds,ax
	ASSUME	DS:NOTHING
	pop	cx			; get back count of 16K chunks
	loop	reload_loop		; keep going until all chunks read in
reload_done:
	pop	ax			; get back the program's return code
	popf				; and carry flag
	ret
$disk_overlay endp

disk_overlay_size equ $ - $disk_overlay

;----------------------------------------------------------------
; section of code to copy over top of the reload loop when using XMS for
; swapping
;  IMPORTANT:  must be no larger than the code in $disk_overlay
;
; on entry,
;    BX = OrigSize
;    CX = SwapSize <- XMS entry pt offset
;    DX = SwapCount <- XMS entry pt segment
;    SI = OverlayFD
;    DS = LoadSeg
;    ES = PSPseg
;
$xms_overlay proc far
relocation_factor = (offset $xms_overlay - relocated_swap_in_code_addr - overlay_size)
	ASSUME	DS:NOTHING
	push	cs
	pop	ds
	ASSUME	DS:_TEXT
	mov	si,offset $xms_done - relocation_factor
	push	cs
	push	si
	mov	si,offset XMSswapin_cmd - relocation_factor
	mov	ah,0Bh		       ; XMS copy memory command
	push	dx
	push	cx
	ret				; invoke XMS driver
$xms_done:
	or	ax,ax
	je	short (abort_overlay + ($xms_overlay - reload_memory))
	pop	ax		       ; get back the program's return code
	popf			       ; and carry flag
	ret

;XMS_entry   DD ?

XMSswapin_cmd _XMS_command <>

$xms_overlay endp

xms_overlay_size equ $ - $xms_overlay

;----------------------------------------------------------------
; section of code to copy over top of the reload loop when using EMS for
; swapping
;
; on entry:
;    BX = OrigSize
;    CX = SwapSize
;    DX = SwapCount
;    SI = OverlayFD
;    DS = LoadSeg
;    ES = PSPseg
;
$ems_overlay proc far
	ASSUME	DS:NOTHING
	push	cx			; remember partial last page
	mov	cx,dx			; CX = SwapCount
	mov	dx,si			; DX = EMS handle
	push	ds
	pop	es			; ES = LoadSeg
__$ems$__:
	mov	ax,0			; will be patched with EMS page frame seg
$ems_page_frame equ word ptr (__$ems$__ + 1)  ;for patching prev instruction
	mov	ds,ax
	ASSUME	DS:NOTHING,ES:NOTHING
	xor	bx,bx			; start on logical page 0
	cld				; make sure we copy from low to high
	jcxz	ems_partial		; skip down if no full pages
ems_reload_loop:
	push	cx			; remember loop counter
	mov	cx,2000h		; 16Kbytes per page
	call	ems_copypage		; copy first CX words from page BX
	mov	ax,es
	add	ax,400h			; move higher in memory
	mov	es,ax
	inc	bx			; go to next logical page
	pop	cx
	loop	ems_reload_loop
ems_partial:
	pop	cx			; retrieve last page's size
	jcxz	ems_reload_done		; do we have a partial page?
	call	ems_copypage		; copy first CX words from page BX
ems_reload_done:
	pop	ax			; get back program's return code
	popf				; and carry flag
	ret

ems_copypage proc near
	ASSUME	DS:NOTHING,ES:NOTHING
	mov	ax,4400h		; map logical page
	int	67h
	or	ah,ah			; was EMS call successful?
	jne short (abort_overlay + ($ems_overlay - reload_memory))
	xor	si,si			; page starts at offset 0
	xor	di,di
	rep	movsw			; copy the page
	ret
ems_copypage endp
$ems_overlay endp

ems_overlay_size equ $ - $ems_overlay

;----------------------------------------------------------------
; compute the size of the resident portion for disk/XMS and EMS

res_size_disk equ relocated_swap_in_code_addr + overlay_size + disk_overlay_size
res_para_disk equ (res_size_disk + 15) / 16

res_size_ems  equ relocated_swap_in_code_addr + overlay_size + ems_overlay_size
res_para_ems  equ (res_size_ems + 15) / 16

;----------------------------------------------------------------

pname	    DD ?
OriginalSS  DW ?
OriginalSP  DW ?
ShrunkSize  DW ?
SwapCount   DW 0
SwapSize    DW ?
OrigSize    DW ?
LoadSeg	    DW ?
using_XMS   DW ?
XMSentry    DD ?

;----------------------------------------------------------------
; return the number of EMS pages which are available (0 if no EMS)

EMS_available proc near
	push	es
	push	si
	push	di
	mov	ax,3567h
	int	21h		; get vector for EMS driver
	mov	di,10		; ES:DI -> driver name
	mov	si,offset DGROUP:emm_name
	mov	cx,4		; length of name in words
	cld
	xor	ax,ax		; assume no match
	repz	cmpsw
	jnz	EMS_avail_done
	mov	ah,42h
	int	67h
	or	ah,ah		; was call successful?
	mov	ax,0		; assume not
	jnz	EMS_avail_done
	mov	ax,bx		; return number of pages available
EMS_avail_done:
	pop	di
	pop	si
	pop	es
	ret
EMS_available endp

;----------------------------------------------------------------
; convert a WORD in DL into four hex digits at ES:DI

$itoh proc near
	mov	al,dh
	call	$itoh2
	mov	al,dl
$itoh2:
	push	ax
	mov	cl,4
	shr	al,cl
	call	$itoh1
	pop	ax
	and	al,0Fh
$itoh1:
	add	al,90h			; convert AL to hex digit
	daa
	adc	al,40h
	daa
	stosb
	ret
$itoh endp

;------------------
;
preserve_PSP proc near
	push	es
	push	ds
	push	si
	push	di
	mov	si,offset PSP_SEG:PSPstart
	mov	di,offset DGROUP:swapstack_buffer
	push	ds
	pop	es
	ASSUME	ES:DGROUP
	mov	ds,_psp@
	ASSUME	DS:PSP_SEG
copy_PSP:
	ASSUME	ES:NOTHING
	push	cx
	mov	cx,PSP_data_size
	cld
	rep	movsb
	pop	cx
	pop	di
	pop	si
	pop	ds
	ASSUME	DS:DGROUP
	pop	es
	ASSUME	ES:NOTHING
	ret
preserve_PSP endp

;------------------
;
restore_PSP proc near
	push	es
	push	ds
	push	si
	push	di
	mov	si,offset DGROUP:swapstack_buffer
	mov	di,offset PSP_SEG:PSPstart
	MOV_ES_PSP
	ASSUME	ES:PSP_SEG
	jmp	copy_PSP
restore_PSP endp

;------------------
; swap the CX bytes of memory starting at DS:SI with the CX bytes starting
; at ES:DI
; destroys CX, SI, DI
;
xchg_memory proc near
	mov	al,[si]
	mov	ah,es:[di]
	mov	es:[di],al
	mov	[si],ah
	inc	si
	inc	di
	loop	xchg_memory
	ret
xchg_memory endp

;------------------
; swap the code for the resident stub with whatever happens to be in its
; final location in the PSP
; destroys CX
;
xchg_code proc near
	ASSUME	DS:DGROUP
	push	es
	push	si
	push	di
	push	ax
	push	ds
	MOV_ES_PSP
	ASSUME	ES:PSP_SEG
	mov	di,offset PSP_SEG:relocated_swap_in_code
ifndef __TINY__
	push	cs
	pop	ds
	ASSUME	DS:_TEXT
endif ;ndef __TINY__
	mov	si,offset $overlay
	mov	cx,overlay_size
	call	xchg_memory
ifndef __TINY__
	pop	ds
	ASSUME	DS:DGROUP
	push	ds
endif ;ndef __TINY__
	cmp	swap_type,SWAP_XMS	; find out where we are swapping to
	mov	si,offset _TEXT:$disk_overlay
	mov	cx,disk_overlay_size	; first case: swapping to disk
	jb	go_xchg_code		; swap type 0 = disk
xchg_code_xms:
	mov	si,offset _TEXT:$xms_overlay
	mov	cx,xms_overlay_size	; second case: swapping to XMS
	je	go_xchg_code		; swap type 1 = XMS
xchg_code_ems:
	mov	si,offset _TEXT:$ems_overlay
	mov	cx,ems_overlay_size	; third case: swapping to EMS
go_xchg_code:
ifndef __TINY__
	push	cs
	pop	ds
endif ;__TINY__
	ASSUME	DS:_TEXT
	mov	di,offset PSP_SEG:relocated_swap_in_code + overlay_size
	call	xchg_memory
xchg_code_done:
	pop	ds
	ASSUME	DS:DGROUP
	pop	ax
	pop	di
	pop	si
	pop	es
	ret
xchg_code endp

;----------------------------------------------------------------
; int pascal __spawnv(char *overlay_path, char *name, arg_list args, int env) ;
;----------------------------------------------------------------
PubProc@  __SPAWNV, __PASCAL__
; parameters
 @overlay_path = DPTR_ [bp+4+cPtrSize+(2*dPtrSize)]
 @name = DPTR_ [bp+4+cPtrSize+dPtrSize]
 @args = DPTR_ [bp+4+cPtrSize]
 @env = word ptr [bp+2+cPtrSize]
; start of actual subroutine
	push	bp
	mov	bp,sp
	push	es
	push	di
	push	ds
	push	si
IFDEF __HUGE__
	mov	ax,DGROUP
	mov	ds,ax
ENDIF ;__HUGE__
	ASSUME	DS:DGROUP
; store the MSDOS 5+ UMB link state and force it to the desired value
	mov	ax,5802h
	int	21h
	mov	UMBs_linked,al		; remember original UMB linkage state
	mov	ax,5803h
	mov	bl,__spawn_swap_UMA@	; specify whether UMBs in DOS mem chain
	xor	bh,bh
	int	21h
; store the critical interrupt and ^Break vectors, which will be changed
	mov	ax,3524h
	int	21h			; get vector 24h into ES:BX
	mov	word ptr OldINT24h,bx	; and store it
	mov	word ptr OldINT24h + 2,es
	mov	ax,3523h
	int	21h			; get vector 23h into ES:BX
	mov	word ptr OldINT23h,bx	; and store it
	mov	word ptr OldINT23h + 2,es
; preserve the contents of the PSP and some of the start of the program
	call	preserve_PSP
; set up addressing to PSP, which will store the data we'll pass to the program
	MOV_ES_PSP
	ASSUME	es:PSP_SEG
	push	ds
	LDS_	ax,@name		; get pointer to name of program to exec
	ASSUME	DS:NOTHING
	mov	word ptr pname,ax	; and store where we can get at it
	mov	word ptr pname+2,ds	;   later
; fill in the EXEC data block
	push	es
	pop	ds
	ASSUME	DS:PSP_SEG
	mov	ax,@env
	mov	exec_block.envseg,ax
	mov	word ptr exec_block.firstfcb,offset PSP_SEG:FCB1
	mov	word ptr exec_block.firstfcb + 2,es
	mov	word ptr exec_block.secondfcb,offset PSP_SEG:FCB2
	mov	word ptr exec_block.secondfcb + 2,es
	mov	word ptr exec_block.cmdline,offset PSP_SEG:CMD_LEN
	mov	word ptr exec_block.cmdline + 2,es
	pop	ds
	ASSUME	DS:DGROUP
; first, concatenate the args to form the commandline
	pushDS_
	mov	di,offset PSP_SEG:cmd_line
	LDS_	si,@args		; point at list of pointers to args
	ASSUME DS:NOTHING
	cld
	mov	ax,[si]			; check if argv[0] is NULL, in which
IF LDATA				;   case we skip parameter list
	or	ax,[si+2]		;   entirely
ELSE
	or	ax,ax
ENDIF
	jz	parm_cat_done2
parm_cat_loop:
	add	si,dPtrSize		; move to next parameter
	pushDS_
	push	si
	LDS_	si,[si]
if LDATA
	mov	ax,ds
	or	ax,si
else
	or	si,si
endif
	je	parm_cat_done
another_parm:
	cmp	di,offset PSP_SEG:end_cmdline-1
	jae	parm_cat_done
	mov	al,' '			; separate parameters by blanks
	stosb
parm_copy_loop:
	lodsb
	or	al,al
	jz	parm_copy_done
	stosb
	cmp	di,offset PSP_SEG:end_cmdline-1
	jb	parm_copy_loop
	jmp short parm_cat_done
parm_copy_done:
	pop	si
	popDS_
	jmp	parm_cat_loop
parm_cat_done:
	pop	si
	popDS_				; pop the DS:SI stored during the loop
parm_cat_done2:
	popDS_
	ASSUME DS:DGROUP
; append the needed CR to the command line, then set the length
	mov	al,13
	stosb
	mov	ax,di
	sub	ax,offset PSP_SEG:cmd_line+1 ; don't count the last CR
	mov	es:cmd_len,al
; now set up the FCBs
	push	ds
	push	es
	pop	ds
	ASSUME	DS:PSP_SEG
	mov	si,offset PSP_SEG:cmd_line
	mov	di,offset PSP_SEG:FCB1	; parse first arg into FCB1
	mov	ax,2901h		;   skipping leading separator chars
	int	21h
	jc	unparseable_name
	cmp	al,0FFh
	je	unparseable_name
	mov	di,offset PSP_SEG:FCB2	; if successful, parse second arg into
	mov	ax,2901h		;   FBC2, again skipping leading
	int	21h			;   separator characters
unparseable_name:
	pop	ds
	ASSUME	DS:DGROUP
; calculate memory requirements
	mov	dx,res_para_disk	; min resident paragraphs for swapping code
	mov	ax,__spawn_resident@
	cmp	dx,ax
	jae	have_res_paras
	mov	dx,ax
have_res_paras:
	mov	ax,dx
	mov	ShrunkSize,ax		; number of paragraphs to keep resident
	mov	cx,_psp@		; add to PSP to get the
	add	dx,cx			;   segment at which to load new program
	mov	LoadSeg,dx		; store that address for later
	dec	cx			; CX <- segment of memory block header
	push	ds			; save DS
	mov	ds,cx			; point DS at our mem block header
	ASSUME	DS:NOTHING
	mov	cx,word ptr ds:[3]	; get current size of memory block
	pop	ds			; restore DS
	ASSUME	DS:DGROUP
	mov	OrigSize,cx
	sub	cx,ax			; how many paragraphs do we have to save?
	jnc	shrunksize_adjusted
	xor	cx,cx			; don't swap any of the PSP block
	mov	ax,OrigSize
	mov	ShrunkSize,ax
shrunksize_adjusted:
	mov	dx,cx			; need to compute both chunks and K
	mov	SwapParas,cx		; store for XMS swap count
	mov	al,ch
	xor	ah,ah			; AX <- CX shr 8
	shr	ax,1			; determine number of 16K blocks
	shr	ax,1			; AX <- paragraphs / 1024
	mov	SwapCount,ax		; and store for later
	mov	si,ax			; (also squirrel it away for later)
	and	cx,03ffh		; find remainder less than 16K
	shl	cx,1
	shl	cx,1
	shl	cx,1
	shl	cx,1			; turn paragraphs into bytes
	mov	SwapSize,cx		; and store for later
	; at this point, DX = paragraphs to swap out from PSP segment
	push	es
	ASSUME	ES:NOTHING
	mov	ah,48h
	mov	bx,0FFFFh		; try to alloc 1M, to force DOS to
	int	21h			;   coalesce free blocks
	jc	coalesced
	mov	es,ax
	mov	ah,49h
	int	21h			; free memory block if alloc successful
coalesced:
	mov	ah,52h
	int	21h			; get list of lists
	mov	es,es:[bx-2]		; get address of first MCB into ES
	xor	bx,bx			; initially no extra blocks of memory
	mov	cx,_psp@
extra_count_loop:
	inc	bx			; we have another block of memory
	inc	dx			; always swap at least the mem arena
	mov	ax,es
	inc	ax			; memory block follows MCB
	cmp	ax,cx			; is this our PSP memory block?
	je	extra_count_loop_end	; if yes, skip
	mov	ax,es:[1]
	cmp	ax,cx			; do we own the block?
	jne	extra_count_loop_end	; if not, go immediately to next
	add	dx,es:[3]		; is ours, so swap out the block
extra_count_loop_end:
	cmp	byte ptr es:[0],'Z'	; was this the last memory block?
	je	extra_count_done
	mov	ax,es
	add	ax,es:[3]		; move to next MCB
	inc	ax
	mov	es,ax
	jmp	extra_count_loop
extra_count_done:
	mov	extra_segs,bx
	pop	es
	; now DX = total paragraphs needed on swap device
	mov	ax,dx			; need two copies of # of paragraphs
	add	dx,63			; round up to next K's worth of paragraphs
	mov	cl,6
	shr	dx,cl			; turn paragraphs into KBytes
	mov	XMS_size,dx		; and store for later
	add	ax,1023			; round paragraphs up to next multiple
	mov	cl,10			;   of 16K to get EMS pages
	shr	ax,cl
	mov	EMS_pages,ax		; store EMS pages for later
; check whether XMS is available
	cmp	__spawn_xms@,0		; are we allowed to use XMS?
	je	no_XMS			; if not, try EMS
check_XMS:
	mov	ax,4300h		; check if XMS driver installed
	int	2Fh
	cmp	al,80h			; installed?
	jne	no_XMS			; if not, try EMS
	mov	ax,4310h
	int	2Fh			; get XMS driver entry point
	ASSUME	ES:NOTHING
	mov	word ptr XMSentry,bx
	mov	word ptr XMSentry+2,es
	mov	ah,08h			; how much XMS memory is available?
	call	XMSentry
	mov	dx,XMS_size		; enough XMS for swapping?
	cmp	ax,dx
	jb	no_XMS			; if not, try EMS
	mov	ah,09h			; allocate XMS memory block
	call	XMSentry
	or	ax,ax
	jz	no_XMS			; if failed, try EMS
	mov	di,offset _TEXT:XMSswapin_cmd
	call	clear_XMS_cmd_CS	; prepare the XMS command blocks
	mov	di,offset DGROUP:XMSswapout_cmd
	call	clear_XMS_cmd
	mov	di,dx			; squirrel away handle for later
	mov	XMSswapin_cmd.?XMSsrc,dx
	mov	XMSswapout_cmd.?XMSdest,dx
	mov	swap_type,SWAP_XMS
	mov	ax,LoadSeg
	mov	word ptr XMSswapin_cmd.?XMSdoffset+2,ax
	mov	ax,word ptr XMSentry+2
	mov	SwapCount,ax		; SwapCount:SwapSize holds entry pt
	mov	ax,word ptr XMSentry	; of XMS driver
	mov	SwapSize,ax
	mov	ax,16
	mul	SwapParas		; paragraphs to bytes
	mov	word ptr XMSswapin_cmd.?XMSbytes,ax
	mov	word ptr XMSswapin_cmd.?XMSbytes+2,dx
	xor	ax,ax
	jmp	create_complete
no_XMS:
	cmp	__spawn_ems@,0		; are we allowed to use EMS?
	je	no_EMS			; if not, swap to disk
	call	EMS_available		; find out how many EMS pages are free
	cmp	ax,EMS_pages		; enough EMS memory for swapping?
	jb	no_EMS			; if not, swap to disk
	mov	ah,41h			; get EMS page frame segment
	int	67h
	or	ah,ah			; was call successful
	jnz	no_EMS			; swap to disk if not
	mov	$ems_page_frame,bx	; store page frame segment in swap-in code
	mov	page_frame_seg,bx
	mov	ah,43h			; allocate EMS for swapping
	mov	bx,EMS_pages
	int	67h
	or	ah,ah
	jnz	no_EMS			; if error, swap to disk
; the EMS swapping code is larger than the disk or EMS code, so adjust things
	mov	ax,res_para_ems - res_para_disk
	add	ShrunkSize,ax
	add	LoadSeg,ax
	sub	SwapParas,ax
	mov	ax,SwapSize		; reduce number of bytes to swap out
	sub	ax,16*(res_para_ems - res_para_disk)
	sbb	SwapCount,0
	and	ax,3FFFh		; limit modulus to 16K
	shr	ax,1			; convert modulus from bytes to words
	mov	SwapSize,ax
	shr	ax,1			; convert to paragraphs
	shr	ax,1
	shr	ax,1
	mov	SwapEMSExtraPara,ax
	mov	ax,SwapCount		; preserve EMS pointer until after swap
	mov	SwapEMSPages,ax
	mov	ax,dx			; squirrel away handle for later
	mov	swap_type,SWAP_EMS	; remember that we are swapping to EMS
	jmp short create_complete

; create a temporary file if we can't use XMS or EMS memory
no_EMS:
	pushDS_
	push	si
	mov	dx,_psp@		; get PSP while DS is still DGROUP
	push	ds
	pop	es
	ASSUME	ES:DGROUP
	mov	di,offset DGROUP:OverlayFile
	LDS_	si,@overlay_path	; get pointer to overlay path
	ASSUME	DS:NOTHING
	cld
	mov	cx,size OverlayFile-14	; reserve space for file's name+ext
copy_path_loop:				; copy ASCIZ string
	lodsb
	or	al,al			; directory name ends with NUL
	jz	copy_path_done
	cmp	al,';'			;   or with a semicolon
	jz	copy_path_done
	stosb
	loop	copy_path_loop
copy_path_done:
	pop	si
	popDS_
	ASSUME	DS:DGROUP
	mov	al,byte ptr [di-1]	; check if last character was / or \
	cmp	al,'/'
	je	has_slash
	cmp	al,'\'
	je	has_slash
	mov	al,'/'			; if not, add the slash
	stosb
has_slash:
	mov	ax,'VO'			; append "OVRL" to path
	stosw
	mov	ax,'LR'
	stosw
	call	$itoh			; append hex of PSP in DX to path
	mov	ax,'$.'			; finally, append ".$$$" to path
	stosw
	mov	al,'$'
	stosw
	mov	al,0			; and zero-terminate the string
	stosb
	mov	swap_type,SWAP_DISK
	mov	ax,3C00h		; create the overlay file
	mov	dx,offset DGROUP:OverlayFile
	mov	cx,0			; normal file, no special attributes
	int	21h
	jnc	create_OK		; were we able to create the file?
create_failed:
	mov	ax,-1			; if not, indicate error
	push	ax			; (need return value on stack)
	jmp	spawn_complete		; and return to caller
create_OK:
	mov	bx,ax
	mov	ah,3Eh
	int	21h			; close the newly-created file
	jc	create_failed
	mov	ax,3D02h		; re-open in read-write mode
	cmp	_osmajor@,2
	je	create_noshare		; if DOS 3+, also add sharing modes
	mov	al,0A2h			; read/write, deny write, no-inherit
create_noshare:
	int	21h
	jc	create_failed
create_complete:
	ASSUME	ES:NOTHING
	mov	OverlayFD,ax		; save the returned handle
	mov	DGROUP:_doserrno@,0	; assume no error will occur
	push	ds			; will be destroyed by overlay()
	mov	cs:OriginalSS,ss	; store current stack so that we can
	mov	cs:OriginalSP,sp	; switch back after running child prog
	call	xchg_code		; preserve clobbered memory
; switch to temporary stack
	mov	ax,_psp@
	mov	es,ax			; set up addressing to PSP
	ASSUME	ES:PSP_SEG
	cli				; no interrupts while switching stacks
	mov	ss,ax			; AX is still _psp@
	mov	sp,offset PSP_SEG:StackTop
	sti				; on new stack, OK to interrupt now
; copy the program's memory to the swap device
; Note: we MUST be on the temporary stack when swapping out, so that the stack
;	will be properly preserved when we swap the code back in....
	ASSUME	DS:DGROUP,ES:PSP_SEG
	push	es
	mov	es,LoadSeg
	ASSUME	ES:NOTHING
	mov	dx,SwapParas		; number of paragraphs to write out
	call	write_mem_block
	pop	es
	ASSUME	ES:PSP_SEG
	jcxz	write_err
	call	write_other_blocks
	mov	bx,OverlayFD		; do_overlay expects BX to be OverlayFD
	cmp	swap_type,SWAP_DISK	; find out where we are swapping to
	jne	do_overlay		; we're ready unless swapping to disk
swap_out_disk:
	xor	cx,cx			; reposition back at beginning of file
	xor	dx,dx
	mov	ax,4200h		; LSEEK, method = absolute position
	int	21h			; BX is still OverlayFD
	jnc	commit_swap_file	; return error if unable to seek to BOF

write_err:
	mov	ax,EWRITE_FAULT		; indicate error "write fault"
	stc
	jmp	no_swap_out		; and return to caller

commit_swap_file:
	mov	ah,45h			; make a duplicate of our file handle
	int	21h
	jc	do_overlay		; but make sure not to close only copy
	xchg	bx,ax
	push	ax
	mov	ah,3Eh			; close duplicate fd
	int	21h
	pop	bx			; get back original fd
do_overlay:
	mov	si,bx			; save OverlayFD for call to $overlay
	lds	dx,es:OrigINT24h	; deactivate critical error handler
	ASSUME	DS:NOTHING
	mov	ax,2524h		; by pointing vector where it was before
	int	21h			; this program started
	lds	dx,es:OrigINT23h	; deactivate ^Break handler
	mov	ax,2523h
	int	21h
	mov	ah,4Ah			; set up for call to shrink memory block
	mov	bx,ShrunkSize		; ditto
	int	21h			; adjust size of memory block
	jc	return_from_swap	; don't EXEC if resize failed
	ASSUME	DS:NOTHING
	mov	ax,offset _TEXT:return_from_swap
	push	cs
	push	ax
; now push all the values needed by $overlay after the EXEC
	push	LoadSeg			; where to start reloading
	push	si			; OverlayFD for reloading from disk
	push	cs:OrigSize		; for restoring memory block
	push	cs:SwapSize		; swapped amount modulo 16K
	push	cs:SwapCount		; number of 16K chunks to reload
	; ES already contains _psp@
	push	es			; put PSP seg on stack for $overlay
	mov	bx,offset PSP_SEG:exec_block ;ES:BX -> EXEC param block in PSP
	lds	dx,cs:pname		; name of program to EXEC
	mov	ax,4B00h		; load the program to be spawned
	push	es
	mov	cx,offset relocated_swap_in_code
	push	cx
goto_overlay proc far
	ret				; jump to resident code to do the EXEC
goto_overlay endp

return_from_swap:
	jc	no_swap_out
	mov	ah,4Dh
	int	21h			; get child program's return code
no_swap_out:
	cli				; don't interrupt while switching stacks
	mov	ss,cs:OriginalSS	; switch back to user stack
	mov	sp,cs:OriginalSP
	sti				; OK to interrupt now
	pop	ds			; restore DS saved before first stack switch
	ASSUME	DS:DGROUP
	jnc	spawn_successful	; $overlay sets CF if there was an error
	push	ax
	call	__SPAWN_ERRNO		; set _doserrno and errno, return AX=-1

spawn_successful:
	push	ax			; preserve function return value
	call	xchg_code		; restore clobbered memory
	push	ds
	lds	dx,OldINT24h
	ASSUME	DS:NOTHING
	mov	ax,2524h		; restore INT 24h to reactivate
	int	21h			;   critical error handler
	pop	ds
	ASSUME	DS:DGROUP
	push	ds
	lds	dx,OldINT23h
	ASSUME	DS:NOTHING
	mov	ax,2523h		; restore INT 23h to reactivate
	int	21h			;   ^Break handler
	pop	ds
	ASSUME	DS:DGROUP
	mov	ax,SwapEMSPages		; restore EMS pointer
	mov	ems_page_num,ax
	mov	ax,SwapEMSExtraPara
	mov	ems_page_offset,ax
	call	read_other_blocks	; restore all other memory owned by prg
	cmp	swap_type,SWAP_XMS	; check where we swapped to
	jb	cleanup_disk		; type 0 = disk
	ja	cleanup_ems		; type 2 = EMS
cleanup_xms:
	mov	ah,0Ah
	mov	dx,XMSswapout_cmd.?XMSdest
	call	XMSentry		; free XMS memory block
	jmp short spawn_complete

cleanup_ems:
	mov	dx,OverlayFD
	mov	ah,45h
	int	67h			; free EMS handle and memory
	jmp short spawn_complete

cleanup_disk:
	mov	bx,OverlayFD
	mov	ah,3Eh			; close temporary file
	int	21h
	mov	dx,offset DGROUP:OverlayFile
	mov	ah,41h			; and then delete it
	int	21h

spawn_complete:
	mov	ax,5803h		; restore UMB link state to original
	mov	bl,UMBs_linked		;   value
	xor	bh,bh
	int	21h
spawn_done:
	call	restore_PSP
	pop	ax			; get back function's return value
	pop	si
	pop	ds
	pop	di
	pop	es
	pop	bp
	ret	2+(3*dPtrSize)
EndProc@	__SPAWNV, __PASCAL__

CSegEnd@

DSeg@
copyright_notice db "SPAWNO v3.5 (c) Copyright 1990, 1993 Ralf Brown",0
DSegEnd@

	END
