/************************************************************************/
/*									*/
/*	DGTAROM - Dump/Patch Kodak DC2xx DigitaOS ROM image		*/
/*	Copyright (c) 1999,2000 by Ralf Brown				*/
/*	Version 1.04							*/
/*	Last Edit: 09jul00						*/
/*									*/
/************************************************************************/

#include <ctype.h>
#include <fcntl.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

typedef unsigned char LONGbuffer[4] ;

#define loadlong(x) \
   ((((unsigned long)x[0]) << 24) | (((unsigned long)x[1]) << 16) |  \
    ((unsigned long)(x[2]) << 8) | ((unsigned long)x[3]))

#define storelong(x,loc) \
   loc[0] = (unsigned char)(x >> 24) ; \
   loc[1] = (unsigned char)(x >> 16) ; \
   loc[2] = (unsigned char)(x >> 8) ; \
   loc[3] = (unsigned char)x ;

#ifndef FALSE
#  define FALSE 0
#endif
#ifndef TRUE
#  define TRUE (!FALSE)
#endif

#define DOSFNAME_LEN	16
#define STRING_LEN	32

//----------------------------------------------------------------------

static long checksum_offset = 0 ;
static long version_marker = 0 ;
static int is_code = FALSE ;
static int verbose = FALSE ;

//----------------------------------------------------------------------

static void banner()
{
   printf("Kodak DigitaOS ROM Image Dumper v1.04\tCopyright 2000 Ralf Brown\n"
	  "\n");
   return ;
}

//----------------------------------------------------------------------

static void usage(const char *argv0)
{
   fprintf(stderr,
	   "Usage:\t%s [options] imagefile\n"
	   "\t%s -p[t] imagefile patchspec [patchspec ...]\n"
	   "Options:\n"
	   "\t-i\tdump image tags instead of camera variables\n"
	   "\t-oOFS\tassume table is OFS bytes into file instead of searching\n"
	   "\t-t\ttest ROM image (-pt tests after patching)\n"
	   "\t-vN.N.N\tset ROM version marker to N.N.N to allow downgrades\n"
           "\n"
	   "Patchspec is in the form\n"
	   "\tname=min,max,default\n"
	   "where any of 'min', 'max', and 'default' may be left unchanged\n"
	   "by giving an empty string (i.e. shut=,,1000000 would set the default\n"
	   "shutter time to one second while leaving min and max untouched)\n",
	   argv0,argv0) ;
   exit(1) ;
}
	  
//----------------------------------------------------------------------

static int extract_version_marker(const char *option, const char *argv0)
{
   if (option && *option)
      {
      int major = 0, minor = 0, patch = 0, update = 0 ;
      char *end = 0 ;
      major = (int)strtol(option,&end,10) ;
      if (end && end != option && major > 0)
	 {
	 if (*end == '.')
	    {
	    option = end+1 ;
	    end = 0 ;
	    minor = (int)strtol(option,&end,10) ;
	    if (end && end != option && minor >= 0)
	       {
	       if (*end == '.')
		  {
		  option = end + 1 ;
		  end = 0 ;
		  patch = (int)strtol(option,&end,10) ;
		  if (end && end != option && patch >= 0)
		     {
		     if (*end == '.')
			{
			option = end + 1 ;
			end = 0 ;
			update = (int)strtol(option,&end,10) ;
			if (update < 0)
			   {
			   printf("--- update (fourth number) must be non-negative ---\n") ;
			   return FALSE ;
			   }
			}
		     }
		  else if (patch < 0)
		     {
		     printf("--- patch (third number) must be non-negative ---\n") ;
		     return FALSE ;
		     }
		  }
	       }
	    else if (minor < 0)
	       {
	       printf("--- minor version must be non-negative ---\n") ;
	       return FALSE ;
	       }
	    }
	 version_marker = (((long)(major & 0xFF)) << 24) |
			  (((long)(minor & 0xFF)) << 16) |
			  ((patch & 0xFF) << 8) | (update & 0xFF) ;
         return TRUE ;
	 }
      else if (major <= 0)
	 {
	 printf("--- major version must be at least 1 ---\n") ;
	 return FALSE ;
	 }
      }
   // if we get to this point, there was a problem with the version marker
   usage(argv0) ;
   return FALSE ;
}

//----------------------------------------------------------------------

static void pad_string(char string[32])
{
   unsigned len = strlen(string) ;
   while (len < STRING_LEN-1)
      string[len++] = ' ' ;
   string[len] = '\0' ;
   return ;
}

//----------------------------------------------------------------------

static void dump_var_type(unsigned long type, int add_CR)
{
   switch (type)
      {
      case 1:	printf("unsigned int (u)") ;         break ;
      case 2:	printf("signed int (i)") ;           break ;
      case 3:	printf("fixed-point (f)") ;          break ;
      case 5:	printf("boolean (b)") ;              break ;
      case 6:	printf("PName (4-char param name)"); break ;
      case 7:	printf("DOS filename (n)") ;         break ;
      case 8:	printf("string (s)") ;               break ;
      default:	printf("unknown (%lu)",type) ;       break ;
      }
   if (add_CR)
      fputc('\n',stdout) ;
   return ;
}

//----------------------------------------------------------------------

static void dump_name_value_list(FILE *fp, unsigned long numbytes)
{
   LONGbuffer num_values, def_value ;
   unsigned long count, defval ;
   (void)fread(num_values,1,sizeof(num_values),fp) ;
   (void)fread(def_value,1,sizeof(def_value),fp) ;
   count = loadlong(num_values) ;
   defval = loadlong(def_value) ;
   if (numbytes != 2*sizeof(num_values) + count * 36)
      {
      printf("\terror: should have had %lu bytes of data, found %lu\n",
	     numbytes,2*sizeof(num_values) + count * 36) ;
      return ;
      }
   for ( ; count > 0 ; count--)
      {
      char name[STRING_LEN] ;
      LONGbuffer enum_ID ;
      unsigned long count ;

      (void)fread(name,1,sizeof(name),fp) ;
      (void)fread(enum_ID,1,sizeof(enum_ID),fp) ;
      printf("\t%s\t%lu\t%.32s\n",
	     (loadlong(enum_ID)==defval)?"  (def)":"",loadlong(enum_ID),name) ;
      }
   return ;
}

//----------------------------------------------------------------------

static void dump_min_max(FILE *fp, unsigned long numbytes,unsigned long type)
{
   LONGbuffer def_value, min_value, max_value ;

   if (numbytes != 3*sizeof(LONGbuffer))
      {
      printf("\terror: should have had 12 bytes of data, found %lu\n",numbytes);
      return ;
      }
   (void)fread(min_value,1,sizeof(min_value),fp) ;
   (void)fread(max_value,1,sizeof(max_value),fp) ;
   (void)fread(def_value,1,sizeof(def_value),fp) ;
   switch (type)
      {
      case 1:				/* unsigned int */
      case 5:				/* boolean */
	 printf("\tmin = %lu, max = %lu, default = %lu\n",
		loadlong(min_value),loadlong(max_value),loadlong(def_value)) ;
	 break ;
      case 2:				/* signed int */
	 printf("\tmin = %ld, max = %ld, default = %ld\n",
		loadlong(min_value),loadlong(max_value),loadlong(def_value)) ;
	 break ;
      case 3:				/* fixed-point */
	 printf("\tmin = %g, max = %g, default = %g\n",
		loadlong(min_value)/65536.0,loadlong(max_value)/65536.0,
		loadlong(def_value)/65536.0) ;
	 break ;
      default:
	 printf("\tinvalid type for range variable (%lu)\n",type) ;
	 break ;
      }
   return ;
}

//----------------------------------------------------------------------

static int is_version_var(LONGbuffer var)
{
   if (memcmp(var,"iprn",4) == 0) return 1 ;
   if (memcmp(var,"ucrv",4) == 0) return 1 ;
   if (memcmp(var,"carv",4) == 0) return 1 ;
   if (memcmp(var,"ccsv",4) == 0) return 1 ;
   if (memcmp(var,"fwrv",4) == 0) return 1 ;
   if (memcmp(var,"hwrv",4) == 0) return 1 ;
   if (memcmp(var,"oerv",4) == 0) return 1 ;
   if (memcmp(var,"iirv",4) == 0) return 1 ;
   return 0 ;
}

//----------------------------------------------------------------------

static void dump_simple_var(FILE *fp, unsigned long numbytes, LONGbuffer var)
{
   LONGbuffer var_type ;
   LONGbuffer int_value ;
   char DOSname[DOSFNAME_LEN] ;
   char string[STRING_LEN] ;
   unsigned long type ;

   (void)fread(var_type,1,sizeof(var_type),fp) ;
   type = loadlong(var_type) ;
   if ((type == 7 && numbytes < sizeof(DOSname) + sizeof(var_type)) ||
       (type == 8 && numbytes < sizeof(string) + sizeof(var_type)) ||
       (type < 7 && numbytes != sizeof(int_value) + sizeof(var_type)))
      {
      printf("wrong length (%lu) stored for '%4.4s'\n",numbytes,var) ;
      return ;
      }
#if 0
   /* this should ALWAYS be the same as type listed in header for variable */
   fputc('\t',stdout) ;
   dump_var_type(type,TRUE) ;
#endif /* 0 */
   switch(type)
      {
      case 1:				/* unsigned int */
	 (void)fread(int_value,1,sizeof(int_value),fp) ;
	 if (is_version_var(var))
	    printf("\tvalue = %lu (%d.%d.%d.%d)\n",loadlong(int_value),
		   int_value[0],int_value[1],
		   int_value[2],int_value[3]) ;
	 else
	    printf("\tvalue = %lu\n",loadlong(int_value)) ;
	 break ;
      case 2:				/* signed int */
	 (void)fread(int_value,1,sizeof(int_value),fp) ;
	 printf("\tvalue = %ld\n",loadlong(int_value)) ;
	 break ;
      case 3:
	 (void)fread(int_value,1,sizeof(int_value),fp) ;
	 printf("\tvalue = %g\n",loadlong(int_value)/65536.0) ;
	 break ;
      case 7:
	 (void)fread(DOSname,1,sizeof(DOSname),fp) ;
	 printf("\tvalue = '%.14s'\n",DOSname) ;
	 break ;
      case 8:
	 (void)fread(string,1,sizeof(string),fp) ;
	 printf("\tvalue = '%.32s'\n",string) ;
	 break ;
      default:
	 // can't print unknown value!
	 break ;
      }
   return ;
}

//----------------------------------------------------------------------

static int dump_var(FILE *fp)
{
   LONGbuffer tag, type, flags, var_type, extra_bytes ;
   char name[STRING_LEN] ;
   char tmpstring[STRING_LEN] ;
   unsigned long numbytes ;
   unsigned long flagbits ;
   if (fread(tag,1,sizeof(tag),fp) < sizeof(tag) ||
       fread(type,1,sizeof(type),fp) < sizeof(type) ||
       fread(name,1,sizeof(name),fp) < sizeof(name) ||
       fread(flags,1,sizeof(flags),fp) < sizeof(flags) ||
       fread(var_type,1,sizeof(var_type),fp) < sizeof(var_type) ||
       fread(extra_bytes,1,sizeof(extra_bytes),fp) < sizeof(extra_bytes))
      {
      printf("--- file read error ---\n") ;
      return FALSE ;
      }
   if (!isalpha(tag[0]) || !isalpha(tag[1]))
      {
      printf("--- end of variable list ---\n") ;
      return FALSE ;
      }
   memcpy(tmpstring,tag,sizeof(tag)) ;
   tmpstring[sizeof(tag)] = '\0' ;
   pad_string(name) ;
   printf("'%s'\t%s\t ",tmpstring,name) ;
   dump_var_type(loadlong(var_type),TRUE) ;
   flagbits = loadlong(flags) ;
   printf("\tflags = %8.08lX",flagbits) ;
   switch (flagbits & 0x000F0000)
      {
      case 0x00050000:
	 printf(" (access via GetProductInfo)") ;
	 break ;
      case 0x00040000:
      case 0x000C0000:
	 printf(" (not available on this camera?)") ;
	 break ;
      case 0x000D0000:
	 printf(" (GetProductInfo; preserve when updating ROM?)") ;
	 break ;
      default:
	 /* do nothing */
	 break ;
      }
   fputc('\n',stdout) ;
   numbytes = loadlong(extra_bytes) ;
   switch (loadlong(type))
      {
      case 1:	dump_name_value_list(fp,numbytes) ;		break ;
      case 2:	dump_min_max(fp,numbytes,loadlong(var_type)) ;	break ;
      case 3:	dump_simple_var(fp,numbytes,tag) ;		break ;
      default:	printf("\tunknown type (%d)\n",loadlong(type)) ; return FALSE ;
      }
   return TRUE ;
}

//----------------------------------------------------------------------

static void search_var_table(FILE *fp, long &offset)
{
   LONGbuffer buf ;
   int count = 0 ;
   do {
      count = fread(buf,1,sizeof(buf),fp) ;
      } while (count == sizeof(buf) &&
	       (memcmp(buf,"aagc",4) != 0 && memcmp(buf,"ALBM",4) != 0)) ;
   if (count < sizeof(buf))
      {
      offset = -1 ;
      printf("unable to find variable table -- did you use the RS*.BIN file?\n") ;
      return ;
      }
   fseek(fp,-4L,SEEK_CUR) ;
   offset = ftell(fp) ;
   printf("found tag '%s' at offset 0x%8.08lX\n\n",
	  ((memcmp(buf,"aagc",4)==0) ? "aagc" : "ALBM"), offset) ;
   return ;
}

//----------------------------------------------------------------------

static void dump(FILE *fp, long offset, int search)
{
   int done = 0 ;
   if (search)
      {
      search_var_table(fp,offset) ;
      if (offset == -1)
	 return ;
      }
   else
      fseek(fp,offset,SEEK_SET) ;
   do {
      done = !dump_var(fp) ;
      } while (!done) ;
   return ;
}

//----------------------------------------------------------------------

static int dump_itag(FILE *fp)
{
   LONGbuffer tag, type, flags, size, offset ;
   char tmpstring[5] ;
   unsigned long flagbits ;
   if (fread(tag,1,sizeof(tag),fp) < sizeof(tag) ||
       fread(type,1,sizeof(type),fp) < sizeof(type) ||
       fread(flags,1,sizeof(flags),fp) < sizeof(flags) ||
       fread(size,1,sizeof(size),fp) < sizeof(size) ||
       fread(offset,1,sizeof(offset),fp) < sizeof(offset))
      {
      printf("--- file read error ---\n") ;
      return FALSE ;
      }
   if (!isalpha(tag[0]) || !isalpha(tag[1]))
      {
      printf("--- end of tag list ---\n") ;
      return FALSE ;
      }
   memcpy(tmpstring,tag,sizeof(tag)) ;
   tmpstring[sizeof(tag)] = '\0' ;
   flagbits = loadlong(flags) ;
   printf("%s\t%3lu\t0x%4.04lX\t0x%8.08lX\t",tmpstring,loadlong(size),
	  loadlong(offset),flagbits) ;
   dump_var_type(loadlong(type),TRUE) ;
   if (flagbits != 0)
      {
      printf("\t\t\t") ;
      if ((flagbits & 0x01000000) != 0)
	 printf("writeable ") ;
      printf("\n") ;
      }
   return TRUE ;
}

//----------------------------------------------------------------------

static void dump_itags(FILE *fp, long offset, int search)
{
   int done = 0 ;
   if (search)
      {
      LONGbuffer buf ;
      int count = 0 ;
      do {
	 count = fread(buf,1,sizeof(buf),fp) ;
	 } while (count == sizeof(buf) && memcmp(buf,"imiv",4) != 0) ;
      fseek(fp,-4L,SEEK_CUR) ;
      offset = ftell(fp) ;
      printf("found tag 'imiv' at offset 0x%8.08lX\n\n",offset) ;
      }
   else
      fseek(fp,offset,SEEK_SET) ;
   printf(" Tag\tSize\tOffset\t   Flags \tType\n") ;
   do {
      done = !dump_itag(fp) ;
      } while (!done) ;
   return ;
}

//----------------------------------------------------------------------

static unsigned long add_string_to_checksum(const char *string,
					    unsigned long checksum)
{
   int i ;
   for (i = 0 ; i < STRING_LEN ; i++)
      checksum += (unsigned char)string[i] ;
   return checksum ;
}

//----------------------------------------------------------------------

static int check_image_type(FILE *fp)
{
   LONGbuffer type ;
   long offset = ftell(fp) ;
   if (fread(type,1,sizeof(type),fp) < sizeof(type))
      {
      printf("--- error reading file type ---\n") ;
      return FALSE ;
      }
   fseek(fp,offset,SEEK_SET) ;
   return type[0] == 'C' && type[1] == 'O' && type[2] == 'D' ;
}

//----------------------------------------------------------------------

static void test_ROM_image(FILE *fp)
{
   LONGbuffer type, startoffset, endoffset, ck ;
   LONGbuffer buf ;
   LONGbuffer tag ;
   char tmpstring[5] ;
   char string[STRING_LEN] ;
   long offset, start_offset, end_offset ;
   unsigned long stored_checksum ;
   unsigned long checksum = 0 ;
   unsigned long extra_check = 0 ;
   int OK = TRUE ;
   int file_complete = FALSE ;
   if (fread(type,1,sizeof(type),fp) < sizeof(type) ||
       fread(startoffset,1,sizeof(startoffset),fp) < sizeof(startoffset) ||
       fread(endoffset,1,sizeof(endoffset),fp) < sizeof(endoffset) ||
       fread(ck,1,sizeof(ck),fp) < sizeof(ck))
      {
      printf("--- file read error (file header) ---\n") ;
      return ;
      }
   offset = 4 * sizeof(type) ;
   memcpy(tmpstring,type,sizeof(type)) ;
   tmpstring[sizeof(type)] = '\0' ;
   start_offset = loadlong(startoffset) ;
   end_offset = loadlong(endoffset) ;
   extra_check += type[0] + type[1] + type[2] + type[3] ;
   extra_check += startoffset[0] + startoffset[1] + startoffset[2] + startoffset[3] ;
   extra_check += endoffset[0] + endoffset[1] + endoffset[2] + endoffset[3] ;
   extra_check += ck[0] + ck[1] + ck[2] + ck[3] ;
   printf("File type %s, ROM data starts at 0x%8.08lX, ends at 0x%8.08lX\n",
	  tmpstring,start_offset,end_offset) ;
   stored_checksum = loadlong(ck) ;
   // skip all extra data between the above header and the actual start of the
   //	ROM data.  The next two DWORDs seem to be the start and end addresses
   //	of the data once relocated inside the camera.
   while (offset < start_offset)
      {
      if (fread(buf,1,sizeof(buf),fp) < sizeof(buf))
	 {
	 printf("--- file read error (extra data) ---\n") ;
	 return ;
	 }
      extra_check += buf[0] + buf[1] + buf[2] + buf[3] ; 
      offset += sizeof(buf) ;
      }
//   fseek(fp,start_offset,SEEK_SET) ;
   while (offset < end_offset)
      {
      unsigned long remaining = end_offset - offset ;
      if (remaining < sizeof(buf))
	 {
	 if (fread(buf,1,1,fp) < 1)
	    {
	    printf("--- file read error (main data) ---\n") ;
	    return ;
	    }
	 checksum += buf[0] ;
	 offset++ ;
	 continue ;
	 }
      else if (fread(buf,1,sizeof(buf),fp) < sizeof(buf))
	 {
	 printf("--- file read error (main data) ---\n") ;
	 return ;
	 }
      checksum += buf[0] + buf[1] + buf[2] + buf[3] ; 
      offset += sizeof(buf) ;
      }
   printf("\tstored data checksum = 0x%8.08lX, computed checksum = 0x%8.08lX\n",
	  stored_checksum,checksum) ;
   if (stored_checksum != checksum)
      OK = FALSE ;
   /* at this point, we're looking at the version-control variables, so dump */
   /*   them */
   checksum += extra_check ;
   memset(ck,0,sizeof(ck)) ;
   while (!feof(fp))
      {
      if (fread(tag,1,sizeof(tag),fp) < sizeof(tag))
	 {
	 printf("--- end of file or read error in middle of tag ---\n") ;
	 break ;
	 }
//	if (verbose)
//	   printf("tag: %4.4s\n",tag) ;
      checksum += tag[0] + tag[1] + tag[2] + tag[3] ; 
      if (memcmp(tag,"fwrv",4) == 0)
	 {
	 (void)fread(buf,1,sizeof(buf),fp) ;
	 checksum += buf[0] + buf[1] + buf[2] + buf[3] ; 
	 printf("\tfirmware version %d.%d.%d.%d\n",buf[0],buf[1],buf[2],buf[3]);
	 }
      else if (memcmp(tag,"ccsv",4) == 0)
	 {
	 (void)fread(buf,1,sizeof(buf),fp) ;
	 checksum += buf[0] + buf[1] + buf[2] + buf[3] ; 
	 printf("\tCCS Version = %d.%d.%d.%d\n",buf[0],buf[1],buf[2],buf[3]);
	 }
      else if (memcmp(tag,"INSV",4) == 0)
	 {
	 (void)fread(buf,1,sizeof(buf),fp) ;
	 checksum += buf[0] + buf[1] + buf[2] + buf[3] ; 
	 printf("\tINSV = %d.%d.%d.%d\n",buf[0],buf[1],buf[2],buf[3]);
	 }
      else if (memcmp(tag,"rgnc",4) == 0)
	 {
	 (void)fread(buf,1,sizeof(buf),fp) ;
	 checksum += buf[0] + buf[1] + buf[2] + buf[3] ; 
	 printf("\tRegion Code = %d (",loadlong(buf)) ;
	 switch (loadlong(buf))
	    {
	    case 1:	printf("US English)\n") ; break ;
	    case 2:	printf("UK English)\n") ; break ;
	    case 3:	printf("French)\n") ;     break ;
	    case 4:	printf("Italian)\n") ;    break ;
	    case 5:	printf("Spanish)\n") ;    break ;
	    case 6:	printf("German)\n") ;     break ;
	    case 7:	printf("Swedish)\n") ;    break ;
	    case 8:	printf("Japanese)\n") ;   break ;
	    case 9:	printf("Dutch)\n") ;      break ;
	    case 10:	printf("Simplified Chinese)\n") ; break ;
	    default:	printf("unknown)\n") ;    break ;
	    }
	 }
      else if (memcmp(tag,"vdid",4) == 0)
	 {
	 (void)fread(string,1,sizeof(string),fp) ;
	 checksum = add_string_to_checksum(string,checksum); 
	 printf("\tVendorID = %s\n",string) ;
	 }
      else if (memcmp(tag,"ptid",4) == 0)
	 {
	 (void)fread(string,1,sizeof(string),fp) ;
	 checksum = add_string_to_checksum(string,checksum); 
	 printf("\tProductID = %s\n",string) ;
	 }
      else if (memcmp(tag,"SNMN",4) == 0)
	 {
	 (void)fread(buf,1,sizeof(buf),fp) ;
	 checksum += buf[0] + buf[1] + buf[2] + buf[3] ; 
	 printf("\tminimum serial number = %lu\n",loadlong(buf)) ; 
	 }
      else if (memcmp(tag,"SNMX",4) == 0)
	 {
	 (void)fread(buf,1,sizeof(buf),fp) ;
	 checksum += buf[0] + buf[1] + buf[2] + buf[3] ; 
	 printf("\tmaximum serial number = %lu\n",loadlong(buf)) ; 
	 }
      else if (memcmp(tag,"DONE",4) == 0)
	 {
	 (void)fread(ck,1,sizeof(ck),fp) ;
	 file_complete = TRUE ;
	 break ;
	 }
      }
   if (!file_complete)
      printf("*** end of file reached without a DONE tag!\n") ;
   printf("\tstored file checksum = 0x%8.08lX, computed checksum = 0x%8.08lX\n",
	  loadlong(ck),checksum) ;
   if (loadlong(ck) != checksum)
      OK = FALSE ;
   if (OK)
      printf("The image file appears to be OK\n") ;
   else
      printf("The image file has been corrupted\n") ;
   return ;
}

//----------------------------------------------------------------------

static int split_patch_arg(const char *arg, char **name, int *have_min,
			   long *min, int *have_max, long *max,
			   int *have_def, long *def)
{
   char *equalsign, *comma1, *comma2 ;
   char *end ;
   
   equalsign = strchr(arg,'=') ;
   if (!equalsign)
      return FALSE ;
   comma1 = strchr(equalsign,',') ;
   if (!comma1)
      return FALSE ;
   comma2 = strchr(comma1+1,',') ;
   if (!comma2)
      return FALSE ;
   *name = (char*)malloc(equalsign-(char*)arg+1) ;
   *equalsign++ = '\0' ;
   *comma1++ = '\0' ;
   *comma2++ = '\0' ;
   memcpy(*name,arg,equalsign-(char*)arg+1) ;
   end = 0 ;
   *min = strtol(equalsign,&end,0) ;
   *have_min = (end && end != equalsign) ;
   end = 0 ;
   *max = strtol(comma1,&end,0) ;
   *have_max = (end && end != comma1) ;
   *def = strtol(comma2,&end,0) ;
   *have_def = (end && end != comma2) ;
   return TRUE ;
}

//----------------------------------------------------------------------

static int update_var(FILE *fp,int have_value,long *value,
		      LONGbuffer curr_value)
{
#if OLD
   if (have_value)
      {
      LONGbuffer new_value ;
      int i ;
      int success ;
      storelong(*value,new_value) ;
      success = (fwrite(new_value,1,sizeof(new_value),fp) == sizeof(new_value));
      for (i = 0 ; i < sizeof(new_value) ; i++)
	 checksum_offset += ((unsigned long)new_value[i] -
			     (unsigned long)curr_value[i]) ;
      return success ;
      }
   else
      {
      fseek(fp,sizeof(curr_value),SEEK_CUR) ;
      *value = loadlong(curr_value) ;
      return TRUE ;
      }
#else
   LONGbuffer new_value ;
   int i ;
   int success ;
   if (!have_value)
      *value = loadlong(curr_value) ;
   storelong(*value,new_value) ;
   success = (fwrite(new_value,1,sizeof(new_value),fp) == sizeof(new_value));
   for (i = 0 ; i < sizeof(new_value) ; i++)
      checksum_offset += ((unsigned long)new_value[i] -
			  (unsigned long)curr_value[i]) ;
   return success ;

#endif
}

//----------------------------------------------------------------------

static int patch_variable(FILE *fp, const char *arg)
{
   char *var_name ;
   int have_min, have_max, have_def ;
   long min, max, def ;
   /* split the argument into variable name, min, max, and default */
   if (!split_patch_arg(arg,&var_name,&have_min,&min,&have_max,&max,
			&have_def,&def))
      {
      printf("malformed patch specification '%s'; should be:\n"
	     "\tname=min,max,default\n"
	     "where 'min', 'max', and/or 'default' may be empty to retain current\n"
	     "value.\n",arg) ;
      return FALSE ;
      }
   for ( ; ; )
      {
      /* read the variable header */
      LONGbuffer tag, type, flags, var_type, extra_bytes ;
      char name[STRING_LEN] ;
      if (fread(tag,1,sizeof(tag),fp) < sizeof(tag) ||
	  fread(type,1,sizeof(type),fp) < sizeof(type) ||
	  fread(name,1,sizeof(name),fp) < sizeof(name) ||
	  fread(flags,1,sizeof(flags),fp) < sizeof(flags) ||
	  fread(var_type,1,sizeof(var_type),fp) < sizeof(var_type) ||
	  fread(extra_bytes,1,sizeof(extra_bytes),fp) < sizeof(extra_bytes))
	 {
	 printf("--- file read error (variable header) ---\n") ;
	 free(var_name) ;
	 return FALSE ;
	 }
      if (!isalpha(tag[0]) || !isalpha(tag[1]))
	 {
	 printf("Capability '%s' not found\n",var_name) ;
	 break ;
	 }
      if (memcmp(var_name,tag,sizeof(tag)) == 0)
	 {
	 if (loadlong(type) == 2)  // range
	    {
	    LONGbuffer curr_min, curr_max, curr_def ;
	    long min_now, max_now, def_now, position ;
	    if (loadlong(extra_bytes) < 3*sizeof(curr_min))
	       {
	       printf("--- variable entry missing fields! ---\n") ;
	       break ;
	       }
	    position = ftell(fp) ;
            (void)fread(curr_min,1,sizeof(curr_min),fp) ;
	    (void)fread(curr_max,1,sizeof(curr_max),fp) ;
	    (void)fread(curr_def,1,sizeof(curr_def),fp) ;
	    fseek(fp,position,SEEK_SET) ;
	    min_now = loadlong(curr_min) ;
	    max_now = loadlong(curr_max) ;
	    def_now = loadlong(curr_def) ;
	    update_var(fp,have_min,&min,curr_min) ;
	    update_var(fp,have_max,&max,curr_max) ;
	    update_var(fp,have_def,&def,curr_def) ;
	    if (loadlong(var_type) == 2)
	       printf("changed '%s' from %ld...%ld (def %ld) to\n"
		      "\t\t    %ld...%ld (def %ld)\n",
		      var_name,min_now,max_now,def_now,min,max,def) ;
	    else
	       printf("changed '%s' from %lu...%lu (def %lu) to\n"
		      "\t\t    %lu...%lu (def %lu)\n",
		      var_name,min_now,max_now,def_now,min,max,def) ;
	    }
	 else if (loadlong(type) == 3) // simple value
	    {
	    LONGbuffer curr_value ;
	    long value_now ;
	    (void)fread(curr_value,1,sizeof(curr_value),fp) ;
	    fseek(fp,-(long)sizeof(curr_value),SEEK_CUR) ;
	    value_now = loadlong(curr_value) ;
	    update_var(fp,have_def,&def,curr_value) ;
	    if (loadlong(var_type) == 2)
	       printf("changed '%s' from %ld to %ld\n",var_name,value_now,def) ;
	    else
	       printf("changed '%s' from %lu to %lu\n",var_name,value_now,def) ;
	    }
	 else
	    {
	    printf("can't patch '%s' because it is neither a range nor simple value\n",
		   var_name) ;
	    }
	 break ;
	 }
      /* skip the remainder of the variable entry */
      fseek(fp,loadlong(extra_bytes),SEEK_CUR) ;
      }
   free(var_name) ;
   return TRUE ;
}

//----------------------------------------------------------------------

static void patch_version_marker(FILE *fp, long ver)
{
   // find the version marker in the ROM image's envelope
   //  start by skipping the header and data block
   LONGbuffer endoffset ;
   fseek(fp,2*sizeof(LONGbuffer),SEEK_SET) ;
   if (fread(endoffset,1,sizeof(endoffset),fp) < sizeof(endoffset))
      {
      printf("--- file read error (file header) ---\n") ;
      return ;
      }
   long end_offset = loadlong(endoffset) ;
   fseek(fp,end_offset,SEEK_SET) ;
   // now that we've reached the version-control variables, scan them for the
   //  firmware revision
   while (!feof(fp))
      {
      LONGbuffer tag ;
      if (fread(tag,1,sizeof(tag),fp) < sizeof(tag))
	 break ;
      if (memcmp(tag,"fwrv",4) == 0)
	 {
	 if ((ver & 0xFF) == 0)
	    ver++ ;
	 LONGbuffer curr_ver ;
	 fread(curr_ver,1,sizeof(curr_ver),fp) ;
	 fseek(fp,-(long)sizeof(curr_ver),SEEK_CUR) ;
	 if (!update_var(fp,TRUE,&ver,curr_ver))
	    printf("--- error updating version marker ---\n") ;
	 else
	    printf("version marker change successful\n") ;
         return ;
	 }
      else if (memcmp(tag,"ccsv",4) == 0 || memcmp(tag,"INSV",4) == 0 ||
	       memcmp(tag,"rgnc",4) == 0 || memcmp(tag,"SNMN",4) == 0 ||
	       memcpy(tag,"SNMX",4) == 0 || memcmp(tag,"DONE",4) == 0)
	 fseek(fp,sizeof(LONGbuffer),SEEK_CUR) ;
      else if (memcmp(tag,"vdid",4) == 0 || memcpy(tag,"ptid",4) == 0)
	 fseek(fp,STRING_LEN,SEEK_CUR) ;
      else
	 {
	 printf("--- unknown version-control tag -- must give up ---\n") ;
	 return ;
	 }
      }
   // if we get here, we weren't able to find the firmware revision tag, so
   //	complain
   printf("--- unable to set firmware revision marker ---\n") ;
   return ;
}

//----------------------------------------------------------------------

static void patch_image(FILE *fp, long offset, int argc, char **argv)
{
   if (argc == 0 || !argv)
      {
      if (is_code && version_marker != 0)
	 argc = 0 ;  // ignore any patchspecs
      else
	 {
	 printf("nothing to patch\n") ;
	 return ;
	 }
      }
   while (argc > 0)
      {
      (void)fseek(fp,offset,SEEK_SET) ;
      /* scan down the variable table for the next variable to be patched */
      if (!patch_variable(fp,argv[0]))
	 {
	 printf("unable to patch '%s'\n",argv[0]) ;
	 break ;
	 }
      argv++ ;
      argc-- ;
      }
   if (checksum_offset != 0 || version_marker != 0)
      {
      printf("patching file's checksums\n") ;
      /* fix the file's checksums */
      LONGbuffer chksum ;
      long checksum ;
      fseek(fp,12L,SEEK_SET) ;
      (void)fread(chksum,1,sizeof(chksum),fp) ;
      fseek(fp,-4L,SEEK_CUR) ;
      checksum = loadlong(chksum) + checksum_offset ;
      if (!update_var(fp,TRUE,&checksum,chksum))
	 {
	 printf("checksum update failed!\n") ;
	 return ;
	 }
      if (version_marker != 0)
	 patch_version_marker(fp,version_marker) ;
      // patch the outer checksum
      fseek(fp,-4L,SEEK_END) ;
      (void)fread(chksum,1,sizeof(chksum),fp) ;
      checksum = loadlong(chksum) + checksum_offset ;
      storelong(checksum,chksum) ;
      fseek(fp,-4L,SEEK_CUR) ;
      if (fwrite(chksum,1,sizeof(chksum),fp) < sizeof(chksum))
	 {
	 printf("checksum update failed!\n") ;
	 return ;
	 }
      }
   else
      printf("file checksums do not require changing\n") ;
   return ;
}

//----------------------------------------------------------------------

int main(int argc, char **argv)
{
   char *filename, *end ;
   long offset ;
   FILE *fp ;
   int dump_image_tags = FALSE ;
   int search = TRUE ;
   int test = FALSE ;
   int patch = FALSE ;
   char *argv0 = argv[0] ;
   int required_argc = 2 ;

   banner() ;
   while (argc > 1 && argv[1][0] == '-')
      {
      switch (argv[1][1])
	 {
	 case 'i':
	    dump_image_tags = TRUE ;
	    break ;
	 case 'p':
	    patch = TRUE ;
	    if (argv[1][2] == 't')
	       test = TRUE ;
	    break ;
	 case 'o':
	    search = FALSE ;
	    if (!argv[1][2])
	       usage(argv0) ;
	    offset = strtol(argv[1]+2,&end,0) ;
	    break ;
	 case 't':
	    test = TRUE ;
	    required_argc = 1 ;
	    break ;
	 case 'v':
	    patch = TRUE ;
            if (extract_version_marker(argv[1]+2,argv0))
	       required_argc = 0 ;
	    break ;
	 case 'V':
	    verbose = TRUE ;
	    break ;
	 default:
	    usage(argv0) ;
	    break ;
	 }
      argc-- ;
      argv++ ;
      }
   if (argc < required_argc)
      {
      usage(argv0) ;
      exit(1) ;
      }
   filename = argv[1] ;
   fp = fopen(filename,patch?"rb+":"rb") ;
   if (fp)
      {
      is_code = check_image_type(fp) ;
      if (patch)
	 {
	 if (search)
	    {
	    search_var_table(fp,offset) ;
	    if (offset == -1)
	       return 1 ;
	    }
	 patch_image(fp,offset,argc-2,argv+2) ;
	 if (test)
	    {
	    fseek(fp,0L,SEEK_SET) ;
	    test_ROM_image(fp) ;
	    }
	 }
      else if (test)
         test_ROM_image(fp) ;
      else if (dump_image_tags)
         dump_itags(fp,offset,search) ;
      else
	 dump(fp,offset,search) ;
      }
   else
      fprintf(stderr,"Unable to open '%s'\n",filename) ;
   return 0 ;
}

/* end of file DGTAROM.cpp */
