
structure Codegen :> CODEGEN =
   struct

      open Variable
      open Prim
      open ILAlloc
      open TextIO

      structure D = VariableDict

      fun find' env x =
          (case D.find env x of
              SOME x' => x'
            | NONE => raise Misc.TypeError)

      fun ident n =
          "var[" ^ Int.toString n ^ "]"

      fun codegenVal outs env v =
          (case v of
              Vvar x =>
                 output (outs, find' env x)
            | Vbool false =>
                 output (outs, "ntos(0)")
            | Vbool true =>
                 output (outs, "ntos(1)")
            | Vint i =>
                 if i >= 0 then
                    (
                    output (outs, "ntos(");
                    output (outs, Int.toString i);
                    output (outs, ")")
                    )
                 else
                    (
                    output (outs, "ntos(-");
                    output (outs, Int.toString (~i));
                    output (outs, ")")
                    )
            | Vchar ch =>
                 (
                 output (outs, "ntos(");
                 output (outs, Int.toString (Char.ord ch));
                 output (outs, ")")
                 ))

      fun codegenPrim outs depth env name prim vl =
          (case (prim, vl) of
              (Neg, [v]) =>
                 (
                 output (outs, name);
                 output (outs, " = ntos(-ston(");
                 codegenVal outs env v;
                 output (outs, "));\n")
                 )
            | (Plus, [v1, v2]) =>
                 (
                 output (outs, name);
                 output (outs, " = ntos(ston(");
                 codegenVal outs env v1;
                 output (outs, ") + ston(");
                 codegenVal outs env v2;
                 output (outs, "));\n")
                 )
            | (Minus, [v1, v2]) =>
                 (
                 output (outs, name);
                 output (outs, " = ntos(ston(");
                 codegenVal outs env v1;
                 output (outs, ") - ston(");
                 codegenVal outs env v2;
                 output (outs, "));\n")
                 )
            | (Times, [v1, v2]) =>
                 (
                 output (outs, name);
                 output (outs, " = ntos(ston(");
                 codegenVal outs env v1;
                 output (outs, ") * ston(");
                 codegenVal outs env v2;
                 output (outs, "));\n")
                 )
            | (Div, [v1, v2]) =>
                 (
                 output (outs, name);
                 output (outs, " = ntos(ston(");
                 codegenVal outs env v1;
                 output (outs, ") / ston(");
                 codegenVal outs env v2;
                 output (outs, "));\n")
                 )
            | (Mod, [v1, v2]) =>
                 (
                 output (outs, name);
                 output (outs, " = ntos(ston(");
                 codegenVal outs env v1;
                 output (outs, ") % ston(");
                 codegenVal outs env v2;
                 output (outs, "));\n")
                 )
            | (EqInt, [v1, v2]) =>
                 (
                 output (outs, name);
                 output (outs, " = ntos(ston(");
                 codegenVal outs env v1;
                 output (outs, ") == ston(");
                 codegenVal outs env v2;
                 output (outs, "));\n")
                 )
            | (LtInt, [v1, v2]) =>
                 (
                 output (outs, name);
                 output (outs, " = ntos(ston(");
                 codegenVal outs env v1;
                 output (outs, ") < ston(");
                 codegenVal outs env v2;
                 output (outs, "));\n")
                 )
            | (LeqInt, [v1, v2]) =>
                 (
                 output (outs, name);
                 output (outs, " = ntos(ston(");
                 codegenVal outs env v1;
                 output (outs, ") <= ston(");
                 codegenVal outs env v2;
                 output (outs, "));\n")
                 )
            | (IntToString, [v]) =>
                 (
                 output (outs, name);
                 output (outs, " = intToString(");
                 codegenVal outs env v;
                 output (outs, ", var, ");
                 output (outs, Int.toString depth);
                 output (outs, ");\n")
                 )                 
            | (EqChar, [v1, v2]) =>
                 (
                 output (outs, name);
                 output (outs, " = ntos(ston(");
                 codegenVal outs env v1;
                 output (outs, ") == ston(");
                 codegenVal outs env v2;
                 output (outs, "));\n")
                 )
            | (Concat, [v1, v2]) =>
                 (
                 output (outs, name);
                 output (outs, " = concat(&(");
                 codegenVal outs env v1;
                 output (outs, "), &(");
                 codegenVal outs env v2;
                 output (outs, "), var, ");
                 output (outs, Int.toString depth);
                 output (outs, ");\n")
                 )                 
            | (EqString, [v1, v2]) =>
                 (
                 output (outs, name);
                 output (outs, " = ntos(strcmp(stostr(");
                 codegenVal outs env v1;
                 output (outs, "), stostr(");
                 codegenVal outs env v2;
                 output (outs, ")) == 0);\n")
                 )
            | (LtString, [v1, v2]) =>
                 (
                 output (outs, name);
                 output (outs, " = ntos(strcmp(stostr(");
                 codegenVal outs env v1;
                 output (outs, "), stostr(");
                 codegenVal outs env v2;
                 output (outs, ")) < 0);\n")
                 )
            | (Print, [v]) =>
                 (
                 output (outs, name);
                 output (outs, " = print(");
                 codegenVal outs env v;
                 output (outs, ", var, ");
                 output (outs, Int.toString depth);
                 output (outs, ");\n")
                 )                 
            | _ =>
                 raise Misc.TypeError)

      fun codegenOper outs depth env oper =
          (case oper of
              Olet (x, v) =>
                 let val x' = ident depth
                     val env' = D.insert env x x'
                 in
                    (depth+1, env', 
                     (fn () =>
                            (
                            output (outs, x');
                            output (outs, " = ");
                            codegenVal outs env v;
                            output (outs, ";\n")
                            )))
                 end
            | Oalloc (x, i) =>
                 let val x' = ident depth
                     val env' = D.insert env x x'
                 in
                    (depth+1, env', 
                     (fn () =>
                            (
                            output (outs, x');
                            output (outs, " = alloc(");
                            output (outs, Int.toString i);
                            output (outs, ", 0, var, ");
                            output (outs, Int.toString depth);
                            output (outs, ");\n")
                            )))
                 end
            | Oread (x, y, i) =>
                 let val x' = ident depth
                     val env' = D.insert env x x'
                     val y' = find' env y
                 in
                    (depth+1, env', 
                     (fn () =>
                            (
                            output (outs, x');
                            output (outs, " = sub(");
                            output (outs, y');
                            output (outs, ", ");
                            output (outs, Int.toString (i+1));
                            output (outs, ");\n")
                            )))
                 end
            | Owrite (y, i, v) =>
                 let val y' = find' env y
                 in
                    (depth, env, 
                     (fn () =>
                            (
                            output (outs, "sub(");
                            output (outs, y');
                            output (outs, ", ");
                            output (outs, Int.toString (i+1));
                            output (outs, ") = ");
                            codegenVal outs env v;
                            output (outs, ";\n")
                            )))
                 end
            | Ostring (x, str) =>
                 let val x' = ident depth
                     val env' = D.insert env x x'
                 in
                    (depth+1, env',
                     (fn () =>
                            (
                            output (outs, x');
                            output (outs, " = (scalar)\"@@@@");
                            output (outs, String.toCString str);
                            output (outs, "\";\n")
                            )))
                 end
            | Oprim (x, prim, vl) =>
                 let val x' = ident depth
                     val env' = D.insert env x x'
                 in
                    (depth+1, env',
                     (fn () => codegenPrim outs depth env x' prim vl))
                 end
            | Optreq (x, y, z) =>
                 let val x' = ident depth
                     val env' = D.insert env x x'
                     val y' = find' env y
                     val z' = find' env z
                 in
                    (depth+1, env',
                     (fn () =>
                            (
                            output (outs, x');
                            output (outs, " = ntos(");
                            output (outs, y');
                            output (outs, " == ");
                            output (outs, z');
                            output (outs, ");\n")
                            )))
                 end)

      fun codegenExp outs depth env e =
          (case e of
              Edo (l, e) =>
                 let
                    fun loop depth env l =
                        (case l of
                            nil =>
                               codegenExp outs depth env e
                          | oper :: rest =>
                               let val (depth', env', write1) = codegenOper outs depth env oper
                                   val (space, write2) = loop depth' env' rest
                               in
                                  (space,
                                   (fn () =>
                                          (
                                          write1 ();
                                          write2 ()
                                          )))
                               end)
                 in
                    loop depth env l
                 end
            | Eapp (f, v) =>
                 let val f' = find' env f
                 in
                    (depth, (fn () =>
                                   (
                                   output (outs, "*argp = ");
                                   codegenVal outs env v;
                                   output (outs, ";\n*nextfn = (codeptr)");
                                   output (outs, f');
                                   output (outs, ";\nreturn;\n")
                                   )))
                 end
            | Ecase (x, l) =>
                 let 
                    val x' = find' env x

                    fun loop i l =
                        (case l of
                            nil =>
                               (0, (fn () => ()))
                          | e :: rest =>
                               let val (space1, write1) = codegenExp outs depth env e
                                   val (space2, write2) = loop (i+1) rest
                               in
                                  (Int.max (space1, space2),
                                   (fn () =>
                                          (
                                          output (outs, "case ");
                                          output (outs, Int.toString i);
                                          output (outs, ":\n");
                                          write1 ();
                                          write2 ()
                                          )))
                               end)

                    val (space, write) = loop 0 l
                 in
                    (space,
                     (fn () => 
                            (
                            output (outs, "switch (");
                            output (outs, x');
                            output (outs, " >> 1)\n{\n");
                            write ();
                            output (outs, "}\n")
                            )))
                 end
            | Eif (x, e1, e2) =>
                 let val (space1, write1) = codegenExp outs depth env e1
                     val (space2, write2) = codegenExp outs depth env e2
                     val x' = find' env x
                 in
                    (Int.max (space1, space2),
                     (fn () =>
                            (
                            output (outs, "if (");
                            output (outs, x');
                            output (outs, " >> 1)\n{\n");
                            write1 ();
                            output (outs, "}\nelse\n{\n");
                            write2 ();
                            output (outs, "}\n")
                            )))
                 end
            | Ehalt =>
                 (depth, (fn () => output (outs, "halt();\n"))))

      fun codegenFunction outs env name x e =
          let val x' = ident 0

              val (space, writeCode) = 
                 codegenExp outs 1 (D.insert env x x') e
          in
             output (outs, "void ");
             output (outs, name);
             output (outs, " (scalar *argp, codeptr *nextfn)\n{\n");
             
             output (outs, "scalar var[");
             output (outs, Int.toString space);
             output (outs, "];\nvar[0] = *argp;\n");
             
             writeCode ();

             output (outs, "}\n\n")
          end

      fun codegenProgram outs fns env p =
          (case p of
              Pbody e =>
                 codegenFunction outs env "mlmain" (newvar ()) e
            | Pfun (f, x, e, p') =>
                 let val f' = "fn" ^ Int.toString fns
                     val f'' = "(scalar)&" ^ f'
                 in
                    codegenFunction outs env f' x e;
                    codegenProgram outs (fns+1) (D.insert env f f'') p'
                 end)

      val prelude = "#include <string.h>\n#include \"heap.h\"\n#include \"runtime.h\"\n\n"

      fun codegen fname p =
          let val outs = openOut fname
          in
             output (outs, prelude);
             codegenProgram outs 0 D.empty p;
             closeOut outs
          end

   end
