
structure Allocate :> ALLOCATE =
   struct

      open Variable

      structure S = ILHoist
      structure T = ILAlloc

      fun translateExp e =
         (case e of
             S.Eapp (v1, v2) =>
                let
                   val (ops1, x1) = translateValueVariable v1
                   val (ops2, v2') = translateValue v2
                in
                   T.Edo (ops1, T.Edo (ops2, T.Eapp (x1, v2')))
                end

           | S.Eunpack (x, v, e) =>
                let
                   val (ops, v') = translateValue v
                   val e' = translateExp e
                in
                   T.Edo (ops, T.Edo ([T.Olet (x, v')], e'))
                end

           | S.Eproj (x, v, i, e) =>
                let
                   val (ops, y) = translateValueVariable v
                   val e' = translateExp e
                in
                   T.Edo (ops, T.Edo ([T.Oread (x, y, i)], e'))
                end

           | S.Ecase (v, arms) =>
                let
                   val (ops, y) = translateValueVariable v

                   val z = newvar ()

                   val arms' =
                      map
                      (fn (xi, ei) =>
                          T.Edo ([T.Oread (xi, y, 1)], translateExp ei))
                      arms
                in
                   T.Edo (ops, T.Edo ([T.Oread (z, y, 0)], T.Ecase (z, arms')))
                end

           | S.Eiftag (v1, v2, x, e1, e2) =>
                let
                   val (ops1, y1) = translateValueVariable v1
                   val (ops2, y2) = translateValueVariable v2
                   val e1' = translateExp e1
                   val e2' = translateExp e2

                   val z = newvar ()  (* v2's tag *)
                   val w = newvar ()  (* bool *)
                in
                   T.Edo (ops1,
                          T.Edo (ops2,
                                 T.Edo ([T.Oread (z, y2, 0),
                                         T.Optreq (w, y1, z)],
                                        T.Eif (w,
                                               T.Edo ([T.Oread (x, y2, 1)], e1'),
                                               e2'))))
                end

           | S.Enewtag (x, _, e) =>
                T.Edo ([T.Oalloc (x, 0)], translateExp e)

           | S.Eref (x, v, e) =>
                let
                   val (ops, v') = translateValue v
                   val e' = translateExp e
                in
                   T.Edo (ops,
                          T.Edo ([T.Oalloc (x, 1), T.Owrite (x, 0, v')],
                                 e'))
                end

           | S.Ederef (x, v, e) =>
                let
                   val (ops, y) = translateValueVariable v
                   val e' = translateExp e
                in
                   T.Edo (ops, T.Edo ([T.Oread (x, y, 0)], e'))
                end

           | S.Eassign (v1, v2, e) =>
                let
                   val (ops1, x1) = translateValueVariable v1
                   val (ops2, v2') = translateValue v2
                   val e' = translateExp e
                in
                   T.Edo (ops1,
                          T.Edo (ops2,
                                 T.Edo ([T.Owrite (x1, 0, v2')],
                                        e')))
                end

           | S.Eif (v, e1, e2) =>
                let
                   val (ops, x) = translateValueVariable v
                   val e1' = translateExp e1
                   val e2' = translateExp e2
                in
                   T.Edo (ops, T.Eif (x, e1', e2'))
                end

           | S.Elet (x, v, e) =>
                let
                   val (ops, v') = translateValue v
                   val e' = translateExp e
                in
                   T.Edo (ops, T.Edo ([T.Olet (x, v')], e'))
                end

           | S.Eprim (x, prim, vl, e) =>
                let
                    fun loop acc vl =
                        (case vl of
                            nil =>
                               T.Edo ([T.Oprim (x, prim, rev acc)],
                                      translateExp e)
                          | v :: rest =>
                               let 
                                  val (ops, v') = translateValue v
                               in
                                  T.Edo (ops,
                                         loop (v' :: acc) rest)
                               end)
                 in
                    loop [] vl
                 end

           | S.Ehalt =>
                T.Ehalt)

      and translateValue v =
         (case v of
             S.Vvar x =>
                ([], T.Vvar x)

           | S.Vpapp (v, _) =>
                translateValue v

           | S.Vpack (_, v, _) =>
                translateValue v

           | S.Vtuple vl =>
                let
                   val x = newvar ()

                   fun loop i vl =
                      (case vl of
                          nil =>
                             []
                        | vi :: rest =>
                             let
                                val (opsi, vi') = translateValue vi
                                val opsrest = loop (i+1) rest
                             in
                                opsi @ T.Owrite (x, i, vi') :: opsrest
                             end)

                   val ops = loop 0 vl
                in
                   (T.Oalloc (x, length vl) :: ops, T.Vvar x)
                end

           | S.Vinj (v, i, _) =>
                let
                   val (ops, v') = translateValue v
                   
                   val x = newvar ()
                in
                   (ops @ [T.Oalloc (x, 2),
                           T.Owrite (x, 0, T.Vint i), 
                           T.Owrite (x, 1, v')],
                    T.Vvar x)
                end

           | S.Vroll (v, _) =>
                translateValue v

           | S.Vunroll v =>
                translateValue v

           | S.Vtag (v1, v2) =>
                let
                   val (ops1, v1') = translateValue v1
                   val (ops2, v2') = translateValue v2
                   
                   val x = newvar ()
                in
                   (ops1 @ ops2 @ [T.Oalloc (x, 2),
                                   T.Owrite (x, 0, v1'),
                                   T.Owrite (x, 1, v2')],
                    T.Vvar x)
                end

            | S.Vbool b =>
                 ([], T.Vbool b)

            | S.Vint i =>
                 ([], T.Vint i)

            | S.Vchar ch =>
                 ([], T.Vchar ch)

            | S.Vstring str =>
                 let 
                    val x = newvar ()
                 in
                    ([T.Ostring (x, str)], T.Vvar x)
                 end)

      and translateValueVariable v =
         (case translateValue v of
             (ops, T.Vvar x) =>
                (ops, x)
           | _ =>
                raise Misc.TypeError)

      fun translateFunction f =
         (case f of
             S.Flam (x, _, e) =>
                (x, translateExp e)
           | S.Fplam (_, f) =>
                translateFunction f)

      fun translateProgram p =
         (case p of
             S.Pbody e =>
                T.Pbody (translateExp e)

           | S.Plet (x, f, p) =>
                let
                   val (y, e') = translateFunction f
                   val p' = translateProgram p
                in
                   T.Pfun (x, y, e', p')
                end)

      val translate = translateProgram

   end
