
structure InterpretModule :> INTERPRET_MODULE =
   struct

      open ILModule
      open ResultModule

      structure D = VariableDict

      type env = result D.dict * mresult D.dict

      structure PrimEval =
         PrimEvalFun
         (structure Param =
             struct
                open ResultModule

                val Runit = Rtuple []

                fun destRbool k =
                    (case k of
                        Rbool x => SOME x
                      | _ => NONE)
          
                fun destRint k =
                    (case k of
                        Rint x => SOME x
                      | _ => NONE)
          
                fun destRchar k =
                    (case k of
                        Rchar x => SOME x
                      | _ => NONE)
          
                fun destRstring k =
                    (case k of
                        Rstring x => SOME x
                      | _ => NONE)
             end)

      fun eval (env as (tenv, menv)) e =
         (case e of
             Tvar v =>
                (case D.find tenv v of
                    SOME r => r
                  | NONE =>
                       raise Wrong)

           | Tlam (v, _, e') =>
                Rfn (fn r => eval (D.insert tenv v r, menv) e')

           | Tapp (e1, e2) =>
                (case eval env e1 of
                    Rfn f =>
                       f (eval env e2)
                  | _ =>
                       raise Wrong)

           | Ttuple el =>
                Rtuple (map (eval env) el)

           | Tproj (e', i) =>
                (case eval env e' of
                    Rtuple rl =>
                       (List.nth (rl, i)
                        handle Subscript => raise Wrong)
                  | _ =>
                       raise Wrong)

           | Tinj (e', i, _) =>
                Rinj (eval env e', i)

           | Tcase (e', arms) =>
                (case eval env e' of
                    Rinj (r, i) =>
                       let
                          val (vi, ei) =
                             List.nth (arms, i)
                             handle Subscript => raise Wrong
                       in
                          eval (D.insert tenv vi r, menv) ei
                       end
                  | _ =>
                       raise Wrong)

           | Troll (e', _) =>
                Rroll (eval env e')

           | Tunroll e' =>
                (case eval env e' of
                    Rroll r => r
                  | _ =>
                       raise Wrong)

           | Ttag (e1, e2) =>
                (case eval env e1 of
                    Rtag tag =>
                       Rexn (tag, eval env e2)
                  | _ =>
                       raise Wrong)

           | Tiftag (e1, e2, v, e3, e4) =>
                (case (eval env e1, eval env e2) of
                    (Rtag tag, Rexn (tag', r)) =>
                       if tag = tag' then
                          eval (D.insert tenv v r, menv) e3
                       else
                          eval env e4
                  | _ =>
                       raise Wrong)
                       
           | Tnewtag _ =>
                Rtag (ref ())

           | Traise (e', _) =>
                raise (RaiseExn (eval env e'))

           | Thandle (e1, v, e2) =>
                (eval env e1
                 handle RaiseExn r =>
                    eval (D.insert tenv v r, menv) e2)

           | Tref e' =>
                Rref (ref (eval env e'))

           | Tderef e' =>
                (case eval env e' of
                    Rref (ref r) => r
                  | _ =>
                       raise Wrong)

           | Tassign (e1, e2) =>
                (case eval env e1 of
                    Rref rr =>
                       (
                       rr := eval env e2;
                       Rtuple []
                       )
                  | _ =>
                       raise Wrong)

           | Tbool b =>
                Rbool b

           | Tif (e1, e2, e3) =>
                (case eval env e1 of
                    Rbool true =>
                       eval env e2
                  | Rbool false =>
                       eval env e3
                  | _ =>
                       raise Wrong)

           | Tint i =>
                Rint i

           | Tchar ch =>
                Rchar ch

           | Tstring str =>
                Rstring str

           | Tlet (v, e1, e2) =>
                eval (D.insert tenv v (eval env e1), menv) e2

           | Tletm (v, m, e, _) =>
                eval (tenv, D.insert menv v (evalModule env m)) e

           | Tprim (prim, el) =>
                PrimEval.primeval prim (map (eval env) el)

           | Tsnd m =>
                (case evalModule env m of
                    MRval r => r
                  | _ =>
                       raise Wrong))
                       
      and evalModule (env as (tenv, menv)) m =
         (case m of
             Mvar v =>
                (case D.find menv v of
                    SOME r => r
                  | NONE =>
                       raise Wrong)

           | Mval e =>
                MRval (eval env e)

           | Mcon _ =>
                MRcon

           | Munit =>
                MRunit

           | Mpair (m1, m2) =>
                MRpair (evalModule env m1, evalModule env m2)

           | Mdpair (v, m1, m2) =>
                let
                   val r1 = evalModule env m1
                in
                   MRpair (r1, evalModule (tenv, D.insert menv v r1) m2)
                end

           | Mpi1 m =>
                (case evalModule env m of
                    MRpair (r1, r2) =>
                       r1
                  | _ =>
                       raise Wrong)

           | Mpi2 m =>
                (case evalModule env m of
                    MRpair (r1, r2) =>
                       r2
                  | _ =>
                       raise Wrong)

           | Mlam (v, _, m) =>
                MRfn (fn r => evalModule (tenv, D.insert menv v r) m)

           | Mapp (m1, m2) =>
                (case evalModule env m1 of
                    MRfn f =>
                       f (evalModule env m2)
                  | _ =>
                       raise Wrong)

           | Min (_, m) =>
                MRin (evalModule env m)

           | Mout m =>
                (case evalModule env m of
                    MRin r => r
                  | _ =>
                       raise Wrong)

           | Mlet (v, m1, m2, _) =>
                evalModule (tenv, D.insert menv v (evalModule env m1)) m2
                    
           | Mletd (v, m1, m2) =>
                evalModule (tenv, D.insert menv v (evalModule env m1)) m2

           | Mlete (v, e, m) =>
                evalModule (D.insert tenv v (eval env e), menv) m

           | Mseal (m, _) =>
                evalModule env m)

   end
