
structure InterpretHoist :> INTERPRET_HOIST =
   struct

      open ILHoist
      open ResultHoist

      structure D = VariableDict

      type env = result D.dict

      structure PrimEval =
         PrimEvalFun
         (structure Param =
             struct
                open ResultHoist

                val Runit = Rtuple []

                fun destRbool k =
                    (case k of
                        Rbool x => SOME x
                      | _ => NONE)
          
                fun destRint k =
                    (case k of
                        Rint x => SOME x
                      | _ => NONE)
          
                fun destRchar k =
                    (case k of
                        Rchar x => SOME x
                      | _ => NONE)
          
                fun destRstring k =
                    (case k of
                        Rstring x => SOME x
                      | _ => NONE)
             end)

      fun execExp env e =
         (case e of
             Eapp (v1, v2) =>
                (case evalValue env v1 of
                    Rcont f =>
                       f (evalValue env v2)
                  | _ =>
                       raise Wrong)

           | Eunpack (x, v, e) =>
                (case evalValue env v of
                    Rpack r =>
                       execExp (D.insert env x r) e
                  | _ =>
                       raise Wrong)

           | Eproj (x, v, i, e) =>
                (case evalValue env v of
                    Rtuple rl =>
                       let
                          val r =
                             List.nth (rl, i)
                             handle Subscript => raise Wrong
                       in
                          execExp (D.insert env x r) e
                       end
                  | _ =>
                       raise Wrong)

           | Ecase (v, arms) =>
                (case evalValue env v of
                    Rinj (r, i) =>
                       let
                          val (xi, ei) =
                             List.nth (arms, i)
                             handle Subscript => raise Wrong
                       in
                          execExp (D.insert env xi r) ei
                       end
                  | _ =>
                       raise Wrong)

           | Eiftag (v1, v2, x, e1, e2) =>
                (case (evalValue env v1, evalValue env v2) of
                    (Rtag tag, Rexn (tag', r)) =>
                       if tag = tag' then
                          execExp (D.insert env x r) e1
                       else
                          execExp env e2
                  | _ =>
                       raise Wrong)

           | Enewtag (x, _, e) =>
                execExp (D.insert env x (Rtag (ref ()))) e

           | Eref (x, v, e) =>
                execExp (D.insert env x (Rref (ref (evalValue env v)))) e

           | Ederef (x, v, e) =>
                (case evalValue env v of
                    Rref (ref r) =>
                       execExp (D.insert env x r) e
                  | _ =>
                       raise Wrong)

           | Eassign (v1, v2, e) =>
                (case evalValue env v1 of
                    Rref rr =>
                       (
                       rr := evalValue env v2;
                       execExp env e
                       )
                  | _ =>
                       raise Wrong)

           | Eif (v, e1, e2) =>
                (case evalValue env v of
                    Rbool true =>
                       execExp env e1
                  | Rbool false =>
                       execExp env e2
                  | _ =>
                       raise Wrong)

           | Elet (x, v, e) =>
                execExp (D.insert env x (evalValue env v)) e

           | Eprim (x, prim, vl, e) =>
                let
                   val r =
                      PrimEval.primeval prim (map (evalValue env) vl)
                in
                   execExp (D.insert env x r) e
                end

           | Ehalt => ())

      and evalValue env v =
         (case v of
             Vvar x =>
                (case D.find env x of
                    SOME r => r
                  | NONE =>
                       raise Wrong)

           | Vpapp (v, _) =>
                (case evalValue env v of
                    Rpfn f =>
                       f ()
                  | _ =>
                       raise Wrong)

           | Vpack (_, v, _) =>
                Rpack (evalValue env v)

           | Vtuple vl =>
                Rtuple (map (evalValue env) vl)

           | Vinj (v, i, _) =>
                Rinj (evalValue env v, i)

           | Vroll (v, _) =>
                Rroll (evalValue env v)

           | Vunroll v =>
                (case evalValue env v of
                    Rroll r => r
                  | _ =>
                       raise Wrong)

           | Vtag (v1, v2) =>
                (case evalValue env v1 of
                    Rtag tag =>
                       Rexn (tag, evalValue env v2)
                  | _ =>
                       raise Wrong)

           | Vbool b =>
                Rbool b

           | Vint i =>
                Rint i

           | Vchar ch =>
                Rchar ch

           | Vstring str =>
                Rstring str)

      fun evalFunction env f =
         (case f of
             Flam (x, _, e) =>
                Rcont (fn r => execExp (D.insert env x r) e)

           | Fplam (_, f) =>
                Rpfn (fn () => evalFunction env f))

      fun execProgram env p =
         (case p of
             Pbody e =>
                execExp env e

           | Plet (x, f, p) =>
                execProgram (D.insert env x (evalFunction env f)) p)

   end
