
structure InspectionValue =
   struct

      datatype value =
         Data of string * value list
       | Bool of bool
       | Int of int
       | String of string
       | Char of char
       | Unit
       | Tuple of value list
       | List of value list
       | Option of value option
       | Abs of exn

   end


functor InspectFun 
   (val tools : (exn -> (InspectionValue.value * string * bool ref) option) list)
   :>
   INSPECT
   =
   struct

      open InspectionValue
      open Susp

      exception Unfold

      fun unfold ex =
          let
             fun loop l =
                 (case l of
                     nil =>
                        raise Unfold
                   | f :: rest =>
                        (case f ex of
                            SOME x =>
                               x
                          | NONE =>
                               loop rest))
          in
             loop tools
          end



      (* The state tree *)

      datatype node = 
         S of { 
              value : value,
              info : (exn * string * bool ref) option,
              children : node susp list susp,
              parent : node option ref
              }

      val empty =
         S {
           value = Unit,
           info = NONE,
           children = delay (fn () => []),
           parent = ref NONE 
           }

      val nope = ref false


      fun simplifyV info v =
          (case v of
              Abs ex =>
                 simplifyEx ex
            | _ =>
                 (info, v))

      and simplifyEx ex =
          (let val (v', tp, suppress) = unfold ex
           in
              simplifyV (SOME (ex, tp, suppress)) v'
           end
           handle Unfold => (SOME (ex, "unknown", nope), Abs ex))

      fun mkNodeEx parent ex =
          mkNode parent (simplifyEx ex)

      and mkNodeV parent v =
          mkNode parent (simplifyV NONE v)

      and mkNode parent (info, v) =
          let 
             val r = ref NONE
                
             val node =
                S {
                  value = v,
                  info = info,
                  children = delay (fn () => mkChildren r v),
                  parent = parent
                  }
                
             val () = r := SOME node
          in
             node
          end

      and mkChildren parent v =
          (case v of
              Data (_, vl) =>
                 map (fn sv' => delay (fn () => mkNodeV parent sv')) vl
            | Bool _ =>
                 []
            | Int _ =>
                 []
            | String _ =>
                 []
            | Char _ =>
                 []
            | Unit =>
                 []
            | Tuple l =>
                 map (fn sv' => delay (fn () => mkNodeV parent sv')) l
            | List l =>
                 map (fn sv' => delay (fn () => mkNodeV parent sv')) l
            | Option opt =>
                 (case opt of
                     NONE =>
                        []
                   | SOME sv' =>
                        [delay (fn () => mkNodeV parent sv')])
            | Abs ex =>
                 [])



      (* Pretty printing expressions *)

      structure PP = PrettyPrint
      val width = 80
      val st = PP.makeStream TextIO.stdOut width
      val indent = 2

      fun parens prec f =
          if prec then
             (
             PP.print st "(";
             PP.openBox st PP.Consistent 0;
             f ();
             PP.closeBox st;
             PP.print st ")"
             )
          else
             f ()

      fun apptween f g l =
          (case l of
              nil =>
                 ()
            | x :: rest =>
                 (
                 f x;
                 app (fn x => (
                              g () : unit;
                              f x
                              )) rest
                 ))

      fun reportMain d prec (S {value=v, info, children, ...}) =
          if d = 0 orelse (case info of
                              NONE => false
                            | SOME (_, _, ref b) => b) then
             PP.print st "#"
          else
             (case v of
                 Data (dcon, _) =>
                    (case force children of
                        [] =>
                           PP.print st dcon
                      | [susp] =>
                           parens prec
                           (fn () =>
                                  (
                                  PP.openBox st PP.Consistent indent;
                                  PP.print st dcon;
                                  PP.break st 1;
                                  reportMain (d-1) true (force susp);
                                  PP.closeBox st
                                  ))
                      | _ =>
                           parens prec
                           (fn () =>
                                  (
                                  PP.openBox st PP.Consistent indent;
                                  PP.print st dcon;
                                  PP.break st 1;
                                  PP.print st "(";
                                  PP.openBox st PP.Consistent 0;

                                  apptween 
                                  (fn child => reportMain (d-1) false (force child))
                                  (fn () => (
                                            PP.print st ",";
                                            PP.break st 1
                                            ))
                                  (force children);

                                  PP.closeBox st;
                                  PP.print st ")";
                                  PP.closeBox st
                                  )))
               | Bool b =>
                    if b then
                       PP.print st "true"
                    else
                       PP.print st "false"
               | Int i =>
                    PP.print st (Int.toString i)
               | String s =>
                    (
                    PP.print st "\"";
                    PP.print st (String.toString s);
                    PP.print st "\""
                    )
               | Char ch =>
                    (
                    PP.print st "#\"";
                    PP.print st (Char.toString ch);
                    PP.print st "\""
                    )
               | Unit =>
                    PP.print st "()"
               | Tuple _ =>
                    (
                    PP.print st "(";
                    PP.openBox st PP.Consistent 0;

                    apptween 
                    (fn child => reportMain (d-1) false (force child))
                    (fn () => (
                              PP.print st ",";
                              PP.break st 1
                              ))
                    (force children);

                    PP.closeBox st;
                    PP.print st ")"
                    )
               | List l =>
                    (
                    PP.print st "[";
                    PP.openBox st PP.Consistent 0;

                    apptween 
                    (fn child => reportMain (d-1) false (force child))
                    (fn () => (
                              PP.print st ",";
                              PP.break st 1
                              ))
                    (force children);

                    PP.closeBox st;
                    PP.print st "]"
                    )
               | Option NONE =>
                    PP.print st "NONE"
               | Option (SOME _) =>
                    (case force children of
                        [susp] =>
                           parens prec
                           (fn () =>
                                  (
                                  PP.openBox st PP.Consistent indent;
                                  PP.print st "SOME";
                                  PP.break st 1;
                                  reportMain (d-1) true (force susp);
                                  PP.closeBox st
                                  ))
                      | _ =>
                           raise (Fail "Children != 1 in reportMain."))
               | Abs ex =>
                    PP.print st "-")



      (* Printing types *)

      fun printType (S {value=v, info, children, ...}) =
          (case info of
              SOME (_, tp, _) =>
                 print tp
            | NONE =>
                 (case v of
                     Data _ =>
                        print "unknown"
                   | Bool _ =>
                        print "bool"
                   | Int _ =>
                        print "int"
                   | String _ =>
                        print "string"
                   | Char _ =>
                        print "char"
                   | Unit =>
                        print "unit"
                   | Tuple _ =>
                        (case force children of
                            [] =>
                               print "{}"
                          | [node] =>
                               (
                               print "{";
                               printType (force node);
                               print "}"
                               )
                          | nodes =>
                               apptween
                               (fn node => printType (force node))
                               (fn () => print " * ")
                               nodes)
                   | List _ =>
                        (case force children of
                            [] =>
                               print "_ list"
                          | susp :: _ =>
                               (
                               print "(";
                               printType (force susp);
                               print ") list"
                               ))
                   | Option _ =>
                        (case force children of
                            [] =>
                               print "_ option"
                          | susp :: _ =>
                               (
                               print "(";
                               printType (force susp);
                               print ") option"
                               ))
                   | Abs _ =>
                        print "unknown"))



      (* UI *)

      val state : (node * int list) ref = ref (empty, [])

      val depth = ref 10

      fun printPath path =
         (
         print "^";
         app (fn n => (print "."; print (Int.toString n))) (rev path)
         )

      fun report (node, path) =
          (
          printPath path;
          print "\n";
          reportMain (!depth) false node;
          PP.newline st;
          PP.flush st;
          print ": ";
          printType node;
          print "\n"
          )

      fun load ex =
          let val node = mkNodeEx (ref NONE) ex
          in
             state := (node, []);
             report (node, [])
          end

      fun cur () =
          report (!state)

      exception Navigation

      fun width () =
         let
            val (S {children, ...}, _) = !state
         in
            length (force children)
         end

      fun path () =
         let
            val (_, l) = !state
         in
            l
         end

      fun down l =
          let
             fun loop node l =
                 (case l of
                     nil =>
                        node
                   | n :: rest =>
                        let val S {children, ...} = node

                            val node' = 
                               force (List.nth (force children, n))
                               handle Subscript => raise Navigation
                        in
                           loop node' rest
                        end)
                 
             val (oldnode, oldpath) = !state
             val node = loop oldnode l
             val path = rev l @ oldpath
          in
             state := (node, path);
             report (node, path)
          end

      fun down1 n = down [n]

      fun up n =
          let
             val () =
                if n < 0 then
                   raise Navigation
                else
                   ()

             fun loop node n =
                 if n = 0 then
                    node
                 else
                    let val S {parent, ...} = node
                    in
                       (case !parent of
                           NONE =>
                              raise Navigation
                         | SOME node' =>
                              loop node' (n-1))
                    end

             val (oldnode, oldpath) = !state
             val node = loop oldnode n
             val path = List.drop (oldpath, n)
          in
             state := (node, path);
             report (node, path)
          end

      fun top () =
          let
             fun loop (node as S {parent, ...}) =
                 (case !parent of
                     NONE =>
                        node
                   | SOME node' =>
                        loop node')

             val (oldnode, _) = !state
             val node = loop oldnode
          in
             state := (node, []);
             report (node, [])
          end

      exception Extract

      fun extract () =
          let val (S {info=info, ...}, _) = !state
          in
             (case info of
                 SOME (ex, _, _) => ex
               | NONE => raise Extract)
          end
          

      type state = node * int list

      fun saveState () = !state

      fun restoreState node =
          state := node

   end
