
structure InspectClosure :> INSPECT_CLOSURE =
   struct

      open InspectionValue
      open ILClosure

      exception TagKind of kind
      exception TagCon of con
      exception TagExp of exp
      exception TagValue of value
      exception TagPrim of Prim.prim

      val TK = Abs o TagKind
      val TC = Abs o TagCon
      val TE = Abs o TagExp
      val TV = Abs o TagValue

      fun Data0 str = Data (str, [])

      val nope = ref false  (* don't modify this *)
      val suppressCon = ref false

      fun kindTool ex =
         (case ex of
             TagKind k =>
                SOME
                ((case k of
                     Ktype =>
                        Data0 "Ktype"
                   | Ksing c =>
                        Data ("Ksing", [TC c])
                   | Kpi (k1, k2) =>
                        Data ("Kpi", [TK k1, TK k2])
                   | Ksigma (k1, k2) =>
                        Data ("Ksigma", [TK k1, TK k2])
                   | Kunit =>
                        Data0 "Kunit"),
                 "kind", nope)
           | _ =>
                NONE)

      fun conTool ex =
         (case ex of
             TagCon c =>
                SOME
                ((case c of
                     Cvar (i, jo) =>
                        Data ("Cvar", [Int i, Option (Option.map Int jo)])
                   | Clam (k, c) =>
                        Data ("Clam", [TK k, TC c])
                   | Capp (c1, c2) =>
                        Data ("Capp", [TC c1, TC c2])
                   | Cpair (c1, c2) =>
                        Data ("Cpair", [TC c1, TC c2])
                   | Cpi1 c =>
                        Data ("Cpi1", [TC c])
                   | Cpi2 c =>
                        Data ("Cpi2", [TC c])
                   | Cunit =>
                        Data0 "Cunit"
                   | Cexists (k, c) =>
                        Data ("Cexists", [TK k, TC c])
                   | Cnot c =>
                        Data ("Cnot", [TC c])
                   | Cprod cl =>
                        Data ("Cprod", [List (map TC cl)])
                   | Csum cl =>
                        Data ("Csum", [List (map TC cl)])
                   | Crec c =>
                        Data ("Crec", [TC c])
                   | Ctag c =>
                        Data ("Ctag", [TC c])
                   | Cref c =>
                        Data ("Cref", [TC c])
                   | Cexn =>
                        Data0 "Cexn"
                   | Cbool =>
                        Data0 "Cbool"
                   | Cint =>
                        Data0 "Cint"
                   | Cchar =>
                        Data0 "Cchar"
                   | Cstring =>
                        Data0 "Cstring"),
                 "con", suppressCon)
           | _ =>
                NONE)

      fun expTool ex =
         (case ex of
             TagExp e =>
                SOME
                ((case e of
                     Eapp (e1, e2) =>
                        Data ("Eapp", [TV e1, TV e2])
                   | Eunpack (v, e1, e2) =>
                        Data ("Eunpack", [Int v, TV e1, TE e2])
                   | Eproj (v, e1, i, e2) =>
                        Data ("Eproj", [Int v, TV e1, Int i, TE e2])
                   | Ecase (e, match) =>
                        Data ("Ecase", [TV e,
                                        List (map
                                                 (fn (vi, ei) =>
                                                     Tuple [Int vi, TE ei])
                                                 match)])
                   | Eiftag (e1, e2, v, e3, e4) =>
                        Data ("Eiftag", [TV e1, TV e2, Int v, TE e3, TE e4])
                   | Enewtag (v, c, e) =>
                        Data ("Enewtag", [Int v, TC c, TE e])
                   | Eref (v, e1, e2) =>
                        Data ("Eref", [Int v, TV e1, TE e2])
                   | Ederef (v, e1, e2) =>
                        Data ("Ederef", [Int v, TV e1, TE e2])
                   | Eassign (e1, e2, e3) =>
                        Data ("Eassign", [TV e1, TV e2, TE e3])
                   | Eif (e1, e2, e3) =>
                        Data ("Eif", [TV e1, TE e2, TE e3])
                   | Elet (v, e1, e2) =>
                        Data ("Elet", [Int v, TV e1, TE e2])
                   | Eprim (v, prim, el, e) =>
                        Data ("Eprim", [Int v, Abs (TagPrim prim), List (map TV el), TE e])
                   | Ehalt =>
                        Data0 "Ehalt"),
                 "exp", nope)
           | _ =>
                NONE)

      fun valueTool ex =
         (case ex of
             TagValue e =>
                SOME
                ((case e of
                     Vvar v =>
                        Data ("Vvar", [Int v])
                   | Vlam (v, c, e) =>
                        Data ("Vlam", [Int v, TC c, TE e])
                   | Vpack (c1, e, c2) =>
                        Data ("Vpack", [TC c1, TV e, TC c2])
                   | Vtuple el =>
                        Data ("Vtuple", [List (map TV el)])
                   | Vinj (e, i, c) =>
                        Data ("Vinj", [TV e, Int i, TC c])
                   | Vroll (e, c) =>
                        Data ("Vroll", [TV e, TC c])
                   | Vunroll e =>
                        Data ("Vunroll", [TV e])
                   | Vtag (e1, e2) =>
                        Data ("Vtag", [TV e1, TV e2])
                   | Vbool b =>
                        Data ("Vbool", [Bool b])
                   | Vint i =>
                        Data ("Vint", [Int i])
                   | Vchar ch =>
                        Data ("Vchar", [Char ch])
                   | Vstring str =>
                        Data ("Vstring", [String str])),
                 "value", nope)
           | _ =>
                NONE)

      val tools = [kindTool, conTool, expTool, valueTool]

      structure Inspect = InspectFun (val tools = tools)
      open Inspect

      val loadKind = Inspect.load o TagKind
      val loadCon = Inspect.load o TagCon
      val loadExp = Inspect.load o TagExp
      val loadValue = Inspect.load o TagValue

      fun extractKind () =
         (case Inspect.extract () of
             TagKind x => x
           | _ => raise Inspect.Extract)

      fun extractCon () =
         (case Inspect.extract () of
             TagCon x => x
           | _ => raise Inspect.Extract)

      fun extractExp () =
         (case Inspect.extract () of
             TagExp x => x
           | _ => raise Inspect.Extract)

      fun extractValue () =
         (case Inspect.extract () of
             TagValue x => x
           | _ => raise Inspect.Extract)

   end
