/* -----------------------------------------------------------------------
 * nl-io.c
 * 
 * IO code for NL-soar
 * Author: Scott Huffman
 * Started: 8-10-92
 * 
 * 8-12: Changed "sentence" link to "incoming-words" link.
 * 4-16-93: Changed type "symbol" to "Symbol" 
 * 94-05-06: rick: Rick's last changes, probably addition of formatter
 * 94-02-09: ghn:  Got rid of Scott's original code, adding in phonological buf
 * ------------------------------------------------------------------------ */

#include <ctype.h>
#include <strings.h>
#include "soar.h"
#include "hearing.h"
#include "nl-io.h"
#include "track-lexicon.h"
#include "dump-utt.h"
#include "dump-sit.h"

/* Initial values for phonological buffer. */
#define CAPACITY 60000	/* Capacity of buffer in total ms	*/
#define DECAY 60000	/* Time before a word decays out (ms)	*/

/* Private functions declarations for interface functions. */
char *help_on_nl[];
static bool nl_io_interface_routine (void);
char *help_on_nl_io[];

static bool set_bool_setting (int *flag);

void add_nl_commands(void) {
  add_command("nl-io", nl_io_interface_routine);
  add_help("nl-io", help_on_nl_io);
  AddLexicon();
  AddUModelDump();
  AddSModelDump();
}

void init_nl_io (void) {
  fprintf(stderr, "ModSAF interface compiled with CMU phonological buffer.\n");

  current_agent(IOC_auditory_input) = get_io_sym_constant("auditory-input");
  current_agent(IOC_first) = get_io_sym_constant("first");
  current_agent(IOC_loc_x) = get_io_sym_constant("loc-x");
  current_agent(IOC_marked) = get_io_sym_constant("marked");
  current_agent(IOC_next) = get_io_sym_constant("next");
  current_agent(IOC_shape) = get_io_sym_constant("shape");
  current_agent(IOC_speaker) = get_io_sym_constant("speaker");
  current_agent(IOC_type) = get_io_sym_constant("type");
  current_agent(IOC_value) = get_io_sym_constant("value");
  current_agent(IOC_yes) = get_io_sym_constant("yes");

  current_agent(pbuf_capacity)	= CAPACITY;
  current_agent(pbuf_decay)	= DECAY;

  current_agent(smodel_debug) = (getenv("DEBUG_SMODEL") != NULL);
  current_agent(umodel_debug) = (getenv("DEBUG_UMODEL") != NULL);
  InitLexicon();
  InitUModelDump();
  InitSModelDump();
}

static char *help_on_nl[] = {
"The following environment variables will change running parameters:",
"  DEBUG_SMODEL	display debugging information during s-model dump",
"  DEBUG_UMODEL	display debugging information during u-model dump",
"  SIT_WIN_WIDTH controls the default width of the situation window",
"  SIT_WIN_HEIGHT controls the default height of the situation window",
"  SIT_GEOMETRY controls the default geometry of the situation window",
"  UTT_WIN_WIDTH controls the default width of the utterance window",
"  UTT_WIN_HEIGHT controls the default height of the utterance window",
"  UTT_GEOMETRY controls the default geometry of the utterance window",
"",
"New commands are `nl-io', `dump-utt', `dump-sit', and `lexicon'.",
"Check out the help for these."
"",
0
};

static char *help_on_nl_io[] = {
"Command: nl-io",
"",
"Syntax: (nl-io arg*)",
"        arg  ::=  :smodel-debug {on|off}",
"        arg  ::=  :umodel-debug {on|off}",
"        arg  ::=  :smodel-size x y",
"        arg  ::=  :umodel-size x y",
"        arg  ::=  :phono-buf capacity decay",
"",
"This command controls a number of user-selectable parameters of the nl I/O",
"system.",
"The umodel (utterance model) and smodel (situation model) debug flags",
"determine whether debugging information is printed during calls to the",
"graphing routines for that model.",
"The size arguments control the size of the display window, in pixels.",
"The phono-buf argument controls the size of the phonological buffer",
"capacity and decay rate, in msec.",
0
};

bool nl_io_interface_routine (void) {
  set_lexer_allow_ids(FALSE);
  get_lexeme();			/* Move past command name. */

  if (current_agent(lexeme).type == R_PAREN_LEXEME) {
    print("\nnl-io status: ");
    if (current_agent(smodel_debug)) print(":smodel-debug on ");
    if (current_agent(umodel_debug)) print(":umodel-debug on ");
    print("\ns-model: %dx%d\tu-model: %dx%d",
	  current_agent(sit_win_wid), current_agent(sit_win_hei),
	  current_agent(utt_win_wid), current_agent(utt_win_hei));
    print("\nPhonological buffer has a capacity of %d msec and decays after %d msec.",
	  current_agent(pbuf_capacity), current_agent(pbuf_decay));
    return TRUE;
  }
  while (current_agent(lexeme).type!=R_PAREN_LEXEME) {
    if (current_agent(lexeme).type==SYM_CONSTANT_LEXEME) {
      if (!strcmp(current_agent(lexeme).string,":smodel-debug")) {
        if (set_watch_setting (&current_agent(smodel_debug))) continue;
        else goto nlioinval;
      }
      if (!strcmp(current_agent(lexeme).string,":umodel-debug")) {
        if (set_watch_setting (&current_agent(umodel_debug))) continue;
        else goto nlioinval;
      }
      if (!strcmp(current_agent(lexeme).string,":phono-buf")) {
	get_lexeme(); /* consume :phono-buf */
	if (current_agent(lexeme).type != INT_CONSTANT_LEXEME)
	  goto nlioinval;
	current_agent(pbuf_capacity) = current_agent(lexeme).int_val;
	get_lexeme(); /* consume capacity value */
	if (current_agent(lexeme).type != INT_CONSTANT_LEXEME)
	  goto nlioinval;
	current_agent(pbuf_decay) = current_agent(lexeme).int_val;
	get_lexeme(); /* consume decay value */
	continue;
      }
      if (!strcmp(current_agent(lexeme).string,":smodel-size")) {
	get_lexeme(); /* consume :smodel-size */
	if (current_agent(lexeme).type != INT_CONSTANT_LEXEME)
	  goto nlioinval;
	current_agent(sit_win_wid) = current_agent(lexeme).int_val;
	get_lexeme(); /* consume x value */
	if (current_agent(lexeme).type != INT_CONSTANT_LEXEME)
	  goto nlioinval;
	current_agent(sit_win_hei) = current_agent(lexeme).int_val;
	get_lexeme(); /* consume y value */
	current_agent(sit_geometry) = NULL;
	current_agent(sit_winch) = TRUE;
	continue;
      }
      if (!strcmp(current_agent(lexeme).string,":umodel-size")) {
	get_lexeme(); /* consume :umodel-size */
	if (current_agent(lexeme).type != INT_CONSTANT_LEXEME)
	  goto nlioinval;
	current_agent(utt_win_wid) = current_agent(lexeme).int_val;
	get_lexeme(); /* consume x value */
	if (current_agent(lexeme).type != INT_CONSTANT_LEXEME)
	  goto nlioinval;
	current_agent(utt_win_hei) = current_agent(lexeme).int_val;
	get_lexeme(); /* consume y value */
	current_agent(utt_geometry) = NULL;
	current_agent(utt_winch) = TRUE;
	continue;
      }
    }
  nlioinval:
    print ("Invalid argument to 'nl-io' command.\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  return TRUE;
}

static bool set_bool_setting (int *flag) {
  get_lexeme(); /* consume :keyword name */
  if (current_agent(lexeme).type==SYM_CONSTANT_LEXEME) {
    if (!strcmp(current_agent(lexeme).string,"on")) {
      *flag = TRUE;
      get_lexeme(); /* consume on/off flag */
      return TRUE;
    }
    if (!strcmp(current_agent(lexeme).string,"off")) {
      *flag = FALSE;
      get_lexeme(); /* consume on/off flag */
      return TRUE;
    }
  }
  print ("Expected 'on' or 'off' for new nl-io setting\n");
  print_location_of_most_recent_lexeme();
  return FALSE;
}
