/* =====================================================================
   hearing.c

   These routines provide the auditory input interface to TacAir with NL.
===================================================================== */

/* Structure of auditory input:
/* <top_state> ^io <barf>					 
   (<barf> ^auditory-input <f> &)  ; multiattribute of word features
   (<f> ^type shape
        ^value <word-name>
	^speaker <radio-name>
	^loc-x <time>
	^marked yes)
*/

#include <string.h>
#include <ctype.h>
#include <math.h>
#include <libtime.h>
#include "soar.h"
#include "nl-io.h"
#include "hearing.h"


/* Private functions to update phonological buffer. */
void pbuf_append(char *word, char *spkr, int time, int dur);
void check_decay(void);
void check_overflow(int dur);
void add_newest(PBList new);
void discard_oldest(void);
void downcase(char *str);

/************************ PUBLIC FUNCTION DEFINITIONS ***********************/

void EnterWord (int duration, char *word, int radio) {
  char radionum[15];

  sprintf(radionum, "radio-%d", radio);
  print("\n%7d Soar hears '%s' over radio-%d\n",
	time_last_simulation_clock, word, radio);
  pbuf_append(word, radionum, time_last_simulation_clock-duration, duration);
  return;

 enterwordbadarg:
  fprintf(stderr, "Bad arguments to EnterWord.\n");
  return;
}

/*********************** PRIVATE FUNCTION DEFINITIONS ***********************/
void pbuf_append(char *word, char *spkr, int time, int dur) {
  PBList new, prev;

  if (current_agent(phono_buf) != NULL) check_overflow(dur);

  if ((new = (PBList) malloc(sizeof(PBEntry))) == NULL) {
    fprintf(stderr, "Couldn't allocate pbuf entry.\n");
    exit(-1);
  }
  DEBUGMALLOC(new);

  assert(new->word = (char *)malloc(strlen(word)+1));
  DEBUGMALLOC(new->word);
  strcpy(new->word, word);
  downcase(new->word);
  assert(new->speaker = (char *)malloc(strlen(spkr)+1));
  DEBUGMALLOC(new->speaker);
  strcpy(new->speaker, spkr);
  downcase(new->speaker);
  new->wfeat = NIL;
  new->wwme = NIL;
  new->arrtime = time;
  new->dectime = time+current_agent(pbuf_decay);
  new->duration = dur;
  prev = new->next = current_agent(phono_buf);
  current_agent(phono_buf) = new;

  add_newest(new);
  if (prev && prev->wfeat && new->wfeat)
    add_input_wme(prev->wfeat, current_agent(IOC_next), new->wfeat);
  if (!prev || !prev->wfeat) {
    add_input_wme(prev->wfeat, current_agent(IOC_first), current_agent(IOC_yes));
}

void add_newest(PBList new) {
  /* Create new word feature. */
  assert(new->wfeat = get_new_io_identifier('F'));
  DEBUGGETIO(new->wfeat);
  assert(new->wwme = add_input_wme(current_agent(io_header),
				   current_agent(IOC_auditory_input),
				   new->wfeat));
  add_input_wme(new->wfeat, current_agent(IOC_marked), current_agent(IOC_yes));
  add_input_wme(new->wfeat, current_agent(IOC_type), current_agent(IOC_shape));
  GENDWI_INT(new->wfeat, current_agent(IOC_loc_x), new->arrtime);
  GENDWI_SYM(new->wfeat, current_agent(IOC_value), new->word);
  GENDWI_SYM(new->wfeat, current_agent(IOC_speaker), new->speaker);
}

void discard_oldest(void) {
  PBList temp, prev, rmtemp, rmprev;
  int age;

  if (current_agent(phono_buf) == NULL) return;

  age = (current_agent(phono_buf))->dectime;
  rmtemp = temp = current_agent(phono_buf);
  rmprev = prev = NULL;
  while (temp) {
    if (temp->dectime < age) {
      age = temp->dectime;
      rmtemp = temp;
      rmprev = prev;
    }
    prev = temp;
    temp = temp -> next;
  }
  
  if (rmprev == NULL) {
    temp = rmtemp;
    current_agent(phono_buf) = rmtemp->next;
  } else {
    temp = rmtemp;
    rmprev->next = rmtemp->next;
  }
  
  print("\n%7d The word %s decays from the phonological buffer\n", 
	time_last_simulation_clock, temp->word);
  /* If word has reached WM, remove it */
  if (temp->wfeat) {
    /* This is supposed to clean up WME's underneath.
       If it doesn't, then we need to remove temp->wwme first, I guess. */
    DEBUGRELIO(temp->wfeat);
    release_io_symbol(temp->wfeat);
  }

  DEBUGFREE(temp->word);
  free(temp->word);
  DEBUGFREE(temp->speaker);
  free(temp->speaker);
  DEBUGFREE(temp);
  free(temp);
}

void check_decay(void) {
  PBList temp, curr, prev;

  if (current_agent(phono_buf) == NULL) return;

  curr = current_agent(phono_buf);
  prev = NULL;
  while (curr) {
    if (time_last_simulation_clock > curr->dectime) {
      temp = curr;
      if (prev == NULL) {
	current_agent(phono_buf) = curr->next;
	curr = curr->next;
      } else {
	prev->next = curr->next;
	curr = curr->next;
      }
      print("\n%7d The word %s decays from the phonological buffer\n", 
	    time_last_simulation_clock, temp->word);
      /* If word has reached WM, remove it */
      if (temp->wfeat) {
	/* This is supposed to clean up WME's underneath.
	   If it doesn't, then we need to remove temp->wwme first, I guess. */
	DEBUGRELIO(temp->wfeat);
	release_io_symbol(temp->wfeat);
      }

      DEBUGFREE(temp->word);
      free(temp->word);
      DEBUGFREE(temp->speaker);
      free(temp->speaker);
      DEBUGFREE(temp);
      free(temp);
    } else {
      prev = curr;
      curr = curr -> next;
    }
  }
}

void check_overflow(int dur) {
  int timesum;
  PBList temp;

  for (temp = current_agent(phono_buf), timesum = dur;
       temp && timesum <= current_agent(pbuf_capacity);
       temp = temp->next)
    timesum += temp->duration;
    
  if (timesum > current_agent(pbuf_capacity)) {
    if (dur > current_agent(pbuf_capacity))
      fprintf(stderr, "Entry too large for phonological buffer.\n");
    else {
      discard_oldest();
      check_overflow(dur);
    }
  }
}

void downcase(char *str) {
  for (;*str != '\0'; str++) *str = isupper(*str)?(*str+32):*str;
}
