;;;
;;; All this file does is load in the other files for Prodigy. If you
;;; want to have dependencies and stuff so you can maintain the
;;; system, use "prodigy.system" in this directory. But this one will
;;; load Prodigy slightly faster if that's all you're looking for.
;;; However, note that this file assumes you have previously compiled
;;; all the files, using prodigy.system and defsystem.

;;; I have an assumption through this code that the "user" package exists.
#+CCL (rename-package "COMMON-LISP-USER" "COMMON-LISP-USER" '("CL-USER" "USER"))
#+CCL (rename-package "COMMON-LISP" "COMMON-LISP" '("CL" "LISP"))

(in-package "USER")

(unless (find-package "PRODIGY4")
  (make-package "PRODIGY4" :nicknames '("P4")))

(use-package "PRODIGY4")

;;; This line fixes a load-time importing error
(import 'p4::goal)

;;; If you run on another operating system, or apple with unix, you
;;; will need to change this.
(defvar *directory-separator*
  #+APPLE #\:
  #+UNIX  #\/
  "Character that separates directories (this should really use the
Common Lisp file system interface)")

;;; *** LOOK HERE WHEN YOU INSTALL OR COPY PRODIGY ***
;;; Change the value of this variable appropriately if copying this file,
;;; or installing 4.0
(defparameter *prodigy-root-directory* 
  #+UNIX "/afs/cs/project/prodigy/version4.0/working/"
  #+APPLE "Macintosh HD:MCL 2.0:Prodigy:"
  "Base pathname for the prodigy 4.0 system.")

(defparameter *system-directory*
  (concatenate 'string *prodigy-root-directory*
	       "system" (string *directory-separator*))
  "Directory holding all source file directories.")

(defparameter *world-path*
  (concatenate 'string *prodigy-root-directory*
	       "domains" (string *directory-separator*))
  "Default directory to search for domain definitions and problems")

;;; First, work out what system this is and whether we have the right
;;; compiled files. If we don't, loading this file will fail with an
;;; unhelpful message at this point. Oh well.
(defparameter *binary-pathname*
  (concatenate 'string *system-directory*
	       #+APPLE "fasl:"
	       #+(and CMU IBM-RT-PC) ".ibm-rt/"
	       #+(and ALLEGRO SUN3 ALLEGRO-V3.1)  ".sun3-allegro-3.1/"
	       #+(and ALLEGRO DEC3100) ".pmax-allegro/"
	       #+(and CMU PMAX) ".pmax-cmu/"
	       #+(and ALLEGRO SPARC ALLEGRO-V4.0) ".sparc-allegro-4.0/"
	       #+(and ALLEGRO SPARC ALLEGRO-V3.1) ".sparc-allegro-3.1/"	       
	       #+(and CMU SPARC) ".sparc-cmu/"
	       ))

;;; See above about unhelpful error messages.
(defparameter *binary-extension*
  #+APPLE "fasl"
  #+IBM-RT-PC "fasl"
  #+(and ALLEGRO SUN3) "fasl"
  #+(and ALLEGRO DEC3100) "decf"
  #+(and CMU PMAX) "pmaxf"
  #+SPARC "fasl")

(defparameter *prodigy-modules*
  '(("planner" "load-domain" "types" "assertions" "misc" "load-problem"
     "goal-loop" "extract-functions" "extract-static" "extract-cr"
     "my-extract-cr" "generator" "node" "instantiate-op"
     "load-graphics" "circular" "macros" "matcher-interface" "search"
     "apply" "treeprint" "matching" "rete-match" "meta-predicates"
     "interrupt" "comforts")
    ("abstraction" "gen-hiers" "utils" "hier" "augment"))
  "A list of directory - file list pairs, whose files should be loaded
in order.")

;;; Here is where the assumption that it is previously compiled comes
;;; into play.
(defun load-compiled ()
  (dolist (module *prodigy-modules*)
    ;; Note that in fact the module directory is ignored - the compiled
    ;; files all have a flat structure here although that need not be
    ;; the case.
    (let ((ender (concatenate 'string "." *binary-extension*))
          (*load-verbose* t))
      (dolist (file (cdr module))
        (load (concatenate 'string *binary-pathname* file ender))))))

;;; Here are some functions you can use to quickly load up the source
;;; and compile all functions (you must have the source loaded to do this).

(defun load-source ()
  (dolist (module *prodigy-modules*)
    (dolist (file (cdr module))
      (format t "~%Loading ~S" file)
      (load (concatenate 'string *system-directory* (car module)
			 (string *directory-separator*) file ".lisp")))))
      

(defun compile-all (&key (verbose t) (load t))
  (dolist (module *prodigy-modules*)
    (dolist (file (cdr module))
      (compile-file
       (concatenate 'string *system-directory* (car module)
			 (string *directory-separator*) file)
       :output-file
       (concatenate 'string *binary-pathname* file "." *binary-extension*)
       :verbose verbose :load load))))

(defun compile-new (&key test (verbose t) (load t))
  (dolist (module *prodigy-modules*)
    (dolist (file (cdr module))
      (let ((src-file (concatenate 'string *system-directory* (car module)
                                   (string *directory-separator*) file ".lisp"))
            (dst-file (concatenate 'string *binary-pathname* file "." *binary-extension*)))
        (if (> (file-write-date src-file) (file-write-date dst-file))
          (if test
            (format t "Compiling ~S to ~S~%" src-file dst-file)
            (compile-file src-file :output-file dst-file :verbose verbose :load load)))))))
