#-clisp
(unless (find-package "PRODIGY4")
  (make-package "PRODIGY4" :nicknames '("P4") :use '("LISP")))
(in-package "PRODIGY4")

#|
================================================
Alpine interface: Functions called from outside.
================================================

These functions are available to the user and Prodigy for using the
Alpine abstraction-generator. Prodigy usually does not call any other
of Alpine's functions directly, with a notable exception of 
"zero-precond-exp" and "zero-effects".

For the details of the data structures used by Alpine, see the
"struct.lisp" file. For the details of the PDL syntax for describing
abstraction hierarchies, see the comment in the beginning of the
"Converting the Abstraction Graph" part in the "build.lisp" file.

The following main functions are for the access by the user and
outside functions (even though some auxiliary functions may also
accessed from outside, because of the code spagettiness):

show-abstraction   print out the abstraction, in the PDL language
build-graph        build an asbtraction graph
abstraction        read-in PDL abstraction description
current-level      current abstraction level (at search time)
literal-level      abstraction level of a literal
expression-level   abstraction level of a precondition expression
highest-interesting   highest level at which a given precond exp not ignored
name-level         highest level of among literals with this name
annotate-rules     annotate rule preconditions with abstraction levels
zero-rules         annotate all rules with zero level (abstraction not used)
fast-highest-level, slow-highest-level   highest level among literals for a
                   give predicates (encoded as abs-node)
|#

(export '(gen-problem-independent gen-problem-specific gen-problem-dependent
  clear-problem-independent clear-problem-specific clear-problem-dependent
  show-abstraction build-graph abstraction abs-level))

;; The following FOUR functions are for Mike's specifications.

;; Build the problem-independent abstraction; return nil if it collapses
;; or there is no problem space, and T otherwise.
(defun gen-problem-independent ()
  (declare (special *current-problem-space*))
  (when (boundp '*current-problem-space*)
    (build-graph *current-problem-space* :for-real t :goal-exp nil)
    (setf (problem-space-property :abs-graph-for) t)
    (not (eq (problem-space-abs-graph *current-problem-space*) :collapse))))

;; Build the problem-specific abstraction; return nil if it collapses
;; or there is no problem space or current problem, and T otherwise.
(defmacro gen-problem-specific () `(gen-problem-dependent))
(defun gen-problem-dependent ()
  (declare (special *current-problem-space*))
  (when (and (boundp '*current-problem-space*) (current-problem))
    (build-graph *current-problem-space* :for-real t :goal-exp t)
    (setf (problem-space-property :abs-graph-for) (current-problem))
    (not (eq (problem-space-abs-graph *current-problem-space*) :collapse))))

;; Reset the problem-independent abstraction to nil, to avoid its
;; use with a new domain.
(defun clear-problem-independent ()
  (declare (special *current-problem-space*))
  (when (boundp '*current-problem-space*)
    (set-nil (problem-space-abs-graph *current-problem-space*))
    (set-nil (problem-space-property :abs-graph-for))
    (set-nil (problem-space-property :rel-graph-for))
    (set-nil (problem-space-property :rel-table-for))))

;; Reset the problem-specific abstraction to nil, to avoid its use
;; with a new problem.
(defmacro clear-problem-specific () `(clear-problem-dependent))
(defun clear-problem-dependent ()
  (declare (special *current-problem-space*))
  (when (boundp '*current-problem-space*)
    (unless (eq (problem-space-property :abs-graph-for) t)
      (set-nil (problem-space-abs-graph *current-problem-space*))
      (set-nil (problem-space-property :abs-graph-for)))
    (set-nil (problem-space-property :rel-tables-for))))


;; Print out the abstraction for "pspace" in the PDL language.
;; The function either generates a new abstraction graph or reads
;; the abstraction graph from the abs-graph slot of "pspace".
;; If the new abstraction is generated, it is not stored.
;;
;; pspace        the problem space
;; new           takes one of the following two values:
;;                   T  build a new abstraction graph for *pspace*
;;                   nil  read the abstraction graph from the abs-graph slot
;;                 if goal-exp is not nil, build a new abstraction regardless
;;                 of the value of "new"
;; use-primary   used only when generating new a new abstraction graph;
;;                 nil (do not use), :side (treat rest like side), 
;;                 or :prim (treat rest like primary)
;; goal-exp      goal experession for which abstraction is generated;
;;                 T or :default  if abstraction is for the current problem;
;;                 nil  if abstraction is problem-independent
;; decs          declarations of the goal expression (if goal is T, :default, 
;;                 or nil, this argument is ignored;note that the epmty list
;;                 of declarations, for an instantiatied goal, must be
;;                 explicitly specified)

(defun show-abstraction (&key (pspace *current-problem-space*) (new nil) 
                          (use-primary nil) goal-exp (decs :default))
  (declare (type problem-space pspace))
  (if (or new goal-exp)
    ;; Generate and show a new abstraction graph (without storing it).
    (let ((*pspace* pspace)
	  (graph (build-graph pspace :use-primary use-primary 
                   :goal-exp goal-exp :decs decs :for-real nil)))
      (declare (type problem-space *pspace*) (special *pspace*))
      (if (eq graph :collapse)
        (format t "~&Abstraction has collapsed to a single level.")
        (format t "~&~A" (alpine-encode-graph graph)))) ;; Print abstration.
    ;; Show an abstraction graph in the abs-graph slot of the problem space.
    (let ((*pspace* pspace)
	  (graph (problem-space-abs-graph pspace)))
      (declare (type problem-space *pspace*) (special *pspace*))
      (cond
        ((null graph) (format t "~&Abstraction is not defined."))
        ((eq graph :collapse) 
          (format t "~&Abstraction has collapsed to a single level."))
	((and (abs-graph-p graph) (singleton-p (abs-graph-components graph)))
	  (format t "~&Warning: Abstraction has only one non-static level."))
	((abs-graph-p graph)
	  (format t "~&~A" (alpine-encode-graph graph))) ;; Print abstraction.
	(t
	  (format t
	    "~&Warning: The abs-graph slot of the problem space contains ~
            `~A', which is not an abstraction graph." graph))))))


;; Build an abstraction graph. If the graph has more than one
;; nonstatic level, then annotate the preconditions of rules with
;; their abstraction levels.
;;
;; *pspace*      the problem space
;; name          name of abstraction graph
;; use-primary   nil (do not use), :side (treat rest like side), 
;;                 or :prim (treat rest like primary)
;; goal-exp      goal experession for which abstraction is generated;
;;                 T or :default  if abstraction is for the current problem;
;;                 nil if abstraction is problem-independent
;; decs          declarations of the goal expression (if goal is T, :default, 
;;                 or nil, this argument is ignored;note that the epmty list
;;                 of declarations, for an instantiatied goal, must be
;;                 explicitly specified)
;; for-real      takes one of the following two values:
;;		   T  graph will be stored and used in abstraction planning;
;;                    after building an abstraction, enumerate levels and
;;                    annotated the precondtions and effects of the rules
;;		   nil  graph will be shown to the user and forgotten;
;;                    do not enumerate levels nor annotate rules
;;
;; returned value: 
;; the resulting abstraction graph, if the graph has more then one level;
;; :collapse otherwise

(defun build-graph (*pspace* &key name (use-primary nil) 
                                  goal-exp (decs :default) (for-real t))
  (declare (type problem-space *pspace*) (special *pspace*))
  (let (graph)
    (if (getf (problem-space-plist *pspace*) :forall-effects-p)
      (setf graph :collapse)
      (progn
        ;; Set "use-primary" and update rel-tables.
        (if (eq use-primary :default)
          (setf use-primary (any-space-property :use-primary *pspace*)))
        (update-rel-tables :goal-exp goal-exp :decs decs 
          :pspace *pspace* :use-primary use-primary)
        ;; Build the abstraction graph.
        (setf graph (initial-graph :name name :use-primary use-primary))
        (unless (final-graph graph) ;; many levels?
          (setf graph :collapse))))
    (when for-real
      (setf (problem-space-abs-graph *pspace*) graph)
      (setf (problem-space-abs-graph-for *pspace*)
        (case goal-exp
          ((nil) t)
          ((t :default) (current-problem *pspace*))
          (otherwise nil)))
      (unless (eq graph :collapse)
        (alpine-enumerate-components graph)
        (annotate-rules graph :fast-p t)))
    graph))


;; Convert a description of an abstraction hierarchy into an 
;; abstraction graph, store it in the "abs-graph" slot of 
;; "*current-problem-space*", annotate the  preconditions 
;; of rules with their abstraction levels, and set the
;; ":use-abstraction" slot in the plist of 
;; "*current-problem-space* to T.
;; 
;; This macro is called when the domain description includes the 
;; description of an abstraction hierarchy, since the description
;; begins with the word "Abstraction".
;;
;; The description is a list of abstraction-graph components
;; (i.e. abstraction levels) and ordering constraints, with the keyword
;; :order between.
;;
;; Note that we use the slow version of the "annotate-operators"
;; function, which is necessary for a hierarchy specified by the user,
;; in case some precondition predicates do not correspond to the nodes
;; in the user's hierarchy. If the hierarchy was generated automatically, 
;; we may use the fast version of "annotate-operators".
;;
;; graph-code   PDL encoding of an abstraction hierarchy

(defmacro abstraction (&body graph-code)
  (declare (special *current-problem-space*)
           (type problem-space *current-problem-space*))
  (alpine-syntax-graph graph-code *current-problem-space*)
  `(let ((*pspace* ,*current-problem-space*))
    (declare (type problem-space *pspace*) (special *pspace*))
    (let ((graph (alpine-decode-graph ',graph-code)))
      (setf (problem-space-abs-graph *pspace*) graph)
      (annotate-rules graph :fast-p t)
      (setf (getf (problem-space-plist *pspace*) :use-abstraction) t))))

;;-------------------------------------------------------------------------

;; Annotate the preconditions of the *finish* operator (goal).

(defun annotate-finish ()
  (declare (special *current-problem-space*)
           (type problem-space *current-problem-space*))
  (let ((*pspace* *current-problem-space*))
    (declare (type problem-space *pspace*) (special *pspace*))
    (unless (problem-space-abs-graph *pspace*)
      (error "annotate-finish: Abstraction is undefined."))
    (annotate-rule (rule-name-to-rule '*finish* *pspace*)
                   (problem-space-abs-graph *pspace*))))


;; Return the annotated precondition expression of the rule,
;; or the zero-annotated expression if we do not use abstraction.

(defmacro annotated-preconds (rule)
  (declare (type rule rule))
  `(if (problem-space-property :use-abstraction)
    (or (getf (rule-plist ,rule) :annotated-preconds)
        (error "annotated-preconds: Rule `~S' does not have annotated ~
                precondition expression." (rule-name ,rule)))
    (or (getf (rule-plist ,rule) :zeroed-preconds)
        (error "annotated-preconds: Rule `~S' does not have zero-annotated ~
                precondition expression." (rule-name ,rule)))))


;; Return the annotated conditional effects of the rule,
;; or the zero-annotated effects if we do not use abstraction.

(defmacro annotated-effects (rule)
  (declare (type rule rule))
  `(if (problem-space-property :use-abstraction)
    (getf (rule-plist ,rule) :annotated-effects)
    (getf (rule-plist ,rule) :zeroed-effects)))

;;-------------------------------------------------------------------------

;; Determine the current abstraction level (during problem solving).

(defmacro current-level ()
  '(problem-space-property :abstraction-level))


;; Determine the abstraction level of a literal.
;; Return 0 if not using abstraction

(defmacro literal-level (literal)
  `(if (problem-space-property :use-abstraction)
    (let ((*pspace*  *current-problem-space*))
      (declare (type problem-space *pspace*) (special *pspace*))
      (abstraction-level ,literal (problem-space-abs-graph *pspace*)))
    0))


;; Determine the highest abstraction level that has a 
;; node (predicate) with the inidcated name. 
;; Return 0 if not using abstraction

(defmacro name-level (name)
  `(if (problem-space-property :use-abstraction)
    (let ((*pspace*  *current-problem-space*))
      (declare (type problem-space *pspace*) (special *pspace*))
      (name-highest-level ,name (problem-space-abs-graph *pspace*)))
    0))


;; Determine the abstraction level of a precondtion expression.
;;
;; expression   precondition expression to be annotated
;; decs         declarations of the varaiable types in the expression
;; negated-p    takes one of the two values:
;;                T  if the expression is inside a negation (which may happen 
;;                     when it is a subexpression of a larger expression)
;;                nil  otherwise
;; 
;; returned value: the level of the expression (0 if not using abstraction)

(defmacro expression-level (expression decs negated-p)
  `(if (problem-space-property :use-abstraction)
    (let ((*pspace* *current-problem-space*))
      (declare (type problem-space *pspace*) (special *pspace*))
      (car (annotate-precond-exp ,expression ,decs 
        (problem-space-abs-graph *pspace*) :negated-p ,negated-p)))
    0))


;; Determine the highest level of abstraction at which the given annoteted
;; precondition expression is not ignored.
;;
;; annotated-preconds   annotated precondition expression
;;
;; returned value: 
;; the highest level at which this expression is not ignored 

(defun highest-interesting (annotated-preconds)
  (case (second annotated-preconds)
    (user::~
      (highest-interesting (third annotated-preconds)))
    ((user::and user::or)
      (apply #'max (mapcar #'highest-interesting (cddr annotated-preconds))))
    ((user::forall user::exists)
      (apply #'max (mapcar #'highest-interesting 
                           (fourth annotated-preconds))))
    (otherwise  ;; it is an annotated predicate
      (car annotated-preconds))))


#|
=============================================
Run-time functions for using the abstraction.
=============================================

We call these functions when loading or solving the problem, not only
when loading the domain or building the abstraction graph.

The functions include annotating the preconditions and effects of
operators with their abstraction levels, and determining the
abstraction levels of predicates and literals.

The functions for determining the abstraction levels of literals are
the only functions used during the problems solving.
|#


;; Annotate the precondition expressions in the precondtions and
;; conditions of conditional effects of the operators and 
;; non-static inference rules. 
;;
;; Note that the annotation process is very time consuming, because
;; it requires determining abstraction levels of predicates.
;; 
;; See the comments to the "annotate-precond-exp" function for the 
;; details of annotation and explanation why the annotation is slow.
;;
;; graph       the abstraction graph
;; fast-p      takes on of the two values:
;;                 T  if we use "fast-highest-level" to determine node levels
;;                 nil  if we use "slow-highest-level"
;; operators   list of operators (by default, all operators in the domain)
;; eagers      list of eager inference rules (by default, all eager rules)
;; lazys       list of lazy inference rules (by default, all lazy rules)

(defun annotate-rules (graph &key (fast-p t)
                                  (operators :default)
                                  (eagers :default)
                                  (lazys :default))
  (declare (type abs-graph graph)
           (special *pspace*) (type problem-space *pspace*))
  ;; Set default lists of operators and inference rules.
  (if (eq operators :default)
    (setf operators (problem-space-real-ops *pspace*)))
  (if (eq eagers :default)
    (setf eagers (problem-space-eager-inference-rules *pspace*)))
  (if (eq lazys :default)
    (setf lazys (problem-space-lazy-inference-rules *pspace*)))
  ;; Annotate operators and inference rules.
  (dolist (operator operators)
    (annotate-rule operator graph :fast-p fast-p))
  (dolist (inf-rule (append eagers lazys))
    (unless (static-inference-rule-p inf-rule)
      (annotate-rule inf-rule graph :fast-p fast-p))))


;; Annotate the precondition expression and the conditions of conditional
;; effects in an operator or inference rule. Store the annotation in the 
;; "annotated-preconds" and "annotated-effects" slots of the rule's plist.
;;
;; rule     rule to be annotated
;; graph    the abstraction graph
;; fast-p   takes on of the two values:
;;            T  if we use "fast-highest-level" to determine node levels
;;            nil  if we use "slow-highest-level"

(defun annotate-rule (rule graph &key fast-p)
  (declare (type rule rule) (type abs-graph graph))
  (let* ((precond-decs (second (rule-precond-exp rule)))
         (effect-decs (append (second (rule-effects rule)) precond-decs))
         (annotated-effects nil))
    ;; Annotate preconditions.
    (setf (getf (rule-plist rule) :annotated-preconds)
          (annotate-precond-exp (third (rule-precond-exp rule)) 
                                precond-decs graph :fast-p fast-p))
    ;; Annotate conditional effects.
    (dolist (effect (rule-effects-all rule))
      (if (eq (car effect) 'user::if)
        (push (cons effect (annotate-precond-exp (second effect) 
          effect-decs graph :fast-p fast-p)) annotated-effects)))
    (if annotated-effects
      (setf (getf (rule-plist rule) :annotated-effects)
            annotated-effects))))


;; Annotate a precondition expression with abstraction levels of its
;; predicates and subexpressions. For every subexpression, we add the 
;; abstraction level of this subexpression in the beginning; for example,
;; (exists ((<peg> Peg)) (~ (or (on small <peg>) (on medium <peg>)))  -->
;; (0 exists ((<peg> Peg)) (0 ~ (0 or (0 on small <peg>) (1 on medium <peg>)))
;;
;; The annotation rules are as follows:
;;   static predicate  -- the static-level number
;;   non-static predicate -- the highest level corresponding to this predicate
;;   "~" -- the same as the level of the argument
;;   "and" with no arguments -- 
;;     -1 if not under negation; static-level if under negation
;;   "or" with no arguments --
;;     static-level is not under negation; -1 if under negation
;;   "and" with at least one argument --
;;     max of the argument levels if not under negation; min if under negation
;;   "or" with at least one argument --
;;     min of the argument levels if not under negation; max if under negation
;;   "exists", "forall" -- 
;;     the same as the level of the experession inside the quantification
;;
;; If "fast-p" is T, the function assumes that all predicates correspond
;; to nodes in the abstraction graph and call the "fast-highest-level"
;; function for determining the node levels. If "fast-p" is nil, we
;; call the "slow-highest-level" function for determining node levels.
;;
;; Note that the function is very time-consuming, since determining
;; the node levels is time consuming.
;;
;; precond-exp   precondition expression to be annotated
;; decs          declarations of the varaiable types in the expression
;; graph         the abstraction graph
;; negated-p     takes one of the two values:
;;                 T  if the expression is inside a negation (which may happen 
;;                      when it is a subexpression of a larger expression)
;;                 nil  otherwise
;; fast-p        takes on of the two values:
;;                 T  if we use "fast-highest-level" to determine node levels
;;                 nil  if we use "slow-highest-level"
;; 
;; returned value: the annotated expression

(defun annotate-precond-exp (precond-exp decs graph &key negated-p fast-p)
  (declare (type abs-graph graph)
           (special *pspace*) (type problem-space *pspace*))
  (case (car precond-exp)
    (user::~
      (let ((sub-exp (annotate-precond-exp (second precond-exp) decs graph 
                       :negated-p (not negated-p) :fast-p fast-p)))
        (list (car sub-exp) 'user::~ sub-exp)))
    ((user::and user::or)
     (if (singleton-p precond-exp) ;; no arguments?
       (let ((level (if (xor (eq (car precond-exp) 'user::or) negated-p)
                      -1
                      (length (abs-graph-components graph)))))
         (cons level precond-exp))
       (let* ((args (mapcar #'(lambda (arg)
                               (annotate-precond-exp arg decs graph 
                                 :negated-p negated-p :fast-p fast-p))
                            (cdr precond-exp)))
              (level (if (xor (eq (car precond-exp) 'user::or) negated-p)
                       (apply #'min (mapcar #'car args))
                       (apply #'max (mapcar #'car args)))))
         (cons level (cons (car precond-exp) args)))))
    ((user::exists user::forall)
      (let ((sub-exp (annotate-precond-exp (third precond-exp)
                       (append (second precond-exp) decs) graph 
                       :negated-p negated-p :fast-p fast-p)))
        (list (car sub-exp) (car precond-exp) (second precond-exp) sub-exp)))
    (otherwise  ;; it is a predicate
      (cons
        (cond
          ((static-predicate-p precond-exp *pspace*)
            (length (abs-graph-components graph)))
          (fast-p
            (fast-highest-level (predicate-to-node precond-exp decs) graph))
          (t
            (slow-highest-level (predicate-to-node precond-exp decs) graph)))
        precond-exp))))


;; Annotate the precondition expressions in the precondtions and conditions 
;; of conditional effects of the operators and non-static inference rules, 
;; assuming that all predicates are at Level 0.
;;
;; See the comments to the "annotate-precond-exp" function for the 
;; details of annotation and explanation.
;;
;; operators   list of operators (by default, all operators in the domain)
;; eagers      list of eager inference rules (by default, all eager rules)
;; lazys       list of lazy inference rules (by default, all lazy rules)

(defun zero-rules (&key (operators :default)
                        (eagers :default)
                        (lazys :default))
  (declare (special *pspace*) (type problem-space *pspace*))
  ;; Set default lists of operators and inference rules.
  (if (eq operators :default)
    (setf operators (problem-space-real-ops *pspace*)))
  (if (eq eagers :default)
    (setf eagers (problem-space-eager-inference-rules *pspace*)))
  (if (eq lazys :default)
    (setf lazys (problem-space-lazy-inference-rules *pspace*)))
  ;; Annotate operators and inference rules.
  (dolist (operator operators)
    (zero-rule operator))
  (dolist (inf-rule (append eagers lazys))
    (unless (static-inference-rule-p inf-rule)
      (zero-rule inf-rule))))


;; Annotate the precondition expression and the conditions of conditional
;; effects in an operator or inference rule, assuming that all predicates
;; are at Level 0. Store the annotation in the "zeroed-preconds" and
;; "zeroed-effects" slots of the rule's plist.
;;
;; rule     rule to be annotated

(defun zero-rule (rule)
  (declare (type rule rule))
  ;; Annotate preconditions.
  (setf (getf (rule-plist rule) :zeroed-preconds)
        (zero-precond-exp (third (rule-precond-exp rule))))
  ;; Annotate conditional effects.
  (setf (getf (rule-plist rule) :zeroed-effects)
        (zero-effects (rule-effects-all rule))))


;; Annotate conditional effects with abstraction levels, assuming
;; that all predicates are on Level 0.
;;
;; effects    list of effects (primary and side effects, without declarations)
;;
;; returned value: list of annotated conditional effects

(defun zero-effects (effects)
  (let ((zeroed-effects nil))
    (dolist (effect effects)
      (if (eq (car effect) 'user::if)
        (push (cons effect (zero-precond-exp (second effect)))
              zeroed-effects)))
    zeroed-effects))


;; Annotate a precondition expression with abstraction levels, assuming
;; that all predicates are on Level 0. For every subexpression, we add 0
;; in the beginning; for example, (or (on small <peg>) (on medium <peg>))  -->
;; (0 or (0 on small <peg>) (0 on medium <peg>)).
;;
;; precond-exp   precondition expression to be annotated
;; 
;; returned value: the annotated expression

(defun zero-precond-exp (precond-exp)
  (case (car precond-exp)
    (user::~ 
      (list 0 'user::~ (zero-precond-exp (second precond-exp))))
    ((user::and user::or)
      (cons 0 (cons (car precond-exp) 
        (mapcar #'zero-precond-exp (cdr precond-exp)))))
    ((user::exists user::forall)
      (list 0 (car precond-exp) (second precond-exp) 
        (zero-precond-exp (third precond-exp))))
    (otherwise  ;; it is a predicate
      (cons 0 precond-exp))))

;;-------------------------------------------------------------------------

;; Given a predicate, determine the highest abstraction level among
;; all possible literals that match this predicate.
;;
;; We use this function if there is a good chance that the predicate is
;; either static or corresponds to some node in the abstraction hierarchy.
;; The function first calls "sloppy-highest-level", which determines the
;; level if this condition is satisfied and returns -1 otherwise. If
;; "sloppy-highest-level" returns -1, then the function calls 
;; "slow-highest-level", which determines the level of any predicate.
;;
;; Since "sloppy-highest-level" is much faster than "slow-highest-level",
;; trying it first saves time.
;;
;; The function is quite time consuming, since even "sloppy-highest-level" 
;; is time-consuming, and "slow-highest-level" is much more so.
;;
;; node    given predicate, encoded as an abstraction node
;; graph   the abstraction graph
;;
;; returned value:
;; 1 + the highest non-static level, if the predicate is static
;; maximal level among all nodes that intersect the predicate,
;;   if the predicate intersects at least one node of the abstraction graph
;; -1, if the predicate is not static and does not intersect any node of the 
;;   abstraction graph

(defun fast-highest-level (node graph)
  (declare (type abs-node node)
           (type abs-graph graph))
  (let ((sloppy-level (sloppy-highest-level node graph)))
    (if (= sloppy-level -1)
      (slow-highest-level node graph)
      sloppy-level)))


;; Given a predicate, determine the highest abstraction level among
;; all possible literals that match this predicate.
;;
;; The predicate must either be static or correspond to some node in the 
;; abstraction graph. If neither holds, then the function does not determine
;; the highest level of the predicate and returns "-1", in which case the 
;; highest abstraction level must be computed by the "slow-highest-level" 
;; function.
;;
;; The function assumes that the links that mark subset relations between 
;; nodes of the abstraction graph are transitively closed; that is, the 
;; "sub" field of the abstraction node points to all subset nodes.
;;
;; The function is quite time-consuming, since it matches the given predicate 
;; (encoded as a node) with all nodes of the abstraction graph, and each 
;; matching is perform by the time-consuming function "same-node-p".
;;
;; node    given predicate, encoded as an abstraction node
;; graph   the abstraction graph
;;
;; returned value:
;; 1 + the highest non-static level, if the predicate is static
;; maximal level among all nodes that are subsets of the predicate, if the 
;;   predicates matches some node of the abstraction graph
;; -1, if the predicate is not static and does not match any node of the 
;;   abstraction graph 

(defun sloppy-highest-level (node graph)
  (declare (type abs-node node) 
           (type abs-graph graph)
           (special *pspace*) (type problem-space *pspace*))
  (if (member (abs-node-name node) (problem-space-static-preds *pspace*))
    (length (abs-graph-components graph))
    (let ((node-in-graph 
            (find node (abs-graph-nodes graph) :test #'same-node-p)))
      (cond
        ;; node is not in the abstraction graph
        ((null node-in-graph) -1)
        ;; node is in the graph, and it does not have subset nodes
        ((null (abs-node-sub node-in-graph))
          (node-level node-in-graph graph))
        ;; node is in the graph, and it has subset nodes
        (t
          (apply #'max 
            (mapcar #'(lambda (node) (node-level node graph))
              (cons node-in-graph (abs-node-sub node-in-graph)))))))))


;; Determine the abstraction level of a node in the graph.
;;
;; If node is not on the static level, then its level is equal to the 
;; number of the node's component. If the node is on the static level, 
;; then its level is "1 + the number of the highest non-static level".
;;
;; node    abstraction node whose level should be determined
;; graph   the abstraction graph
;;
;; returned value:  abstraction level of the node

(defun node-level (node graph)
  (declare (type abs-node node) (type abs-graph graph))
  (if (eq (abs-node-component node) :static)
     (length (abs-graph-components graph))
     (abs-component-number (abs-node-component node))))


;; Given a predicate, determine the highest abstraction level among
;; all possible literals that match this predicate.
;;
;; The function is very time-consuming, because, for every node of
;; the abstraction graph, it checks whether the predicate intersects
;; with this node. The function that performs the intersection-checking,
;; "node-intersection-p", is quite time-consuming.
;;
;; node    given predicate, encoded as an abstraction node
;; graph   the abstraction graph
;;
;; returned value:
;; 1 + the highest non-static level, if the predicate is static
;; maximal level among all nodes that intersect the predicate,
;;   if the predicate intersects at least one node of the abstraction graph
;; -1, if the predicate is not static and does not intersect 
;;   any nodes of the abstraction graph

(defun slow-highest-level (node graph)
  (declare (type abs-node node) 
           (type abs-graph graph)
           (special *pspace*) (type problem-space *pspace*))
  (cond
    ;; The predicate is static.
    ((member (abs-node-name node) (problem-space-static-preds *pspace*))
      (length (abs-graph-components graph)))
    ;; The predicate intersects some node at the static level.
    ((member node (abs-graph-static graph) :test #'node-intersection-p)
      (length (abs-graph-components graph)))
    ;; The predicate intersects some node at a non-static level.
    ;; We use the fact that the components are stored in the
    ;;   order from the highest component (level) to the lowest.
    ((dolist (component (abs-graph-components graph))
       (if (member node (abs-component-nodes component) 
                   :test #'node-intersection-p)
         (return (abs-component-number component)))))
    ;; The predicate is not static and does not intersect any node.
    (t -1)))


;; Determine the highest abstraction level that has a node (predicate) 
;; with the specified name; that is, the highest possible level of a
;; literal with this name.
;;
;; node-name    name of a node (predicate)
;; graph        the abstraction graph
;;
;; returned value:
;; 1 + the highest non-static level, if the predicate with this name is static
;; maximal level among all nodes that have this name,
;;   if at least one node of the abstraction graph has this name
;; -1, if there is no static predicate with this name and 
;;   no node of the abstraction graph has this name

(defun name-highest-level (node-name graph)
  (declare (symbol node-name) 
           (type abs-graph graph)
           (special *pspace*) (type problem-space *pspace*))
  (cond
    ;; The predicate with this name is static.
    ((member node-name (problem-space-static-preds *pspace*))
      (length (abs-graph-components graph)))
    ;; Some node at the static level has this name
    ((member node-name (abs-graph-static graph) :key #'abs-node-name)
      (length (abs-graph-components graph)))
    ;; Some node at a non-static level has this name
    ;; We use the fact that the components are stored in the
    ;;   order from the highest component (level) to the lowest.
    ((dolist (component (abs-graph-components graph))
       (if (member node-name (abs-graph-static graph) :key #'abs-node-name)
         (return (abs-component-number component)))))
    ;; The predicate is not static and does not intersect any node.
    (t -1)))

;;-------------------------------------------------------------------------

;; Determine the abstraction level of "literal" and store it in
;; the "abs-level" field of the literal (unless already stored).
;;
;; Note that, if the abstraction level of the literal is not already 
;; stored in the "abs-level" field of the literal, then the function is 
;; time-comsuming, because the literal has to be matched against all
;; nodes of the abstraction graph, and each matching uses the 
;; time-consuming function "node-instance-p".
;;
;; We assume that the abstraction nodes are topologically sorted on the 
;; subset relationship; that is, if Node1 is a subset of Node2, then 
;; Node1 is before Node2 in the list of nodes (i.e. in the "nodes"
;; field of the graph).
;;
;; literal   literal (described by the "literal" structure)
;; graph     the abstraction graph
;;
;; returned value:
;; level number, if the literal matches some node at a non-static level
;; 1 + the number of the highest non-static level, 
;;   if the literal is static or matches some static-level node
;; -1, if the literal is not static and does not match any node of the 
;;   abstraction graph 

(defun abstraction-level (literal graph)
  (declare (type literal literal) 
           (type abs-graph graph)
           (special *pspace*) (type problem-space *pspace*))
  (or
    ;; we have already computed the level
    (literal-abs-level literal)  
    ;; the literal matches some node in the abstraction graph
    (let ((node (find literal (abs-graph-nodes graph) 
                      :test #'node-instance-p)))
      (cond
        ((null node) nil)  ;; literal does not match any node
        ((eq (abs-node-component node) :static)  
            ;; literal matches some node at the static level
          (setf (literal-abs-level literal)
                (length (abs-graph-components graph))))
        (t  ;; leteral matches some node at a non-static level
          (setf (literal-abs-level literal)
                (abs-component-number (abs-node-component node))))))
    ;; the literal is static
    (if (member (literal-name literal) (problem-space-static-preds *pspace*))
      (setf (literal-abs-level literal)
            (length (abs-graph-components graph))))
    ;; the literal is not static and does match any node
    (setf (literal-abs-level literal) -1)))


;; Determine whether "literal" is an instance of "node"; that is, "literal" 
;; is a possible instantiation of the typed predicate in the "node".
;;
;; The function may be time-consuming when the literal is indeed an
;; instance of the node, since it matches every argument of the literal
;; with the corresponding argument type of the node, and the argument
;; types may be disjunctions.
;;
;; Note that the "args" field of abs-node is a list, whereas the 
;; "arguments" field of literal is a vector.
;;
;; literal   literal (described by the "literal" structure)
;; node      a node of the abstraction graph 

(defun node-instance-p (literal node)
  (declare (type literal literal) (type abs-node node))
  (and (eq (literal-name literal) (abs-node-name node)) ;; same literal name
       (let ((args (abs-node-args node)))
         (cond
           ((eq args :any) t)  ;; node may have any args
           ((integerp args)  ;; only arg number is specified for the node
              (= (length (literal-arguments literal)) args))
           (t (and (= (length (literal-arguments literal)) (length args))
                   (every #'arg-instance-p 
                     (literal-arguments literal) args)))))))


;; Determine whether the object "object" is an instance the type "arg-type".
;; "arg-type" must an object, :Top-Type, a finite simple type, or a
;;   disjunction of finite simple types.
;; We use the following rules:
;; - Object is an instance of another object is the objects are identical.
;; - Any "object" is an instance of :Top-Type; in this case, 
;;   "obj" may not be a  prodigy object (since instances of 
;;   infinite types are not objects).
;; - Object is an instance of a simple type if .. well .. it is an 
;;   object of this type.
;; - Object is an object of a type disjunction if it is an instance of
;;   one of the elements of the disjunction.
;;
;; obj        prodigy object, or an instance of an infinite type
;; arg-type   prodigy object, simple type, or a disjunction of simple types

(defun arg-instance-p (object arg-type)
  (cond 
    ((prodigy-object-p arg-type)
      (eq object arg-type))
    ((eq arg-type :Top-Type))
    ((and (prodigy-object-p object) (type-p arg-type))
      (object-type-p object arg-type))
    ((prodigy-object-p object) ;; and arg-type is a type disjunction
      (member object (cdr arg-type) :test #'object-type-p))))
