#-clisp
(unless (find-package "PRODIGY4")
  (make-package "PRODIGY4" :nicknames '("P4") :use '("LISP")))
(in-package "PRODIGY4")


#|
===========================================================
Converting the PDL description of abstraction into a graph.
===========================================================

These functions convert a PDL description of an abstraction hierarchy
into an abstraction graph.

The input of the top-level conversion function is the list that 
encodes the abstraction graph (hierarchy), without the word
"Abstraction". The description must not be a one-level hierarchy,
which is described in PDL as (Abstraction Abs-Name :collapse).

The functions do not check the syntax of the abstraction description.
The syntax-check function, "alpine-syntax-graph", must be called 
before converting the abstraction description in the abstraction graph.

The functions, however, provide some semantic checking. We check that
the abstraction contains the same node more then once, the sets of
literals corresponding nodes in different components do not intersect
(unless one of these sets is a subset of the other), and ordering
constraints always point from a higher level to a lower level.

For the details of the PDL syntax for describing abstraction hierarchies,
see the comment in the beginning of the "Converting the Abstraction Graph"
part in the "build.lisp" file.
|#


;; Convert a description of an abstraction hierarchy into an 
;; abstraction graph. The input of the function does not include
;; the word "Abstraction"; it begins with the name of the abstraction
;; graph (hierarchy). 
;;
;; The description is a list of abstraction-graph components
;; (i.e. abstraction levels) and ordering constraints, with the keyword
;; :order between.
;;
;; graph-code   PDL encoding of an abstraction hierarchy
;;
;; returned values: the abstraction graph representing the hierarchy

(defun alpine-decode-graph (graph-code)
  (unless (listp graph-code)
    (error "alpine-decode-graph: Abstraction description is not a list."))
  (let ((graph (create-empty-graph)))
    (declare (type abs-graph graph))
    ;; Decode the graph name.
    (if (symbolp (first graph-code))
      (setf (abs-graph-name graph) (pop graph-code)))
    ;; Decode the static level.
    (if (member (caar graph-code) '(user::static :static))
      (setf (abs-graph-static graph) 
            (alpine-decode-static (pop graph-code))))
    ;; Decode components (non-static abstraction levels).
    (do* ((level (pop graph-code) (pop graph-code)))
         ((or (null level) (member level '(user::order :order))))
      (push (alpine-decode-component level) (abs-graph-components graph)))
    (setf (abs-graph-components graph) 
          (nreverse (abs-graph-components graph)))
    (alpine-enumerate-components graph)
    ;; Decode ordering constraints.
    (cond 
      (graph-code  ;; ordering constraints explicitly described?
        (dolist (order graph-code)
          (alpine-decode-order order graph))
        (dolist (component (abs-graph-components graph))
          (del-duplicates (abs-component-out component))
          (del-duplicates (abs-component-in component)))
        (alpine-check-top-sort graph))
      (t  ;; otherwise, the components totally ordered
        (alpine-total-order graph)))
    ;; Create the list of abstraction nodes, mark subset relations,
    ;; and tolopolgically sort the list of nodes on subset relations.
    (setf (abs-graph-nodes graph)
      (append (abs-graph-static graph)
        (apply #'append 
               (mapcar #'abs-component-nodes (abs-graph-components graph)))))
    (alpine-subsets-and-intersections graph)
    (alpine-subset-sort graph)
    graph))


;; Convert a description of ordering constraints into edges of the 
;; abstraction graph. The ordering constraint is of the form 
;; (Level-Name > Level-Name-1 ... Level-Name-N).
;;
;; We add the components with names Level-Name-1,...,Level-Name-N to
;; the "out" list of the component with name Level-Name, and the 
;; component Level-Name to the "in" list of each of the components
;; Level-Name-1,...,Level-Name-N.
;;
;; If the same ordering constraint is specified several times,
;; the "in" and "out" lists of components will contain duplicates.
;;
;; order   description of ordering constraints
;; graph   the abstraction graph
;;
;; local variables:
;; component         the component with name Level-Name
;; child-component   consecutively takes values of the components with
;;                   names Level-Name-1,...,Level-Name-N

(defun alpine-decode-order (order graph)
  (let ((component (find (car order) 
          (abs-graph-components graph) :key #'abs-component-name))
        child-component)
    (declare (type abs-component component child-component))
    (dolist (component-name (cddr order))
      (setf child-component (find component-name 
        (abs-graph-components graph) :key #'abs-component-name))
      (push child-component (abs-component-out component))
      (push component (abs-component-in component)))))


;; Convert a description of the static abstraction level into the 
;; static level of the abstraction graph. The static level is the
;; list of nodes; the "component" fields of the nodes are set to
;; ":static".
;;
;; The static-level list of nodes include only non-static predicates 
;; that are on the static level. The user must ensure that the values
;; of these predicates do not change during abstraction problem solving.
;;
;; All static predicates are by default at the static level,
;; and we do not store them in the static-level list of nodes.
;;
;; For example, if the description of the static level is
;; (Static = adj-room adj-loc (in-room Object 'Room1)),
;; where "adj-room" and "adj-loc" are static predicates and
;; "(in-room Object 'Room1)" is a non-static predicate that
;; does not change in abstraction problems solving, then
;; the static-level list contains the abstraction node of 
;; "(in-room Object 'Room1)" and does not contain the 
;; nodes for "adj-room" and "adj-loc"
;;
;; static-level   description of the static level
;;
;; returned value: static-level list of abstraction nodes

(defun alpine-decode-static (static-level)
  (declare (special *pspace*) (type problem-space *pspace*))
  (unless (listp static-level)
    (error "alpine-decode-static: ~S is not an abstraction level." 
           static-level))
  (let ((nodes nil) 
        predicate-name node)
    (dolist (pred (cddr static-level))
      ;; Determine the predicate name.
      (if (symbolp pred)
        (setf predicate-name pred)
        (setf predicate-name (car pred)))
      ;; If the predicate is not static, add it to the list of nodes.
      (unless (member predicate-name (problem-space-static-preds *pspace*))
        (setf node (alpine-decode-node pred))
        (setf (abs-node-component node) :static)
        (push node nodes)))
    nodes))


;; Convert a description of a non-static abstraction level into a component
;; of the abstraction graph. Set the name of the component (if any) and
;; the list of nodes.
;;
;; An abstraction-level description is either a list of typed predicates 
;; or a list the first element of which is the component name, the second 
;; element is "=", and the other elements are typed predicates; for example,
;; (Level-0 = (on 'small Peg) (on 'medium Peg)).
;;
;; level   description of a non-static abstraction level
;;
;; returned value: the resulting component

(defun alpine-decode-component (level)
  (unless (listp level)
    (error "alpine-decode-component: ~S is not an abstraction level." level))
  (let ((component (make-abs-component)))
    (declare (type abs-component component))
    ;; Decode the component name.
    (when (eq (second level) 'user::=)
      (setf (abs-component-name component) (car level))
      (setf level (cddr level)))
    ;; Decode the nodes.
    (setf (abs-component-nodes component)
      (mapcar #'alpine-decode-node level))
    ;; Set the "component" filed of the nodes.
    (dolist (node (abs-component-nodes component))
      (setf (abs-node-component node) component))
  component))


;; Convert a typed predicate into a node of the abstraction graph.
;; Set the "name" and "args" fields of the node.
;;
;; A typed predicate is a predicate name (e.g. 'on), a predicate name
;; with the number of arguments (e.g. (on . 2), or a list, the first
;; element of which is the predicate name and the other elements are
;; argument types (e.g. (on Disk Peg)). If only a predicate name is
;; specified, set the "args" field of the node to :any. If only the
;; argument number is specified, set "args" to this number.
;;
;; pred   typed predicate
;; returned value: the node

(defun alpine-decode-node (pred)
  (let ((node (make-abs-node)))
    (cond 
      ((symbolp pred)
        (setf (abs-node-name node) pred)
        (setf (abs-node-args node) :any))
      ((and (listp pred) (symbolp (car pred)) (natural-p (cdr pred)))
        (setf (abs-node-name node) (car pred))
        (setf (abs-node-args node) (cdr pred)))
      ((listp pred)
        (setf (abs-node-name node) (car pred))
        (setf (abs-node-args node)
              (mapcar #'alpine-decode-arg (cdr pred))))
      (t
        (error "alpine-decode-node: ~S is not a predicate." pred)))
    node))


;; Convert a description of an argument type into the argument type
;; itself. An argument type is a Prodigy object, a simple type
;; (the type itself, not its name), or the disjunction of simple types.
;;
;; An object name in an argument description is preceded by "'";
;; for example, 'small, which is identical to (quote small).
;;
;; arg-code   description of an argument type
;;
;; returned value: the resulting argument type

(defun alpine-decode-arg (arg-code)
  (cond 
    ((and (listp arg-code) (eq (first arg-code) 'user::quote))
      (alpine-decode-object arg-code))
    ((symbolp arg-code)
      (alpine-decode-simple-type arg-code))
    ((and (listp arg-code) (eq (first arg-code) 'user::or))
      (alpine-decode-or-type arg-code))
    (t
      (error "alpine-decode-arg: `~S' is not object, simple type, ~
             or type disjunction." arg-code))))


;; Convert an object description into and a prodigy object.
;; An object description is an object name preceded by "'";
;; for example, 'small, which is identical to (quote small).
;;
;; object-code   description of an object (including "'")
;;
;; returned value: the resulting object

(defun alpine-decode-object (object-code)
  (declare (special *pspace*) (type problem-space *pspace*))
  (unless (and (listp object-code) (eq (car object-code) 'user::quote))
    (error "alpine-decode-object: ~S is not an object description." 
           object-code))
  (let ((object (object-name-to-object (second object-code) *pspace*)))
    (unless object
      (error "alpine-decode-object: ~S is not a valid object."
             (second object-code)))
    object))


;; Convert a description of a type disjunction into a type disjunction.
;; The resulting type disjunction is a list, the first element of which
;; is 'user::or and the other elements are simple types (the types 
;; themselves, not their names.
;;
;; or-code   the description of a type disjunction
;;
;; returned value: the resulting type disjunction

(defun alpine-decode-or-type (or-code)
  (unless (and (listp or-code) (eq (car or-code) 'user::or))
    (error "alpine-decode-or-type: `~S' is not a type disjunction." or-code))
  (cons 'user::or 
    (mapcar #'alpine-decode-simple-type (cdr or-code))))


;; Convert the name of a simple type into the type itself.
;; If the type is infinite, replace it with :Top-Type.
;;
;; simple-name   name of a simple type
;;
;; returned value: the resulting simple type

(defun alpine-decode-simple-type (simple-name)
  (declare (symbol simple-name) 
           (special *pspace*) (type problem-space *pspace*))
  (let ((simple-type (type-name-to-type simple-name *pspace*)))
    (unless simple-type
      (error "alpine-decode-simple-type: ~S is not a simple type." 
             simple-name))
    (if (functionp simple-type)  ;; infinite type?
      (setf simple-type (type-name-to-type :Top-Type *pspace*)))
    simple-type))

;;-------------------------------------------------------------------------

;; Enumerate the components of the abstraction graph (abstraction levels),
;; from the lowest level up; the lowest level is 0.
;;
;; The last component (in the topologically-sorted order) is the lowest
;; level and the first component is the highest level.
;;
;; graph   the abstraction graph

(defun alpine-enumerate-components (graph)
  (declare (type abs-graph graph))
  (let ((component-number 
          (length (abs-graph-components graph))))
    (dolist (component (abs-graph-components graph))
      (decf component-number)
      (setf (abs-component-number component) 
        component-number))))


;; Check that the components of the abstraction graph are in 
;; topologically-sorted order.
;;
;; A PDL abstraction description is valid only if the components 
;; (abstraction levels) are listed in a topologically-sorted order. 
;; That is, for every ordering constraint (Level > Level-1 ... Level-N), 
;; Level must be listed before Level-1 ... Level-N in the description of 
;; components (abstraction levels).
;;
;; Since the "components" field of the graph contains the components in
;; the same order as they are listed in the PDL description, the converting
;; of a valid description into the graph results in a topologically-sorted
;; order of components.
;;
;; graph   the abstraction graph

(defun alpine-check-top-sort (graph)
  (declare (type abs-graph graph))
  (dolist (component (abs-graph-components graph))
    (dolist (child-component (abs-component-out component))
      (unless (> (abs-component-number component)
                 (abs-component-number child-component))
        (error "Level `~S' is described after level `~S' in abstraction, ~
               whereas ordering description shows that `~S' is above `~S'."
               (abs-component-name component)
               (abs-component-name child-component)
               (abs-component-name component)
               (abs-component-name child-component))))))


;; Add edges that make components totally ordered. That is, add an 
;; edge from each component to the following component, in the order 
;; in which components are listed in the abstraction description.
;;
;; graph   the abstraction graph

(defun alpine-total-order (graph)
  (declare (type abs-graph graph))
  (mapcar #'(lambda (component1 component2)
              (push component2 (abs-component-out component1))
              (push component1 (abs-component-in component2)))
          (abs-graph-components graph) (cdr (abs-graph-components graph))))


;; Find all subset-relations between nodes and mark them by subset-links.
;;
;; If the same node occurs twice in the graph, or if two nodes from
;; different components intersect and neither of them is a subset of the 
;; other, signal an error. (See the comment to the "Operations related to 
;; the subset relationship" in the "build.lisp" graph for the definition 
;; of subset and intersection relationship between nodes.)
;;
;; Note that the graph of subset-links becomes transitively-closed:
;; if Node3 is a subset of Node2 and Node2 is a subset of Node1,
;; there will be a subset-link from Node1 to Node3.
;;
;; This function makes quadratic number (to be more precise, |graph|^2) 
;; of subset checks and intersection checks on nodes, which may be
;; time-consuming operations. The function is, therefore, time-consuming.
;;
;; graph   the graph, represented as a list of the abstract nodes
;;
;; local variables (parameters of the main loop):
;; node1         a node from the "graph" list (goes through all nodes)
;; other-nodes   the list of nodes following node1 in the "graph" list

(defun alpine-subsets-and-intersections (graph)
  (do ((node1 (car (abs-graph-nodes graph)) (car other-nodes))
       (other-nodes (cdr (abs-graph-nodes graph)) (cdr other-nodes)))
      ((null other-nodes))
    (dolist (node2 other-nodes)
      (cond 
        ((same-node-p node1 node2)
          (error "Predicate `~S' occurs more then once in abstraction."
                 (alpine-node-to-encoding node1)))
        ((node-subset-p node1 node2)
          (add-subset-link node1 node2))
        ((node-subset-p node2 node1)
          (add-subset-link node2 node1))
        ((and (diff (abs-node-component node1) (abs-node-component node2))
              (node-intersection-p node1 node2))
          (error "In abstraction, set of literals corresponding to ~
            predicate `~S' intersects with set of literals corresponding ~
            to predicate `~S', these sets are on different levels, and ~
            neither of them is a subset of the other." 
                 (alpine-node-to-encoding node1)
                 (alpine-node-to-encoding node2)))))))


;; Convert an abstraction node into its PDL encoding.
;; We use the "alpine-encode-node" function, which converts the node
;; into its PDL encoding and adds the encoding to the end of the
;; dynamically-scoped global string "*encoding*".
;;
;; node   node to be converted into PDL
;;
;; returned value: the PDL encoding of the node

(defun alpine-node-to-encoding (node)
  (declare (type abs-node node))
  (let ((*encoding* (make-array 0 :fill-pointer t :adjustable t 
                                    :element-type 'excl::string-char)))
    (declare (type string *encoding*) (special *encoding*))
    (alpine-encode-node node)
    *encoding*))


#|
==========================================================
Check the syntax of the abstraction-graph PDL description.
==========================================================

These functions check the syntactic validity of the abstraction-hierarchy
description in PDL. Since the hierarchy is encoded as a graph, I usually
say "abstraction graph" instead of "abstraction hierarchy" and "component
of the abstraction graph" instead of "non-static abstraction level."

The input of the top-level syntax-checking function is the list that
encode the abstraction graph (hierarchy), without the word "Abstraction".

The functions only check the syntax; they do not create any data structures.
For the details of the PDL syntax for describing abstraction hierarchies,
see the comment in the beginning of the "Converting the Abstraction Graph"
part in the "build.lisp" file.
|#


;; Check the validity of the description of an abstraction graph (hierarchy).
;; The input of the function does not include the word "Abstraction".
;; The description is valid if it is a list of abstraction-graph components
;;   (i.e. abstraction levels) and ordering constraints, with the keyword
;;   :order between. If there is the static level, it must be first in 
;;   the list of abstraction levels:
;;     (static-level level ... level {ORDER | :ORDER} order ... order).
;; For more details on the description syntax, see the comment in the 
;;   beginning of the "Converting the Abstraction Graph" part in the
;;   "build.lisp" file.
;;
;; graph   description of the abstraction graph (hierarchy)
;;
;; dynamically-scoped global variables:
;; *component-names*   list of the names of the already-checked components
;; *infinite-types*    list of infinite-type names used in typed predicates

(defun alpine-syntax-graph (graph *pspace*)
  (declare (type problem-space *pspace*) (special *pspace*))
  (let ((*component-names* nil)
        (*infinite-types* nil))
    (declare (special *component-names* *infinite-types*))
    ;; Check that the abstraction description is a list.
    (unless (listp graph)
      (error "Abstraction description is not a list."))
    ;; Check that the abstraction is not collapsed.
    (if (or (member (first graph) '(user::collapse :collapse))
            (member (second graph) '(user::collapse :collapse)))
      (error "Abstraction is collapsed."))
    ;; Check the validity of the abstraction name.
    (if (symbolp (car graph))
      (alpine-syntax-graph-name (pop graph)))
    ;; Check the validity of the static level.
    (if (and (listp (car graph))
             (member (caar graph) '(user::static :static))
             (eq (second (car graph)) 'user::=))
               ;; the first level static?
      (alpine-syntax-static-level (pop graph)))
    ;; Check that the abstraction has at least two non-static levels.
    (if (or (null graph) (member (first graph) '(user::order :order)))
      (error "Abstraction has no non-static levels."))
    (if (or (singleton-p graph) (member (second graph) '(user::order :order)))
      (error "Abstraction has only one non-static level."))
    ;; Check the validity of components (non-static abstraction levels).
    (do* ((component (pop graph) (pop graph)))
         ((or (null component) (member component '(user::order :order))))
      (unless (listp component)
        (error "`~S' in abstraction is not an abstraction level." component))
      (alpine-syntax-component component))
    ;; If typed predicates contain infinite types,
    ;; print a warning with the list of these types.
    (when *infinite-types*
      (format t "~&Warning: Abstraction contains infinite types (`~S'~{, ~
                `~S'~}).~%These types will be replaced with :TOP-TYPE." 
                (car *infinite-types*) (cdr *infinite-types*)))
    ;; Check the validity of ordering constraints.
    (when graph  ;; any ordering constraints?
      (dolist (order graph)
        (unless (listp order)
          (error "`~S' in abstraction is not an ordering constraint." 
                 order))
        (alpine-syntax-order order)))))

;; Check the validity of the abstraction graph (hierarchy) name.
;; "graph-name" must be a symbol.
;;
;; graph-name    name of the abstraction graph
;; graph-names   names of other abstraction graphs

(defun alpine-syntax-graph-name (graph-name &optional graph-names)
  (declare (symbol graph-name))
  (unless graph-name
     (error "Abstraction name may not be NIL."))
  (if (member graph-name graph-names)
     (error "More than one abstraction has name `~S'." graph-name)))


;; Check the validity of a description of the static abstraction level.
;; 
;; The description is valid if it is a list, the first element 
;; if which is "STATIC" or ":STATIC'", the second element is "=",
;; and the other elements are (non-static or static) typed predicates.
;; If the static level contains non-static predicates, the user must
;; ensure that the values of these predicates do not change during
;; abstraction problem solving.
;;
;; "static-level" must be a list, the first element of which 
;; is "STATIC" or ":STATIC" and the second element is "=".
;;
;; static-level   description of the static level
;;
;; local variables:
;; non-static-list   list of non-static predicates at the static level

(defun alpine-syntax-static-level (static-level)
  (declare (special *pspace*) (type problem-space *pspace*))
  (let ((non-static-preds nil))
    (dolist (pred (cddr static-level))
      ;; Check the validity of a type predicate.
      ;; If the predicate is not static, add its name to "non-static-preds".
      (if (eq (alpine-syntax-pred pred) :non-static)
        (push pred non-static-preds)))
    ;; If the static level contains non-static predicates,
    ;; print a warning with the list of these predicates.
    (when non-static-preds
      (format t "~&Warning: Static abstraction level contains non-static ~
                predicates ~%         (`~S'" (car non-static-preds))
      (dolist (pred (cdr non-static-preds))
        (format t ", `~S'" pred))
      (format t ")"))))


;; Check the validity of a component (non-static abstraction level).
;;
;; A component description is valid if it is a list, the first element 
;; of which is component name, the second element is "=", and the
;; other elements are typed non-static predicates; for example, 
;; (Level-0 = (on 'small Peg) (on 'medium Peg)).
;; The first two elements, the name and "=", may be absent; 
;; for example, ((on 'small Peg) (on 'medium Peg)).
;;
;; "component" must be a list.
;;
;; component   description of a component of the abstraction graph
;;             (component of abstraction graph is abstraction level)

(defun alpine-syntax-component (component)
  ;; Check the name of the component.
  (when (eq (second component) 'user::=)
    (unless (symbolp (car component))
      (error "`~S' is not an abstraction-level name." (car component)))
    (alpine-syntax-component-name (car component))
    (setf component (cddr component)))
  ;; Check typed predicates.
  (dolist (pred component)
    (alpine-syntax-pred pred :non-static)))


;; Check the validity of name of component (non-static abstraction level).
;; The name is valid if it is not nil, "static", or ":static", and differs 
;;   from the names of the previously checked components.
;; After checking the name, add it to the list of names, *component-names*.
;; "component-name" must be a symbol.
;;
;; component-name   name of a component of the abstraction graph
;;                  (component of abstraction graph is abstraction level)

(defun alpine-syntax-component-name (component-name)
  (declare (symbol component-name) (special *component-names*))
  (unless component-name
    (error "`NIL' is not a valid abstraction-level name."))
  (if (member component-name '(user::static :static))
    (error "Name of a non-static abstraction level cannot be ~
           `STATIC' or `:STATIC'"))
  (if (member component-name *component-names*)
    (error "More than one abstraction component has name `~S'." 
           component-name))
  (push component-name *component-names*))


;; Check the validity of a typed predicate.
;;
;; A typed predicate is valid if it is either the name of an existing 
;; predicate (e.g. "on") or a list, the first element of which is the 
;; name of a predicate and the other elements are argument types.
;;
;; pred   typed non-static predicate
;; static?     takes one of the following three values:
;;               :static       pred is valid only if static
;;               :non-static   pred is valid only if non-static
;;               nil           pred is valid in either case
;;
;; returned value:
;; :static if the predicate is static; :non-static otherwise

(defun alpine-syntax-pred (pred &optional static?)
  (cond 
    ((symbolp pred)
      (alpine-syntax-pred-name pred static?))
    ((and (listp pred) (symbolp (car pred)))
      (let ((really-static? 
              (alpine-syntax-pred-name (car pred) static?)))
        (dolist (arg (cdr pred))
          (alpine-syntax-arg arg))
        really-static?))
    (t
      (error "`~S' is not a valid predicate." pred))))


;; Check the validity of the name of a predicate.
;; "predicate-name" must be an atom.
;;
;; predicate-name   name of a predicate
;; static?          takes one of the following three values:
;;                   :static       predicate-name is valid only if static
;;                   :non-static   predicate-name is valid only if non-static
;;                   nil           predicate-name is valid in either case
;;
;; returned value:
;; :static if the predicate is static; :non-static otherwise

(defun alpine-syntax-pred-name (predicate-name &optional static?)
  (declare (symbol object) 
           (special *pspace*) (type problem-space *pspace*))
  (unless (member predicate-name (problem-space-all-preds *pspace*))
    (error "`~S' is not a valid predicate name." predicate-name))
  (cond
    ((eq static? :static)
      (unless (member predicate-name (problem-space-static-preds *pspace*))
        (error "Predicate `~S' is not static." predicate-name))
      :static)
    ((eq static? :non-static)
      (if (member predicate-name (problem-space-static-preds *pspace*))
        (error "Static predicate `~S' is not on the static level." 
               predicate-name))
      :non-static)
    (t 
      (if (member predicate-name (problem-space-static-preds *pspace*))
        :static
        :non-static))))


;; Check the validity of an argument type.
;; An argument type is valid if it is a prodigy-object name preceded
;;   by "'" (for example, 'small), type name, or type disjunction.
;;
;; arg  argument type

(defun alpine-syntax-arg (arg)
  (cond 
    ((and (listp arg) (eq (first arg) 'user::quote) (symbolp (second arg)))
      (alpine-syntax-object arg))
    ((symbolp arg)
      (alpine-syntax-simple-type arg))
    ((and (listp arg) (eq (first arg) 'user::or))
      (alpine-syntax-or-type arg))
    ((and (listp arg) (eq (first arg) 'user::or))
      (error "You cannot use a conjunction of types in abstraction and ~
        relevance graphs, hence `~S' is not a valid type generator." arg))
    (t
      (error "`~S' is not a valid type generator." arg))))


;; Check the validity of an object name.
;; "object-name" must be a symbol preceded by "'"; for example,
;;   'small, which is identical to (quote small).
;;
;; object-name  name of a prodigy object

(defun alpine-syntax-object (object-name)
  (declare (special *pspace*) (type problem-space *pspace*))
  (unless (object-name-to-object (second object-name) *pspace*)
    (error "`~S' is not a valid object." object-name)))


;; Check the validity of a type disjunction.
;; "or-type" must be a list, whose first element is 'user::or.
;;
;; or-type   encoding of a type disjunction

(defun alpine-syntax-or-type (or-type)
  (dolist (simple-type (cdr or-type))
    (unless (symbolp simple-type)
      (error "`S' is not a type name." simple-type))
    (alpine-syntax-or-type simple-type)))
  

;; Check the validity of a simple-type name.
;; If the type is infinite, add its name to the list of 
;;   infinite-type names, *infinite-types*.
;; "simple-type" must be a symbol.
;;
;; simple-type  name of a simple type

(defun alpine-syntax-simple-type (simple-type)
  (declare (symbol simple-type) (special *pspace* *infinite-types*)
           (type problem-space *pspace*))
  (unless (type-name-to-type simple-type *pspace*)
    (error "`~S' is not a valid type." simple-type))
  (if (functionp (type-name-to-type simple-type *pspace*)) ;; infinite type?
    (push simple-type *infinite-types*)))


;; Check the validity of a component ordering.
;;
;; The ordering element is valid if its first element is an existing 
;; component name, the second element is ">", and the other elements
;; are the existing component names distinct from the first component
;; name. There must be at least one component name after ">"; 
;; for example (Level-2 > Level-0 Level-1).
;;
;; "order" must be a list.
;;
;; order   component ordering

(defun alpine-syntax-order (order)
  (declare (special *component-names*))
  (unless (and (eq (second order) 'user::>) (cddr order))
    (error "`~S' in abstraction is not a valid ordering description." order))
  (dolist (component-name (cons (car order) (cddr order)))
    (unless (symbolp component-name)
      (error "`~S' in abstraction ordering description is not an ~
             abstraction-level name." component-name))
    (if (member component-name '(user::static :static))
      (error "Abstraction ordering description must not include ~
              constraints on the static level."))
    (unless (member component-name *component-names*)
      (error "`~S' in abstraction ordering description is not a ~
             valid abstraction-level name." component-name)))
  (dolist (component-name (cddr order))
    (if (eq component-name (car order))
      (error "Abstraction ordering description shows that level `~S' ~
             is below itself." component-name))))