#|
===========================
General Comments to Alpine.
===========================
Required files.
---------------
The use of Alpine with Prodigy requires to load all the files of my
version of Prodigy (from the "planner" subdirectory), the four Alpine
files ("alpine/struct", "alpine/build", "alpine/load", and "alpine/use"), 
and two Margie files ("margie/preg" and "margie/postg").

PDL Assumptions.
----------------
We use the following assumptions about the domain description. My
version of "load-domain" ensures that the domain description satisfies
these assumptions. If the assumptions are not satisfied, my version
of "load-domain" gives an error.

The operator description does not contain "forall" effects.  Every
rule has some effect (primary or side). The list of actions in every
conditional effect is nonempty.

The operator description does not contain type descriptions with
negations, such as (<disk> (not Peg)). Also, the type description
cannot contain nested conjunctions, i.e. (and ...  (and ...)).  The
first element of a conjunction must be a simple (atomic) type or a
disjunction, and the other elements are functions.

We allow only disjunctions of simple (atomic) types in the type
descriptions; the elements of a type disjunctions cannot be functions,
conjunctions, or disjunctions; for example, the type description (or
Small Medium) is legal, whereas (or Small (or Medium Large)), (or
Small (diff <small medium>)) and (or Small (and ...)) are not legal.

When we use the disjunction of types in a declaration of domain
variables, the types in the disjunction are listed in the alphabetical
order of their names. For example, (or Disk Peg) rather than (or Peg
Disk). ("load-domain" sorts the types in disjunctions, so the user may
list the types in any order.)

Possible improvements.
----------------------
We presently represent the abstraction graph as a list of nodes and
list of components. If we operate with large graphs, search for 
a specified node in the list is inefficient. We may improve efficiency
by using a hash-table, hashed on the predicate names, in the same way
as in relevance table of Relator. This improvement is especially useful 
for determining the levels of literals during abstraction problem solving.

We determine static predicates by name only. For example, if there is
an operator that achieves (on small peg-a), then ALL predicates with
name "on" are not considered static. We may improve the effectiveness
of Alpine by implementing a better static-predicate test, which requires
building a table of static predicates (based on their arguments, not only
names) and modifying the function "static-predicate-p" in "general.lisp"
(which determines whether a predicate is static).
|#


#-clisp
(unless (find-package "PRODIGY4")
  (make-package "PRODIGY4" :nicknames '("P4") :use '("LISP")))
(in-package "PRODIGY4")


#|
================
Data structures.
================

We use the following data structures.

abs-graph: 
The abstraction graph, representing the predicates with types and the 
constraints on their levels in the hierarchy.

abs-node:
A node of the abstraction graph, which represents a predicate with types.

abs-component:
A strongly connected component of the abstraction graph; the edges of the 
graph a between components, not between individual nodes.

pred:
The list of all abstraction-graph nodes with the same name, arranged
by the number of argumnets.

argnum:
The list of all abstraction-graph nodes with the same name and the 
same number of arguments.

----------------

We use the following global variables, which must be defined before calling
Alpine. The Alpine-interface functions (see the "alpine/use" file) define
these globale variables before calling Alpine.

*pspace*: 
problem space for which we generate abstraction
|#


;; Node in the abstraction graph, which represents a predicate with argument 
;; types; for example, #<AbsNode: [ON 'SMALL PEG]> (where ' denotes constants 
;; -- see comments to the "print-short-abs-node" function).

(defstruct (abs-node (:print-function print-abs-node))
  (name nil)   ;; predicate name
  (args nil)   ;; list of arguments (objects, simple types, type disjunctions);
               ;;   set to :any if the arguments are not specified;
               ;;   set to integer if only the argument number is specified
  (loop-p nil) ;; T if the nodes in on a loop (that is, there is path
               ;;   from the node into itself)
  (sub nil)    ;; list of nodes that are subsets of this node
  (super nil)  ;; list of nodes that are supersets of this node
  (color nil)  ;; used in graph algorithms, to determine whether the
               ;;   node has been visited by a graph search
  (component nil)  ;; strongly connected component containing the node;
                   ;;   set to :static if the node is at the 
                   ;;   static abstraction level
  (plist nil)) ;; additional information use by Margie; in particular,
               ;; the maybe-nodes that intersect the given maybe-node
               ;; (:inter) and a temporary boolean value (:temp-p)


;; Print out a node, of the type "abs-node", of the abstraction graph.

(defun print-abs-node (node stream z)
  (declare (type abs-node node) (stream stream) (ignore z))
  (princ "#<AbsNode: " stream)
  (print-short-abs-node node stream)
  (princ ">" stream))


;; Print out a shorter version of a node.
;;
;; If an argument is an object (rather than a type), print ' in front of it;
;; for example, [ON 'SMALL PEG]. If the predicate may have any arguments,
;; print "--" for arguments; for example, [ON --]. If the number of arguments
;; is specified, print this number of "*"; for example, [ON * *].

(defun print-short-abs-node (node stream)
  (declare (type abs-node node) (stream stream))
  (princ "[" stream)
  (if (symbolp (abs-node-name node))
    (princ (abs-node-name node) stream)
    (progn (princ "~ " stream) (princ (cdr (abs-node-name node)) stream)))
  (typecase (abs-node-args node)
    (symbol  ;; ":any"
      (princ " --" stream))
    (number  ;; argument number
      (dotimes (i (abs-node-args node)) (princ " *" stream)))
    (list  ;; typed argument list
      (princ #\Space stream)
      (print-args (abs-node-args node) stream)))
  (princ "]" stream))


;; Print out the elements of a typed-argument list.
;;
;; An argument may be a simple type, type disjunction, or object.
;; If it is an object, print "'" in front of it; for example, "'SMALL PEG".

(defun print-args (args stream)
  (declare (type stream stream))
  (print-arg (car args) stream)
  (dolist (arg (cdr args))
    (princ #\Space stream)
    (print-arg arg stream)))

;; Print out a typed argument (simple type, type disjunction, or object).
;;
;; If it is an object, print "'" in front of it; for example, 'SMALL.

(defun print-arg (arg stream)
  (declare (type stream stream))
  (typecase arg
    (type  ;; simple type
      (princ (type-name arg) stream))
    (list  ;; type disjunction
      (princ "(or" stream)
      (mapcar #'(lambda (simple-type) 
                  (princ #\Space stream) 
                  (princ (type-name simple-type) stream))
              (cdr arg))
      (princ ")" stream))
    (prodigy-object  ;; instance (prodigy object)
      (princ "'" stream)
      (princ (prodigy-object-name arg) stream))))


;; Strongly connected component in the abstraction graph.
;; After constructing a hierarchy, each component becomes abstraction level.

(defstruct (abs-component (:print-function print-abs-component))
  (name nil)   ;; component name (optional)
  (number -1)  ;; number of the corresponding level in the hierarchy, 
               ;;   counting from lowest level; lowest level is 0
  (nodes nil)  ;; list of nodes in the component
  (out nil)    ;; list of components connected by outcoming edges
  (in nil)     ;; list of components connected by incoming edges
  ;; The rest of the fields do not contain any permanent information.
  ;; They are for local use in some graph algorithms.
  (color nil)  ;; used in graph algorithms, to determine whether the
               ;;   component has been visited by a graph search
  (time nil)   ;; used in graph algorithms, to determine when the
               ;;   component has been visited
  (super-component nil)
               ;; super-component containing the component; used in graph 
               ;;   algorithms (when combining strongly connected components 
               ;;   in the graph of components, a component (node of the 
               ;;   component graph) becomes an element of some 
               ;;   strongly connected component of this graph, called 
               ;;   "super-component")
  (inherited-out) 
               ;; list of inheritied outcoming edges; used in adding edges
               ;;   based on subset-links
  (inherited-in))
               ;; list of inheritied outcoming edges; used in adding edges
               ;;   based on subset-links


;; Print out a component, of the type abs-component (which is
;; potentially an abstraction level).
;;
;; Print the component name (if any), the number of the corresponding
;; abstraction level in the hierarchy (if any), and the names of nodes
;; in the component; for example, #<AbsLevel: Level-0 [0: ROBOT-AT
;; BALL-AT]>.
              
(defun print-abs-component (component stream z)
  (declare (type abs-component component) (stream stream) (ignore z))
  (princ "#<AbsLevel: " stream)
  ;; Print the component name (if any).
  (when (abs-component-name component)
    (princ (abs-component-name component) stream) (princ " " stream))
  ;; Print the names of abstraction nodes (predicates) in the component.
  (print-short-abs-component component stream)
  (princ ">" stream))


;; Print out a shorter version of a component. 
;;
;; Print the number of the corresponding abstraction level (if any) and the
;; names of abstraction nodes (predicates) in the component;
;; for example, [0: ROBOT-AT BALL-AT]. If the nodes in the component are
;; themselves abs-components (rahter than abs-nodes), we print "AbsLevel"
;; for every node; for example, [AbsLevel AbsLevel].
;;
;; For an abstraction postgraph used in margie, also print the past nodes, in
;; brackets after the real nodes; for example, [0: ROBOT-AT (BALL-AT DOOR)].
;; All past nodes are listed in the ":pasts" keyword.

(defun print-short-abs-component (component stream &key (pasts nil))
  (declare (type abs-component component) (stream stream))
  (princ "[" stream)
  (when (/= -1 (abs-component-number component))
    (princ (abs-component-number component) stream) (princ ": " stream))
  (when (abs-component-nodes component)
    (print-node-names (abs-component-nodes component) stream))
  (if pasts (del component pasts :test-not #'eq :key #'abs-node-component))
  (when pasts
    (print "(" stream)
    (print-node-names pasts stream)
    (print ")"))
  (princ "]" stream))


;; Print out several node names with spaces between them; for example,
;; "ROBOT-AT BALL-AT". The node list, "nodes", must be nonempty.

(defun print-node-names (nodes stream)
  (declare (stream stream))
  (print-node-name (car nodes) stream)
  (dolist (node (cdr nodes))
    (princ " " stream)
    (print-node-name node stream)))


;; Print out a node name; for a negated-predicate node, 
;; print "~" before the name; for example, "~ROBOT-AT".

(defun print-node-name (node stream)
  (declare (type abs-node node) (stream stream))
  (cond
    ((and (abs-node-p node) (symbolp (abs-node-name node)))
       (princ (abs-node-name node) stream))
    ((and (abs-node-p node) (consp (abs-node-name node)))
       (princ "~" stream) 
       (princ (cdr (abs-node-name node)) stream))
    (t (princ "AbsLevel" stream))))


;; Show the data in abstraction component.
;;
;; Print the number of the corresponding abstraction level (if any), and the 
;; names and typed arguments of abstraction nodes (predicates) in the 
;; component; for example, "AbsLevel 0: [ROBOT-AT ROOM] [BALL-AT ROOM]".

(defun show-abs-component (component &optional (stream t))
  (declare (type abs-component component) (stream stream))
  (princ "AbsLevel")
  (when (/= -1 (abs-component-number component))
    (princ " " stream) 
    (princ (abs-component-number component) stream))
  (princ ":" stream)
  (if (abs-component-nodes component)
    (dolist (node (abs-component-nodes component))
      (princ " " stream)
      (if (abs-node-p node)
        (print-short-abs-node node stream)
        (princ "AbsLevel" stream)))
    (princ " No nodes")))
            

;; Abstraction graph (list of nodes and components, in no particular order),
;; which will become an abstraction hierarchy.

(defstruct (abs-graph (:print-function print-abs-graph))
  (name nil)         ;; name of the abstraction hierarchy (optional)
  (nodes nil)        ;; list of all nodes in the graph;
                     ;;   when we use the graph in abstraction planning,
                     ;;   the nodes are topologically sorted on the subset
                     ;;   relationship; that is, if Node1 is a subset of
                     ;;   Node2, then Node1 is before Node2 in the list
  (components nil)   ;; list of all strongly connected components;
                     ;;   when we use the graph in abstraction planning,
                     ;;   the components are in the order from most abstract
                     ;;   level to the least abstract (ground) level
  (static nil)       ;; static abstraction level (optional), which is a
                     ;;   list of nodes for non-static predicates that
                     ;;   must be considered static in abstraction planning
                     ;;   (the assumption is that these predicates do 
                     ;;   not change during abstraction problem solving)
  (plist nil))       ;; additional information used by Margie; in particular,
                     ;;   the list of nodes that may be inserted into the
                     ;;   graph later (:maybes) and the list of nodes removed
                     ;;   from the graph (:pasts)


;; Print out an abstraction graph.
;;
;; Print the graph name (if any), the static-level predicates (if any),
;; and short versions of all components, in the order they are listed in 
;; the "components" field; for example, 
;; #<AbsGraph: ROBOT-ABSTRACTION [Static: NEAR] [DOOR] [ROBOT-AT BALL-AT]>.
;;
;; For Margie's abstraction postgraph, also print past nodes (see the 
;; comment to the "print-short-abs-component" function) and maybe nodes; e.g.
;; #<AbsGraph [ROBOT-AT (BALL-AT DOOR)] [Maybes: ~ROBOT-AT ~BALL-AT]>.

(defun print-abs-graph (graph stream z)
  (declare (type abs-graph graph) (stream stream) (ignore z))
  (let ((pasts (abs-graph-pasts graph))
        (maybes (abs-graph-maybes graph)))
    (princ "#<AbsGraph:" stream)
    ;; Print the graph name (if any).
    (when (abs-graph-name graph)
      (princ " " stream) (princ (abs-graph-name graph) stream))
    ;; Print the static abstraction level.
    (when (abs-graph-static graph)
      (princ " [Static: " stream)
      (print-node-names (abs-graph-static graph) stream)
      (princ "]" stream))
    ;; Print the names of abstraction nodes (predicates) in every component.
    (mapcar #'(lambda (component)
                (princ " " stream)
                (print-short-abs-component component stream :pasts pasts))
            (abs-graph-components graph))
    ;; Print the maybe nodes.
    (when maybes
      (princ " [Maybes: " stream)
      (print-node-names maybes stream)
      (princ "]" stream))
    (princ ">" stream)))


;; List of all nodes in the abstraction graph with some specific
;; predicate name, arranged by the argument number.

(defstruct (pred (:print-function print-pred))
  (name nil)       ;; the predicate name
  (component nil)  ;; component that contains all nodes with this name;
                   ;;   nil if the nodes are not all in the same component
  (color nil)      ;; color of the nodes with this name (black or white);
                   ;;   nil if the nodes are not all in the same color
  (argnums nil))   ;; lists of nodes with this name, each list for a
                   ;;   specific argument number; lists are represented
                   ;;   by "argnum" structures


;; Print out a Pred structure.
;;
;; Print the predicate name and the list of argument numbers that have
;; corresponding node lists; for examle, #<Pred: [ON: :ANY 1 2]>.

(defun print-pred (pred stream z)
  (declare (type pred pred) (stream stream) (ignore z))
  (princ "#<Pred: " stream)
  (print-short-pred pred stream)
  (princ ">" stream))


;; Print out a shorter version of Pred.
;;
;; Print the predicate name and the list of argument numbers that have
;; corresponding node lists; for examle, [ON: :ANY 1 2].

(defun print-short-pred (pred stream)
  (declare (type pred pred) (stream stream))
  (princ "[" stream)
  (if (symbolp (pred-name pred))
    (princ (pred-name pred) stream)
    (progn (princ "~ " stream) (princ (cdr (pred-name pred)) stream)))
  (princ ":" stream)
  (dolist (argnum (pred-argnums pred))
    (princ " " stream)
    (if (= (argnum-num argnum) -1)
      (princ ":ANY" stream)
      (princ (argnum-num argnum) stream)))
  (princ "]" stream))


;; Show the data in a Pred structure.
;;
;; Print the predicate name, argument numbers, and the list of nodes
;; for every argument number.

(defun show-pred (pred &optional (stream t))
  (declare (type pred pred) (stream stream))
  (format stream "~&Pred for ~S" (pred-name pred))
  (if (pred-component pred) (format stream ", in one component"))
  (if (pred-color pred) (format stream ", of ~S color" (pred-color pred)))
  (dolist (argnum (pred-argnums pred))
    (show-argnum argnum stream)))


;; List of all nodes in the abstraction graph with some specific
;; predicate name and number of arguments.
;;
;; For example, it may contain all nodes of the form [on * *],
;; i.e. all two-argument nodes with the predicate name "on".

(defstruct (argnum (:print-function print-argnum))
  (num -2)         ;; the number of arguments in a node; -1 for a node
                   ;;   with any number of arguments, i.e. when a node 
                   ;;   is deterimed only by a predicate name
  (component nil)  ;; component that contains all nodes on the list;
                   ;;   nil if the nodes are not all in the same component
  (color nil)      ;; color of the nodes on the list (black or white);
                   ;;   nil if the nodes are not of the same color
  (nodes nil))     ;; list of nodes, in arbitrary order


;; Print out an Argnum structure.
;;
;; Print the predicate name and a "*" for each of the arguments; for
;; example, #<Argnum: [ON * *]. If the list represents a node with any
;; number of arguments, print ":ANY" instead of the asterisks; for
;; example, #<Argnum: [ON :ANY]>.

(defun print-argnum (argnum stream z)
  (declare (type argnum argnum) (stream stream) (ignore z))
  (princ "#<Argnum: " stream)
  (print-short-argnum argnum stream)
  (princ ">" stream))


;; Print out a shorter version of Argnum.
;; 
;; Print the predicate name and a "*" for each of the arguments; that is,
;; the number of asterisks equals the number of arguments; for example,
;; [ON * *]. If the list represents a node with any number of arguments,
;; print the node name and ":ANY"; for example, [ON :ANY].

(defun print-short-argnum (argnum stream)
  (declare (type argnum argnum) (stream stream))
  (let ((pred-name (abs-node-name (car (argnum-nodes argnum)))))
    (princ "[" stream)
    (if (symbolp pred-name)
      (princ pred-name stream)
      (progn (princ "~ " stream) (princ (cdr pred-name) stream)))
    (if (= (argnum-num argnum) -1)
      (princ " --" stream)
      (dotimes (i (argnum-num argnum)) (princ " *" stream)))
    (princ "]" stream)))


;; Show the data in an Argnum structure.
;;
;; Print the argument number and the list of nodes.

(defun show-argnum (argnum &optional (stream t))
  (declare (type argnum argnum) (stream stream))
  (format stream "~&~D" (argnum-num argnum))
  (if (argnum-component argnum) (format stream ", in one component"))
  (if (argnum-color argnum) 
    (format stream ", of ~S color" (argnum-color argnum)))
  (format stream ":")
  (dolist (node (argnum-nodes argnum))
    (format stream "~%  ")
    (print-short-abs-node node stream)))
