#-clisp
(unless (find-package "PRODIGY4")
  (make-package "PRODIGY4" :nicknames '("P4") :use '("LISP")))
(in-package "PRODIGY4")


#|
=========================================
Creating the initial graph of predicates.
=========================================

We convert preconditions and effect of the operators and inference rules
into nodes of the abstraction graph and impose constraints.

The graph does not contain static predicates. The predicate is static
if ALL predicates with this name are static. That is, the static-property
is determined by name only. For example, if there is an operator that
achieves (on small <disk>) and there is no operator that achieves
(on large <disk>), we do NOT consider (on large <disk>) static,
since the name "on" refers to some non-static predicate.

If a predicate does not have any constraints, we do not add it to the
graph. A predicate does not have constraints only if it may never
become a subgoal (e.g., a condition of a conditional side effect).  
|#


;; Create the initial abstraction graph.
;;
;; Add all the non-static preconditions and effects of operators
;; and inference rules to the graph and impose constraints between
;; preconditions and effects of every rule.
;;
;; name          name of the graph (and of the abstraction hierarchy)
;; operators     list of operators (by default, all operators in the domain)
;; eagers        list of eager inference rules (by default, all eager rules)
;; lazys         list of lazy inference rules (by default, all lazy rules)
;; use-primary   nil (do not use), :side (treat rest like side), 
;;                 or :prim (treat rest like primary)
;;
;; returned value: the initial abstraction graph

(defun initial-graph (&key (name nil)
                           (operators :default) 
                           (eagers :default)
                           (lazys :default)
                           (use-primary nil))
  (declare (type abs-graph graph) (symbol use-primary)
           (special *pspace*) (type problem-space *pspace*))
  ;; Set default lists of operators and inference rules.
  (if (eq operators :default)
    (setf operators (problem-space-real-ops *pspace*)))
  (if (eq eagers :default)
    (setf eagers (problem-space-eager-inference-rules *pspace*)))
  (if (eq lazys :default)
    (setf lazys (problem-space-lazy-inference-rules *pspace*)))
  (if (eq use-primary :default)
    (setf use-primary (any-space-property :use-primary *pspace*)))
  ;; Create the initial abstraction graph.
  (let ((graph (create-empty-graph name)))
    (dolist (operator operators)
      (add-operator-to-graph operator graph use-primary))
    (dolist (inf-rule (append eagers lazys))
      (unless (static-inference-rule-p inf-rule)
        (add-inf-rule-to-graph inf-rule graph use-primary)))
    graph))

;;-------------------------------------------------------------------------

;; Convert all nonstatic relevant preconditions and all relevant
;; effects of an operator into nodes of an abstraction graph, add
;; these nodes to the graph (unless they are already in the graph),
;; and add all necessary edges.
;;
;; We pick some primary effect Prim-Eff-Node and then add edges as follows:
;; - for every primary-effect Node,  Prim-Eff-Node <-> Node
;; - for every side-effect Node,  Prim-Eff-Node -> Node
;; - for every precondition Node,  Prim-Eff-Node -> Node
;;
;; The constraints on the conditions of conditional effects are
;; added in the function "add-operator-eff-list-to-graph".
;;
;; operator      an operator
;; graph         the abstraction graph
;; use-primary   nil (do not use), :side (treat rest like side), 
;;                 or :prim (treat rest like primary)

(defun add-operator-to-graph (operator graph use-primary)
  (declare (type rule operator) (type abs-graph graph) (symbol use-primary)
           (special *pspace*) (type problem-space *pspace*))
  (let* ((precond-decs (second (rule-precond-exp operator)))
         (effect-decs (append (second (rule-effects operator)) precond-decs))
         precond-nodes prim-eff-nodes side-eff-nodes 
         prims-and-sides prim-eff-node)
    (when ;; Does operator have relevant primary effects? If not, don't add it 
          ;; to the graph, since it does not require any constraints.
          (some #'(lambda (effect)
                    (relevant-vared-p (effect-cond-effect effect)
                      effect-decs *pspace* :negated-p :either
                      :use-primary use-primary))
                (subelts-to-elts (append (rule-add-list operator) 
                                         (rule-del-list operator))))
      ;; Add precondition nodes to the graph.
      (setf precond-nodes (delete-duplicates (add-precond-exp-to-graph 
        (third (rule-precond-exp operator)) precond-decs graph use-primary)))
      ;; Add effect nodes to the graph.
      (setf prims-and-sides
        (add-operator-effs-to-graph (rule-add-list operator) 
          (rule-del-list operator) effect-decs graph use-primary))
      (setf prim-eff-nodes (delete-duplicates (first prims-and-sides)))
      (setf side-eff-nodes (delete-duplicates (second prims-and-sides)))
      ;; Pick some primary effect.
      (setf prim-eff-node (car prim-eff-nodes))
      ;; Add constraints on the primary effects.
      (dolist (node (cdr prim-eff-nodes))
        (add-directed-edge prim-eff-node node)
        (add-directed-edge node prim-eff-node))
      ;; Add constraints on the side effects.
      (dolist (node side-eff-nodes)
        (add-directed-edge prim-eff-node node))
      ;; Add constraints on the non-static preconditions.
      (dolist (node precond-nodes)
        (add-directed-edge prim-eff-node node)))))


;; Convert all relevant effects of an operator into nodes of an
;; abstraction graph and add these nodes to the graph (unless they are
;; already in the graph).
;;
;; For conditional primary effects, add their conditions to the graph
;; and add edges from primary actions to conditions.
;;
;; add-list      operator's add-list
;; del-list      operator's del-list
;; decs          declarations from the precond-exp of the operator
;; graph         the abstraction graph
;; use-primary   way of using primary effects (nil, :side, or :prim)
;;
;; returned value:
;; the two-element list, where the first element is primary-effect
;; abs-nodes and the second element is the side-effect abs-nodes; if
;; several effects correspond to the same node, then this node occurs
;; several times in the lists

(defun add-operator-effs-to-graph (add-list del-list decs graph use-primary)
  (declare (type abs-graph graph) (symbol use-primary))
  (let ((del-remains (copy-list del-list))
        (prim-eff-nodes nil)
        (side-eff-nodes nil)
        prims-and-sides del-effect-list)
    ;; Add the nodes for the add-list to the graph. If a conditional effect
    ;; have both add and del effects, we merge them before calling the
    ;; "add-operator-eff-list-to-graph" function, and remove the corresponding
    ;; effect list from del-remains (which is a copy of del-list). This
    ;; merging prevents a double attemp of adding the effect's conditions.
    (dolist (add-effect-list add-list)
      ;; Find del-actions of the same conditional effect.
      (setf del-effect-list
        (find (effect-cond-conditional (car add-effect-list)) del-remains
         :key #'(lambda (eff-list) (effect-cond-conditional (car eff-list)))))
      ;; Remove them from the "del-remains" list.
      (if del-effect-list
        (del del-effect-list del-remains))
      ;; Add the conditional effect to the graph.
      (setf prims-and-sides
        (add-operator-eff-list-to-graph 
          (append del-effect-list add-effect-list) decs graph use-primary))
      (n-push-list (first prims-and-sides) prim-eff-nodes)
      (n-push-list (second prims-and-sides) side-eff-nodes))
    ;; Add nodes for the del-list to the graph. Note that we here only add the
    ;; del-actions of conditional effects that do not have any add-actions.
    (dolist (del-effect-list del-remains)
      (setf prims-and-sides
        (add-operator-eff-list-to-graph 
          del-effect-list decs graph use-primary))
      (n-push-list (first prims-and-sides) prim-eff-nodes)
      (n-push-list (second prims-and-sides) side-eff-nodes))
    (list prim-eff-nodes side-eff-nodes)))


;; Convert all relevant unconditional effects or all relevant actions
;; of a conditional effect of an operator into abstraction nodes and
;; add these nodes to the graph (unless they are already in the
;; graph). If the effect is conditional and has relevant primary
;; actions, add its relevant conditions to the abstraction graph and
;; then add edges from one of the primary actions to the conditions.
;;
;; This function is time-consuming, because it calls the time-consuming
;; function "add-predicate-to-graph".
;;
;; effect-list   list of unconditional effects or contional effect's actions
;;   (from add-list or delete-list of an operator); must be nonempty
;; decs          declarations of the variables in the effect
;; graph         the abstraction graph
;; use-primary   way of using primary effects (nil, :side, or :prim)
;;
;; returned value: 
;; the two-element list, where the first element is primary-effect
;; abs-nodes and the second element is the side-effect abs-nodes; if
;; several effects correspond to the same node, then this node occurs
;; several times in the lists

(defun add-operator-eff-list-to-graph (effect-list decs graph use-primary)
  (declare (type abs-graph graph) (symbol use-primary)
           (special *pspace*) (type problem-space *pspace*))
  (let ((prim-eff-nodes nil)
        (side-eff-nodes nil)
        condition-nodes)
    ;; Add the action nodes to the graph.
    (dolist (effect effect-list)
      (if (relevant-vared-p (effect-cond-effect effect) 
            decs *pspace* :negated-p :either :use-primary use-primary)
         (push (add-predicate-to-graph (effect-cond-effect effect) decs graph)
             prim-eff-nodes)))
    ;; If the effect is conditional and some actions are primary, add 
    ;; condition nodes and edges to them from some primary action.
    (when (and (effect-cond-conditional (car effect-list)) prim-eff-nodes)
      (setf condition-nodes
        (delete-duplicates 
          (add-precond-exp-to-graph 
            (second (effect-cond-conditional (car effect-list))) 
            decs graph use-primary)))
      (dolist (condition-node condition-nodes)
        (add-directed-edge (car prim-eff-nodes) condition-node)))
    (list prim-eff-nodes side-eff-nodes)))
        

;; Convert all relevant nonstatic predicates in a precondition expression 
;; into nodes of an abstraction graph and add these nodes to the graph 
;; (unless they are already in the graph).
;; 
;; This function is time-consuming, because it calls the
;; time-consuming function "add-predicate-to-graph" for every
;; predicate in the precondition expression.
;;
;; precond-exp   precondition expression (to be converted into abs-nodes)
;; decs          declarations of the variables in precond-exp
;; graph         the abstraction graph
;; use-primary   nil, :prim, or :side
;;
;; returned value:
;; the list of abs-nodes corresponding to the predicates of precond-exp;
;; if several predicates correspond to the same node, then this node
;; occurs several times in the list

(defun add-precond-exp-to-graph (precond-exp decs graph use-primary)
  (declare (type abs-graph graph) (symbol use-primary)
           (special *pspace*) (type problem-space *pspace*))
  (case (car precond-exp)
    (user::~
      (add-precond-exp-to-graph (second precond-exp) decs graph use-primary))
    ((user::and user::or)
      (mapcan #'(lambda (sub-exp)
                  (add-precond-exp-to-graph sub-exp decs graph use-primary))
              (cdr precond-exp)))
    ((user::exists user::forall)
      (add-precond-exp-to-graph (third precond-exp)
        (append (second precond-exp) decs) graph use-primary))
    (otherwise  ;; it is a predicate
      (if (and (not (static-predicate-p precond-exp *pspace*))
            (relevant-vared-p precond-exp decs *pspace* 
              :negated-p :either :use-primary use-primary))
        (list (add-predicate-to-graph precond-exp decs graph))))))

;;-------------------------------------------------------------------------

;; Convert all relevant nonstatic preconditions and all relevant
;; effects of an inference rule into nodes of an abstraction graph,
;; add these nodes to the graph (unless they are already in the
;; graph), and add all necessary edges.
;;
;; We pick some precondition or primary effect Precond-or-Prim-Node and 
;; then add edges as follows:
;; - for every precondition or primary-effect Node (including the actions
;;   of conditional primary effects), Precond-or-Prim-Node <-> Node
;; - for every side-effect Node (including the actions of conditional
;;   side effects), Precond-or-Prim-Node -> Node
;;
;; The constraints on the conditions of conditional effects are
;; added in the function "add-inf-rule-eff-list-to-graph".
;;
;; inf-rule   an inference rule
;; graph      the abstraction graph
;; use-primary   nil (do not use), :side (treat rest like side), 
;;                 or :prim (treat rest like primary)

(defun add-inf-rule-to-graph (inf-rule graph use-primary)
  (declare (type rule inf-rule) (type abs-graph graph) (symbol use-primary)
           (special *pspace*) (type problem-space *pspace*))
  (let* ((precond-decs (second (rule-precond-exp inf-rule)))
         (effect-decs (append (second (rule-effects inf-rule)) precond-decs))
         precond-nodes side-eff-nodes prims-and-sides
         precond-and-prim-nodes precond-or-prim-node)
    (when ;; Does relevant preconditions and effects, or relevant
          ;; primary effects?  If not, don't add it to the graph,
          ;; since it does not require any constraints.
          (or (and ;; relevant preconditions
                   (relevant-preconds-p (third (rule-precond-exp inf-rule))
                     precond-decs use-primary)
                 ;; relevant effects
                 (some #'(lambda (effect)
                           (relevant-vared-p (effect-cond-effect effect)
                             effect-decs *pspace* :negated-p :either
                             :use-primary use-primary))
                       (subelts-to-elts (append (rule-add-list inf-rule) 
                                                (rule-del-list inf-rule)))))
              ;; relevant primary effects
              (some #'(lambda (effect)
                        (relevant-vared-p (effect-cond-effect effect)
                          effect-decs *pspace* :negated-p :either
                          :use-primary use-primary))
                    (subelts-to-elts (append (rule-add-list inf-rule) 
                                             (rule-del-list inf-rule)))))
      ;; Add precondition nodes to the graph.
      (setf precond-nodes (delete-duplicates (add-precond-exp-to-graph 
        (third (rule-precond-exp inf-rule)) precond-decs graph use-primary)))
      ;; Add effect nodes to the graph.
      (setf prims-and-sides
        (add-inf-rule-effs-to-graph (rule-add-list inf-rule) 
          (rule-del-list inf-rule) effect-decs graph use-primary))
      (setf precond-and-prim-nodes 
        (nconc precond-nodes (delete-duplicates (first prims-and-sides))))
      (setf side-eff-nodes (delete-duplicates (second prims-and-sides)))
      ;; Pick some precondition or primary effect.
      (setf precond-or-prim-node (car precond-and-prim-nodes))
      ;; Add constraints on the preconditions and primary effects.
      (dolist (node (cdr precond-and-prim-nodes))
        (add-directed-edge precond-or-prim-node node)
        (add-directed-edge node precond-or-prim-node))
      ;; Add constraints on the side effects.
      (dolist (node side-eff-nodes)
        (add-directed-edge precond-or-prim-node node)))))


;; Convert all relevant effects of an inference rule into nodes of an
;; abstraction graph and add these nodes to the graph (unless they are
;; already in the graph).
;;
;; For conditional primary effects, add their conditions to the graph
;; and add edges between actions to conditions.
;;
;; add-list      inference rule's add-list
;; del-list      inference rule's del-list
;; decs          declarations from the precond-exp of the operator
;; graph         the abstraction graph
;; use-primary   way of using primary effects (nil, :side, or :prim)
;;
;; returned value:
;; the two-element list, where the first element is primary-effect
;; abs-nodes and the second element is the side-effect abs-nodes; if
;; several effects correspond to the same node, then this node occurs
;; several times in the lists

(defun add-inf-rule-effs-to-graph (add-list del-list decs graph use-primary)
  (declare (type abs-graph graph) (symbol use-primary))
  (let ((del-remains (copy-list del-list))
        (prim-eff-nodes nil)
        (side-eff-nodes nil)
        prims-and-sides del-effect-list)
    ;; Add the nodes for the add-list to the graph. If a conditional effect
    ;; have both add and del effects, we merge them before calling the
    ;; "add-inf-rule-eff-list-to-graph" function, and remove the corresponding
    ;; effect list from del-remains (which is a copy of del-list). This
    ;; merging prevents a double attemp of adding the effect's conditions.
    (dolist (add-effect-list add-list)
      ;; Find del-actions of the same conditional effect.
      (setf del-effect-list
        (find (effect-cond-conditional (car add-effect-list)) del-remains
         :key #'(lambda (eff-list) (effect-cond-conditional (car eff-list)))))
      ;; Remove them from the "del-remains" list.
      (if del-effect-list
        (del del-effect-list del-remains))
      ;; Add the conditional effect to the graph.
      (setf prims-and-sides
        (add-inf-rule-eff-list-to-graph 
          (append del-effect-list add-effect-list) decs graph use-primary))
      (n-push-list (first prims-and-sides) prim-eff-nodes)
      (n-push-list (second prims-and-sides) side-eff-nodes))
    ;; Add nodes for the del-list to the graph. Note that we here only add the
    ;; del-actions of conditional effects that do not have any add-actions.
    (dolist (del-effect-list del-remains)
      (setf prims-and-sides
        (add-inf-rule-eff-list-to-graph 
          del-effect-list decs graph use-primary))
      (n-push-list (first prims-and-sides) prim-eff-nodes)
      (n-push-list (second prims-and-sides) side-eff-nodes))
    (list prim-eff-nodes side-eff-nodes)))


;; Convert all relevant unconditional effects or all relevant actions
;; of a conditional effect of an inference rule into abstraction nodes
;; and add these nodes to the graph (unless they are already in the
;; graph). If the effect is conditional and has relevant primary
;; actions, add its relevant conditions to the abstraction graph and
;; then add edges from one of the primary actions to the conditions.
;;
;; - If the conditional effect has primary actions, we pick some action 
;;   Effect-Node and then, for every condition Condition-Node, we add the 
;;   edges Effect-Node <-> Node.
;; - If all actions are side, we add an edge from every condition 
;;   to every action.
;;
;; This function is time-consuming, because it calls the time-consuming
;; function "add-predicate-to-graph".
;;
;; effect-list   list of unconditional effects or contional effect's actions
;;   (from add-list or delete-list of an inference rule); must be nonempty
;; decs          declarations of the variables in the effect
;; graph         the abstraction graph
;; use-primary   way of using primary effects (nil, :side, or :prim)
;;
;; returned value: 
;; the two-element list, where the first element is primary-effect
;; abs-nodes and the second element is the side-effect abs-nodes; if
;; several effects correspond to the same node, then this node occurs
;; several times in the lists

(defun add-inf-rule-eff-list-to-graph (effect-list decs graph use-primary)
  (declare (type abs-graph graph) (symbol use-primary)
           (special *pspace*) (type problem-space *pspace*))
  (let ((prim-eff-nodes nil)
        (side-eff-nodes nil)
        condition-nodes)
    ;; Add the action nodes to the graph.
    (dolist (effect effect-list)
      (if (relevant-vared-p (effect-cond-effect effect) 
            decs *pspace* :negated-p :either :use-primary use-primary)
              ;; do not add irrelevant
        (push (add-predicate-to-graph (effect-cond-effect effect) decs graph)
          prim-eff-nodes)))
    ;; If the effect is conditional and has some relevant actions,
    ;; add condition nodes and edges between them and actions.
    (when (and (effect-cond-conditional (car effect-list)) 
               (or prim-eff-nodes side-eff-nodes))
      ;; Add condition nodes.
      (setf condition-nodes
        (delete-duplicates 
          (add-precond-exp-to-graph 
            (second (effect-cond-conditional (car effect-list))) 
            decs graph use-primary)))
      ;; Add edges between them and action nodes.
      (if prim-eff-nodes
        (dolist (condition-node condition-nodes)
          (add-directed-edge (car prim-eff-nodes) condition-node)
          (add-directed-edge condition-node (car prim-eff-nodes)))
        (dolist (condition-node condition-nodes)
          (dolist (effect-node (remove-duplicates side-eff-nodes))
            (add-directed-edge condition-node effect-node)))))
    (list prim-eff-nodes side-eff-nodes)))

;;-------------------------------------------------------------------------

;; Determine whether at least one nonstatic predicate in a precondition
;; expression is relevant. We consider predicates unsigned; that is,
;; if a predicate's negation is relevant, then the predicate is relevant.
;;
;; precond-exp    precondition expression  
;; decs           declarations of the variables in precond-exp
;; use-primary    nil, :prim, or :side
;;
;; returned value: T if at least one nonstatic pred is relevant; nil otherwise

(defun relevant-preconds-p (precond-exp decs use-primary)
  (declare (special *pspace*) (type problem-space *pspace*))
  (case (car precond-exp)
    (user::~
      (relevant-preconds-p (second precond-exp) decs use-primary))
    ((user::and user::or)
      (some #'(lambda (sub-exp)
                (relevant-preconds-p sub-exp decs use-primary))
            (cdr precond-exp)))
    ((user::exists user::forall)
      (relevant-preconds-p (third precond-exp)
        (append (second precond-exp) decs) use-primary))
    (otherwise  ;; it is a predicate
      (unless (static-predicate-p precond-exp *pspace*)
        (relevant-vared-p precond-exp decs *pspace* 
          :negated-p :either :use-primary use-primary)))))

#|
======================================================
Operations on the graph of predicates and components.
======================================================

The abstraction graph is represented as two lists, a list of abs-nodes
(in the "abs-graph-nodes" field) and a list of abs-components (in the
"abs-graph-components" field).

The elements of the "nodes" list are of the type "abs-node;" they
represent typed predicates.

The elements of the "components" list are of the type "abs-component."
Each component includes a list of abs-nodes that belong to this
component and incoming and outcoming edges that connect it to other
components (see the definition of the "abs-components" structure in
the beginning of the file for more details).

Each abs-node belongs to one and only one abs-component. We determine
the graph connectivity (that is, edges) by the "in" and "out" fields
of abs-components. Abs-nodes do not have explicit representation of
incoming and outcoming edges. 

If there is and edge from the component containing Node1 to the
component containing Node2, we assume that there is the edge Node1 ->
Node2. If Node1 and Node2 are in the same component, we assume that
there are the edges Node1 <-> Node2.
|#


;; Convert a predicate into a node and add the node to the abstraction 
;; graph (unless it is already in the graph).
;;
;; This function is time-consuming, because it calls two time-consuming
;; functions "predicate-to-node" (converting a predicate into a node of
;; the abstraction graph) and "add-abs-node" (adding a node to the graph).
;;
;; predicate   predicate to be converted into a node and added to the graph
;;               (e.g. "(on small <pegs>)")
;; decs        declarations of the variable types in the predicate
;;               (e.g. "((<disk> Disk) (<peg> Peg))")
;; graph       the abstraction graph
;; negated-p   T if the predicate is negated; nil otherwise
;;
;; The "negated-p" argument is for Relator, which distinguished between a
;; predicate and its negation. For Alpine, this argument is always nil.
;;
;; returned value:
;; the node corresponding to the predicate;
;;   if the node has already been in the graph, return this old node
;;   otherwise, if the node is added to the graph, return the new node

(defun add-predicate-to-graph (predicate decs graph &key (negated-p nil))
  (declare (type abs-graph graph))
  (add-abs-node (predicate-to-node predicate decs :negated-p negated-p) 
                graph))


;; Convert a predicate, represented as a list (e.g. "(on small <pegs>)"),
;; into a node of the abstraction graph (abs-node). 
;;
;; Note that, if many arguments have complex types (i.e., conjunctions and
;; disjunctions of type), then the type conversion may be time-consuming.
;;
;; predicate   predicate to be converted into a node
;; decs        declarations of the variable types 
;;               (e.g. "((<disk> Disk) (<peg> Peg))")
;; negated-p   T if the predicate is negated; nil otherwise
;;
;; The "negated-p" argument is for Relator, which distinguished between a
;; predicate and its negation. For Alpine, this argument is always nil.
;;
;; returned value: the node

(defun predicate-to-node (predicate decs &key (negated-p nil))
  (make-abs-node 
    :name (if negated-p
            (cons '~ (car predicate)) 
            (car predicate))
    :args (mapcar #'(lambda (arg-name) 
                      (arg-name-to-type arg-name decs))
                  (cdr predicate))))


;; Convert an argument of a predicate into a type or an object.
;;
;; If the type is not simple (i.e., disjunction or conjunction),
;; return the corresponding disjunction with types instead of type names.
;; For example, suppose an argument <object> is of the type 
;; (and (or Disk Peg) (diff <object> <other-object>)). 
;; Then, the function will return (or #<TYPE: disk> #<TYPE: peg>).
;;
;; arg-name    name of an object or variable in the predicate
;; decs        declarations of the variable types
;;
;; returned value:
;;   if arg-name is variable of a finite type, returns its type
;;   if arg-name is variable of an infinite type, returns the top type
;;   if arg-name is name of an explicitly defined object, 
;;     returns the corresponding object.
;;   if arg-name is anything else, concludes that it is an object
;;     of an infinite type and returns the top type

(defun arg-name-to-type (arg-name decs)
  (declare (type atom arg-name) 
           (special *pspace*) (type problem-space *pspace*))
  (cond 
    ((strong-is-var-p arg-name) ;; arg-name is a variable?
      (let ((type-generator (second (assoc arg-name decs))))
        ;; Verify that the type of variable is specified.
        (unless type-generator
          (error "arg-name-to-type: Variable ~
                 `~S' is not in declarations `~S'."
                 arg-name decs))
        ;; Convert the type names into types.
        (alpine-type-names-to-types type-generator)))
    ((is-variable-p arg-name)
       (error "arg-name-to-type: `~S' is neither variable nor object." 
              arg-name))
    (t ;; arg-name is an object name
       (let ((object (and (symbolp arg-name)
                             ;; if not symbol, conversion to object errs
                          (object-name-to-object arg-name *pspace*))))
         ;; If "object" is nil, the arg-name is not an explicitly 
         ;; defined object. We then conclude it is of an infinite type,
         ;; and treat it the way we treat infinite types.
         (if object  ;; explicitly defined object ?
           object ;; return it
           (type-name-to-type :Top-Type *pspace*)))))) ;; return the top type


;; Replace type names in a type generator with types.
;; Also, remove functions from a conjunctive type generator.
;;
;; type-generator   the type generator, such as 
;;                    (and (or Disk Peg) (diff <from-peg> <to-peg>))
;;
;; returned value:
;; the resulting type generator, with types instead of type names

(defun alpine-type-names-to-types (type-generator)
  (cond 
    ((symbolp type-generator)  ;; simple type
      (alpine-type-name-to-type type-generator))
    ((eq (car type-generator) 'user::or)  ;; type disjunction
      (if (= (length type-generator) 2)  ;; one-element disjunction?
        (alpine-type-name-to-type (second type-generator))
        (cons 'user::or 
               (mapcar #'alpine-type-name-to-type (cdr type-generator)))))
    ((eq (car type-generator) 'user::and)  ;; type conjunction
      (alpine-type-names-to-types (second type-generator)))
    (t
      (error "alpine-type-names-to-types: Illegal type `~S'." 
             type-generator))))


;; Convert type name into type; if the type is infinite, return
;; the top type, because objects of an infinite type may also be
;; objects of any other type.
;;
;; type-name   the type name
;;
;; returned value
;; if the name is of the finite type, then returns the corresponding type
;; if the name is of infinite type, then returns the top type

(defun alpine-type-name-to-type (type-name)
  (declare (symbol type-name) 
           (special *pspace*) (type problem-space *pspace*))
  (let ((type-value (type-name-to-type type-name *pspace*)))
    (declare (type type type-value))
    (if (functionp type-value)  ;; infinite type?
      (type-name-to-type :Top-Type *pspace*) 
      type-value)))

;;-------------------------------------------------------------------------

;; Create the empty abstraction graph.
;;
;; name   the name of the graph (and of the abstraction hierarchy)

(defun create-empty-graph (&optional name) 
  (unless (symbolp name)
    (error "create-empty-graph: Graph name `~S' is not a symbol." name))
  (make-abs-graph
    :name name
    :nodes nil
    :components nil))

;; Find a node in the graph identical to the specified node (which may
;; or may not be the same node). The node identity is determined by the
;; predicate name and the types of arguments (see the "same-node-p"
;; function). 
;;
;; A valid graph must not contain identical nodes, so at most one node
;; may be identical to the specified node. 
;;
;; Returned value:
;;   if an identical node is found, return this node; otherwise nil

(defun find-abs-node (node graph)
  (declare (type abs-node node) (type abs-graph graph))
  (find node (abs-graph-nodes graph) :test #'same-node-p))


;; Add a new node to the graph, unless such a node is already in the graph.
;; The node identity is determined by the predicate name and the types
;; of arguments. Also add a connected component, containing this node,
;; to the list of connected components.
;;
;; This function is quite time-consuming because of comparing the
;; node with the other nodes in the graph.
;;
;; node    node to be added
;; graph   the graph of predicates
;;
;; returned values:
;;   if a node identical to "node" is already in the graph (in which 
;;     case "node" is not added, them return this identical node
;;   if "node" is added, then return this added node

(defun add-abs-node (node graph)
  (declare (type abs-node node) (type abs-graph graph))
  (or
    ;; If an identical node is in the graph, return it.
    (find-abs-node node graph)
    ;; Otherwise, add the new node and the corresponding one-node component.
    (let ((component (make-abs-component :nodes (list node))))
      (declare (type abs-component component))
      (push node (abs-graph-nodes graph))
      (setf (abs-node-component node) component)
      (push component (abs-graph-components graph))
      node)))


;; Delete a node from the graph; if the node is not in the graph, err.
;;
;; When this function is used by Margie, the deleted node is added to
;; the ":pasts" slot in the plist of the abstraction graph.
;;
;; The "node" argument is the pointer to the node, not a separate data 
;; structure that represents a node identical to some node in the graph.
;;
;; node    node to be deleted
;; graph   the abstraction graph

(defun del-abs-node (node graph)
  (declare (type abs-node node) (type abs-graph graph))
  (unless (member node (abs-graph-nodes graph))  ;; node in the graph?
    (error "del-abs-node: Node `~S' is not in the graph." node))
  (let ((component (abs-node-component node)))
    (declare (type abs-component component))
    (del node (abs-graph-nodes graph))
    ;; Delete the node from the "sub" fields of its superset-nodes.
    (dolist (superset-node (abs-node-super node))
      (del node (abs-node-sub superset-node)))
    ;; Delete the node from the "super" fields of its sub-nodes. 
    (dolist (subset-node (abs-node-sub node))
      (del node (abs-node-super subset-node)))
    ;; Delete the node from the component.
    (del node (abs-component-nodes component))
    ;; If the function is used by Margie, add the node to the
    ;; ":pasts" slot in the plist of the abstraction graph.
    (when (member :pasts (abs-graph-plist graph))
      (push node (abs-graph-pasts graph)))
    ;; If the component becomes empty, delete it from the graph.
    ;; (In the current implementation, this never happens.)
    (unless (abs-component-nodes component)
      (del-empty-component component graph)
      (setf (abs-node-component node) nil))))


;; Delete an empty component from the graph; if the component is 
;; not in the graph err.
;;
;; In the current implementation, we never delete the last node of 
;; the component; therefore, we never get empty components and
;; this function is never called. It may be useful in the future.
;;
;; component   component to be deleted
;; graph       the abstraction graph

(defun del-empty-component (component graph)
  (declare (type abs-component component) (type abs-graph graph))
  (if (abs-component-nodes component)  
    (error "del-component: Component `~S' is not empty." component))
  (unless (member component (abs-graph-components graph))  
    (error "del-component: Component `~S' is not in the graph." component))
  (del component (abs-graph-components graph))
  ;; Delete the component from the "out" fields of other components.
  (dolist (component-in (abs-component-in component))
    (del component (abs-component-out component-in)))
  ;; Delete the component from the "in" fields of other components.
  (dolist (component-out (abs-component-out component))
    (del component (abs-component-in component-out))))  


;; Checks whether two nodes, of the type "abs-node," represent the same
;; predicate. The nodes are considered identical if they have the same
;; predicate name and list of argument. 
;;
;; Note that, if this function is used by Relator, the node name may 
;; include negation, e.g. (~ . on).
;;
;; node1, node2   nodes to be compared
;;
;; returned value: T if the nodes are identical, nil otherwise

(defun same-node-p (node1 node2)
  (declare (type abs-node node1 node2))
  (let ((args1 (abs-node-args node1))
        (args2 (abs-node-args node2)))
    (and (equal (abs-node-name node1) (abs-node-name node2))  ;; same name
         (cond
           ((eq args1 :any) (eq args2 :any))
             ;; both nodes may have any arguments
           ((integerp args1) (eql args1 args2))
             ;; only argument number is specified, and it is the same
           (t (and (listp args2)  ;; argument types are specifed
                   (= (length args1) (length args2))  ;; same argument number
                   (every #'same-type-p args1 args2)))))))  ;; same types


;; Check whether two types in the node arguments are identical.
;; Every type is an object (object itself, not the object name), a simple 
;; type (type itself, not its name) or a disjunction of simple types, such 
;; as "(or #<TYPE: disk> #<TYPE: peg>)".
;;
;; The function assumes that the elements of every disjunctive types are 
;; sorted in alphabetical order of their names (e.g. "disk" before "peg"),
;; so we can compare two disjunctions element-by-element, without sorting.
;;
;; type1, type2   types to be compared
;;
;; returned values: T if the types are identical, nil otherwise

(defun same-type-p (type1 type2)
  (or (and (prodigy-object-p type1) (prodigy-object-p type2) (eq type1 type2))
        ;; Identical objects.
      (and (type-p type1) (type-p type2) (eq type1 type2))
        ;; Identical simple types.
      (and (listp type1) (listp type2) (every #'eq type1 type2))))
        ;; Identical type disjunctions.


;; Add the directed edge between two nodes of the abstraction graph.
;; If "from-node" and "to-node" are the same, set the "loop-p" 
;; field of the node to T.
;;
;; from-node   the source node of the added edge
;; to-node     the destination node of the added edge
;;
;; returned value:
;; T if new edge is added (i.e. component of from-node is different 
;; from component of to-node and edge was not in the graph); nil otherwise

(defun add-directed-edge (from-node to-node)
  (declare (type abs-node from-node to-node))
  (if (or (eq (abs-node-color from-node) 'maybe)
          (eq (abs-node-color to-node) 'maybe))
    (error "add-directed-edge: Edge from ~S to ~S cannot be created, ~
      because one of them is a `maybe' node." from-node to-node))
  (cond 
    ((eq from-node to-node)  ;; same node?
      (setf (abs-node-loop-p from-node) t)
      nil)
    (t
      (add-directed-edge-between-components
        (abs-node-component from-node) (abs-node-component to-node)))))


;; Add the directed edge between two components in the component graph.
;;
;; from-component   the source node of the added edge
;; to-component     the destination node of the added edge
;;
;; returned value:
;; T if new edge is added (i.e. from-component is different from 
;; to-component and edge was not in graph); nil otherwise

(defun add-directed-edge-between-components (from-component to-component)
  (declare (type abs-component from-component to-component))
  (cond
    ((and (eq from-component to-component)  ;; same component?
          (not (abs-node-loop-p (car (abs-component-nodes to-component)))))
            ;; its node is not looped?
       (format t "~&~S    ~S" (car (abs-component-nodes to-component))
         (abs-node-loop-p (car (abs-component-nodes to-component))))
       (setf (abs-node-loop-p (car (abs-component-nodes to-component))) t))
    ((and (not (eq from-component to-component))  ;; diffirent components?
          (not (member to-component (abs-component-out from-component))))
            ;; edge is not yet in the graph?
      (push to-component (abs-component-out from-component))
      (push from-component (abs-component-in to-component))
      t)))

;;-------------------------------------------------------------------------

;; Given the initial abstraction graph, generate the final graph,
;; whose components corresponds to the abstraction levels.
;;
;; The initial graph was built by converting preconditions and effects
;; of operators and inference rules into abstraction nodes and imposing the 
;; corresponding constraints (directed edges).
;;
;; We now combine strongly connected components of the initial graph,
;; taking into account the subset relationship between its nodes. If
;; the graph has more than one level, we simplify the node
;; description, topologically sort the nodes, in the "nodes" field of
;; the graph, on the subset realtionship, and then sort topologically
;; sort the components.
;;
;; graph   the initial abstraction graph
;;
;; returned value: T if the graph has more than one component; nil otherwise

(defun final-graph (graph)
  (declare (type abs-graph graph))
  ;; Determine which nodes intersect and mark all subset relations.
  (find-subsets-and-intersections graph)
  ;; Combine components and propagate edges to subset nodes.
  ;; Repeat until the propagation does not add new edges.
  (strongly-connected-components graph)
  (do () ((not (edges-for-subsets graph)))
    (strongly-connected-components graph))
  ;; Topologically sort the nodes on the subset relationship and the
  ;; components on the graph's edges.
  (unless (singleton-p (abs-graph-components graph))
    (simplify-graph graph)
    (alpine-subset-sort graph)
    (topological-sort graph)
    t))
  

;; Finding strongly connected components in the graph of components.
;; After finding strongly connected components, they are "collapsed;"
;; that is, every collection of components that form a strongly 
;; connected component is replaced with a single component.
;;
;; It is a classical algorithm for finding strongly connected components,
;; using two DFS on the graph. The procedure is closely follows the
;; pseudocode in Chapters 23.3 and 23.5 of "Introduction to Algorithms" by 
;; Cormen, Leiserson, and Rivest. The algorithm is based on complex theory. 
;; If you are not familiar with it, reading the code may be difficult.
;;
;; Note that we operate with the graph of components (stored in the
;; "components" field of "graph"). We sometimes call the elements of
;; this graph "nodes" in the following explanations (but not in the code).
;; We look of strongly connected components in the graph of components;
;; we will call these strongly connected components "super-components."
;;
;; graph   the abstraction graph

(defun strongly-connected-components (graph)
  (declare (type abs-graph graph))
  ;; Find strongly connected components.
  (first-dfs graph)
  (second-dfs graph)
  ;; Mark every node that is in a loop.
  (mark-loops graph))


;; The purpose of the first DFS is to determine the order of "closing"
;; nodes in the component graph. The node is considered closed after 
;; we have visited all its children. When closing a node, we set its 
;; "color" field to 'black. The *time* variable determines the number 
;; of already closed nodes; we increment each time when we close a node. 
;; When closing a node, we assign the current value of *time* to its 
;; "time" field.

(defun first-dfs (graph)
  (declare (type abs-graph graph))
  (let ((*time* 0))
    (declare (type integer *time*) (special *time*))
    (dolist (component (abs-graph-components graph))
      (setf (abs-component-color component) 'white))
    (dolist (component (abs-graph-components graph))
      (if (eq (abs-component-color component) 'white)
        (first-dfs-visit component)))))


;; Recursively visit all successors of the current node in the
;; component graph; then "close" the node (that is, sets its "color"
;; to 'black) and store the closing time in the "time" field.

(defun first-dfs-visit (component)
  (declare (type abs-component component)    
           (special *time*) (type integer *time*))
  (setf (abs-component-color component) 'gray)
  (dolist (child (abs-component-out component))
    (if (eq (abs-component-color child) 'white)
      (first-dfs-visit child)))
  (setf (abs-component-color component) 'black)
  (setf (abs-component-time component) (incf *time*)))


;; The purpose of the second DFS is to collapse nodes in the component
;; graph that belong to the same strongly connected component.
;; It begins the exploration of a strongly connected component from
;; the node with the latest closing time. The strongly connected 
;; component includes all unvisited parents of this node and nothing else.
;;
;; The algorithm collects all nodes of the same strongly connected component
;; in the "nodes" field of the variable *super-component*, of the type 
;; abs-component. The algorithm makes a list of all strongly connected
;; components (super-components); this list is called "super-components."
;; Then, for every super-component, the algorithm collapses all its
;; components into a single component.

(defun second-dfs (graph)
  (declare (type abs-graph graph))
  (let ((components 
           (sort (abs-graph-components graph) #'> :key #'abs-component-time))
             ;; Sort all nodes of the component graph in the decreasing
             ;; order of closing times. Note that the value of the
             ;; "components" field of "graph" is destroyed.
        (super-components nil))
    ;; Set "colors" of all nodes in the component graph to 'white.
    (dolist (component components)
      (setf (abs-component-color component) 'white))
    ;; For every strongly connected component, collect all its nodes,
    ;; starting from the node with latest closing time, in the
    ;; "nodes" list of "super-component".
    (dolist (component components)
      (when (eq (abs-component-color component) 'white)
        (let ((*super-component* 
                 (make-abs-component :nodes nil)))
          (declare (type abs-component *super-component*)
                   (special *super-component*))
;;D          (setf user::*graph* graph) ;;D
          (second-dfs-visit component)
          (push *super-component* super-components))))
    ;; For every "super-component," collapse all its elements,
    ;; which are components, into a singly component.
    (collapse-within-super-components super-components)
    ;; The resulting components form the new component graph.
    (setf (abs-graph-components graph) super-components)))


;; Recursively visit all ancestors of the current node in the
;; component graph and collect them in the "nodes" field
;; *super-component*. Then "close" the node (set "color" to 'black).

(defun second-dfs-visit (component)
  (declare (type abs-component component)    
           (special *super-component*) 
           (type abs-component *super-component*))
  (setf (abs-component-color component) 'gray)
  (dolist (parent (abs-component-in component))
;;D    (setf user::*comp* component) ;;D
;;D    (format t "~%~S  ~S" component parent) ;;D
    (if (eq (abs-component-color parent) 'white)
      (second-dfs-visit parent)))
  (setf (abs-component-color component) 'black)
  (push component (abs-component-nodes *super-component*))
  (setf (abs-component-super-component component) *super-component*))


;; For every super-component, collapse its components into a single 
;; component. That is, initially each super-component is a list of
;; components, whereas after the execution each super-component is
;; a list of nodes.
;;
;; The "in" and "out" fields of super-components are set accordingly:
;; there is an edge from Super-Component1 to Super-Component2 if and
;; only if there is an edge from some component in Super-Component1 
;; to some component in Super-Component2.
;;
;; super-components   list of all super-components

(defun collapse-within-super-components (super-components)
  (dolist (super-component super-components)
    (setf (abs-component-in super-component) nil)
    (setf (abs-component-out super-component) nil)
    (dolist (component (abs-component-nodes super-component))
      (n-push-list (mapcar #'abs-component-super-component 
                           (abs-component-out component))
                   (abs-component-out super-component))
      (n-push-list (mapcar #'abs-component-super-component 
                           (abs-component-in component))
                   (abs-component-in super-component)))
    (setf (abs-component-out super-component)
      (delete super-component
        (delete-duplicates (abs-component-out super-component))))
    (setf (abs-component-in super-component)
      (delete super-component
        (del-duplicates (abs-component-in super-component)))))
  (dolist (super-component super-components)
    (setf (abs-component-nodes super-component)
      (mapcan #'abs-component-nodes (abs-component-nodes super-component)))
        ;; We destroy the lists of nodes in the old components.
    (dolist (node (abs-component-nodes super-component))
      (setf (abs-node-component node) super-component))))


;; After combining strongly connected components, we may detect some
;; nodes in loops (i.e. nodes that have a path from them to themselves).
;; We find such nodes and set their "loop-p" fields to T.
;;
;; A node is in a loop if either there is an edge from a node to itself, in 
;; which case "loop-p" has already been set to T when adding the node to the 
;; graph, or the node belongs to a component with more than one node.

(defun mark-loops (graph)
  (declare (type abs-graph graph))
  (dolist (component (abs-graph-components graph))
    (unless (singleton-p (abs-component-nodes component))
              ;; Only one node in the component?
      (dolist (node (abs-component-nodes component))
        (setf (abs-node-loop-p node) t)))))

;;-------------------------------------------------------------------------

;; Simplify the description of the abstraction graph using the
;; following two operations:
;; - if, for some predicate name, all nodes with this name are in the
;;   same component, replace them with a single node, that has the
;;   same name and any arguments
;; - if not all nodes with this name are in the same component, but,
;;   for some number of arguments, all nodes with this name and this
;;   number of arguments are in the same component component, replace
;;   them with a single node, which specifies only the name and the
;;   number of arguments
;; 
;; graph   abstraction graph
;; 
;; local variable:
;; *preds*   list of Pred structures, where each structure is for some node; 
;;             we use this list to arrange the nodes by their names and the
;;             number of arguments; this variable is dynamically scoped and
;;             used in the "get-pred" function

(defun simplify-graph (graph)
  (declare (type abs-graph graph))
  (let ((*preds* nil))
    (declare (special *preds*))
    ;; Arrange all nodes of the abstraction graph by
    ;; their names and the number of arguments.
    (dolist (node (abs-graph-nodes graph))
      (add-node-to-preds node))
    ;; Replace groups of nodes with more general nodes.
    (dolist (pred *preds*)
      (component-for-pred pred)
      (maybe-substitute pred graph))))


;; Add an abstraction node to the list of nodes arranged by names and
;; argument numbers.
;;
;; This "list of nodes" is represented as a list of Pred structures,
;; where each structure is for some node name. The nodes within the
;; Pred structure are arranged by the argument number, where each
;; argument number is represented by an Argnum structure. (See the
;; comments to the Pred and Argnum structures in the "Alpine/struct"
;; file for details).
;;
;; node    node to be added

(defun add-node-to-preds (node)
  (declare (type abs-node node))
  (let* ((pred (get-pred (abs-node-name node)))
         (argnum (get-argnum (arg-number node) pred)))
    (push node (argnum-nodes argnum))))


;; Find a Pred structure for a specified node name in the list of
;; Preds. If there is no strucutre with this name, create it and add
;; to the list.
;;
;; name    node (predicate) name
;;
;; returned value: the Pred structure of the specified name

(defun get-pred (name)
  (declare (special *preds*))
  (or (find name *preds* :test #'equal :key #'pred-name)
      (car (push (make-pred :name name :argnums nil) *preds*))))


;; Find an Argnum structure for a specific argument number in the
;; list of Argnums in a Pred structure. If there is no structure
;; for this number of arguments, create it and add to the list.
;;
;; num    number of arguments
;; pred   the Pred strucutre
;;
;; returned value: the Argnum structure for the specified argument number

(defun get-argnum (num pred)
  (declare (type integer num) (type pred pred))
  (or (find num (pred-argnums pred) :key #'argnum-num :test #'=)
      (car (push (make-argnum :num num :nodes nil) (pred-argnums pred)))))


;; Determine if all nodes with some name, which are stored in a
;; Pred structure, are in the same component of the abs-graph.
;; If they are, store their component in the "component" field
;; of Pred; if not, store nil there.
;;
;; The function also stores the component for each Argnum in Pred, by
;; calling "component-for-argnum". The function assumes that the list
;; of Argnums in Pred is not empty.
;;
;; pred   Pred structure, containing all nodes with some name
;; 
;; returned value: 
;; the component, if all nodes are in the same component; nil otherwise

(defun component-for-pred (pred)
  (declare (type pred pred))
  (let* ((argnums (pred-argnums pred))
         (component (component-for-argnum (car argnums))))
    (dolist (argnum (cdr argnums))
      (unless (eq component (component-for-argnum argnum))
        (setf component nil)))
    (setf (pred-component pred) component)))


;; Determine if all nodes with some name and argument number, which
;; are stored in an Argnum structure, are in the same component of the
;; abs-graph. If they are, store their component in the "component"
;; field of Argnum; if not, store nil there.
;;
;; The function assumes that the list of nodes in Argnum is not empty.
;;
;; argnum   Argnum structure, containing all nodes with some name
;;            and argument number
;; 
;; returned value: 
;; the component, if all nodes are in the same component; nil otherwise

(defun component-for-argnum (argnum)
  (declare (type argnum argnum))
  (if (same-p (argnum-nodes argnum) :test #'eq :key #'abs-node-component)
    (setf (argnum-component argnum)
          (abs-node-component (car (argnum-nodes argnum))))
    (setf (argnum-component argnum) nil)))


;; If all nodes with some name, which are stored in a Pred structure,
;; are in the same component, then replace them with a single node.
;; If they are not, but the nodes in some Argnum structures are in
;; the same component, then, for every such Argnum, replace its 
;; nodes with a single node.
;;
;; pred    Pred structure, containing all nodes with some name
;; graph   abstraction graph

(defun maybe-substitute (pred graph)
  (declare (type pred pred) (type abs-graph graph))
  (if (pred-component pred)  ;; all nodes of Pred are in the same component?
    (substitute-pred pred graph)
    (dolist (argnum (pred-argnums pred))
      (if (and (not (eq (argnum-num argnum) -1))
                 ;; Agrnum does not store the node with any number of args?
               (argnum-component argnum))
                 ;; all nodes of Argnum are in the same component?
        (substitute-argnum argnum graph)))))


;; Substitute all nodes with the same name by a single node.
;;
;; This operation is valid only if all these nodes are in the same component.
;; We use the fact that the resulting node does not have subset and superset
;; nodes, so we do not have to take care of subset/superset links.
;;
;; pred    Pred structure, containing all nodes with some name
;; graph   abstraction graph

(defun substitute-pred (pred graph)
  (declare (type pred pred) (type abs-graph graph))
  (let* ((nodes (mapcan #'argnum-nodes (pred-argnums pred)))
         (first-node (car nodes)))
    (declare (type abs-node first-node))
    (if (singleton-p nodes)
      ;; If there is only one node with this name, just 
      ;; replace its arguments types with ":any".
      (setf (abs-node-args first-node) :any)
      ;; If there are several nodes, remove them all and add a
      ;; single node, with this name and any arguments. Note 
      ;; that his node is looped, since all original nodes with
      ;; are in the same component, which make them all looped.
      (let* ((component (pred-component pred))
             (anyarg-node
               (make-abs-node :name (pred-name pred) :args :any :loop-p t
                 :sub nil :super nil :component component)))
        (declare (type abs-component component) (type abs-node anyarg-node))
        ;; Replace nodes in the "nodes" list of the abs-graph.
        (set-dif (abs-graph-nodes graph) nodes)
        (push anyarg-node (abs-graph-nodes graph))
        ;; Replace nodes in the "nodes" list of the component.
        (set-dif (abs-component-nodes component) nodes)
        (push anyarg-node (abs-component-nodes component))))))


;; Substitute all nodes with the same name and argument number by a
;; single node.
;;
;; This operation is valid only if all these nodes are in the same component.
;; We use the fact that the resulting node does not have subset nodes, and
;; its only possible supreset is the node with this name and ":any" arguments,
;; so updating subset/superset links is simple.
;;
;; argnum   Argnum structure, containing all nodes with some name
;;            and argument number
;; graph    abstraction graph

(defun substitute-argnum (argnum graph)
  (declare (type argnum argnum) (type abs-graph graph))
  (let* ((nodes (argnum-nodes argnum))
         (first-node (car nodes)))
    (declare (type abs-node first-node))
    (if (singleton-p nodes)
      ;; If there is only one node with this name and argument number,
      ;; just replace its arguments types with their number.
      (setf (abs-node-args first-node) (arg-number first-node))
      ;; If there are several nodes, remove them all and add a single
      ;; node. Note that his node is looped, since all original nodes
      ;; with are in the same component, which make them all looped.
      (let* ((name (abs-node-name first-node))
             (component (argnum-component argnum))
             (anyarg-node 
               (find-if #'(lambda (node)
                            (and (equal (abs-node-name node) name) 
                                 (equal (abs-node-args node) :any)))
                 (abs-graph-nodes graph)))
                   ;; Node with the same name and ":any" arguments.
             (argnum-node
               (make-abs-node :name name :args (argnum-num argnum) :loop-p t
                 :sub nil :super anyarg-node :component component)))
                   ;; Node to replace original nodes.
        (declare (type abs-component component) (type abs-node argnum-node))
        ;; Replace nodes in the "nodes" list of the abs-graph.
        (set-dif (abs-graph-nodes graph) nodes)
        (push argnum-node (abs-graph-nodes graph))
        ;; Replace nodes in the "nodes" list of the component.
        (set-dif (abs-component-nodes component) nodes)
        (push argnum-node (abs-component-nodes component))
        ;; Replace nodes in the "subset" list of the node with the
        ;; same name and ":any" arguments.
        (when anyarg-node
          (set-dif (abs-node-sub anyarg-node) nodes)
          (push (argnum-nodes argnum) (abs-node-sub anyarg-node)))))))

;;-------------------------------------------------------------------------

;; Topological sort of the graph of components.
;;
;; It is a classical algorithm for topological sort, using DFS. The 
;; procedure closely follows the pseudocode in Chapters 23.3 and 23.4
;; of "Introduction to Algorithms" by Cormen, Leiserson, and Rivest.
;;
;; The DFS search adds a component to in the beginning of the current 
;; topologically sorted list, *components*, after the search visited
;; all successors of the node and added them to the list. 
;;
;; The topological sort is stable. In particular, if graph is already
;; topologically sorted, the order does not change.
;;
;; graph   the abstraction graph

(defun topological-sort (graph)
  (declare (type abs-graph graph))
  (let ((*components* nil))
    (declare (special *components*))
    (dolist (component (abs-graph-components graph))
      (setf (abs-component-color component) 'white))
    (dolist (component (abs-graph-components graph))
      (if (eq (abs-component-color component) 'white)
        (topological-sort-visit component)))
    (setf (abs-graph-components graph) *components*)))


;; Recursively visit all successors of the current component; then
;; set the "color" of the component to 'black and add the component
;; to the beginning of the sotred list.

(defun topological-sort-visit (component)
  (declare (type abs-component component) (special *components*))
  (setf (abs-component-color component) 'gray)
  (dolist (child (abs-component-out component))
    (if (eq (abs-component-color child) 'white)
      (topological-sort-visit child)))
  (setf (abs-component-color component) 'black)
  (push component *components*))


#|
=============================================================
 Operations related to the subset relationship between nodes.
=============================================================

We consider Node2 a "subset" of Node1 if every instantiated literal
that corresponds to the typed predicate in Node1 also corresponds to
the typed predicate in Node2. In other words, the set of instantiated
literals that correspond to Node2 is a subset of literals that
correspond to Node1. For example, if Small is a subtype of Disk, then
[on Small Peg] is a subset of [on Disk Peg].

The "abs-node-sub" field of a node is the list of the nodes that are
its subsets. The "abs-node-super" field of a node is a list of its
supersets in the graph.

If neither of Node1 and Node2 is a subset of the other, but some
instantiated literal corresponds to both nodes, we say that Node1 and
Node2 "intersect." For example, if Small is a subtype of Disk and peg-a
is an object of type Peg, then [on Small Peg] and [on Disk 'peg-a]
intersect.

If two nodes intersect, they must be on the same level; we therefore
establish the both-ways edge between them. If Node2 is a subset of
Node1, they may be on different levels. In this situation, Node2
inherits all incoming and outcoming edges of Node1.
|#


;; Find all subset-relations between nodes and mark them by subset-links.
;; Also, find all pairs of node intersections. Whenever Node1 and Node2 
;; intersect and neither of them is a subset of the other, connect them by 
;; the edges Node1 <-> Node2, since these nodes must be on the same level
;; of the hierarchy.
;;
;; Note that the graph of subset-links becomes transitively-closed:
;; if Node3 is a subset of Node2 and Node2 is a subset of Node1,
;; there will be a subset-link from Node1 to Node3.
;;
;; This function makes quadratic number (to be more precise, |graph|^2) 
;; of subset checks and intersection checks on nodes, which may be
;; time-consuming operations. The function is, therefore, time-consuming.
;; We use it only once during generating a hierarchy.
;;
;; graph   the graph, represented as a list of the abstract nodes
;;
;; local variables (parameters of the main loop):
;; node1         a node from the "graph" list (goes through all nodes)
;; other-nodes   the list of nodes following node1 in the "graph" list

(defun find-subsets-and-intersections (graph)
  (declare (type abs-graph graph))
  (do ((node1 (car (abs-graph-nodes graph)) (car other-nodes))
       (other-nodes (cdr (abs-graph-nodes graph)) (cdr other-nodes)))
      ((null other-nodes))
    (dolist (node2 other-nodes)
      (cond 
        ((node-subset-p node1 node2)
          (add-subset-link node1 node2))
        ((node-subset-p node2 node1)
          (add-subset-link node2 node1))
        ((node-intersection-p node1 node2)
          (add-directed-edge node1 node2)
          (add-directed-edge node2 node1))))))
  

;; Determine whether node1 is a subset of node2 (see the comment above
;; for the definition of the node subset relation). 
;;
;; The function may be time-consuming when the node1 is indeed a subset
;; of node2, since it compares each of the arguments for type subset, which,
;; in turn, involves the element-by-element comparison of type disjunctions.
;;
;; Note that, if this function is used by Relator, the node name may 
;; include negation, e.g. (~ . on).
;;
;; node1, node2   the abs-nodes to be checked for subset relation
;;
;; returned value: T if node1 is a subset of node2; nil otherwise

(defun node-subset-p (node1 node2)
  (declare (type abs-node node1 node2))
  (let ((args1 (abs-node-args node1))
        (args2 (abs-node-args node2)))
    (and (equal (abs-node-name node1) (abs-node-name node2))  ;; same name
         (cond
           ((eq args2 :any) t)  ;; node2 may have any arguments
           ((integerp args2)  ;; only argument number is specified for node2
              (eql (arg-number node1) args2))  ;; node2 have same arg number
           (t (and (listp args1)  ;; argument types are specifed
                   (= (length args1) (length args2))  ;; same argument number
                   (every #'arg-subset-p args1 args2)))))))  ;; subset types


;; Determine whether arg1 is a subset of arg2.
;; Each argument must be an object, a finite simple type,
;;   or a disjunction of finite simple types.
;; The definition of the subset is as follows:
;; - Object is a subset of another object only if the objects are identical.
;; - Object is a subset of a simple type if the object is of this type.
;; - Type is never a subset of object.
;; - Type is subset of another type if the types are identical or the
;;   the first type is a successor of the second.
;; - Object or type is a subset of disjunction if it is a subset of some
;;   element of the disjunction.
;; - Disjunction is a subset of another disjunction if every element of
;;   the first disjunction is a subset of the second.
;;
;; arg1, arg2   arguments of predicates from abs-nodes

(defun arg-subset-p (arg1 arg2)
  (cond
    ((prodigy-object-p arg1)  ;; object?
      (cond 
        ((prodigy-object-p arg2)
          (eq arg1 arg2))
        ((type-p arg2)
          (object-type-p arg1 arg2))
        (t  ;; otherwise, arg2 is a type disjunction
          (member arg1 (cdr arg2) :test #'object-type-p))))
    ((type-p arg1)  ;; simple type?
      (cond 
        ((prodigy-object-p arg2)
          nil)
        ((type-p arg2)
          (or (eq arg1 arg2) (child-type-p arg1 arg2)))
        (t  ;; otherwise, arg2 is a type disjunction
          (some #'(lambda (simple-type) 
                          (or (eq arg1 simple-type) 
                              (child-type-p arg1 simple-type)))
                (cdr arg2)))))
    (t  ;; otherwise, arg1 is a type disjunction
      (cond 
        ((prodigy-object-p arg2)
          nil)
        ((type-p arg2)
          (every #'(lambda (simple-type)
                     (or (eq simple-type arg2)
                         (child-type-p simple-type arg2)))
                 (cdr arg1)))
        (t  ;; otherwise, arg2 is a type disjunction
          (every #'(lambda (simple-type)
                     (arg-subset-p simple-type arg2))
                 (cdr arg1)))))))


;; Determine whether the two nodes intersect (see the comment above
;; for the definition of the node intersection). If one of the nodes
;; is a subset of the other, they are considered intersecting.
;;
;; The function may be time-consuming when the nodes do indeed intersect,
;; since it calls the time consuming "args-intersection-p" function.
;;
;; Note that, if this function is used by Relator, the node name may 
;; include negation, e.g. (~ . on).
;;
;; node1, node2   the abs-nodes to be checked for intersection

(defun node-intersection-p (node1 node2)
  (declare (type abs-node node1 node2))
  (and (equal (abs-node-name node1) (abs-node-name node2))  ;; same name
       (args-intersection-p (abs-node-args node1) (abs-node-args node2))))


;; Determine whether two typed-argument lists intersect.
;;
;; The function may be time-consuming, since it compares each of the
;; arguments for type intersection, which, in turn, involves the
;; element-by-element comparison of type disjunctions.
;;
;; args1, args2   the typed-argument lists to be checked for intersection

(defun args-intersection-p (args1 args2)
  (cond
    ((or (eq args1 :any) (eq args2 :any)) t) ;; one of nodes may have any args
    ((integerp args1)  ;; only arg number is specified for the first node
       (eq args1 (if (integerp args2) args2 (length args2))))
    ((integerp args2)  ;; only arg number is specified for the second node
       (eq args2 (if (integerp args1) args1 (length args1))))
    (t (and (= (length args1) (length args2))  ;; same arg number
            (every #'arg-intersection-p args1 args2)))))
              ;; the pairs of corresponding argument types intersect  


;; Determine whether two arguments intersects.
;; Each argument must be an object, a finite simple type,
;;   or a disjunction of finite simple types.
;; The definition of the intersection is as follows:
;; - Two objects intersect only if they are identical.
;; - An object and a simple type intersect if the object is of this type.
;; - Two simple types intersect if one of them is a subtype of the other.
;; - An object or a simple type intersects with a disjunction if it 
;;   intersects with some element of the disjunction.
;; - Two disjunctions intersect if some element of the first disjunction
;;   intersects with some element of the second disjunction.
;;
;; arg1, arg2   arguments of predicates from abs-nodes

(defun arg-intersection-p (arg1 arg2)
  (cond
    ((prodigy-object-p arg1)  ;; object?
      (cond 
        ((prodigy-object-p arg2)
          (eq arg1 arg2))
        ((type-p arg2)
          (object-type-p arg1 arg2))
        (t  ;; otherwise, arg2 is a type disjunction
          (member arg1 (cdr arg2) :test #'object-type-p))))
    ((type-p arg1)  ;; simple type?
      (cond 
        ((prodigy-object-p arg2)
          (object-type-p arg2 arg1))
        ((type-p arg2)
          (simple-type-intersection-p arg1 arg2))
        (t  ;; otherwise, arg2 is a type disjunction
          (member arg1 (cdr arg2) :test #'simple-type-intersection-p))))
    (t  ;; otherwise, arg1 is a type disjunction
      (cond 
        ((prodigy-object-p arg2)
          (member arg2 (cdr arg1) :test #'object-type-p))
        ((type-p arg2)
          (member arg2 (cdr arg1) :test #'simple-type-intersection-p))
        (t  ;; otherwise, arg2 is a type disjunction
          (member arg2 (cdr arg1) :test #'arg-intersection-p))))))


;; Determine whether two simple types intersect.
;; Type1 and type2 must be simple types.
;; The types intersect they are identical or one of them 
;;   is a parent of the other.

(defun simple-type-intersection-p (type1 type2)
  (declare (type type type1 type2))
  (or (eq type1 type2)
      (child-type-p type1 type2)
      (child-type-p type2 type1)))


;; Determine the number of arguments of the predicate in an abs-node.
;; If the predicate may have any number of arguments, return -1.

(defun arg-number (node)
  (declare (type abs-node node))
  (let ((args (abs-node-args node)))
    (cond
      ((eq args :any) -1)
      ((integerp args) args)
      ((listp args) (length args)))))


;; Add a subset-link between two nodes.
;; The link means that the second node is a subset of the first (see the 
;; extended comment above for the definition of the node subset relation).
;;
;; subset-node     node to mark as a subset of superset-node
;; superset-node   the superset node

(defun add-subset-link (subset-node superset-node)
  (declare (type abs-node subset-node superset-node))
  (unless (or (eq subset-node superset-node)  ;; same node?
              (member subset-node (abs-node-sub superset-node)))
                ;; link is already in the graph?
    (push subset-node (abs-node-sub superset-node))
    (push superset-node (abs-node-super subset-node))))

;;-------------------------------------------------------------------------

;; Check all subset-links and add the corresponding edges as necessary.
;;
;; - If a Superset-Node and its Subset-Node is in the same component, 
;;   remove the subset-node from the graph (since its presence on the
;;   same level of the abstraction hierarchy as superset-node does
;;   not add any information).
;;
;; - If Superset-Node is in the loop, then all its subsets must be 
;;   the same component; therefore, for each Subset-Node that is a subset
;;   of Superset-Node, add the edges Subset-Node <-> Superset-Node.
;;
;; - If Superset-Node is not in the loop, then all of its subset-nodes
;;   inherit all its links. (See comment to the "store-inherited-edges"
;;   procedure for the definition of link inheritance.)
;;
;; graph   the abstraction graph
;;
;; local variables
;; new-edges-p     initially nil; becomes T after some new edge is added
;; superset-node   a node whose subsets we consider (loops through all
;;                 nodes of the graph)
;; subset-node     a subset of "superset-node" (loops through all
;;                 subset-nodes of "superset-node")
;;
;; returned value:
;; T if at least one new edge is added; nil otherwise

(defun edges-for-subsets (graph)
  (declare (type abs-graph graph))
  (let ((new-edges-p nil))
    ;; Initialize the lists of inherited edges to the empty lists.
    (dolist (component (abs-graph-components graph))
      (setf (abs-component-inherited-out component) nil)
      (setf (abs-component-inherited-in component) nil))
    ;; Loop through every pair of superset-node and subset-node.
    (dolist (superset-node (abs-graph-nodes graph))
      (dolist (subset-node (abs-node-sub superset-node))
        (cond 
          ;; If the superset-node and the subset-node are in the same
          ;; component, then remove the subset-node from the graph.
          ((eq (abs-node-component superset-node)
               (abs-node-component subset-node)) 
            (del-abs-node subset-node graph))
          ;; If the superset-node is in the loop, then add the
          ;; edges subset-node <-> superset-node.
          ((abs-node-loop-p superset-node)  
            (if (add-directed-edge superset-node subset-node)
              (setf new-edges-p t))
            (if (add-directed-edge subset-node superset-node)
              (setf new-edges-p t)))
          ;; Otherwise, the subset-node inherits all the edges of the 
          ;; superset-node; store the inherited edges in the "inherited-out" 
          ;; and "inherited-in" fields of the subset-node's component.
          (t
            (store-inherited-edges subset-node superset-node)))))
    ;; Add all inherited edges to the graph of components.
    (or (add-inherited-edges graph) new-edges-p)))


;; Taker-Node inherits all incoming and outcoming edges of Giver-Node.
;; That is, for each Node with edge Note -> Giver-Node, we add the
;; edge Note -> Taker-Node, and for each Node with edge Giver-Node -> Node,
;; we add the edge Taker-Node -> Node.
;;
;; We store all inherited edges in the "inherited-out" (for outcoming 
;; edges) and "inherited-in" (for incoming edges) of a component.
;; We may store the same inherited edge several times. We will
;; remove duplicates and add the inherited edges to the actual
;; component graph after storing all of them, by the 
;; "add-inherited-edges" procedure.
;;
;; I am making handling of inherited nodes more complex then
;; necessary, in hope to make it efficient.
;;
;; taker-node   node of abstraction graph that inherits edges
;; giver-node   node from which "taker-node" inherits edges

(defun store-inherited-edges (taker-node giver-node)
  (declare (type abs-node taker-node giver-node))
  (let ((taker-component (abs-node-component taker-node))
        (giver-component (abs-node-component giver-node)))
    (declare (type abs-component taker-component giver-component))
    (unless (eq taker-component giver-component)  ;; same node?
      ;; Store all outcoming inherited edges.
      (push-list (abs-component-out giver-component)
        (abs-component-inherited-out taker-component))
      (dolist (component (abs-component-out giver-component))
        (push taker-component (abs-component-inherited-in component)))
      ;; Store all incoming inherited edges.
      (push-list (abs-component-in giver-component)
        (abs-component-inherited-in taker-component))
      (dolist (component (abs-component-in giver-component))
        (push taker-component (abs-component-inherited-out component))))))


;; Add all inherited edges to the component graph (unless these
;; edges have already been in the graph before inheriting.)
;; We assume that all inherited edges are stored in the "inherit-out" 
;; and "inherit-in" fields of components.
;;
;; graph   abstraction graph
;;
;; local variables:
;; new-edges-p   initially nil; becomes T after some new edge is added
;;
;; returned value:
;; T if at least one new edge is added; nil otherwise

(defun add-inherited-edges (graph)
  (declare (type abs-graph graph))
  (let ((new-edges-p nil))
    (dolist (component (abs-graph-components graph))
      (let ((inherited-out 
              (delete-duplicates (abs-component-inherited-out component))))
        (setf inherited-out
          (nset-difference inherited-out (abs-component-out component)))
        (if (and inherited-out (add-edges-out component inherited-out))
          (setf new-edges-p t)))
      (let ((inherited-in 
              (delete-duplicates (abs-component-inherited-in component))))
        (setf inherited-in
          (nset-difference inherited-in (abs-component-in component)))
        (if (and inherited-in (add-edges-in component inherited-in))
          (setf new-edges-p t))))
    new-edges-p))


;; Add directed edge from "component" to each component in the 
;; "components-out" list (unless these edges are already in the graph).
;;
;; component        component in the component graph
;; components-out   list of components 
;;
;; local variable:
;; new-edges-p   initially nil; becomes T after some new edge is added
;;
;; returned value:
;; T if at least one new edge is added; nil otherwise

(defun add-edges-out (component components-out)
  (declare (type abs-component component))
  (let ((new-edges-p nil))
    (dolist (component-out components-out)
      (if (add-directed-edge-between-components component component-out)
        (setf new-edges-p t)))
    new-edges-p))


;; Add directed edge from each component in the "components-out" list 
;; to "component" (unless these edges are already in the graph).
;;
;; component        component in the component graph
;; components-in    list of components 
;;
;; local variable:
;; new-edges-p   initially nil; becomes T after some new edge is added
;;
;; returned value:
;; T if at least one new edge is added; nil otherwise

(defun add-edges-in (component components-in)
  (declare (type abs-component component))
  (let ((new-edges-p nil))
    (dolist (component-in components-in)
      (if (add-directed-edge-between-components component-in component)
        (setf new-edges-p t)))
    new-edges-p))

;; -------------------------------------------------------------------------

;; Topologically sort the nodes of the abstraction graph on the 
;; subset relationship. If Node1 is a subset of Node2, then 
;; Node1 must be before Node2 in the list of nodes.
;;
;; It is a classical algorithm for topological sort, using DFS. The 
;; procedure closely follows the pseudocode in Chapters 23.3 and 23.4
;; of "Introduction to Algorithms" by Cormen, Leiserson, and Rivest.
;;
;; The DFS search adds a node to in the beginning of the current 
;; topologically sorted list, *nodes*, after the search visited
;; all successors of the node and added them to the list.
;;
;; graph   the abstraction graph

(defun alpine-subset-sort (graph)
  (declare (type abs-graph graph))
  (let ((*nodes* nil))
    (declare (special *nodes*))
    (dolist (node (abs-graph-nodes graph))
      (setf (abs-node-color node) 'white))
    (dolist (node (abs-graph-nodes graph))
      (if (eq (abs-node-color node) 'white)
        (alpine-subset-visit node)))
    (setf (abs-graph-nodes graph) *nodes*)))


;; Recursively visit all successors of the current node; then
;; set the "color" of the node to 'black and add the node
;; to the beginning of the sotred list.

(defun alpine-subset-visit (node)
  (declare (type abs-node node) (special *nodes*))
  (setf (abs-component-color node) 'gray)
  (dolist (superset-node (abs-node-super node))
    (if (eq (abs-node-color superset-node) 'white)
      (alpine-subset-visit superset-node)))
        ;; Note that we ignore the maybe-nodes, used by Margie,
        ;; since their color is 'maybe rather than 'white.
  (setf (abs-node-color node) 'black)
  (push node *nodes*))


#|
======================================================
Converting the abstraction graph (hierarchy) into PDL.
======================================================

These functions convert the abstraction hierarchy, stored in the
memory as an abs-graph, into the encoding of the abstraction in the
PDL language.

The abstraction hierarchy is represented in PDL in the form shown
below. The upper-case words are the fixed words of PDL and the
lower-case words should be replaced with specific instances. Square
brackets denote the optional parts of the description (they are not a
part of the syntax).

(ABSTRACTION [hierarchy-name]
  [({STATIC | :STATIC} = typed-pred typed-pred ...)]
  ([level-name =] typed-pred typed-pred ...)
  ([level-name =] typed-pred typed-pred ...)
  ...
[{ORDER | :ORDER}
  (level-name > level-name level-name ...)
  (level-name > level-name level-name ...)
  ...)

Every "typed-pred" is a predicate name, predicate name with the 
number of arguments, or predicate with argument types; each argument
type is a prodigy-object name, preceded by "'", a type name, or a
disjunction of type-names:

typed-pred ::= pred-name | (pred-name . arg-number) 
               | (pred-name arg-type arg-type ...)
arg-type ::= 'object-name | type-name | (or type-name type-name ...)

If the abstraction graph contains only one component, we encode it as
(ABSTRACTION [hierarchy-name] {COLLAPSE | :COLLAPSE}).

When converting the internal representation of the graph into PDL, we
write the corresponding PDL statement into the dynamically-scoped
global variable *encoding*. Only the functions called by the
"alpine-encode-graph" function have the access to this variable.
|#


;; Encode the abstraction graph (hierarchy). If the components (levels)
;; are totally ordered, do not encode the order of the components; that is, 
;; do not encode the part beginning from the word :order (see the detailed
;; comment above.
;;
;; The graph of components must be topologically sorted. If the
;; components are not totally ordered by the constraints, then every component
;; must have a name (otherwise, we would not be able to encode the order).
;;
;; Before encoding the abstraction, go to a new line. The word "Abstraction"
;; (in the beginning of the encoding), the graph name, the component names,
;; and the type names are capitalized. The predicate names and the object
;; names are not capitalized.
;;
;; Returned value: PDL encoding of the graph.

(defun alpine-encode-graph (graph)
  (declare (type abs-graph graph)
           (special *pspace*) (type problem-space *pspace*))
  (if (singleton-p (abs-graph-components graph))
    (alpine-encode-collapse graph)
    (let ((*encoding* (make-array 0 :fill-pointer t :adjustable t 
                                    :element-type 'excl::string-char)))
      (declare (type string *encoding*) (special *encoding*))
      (format *encoding* "~%(Abstraction")
      (if (abs-graph-name graph)
        (format *encoding* " ~A" (string-cap (abs-graph-name graph))))
      (if (or (problem-space-static-preds *pspace*)
              (abs-graph-static graph))
        (alpine-encode-static (problem-space-static-preds *pspace*)
                              (abs-graph-static graph)))
      (name-components graph)
      (dolist (component (abs-graph-components graph))
        (alpine-encode-component component))
      (unless (total-order-p graph)
        (unless (every #'abs-component-name (abs-graph-components graph))
          (error "alpine-encode-graph: Graph is not total order and some ~
                 components have no names."))
        (format *encoding* "~% :order")
        (dolist (component (abs-graph-components graph))
          (if (abs-component-out component)
            (alpine-encode-order component))))
      (format *encoding* ")")
      *encoding*)))


;; Encode a one-level graph, in the form 
;; (Abstraction [graph-name] :collapse). 
;;
;; Before encoding the abstraction, go to a new line. 
;; The word "Abstraction" and the graph name are capitalized.
;;
;; Returned value: PDL encoding of the graph.

(defun alpine-encode-collapse (graph)
  (declare (type abs-graph graph))
  (if (abs-graph-name graph)
    (concatenate 'string "(Abstraction " 
      (string-cap (abs-graph-name graph)) " :collapse)")
    "(Abstraction :collapse)"))


;; Name all components (abstraction levels) in the abstraction graph 
;; (unless they are already named).
;;
;; The name of a component is Level-N, where N is the component's 
;; number from the end of the component list, beginning from 0.
;; For example, if there are three components, they will be
;; named Level-2, Level-1, and Level-0.
;;
;; If the graph is topologically sorted, then N is the number of
;; the level in the abstraction hierarchy.
;;
;; graph   the abstraction graph

(defun name-components (graph)
  (let ((num (length (abs-graph-components graph))))
    (dolist (component (abs-graph-components graph))
      (decf num)
      (unless (abs-component-name component)
        (setf (abs-component-name component) (format nil "Level-~D" num))))))


;; Determining whether the component graph is totally ordered;
;; that is, the edges define a total order of the components.
;;
;; The graph must be topologically sorted, but it does not have to
;; be transitively closed.
;;
;; A topologically sorted graph of components is transitively closed
;; if every component, except for the last one in the topologically-sorted
;; order, is a parent of the next node in the topologically sorted order.
;; 
;; The function is quite time consuming; to be more precise, it takes
;; O(E) time, where E is the number of edges in the graph. 
;;
;; graph   the abstraction graph
;;
;; returned value: t if the components are totally ordered; nil otherwise

(defun total-order-p (graph)
  (declare (type abs-graph graph))
  (every #'(lambda (component1 component2)
             (member component2 (abs-component-out component1)))
               ;; component2 child of component1?
         (abs-graph-components graph) (cdr (abs-graph-components graph))))


;; Encode ordering constraints for an abstraction level: for a given 
;; component (level), list components (levels) that are below it.
;;
;; In other words, we list all components that are children of the
;; given component in the component graph, in the form
;; (level-name > level-name level-name ...). 
;;
;; The list may not contain all successors of the component in the graph;
;; storing its children is enough, since we may compute the transitive
;; closure of the ordering constraints when loading the graph.
;;
;; The component must have at least one child. Before encoding the 
;; component, go to a new line and leave two spaces from the beginning. 
;; The component names are capitalized.

(defun alpine-encode-order (component)
  (declare (type abs-component component) 
           (special *encoding*) (type string *encoding*))
  ;; Check that the component has children.
  (unless (abs-component-out component)
    (error "alpine-ecnode-order: Component `~S' has no children."
           (string-cap (abs-component-name component))))
  ;; Encode the name of the component.
  (format *encoding* "~%  (~A >" 
          (string-cap (abs-component-name component)))
  ;; Encode the names of the children of the component.
  (dolist (child (abs-component-out component))
    (format *encoding* " ")
    (format *encoding* "~A" 
            (string-cap (abs-component-name child))))
  (format *encoding* ")"))


;; Encode the top-level of the hierarchy, which contains static predates
;; and the non-static predicates stored in the "static" field of the graph.
;; 
;; The name of this level is "Static"; the static predicates are encode
;; without their argument types; the non-static predicates from the 
;; "static" field are encoded with types; for example, (Static = 
;; adj-room adj-loc (in-room Object 'Room1)), where "adj-room" and 
;; "adj-loc" are static predicates and "(in-room Object 'Room1)" is a 
;; non-static predicate that does not change in abstraction problems solving.
;;
;; Either the list of static predicates, "static-names", or the list of 
;; nodes in the "static" field of the graph, static-level, must be non-empty.
;; All elements of "static-names" must be the names of static predicates.
;;
;; Before encoding the component, go to a new line and leave two spaces
;; from the beginning. The encode of every static-level node begins from
;; the new line. The component name, "Static", is capitalized.
;;
;; static-names   the names of all static predicates in the domain
;; static-level   the list of nodes at the static level ("static" field
;;                of the graph); these nodes are non-static predicates
;;                that do not change during abststraction problem solving

(defun alpine-encode-static (static-names static-level)
  (declare (special *encoding*) (type string *encoding*))
  (format *encoding* "~%  (Static =")
  (dolist (predicate-name static-names)
    (format *encoding* " ~A" (string-downcase predicate-name)))
  (dolist (node static-level)
    (format *encoding* "~%    ")
    (alpine-encode-node node))
  (format *encoding* ")"))
  

;; Encode a component of an abstraction graph (which is an abstraction level);
;; that is, encode the name of the component and the nodes in the component.
;; Each node (predicate) begins from a new line; for example
;;   (Level-1 = 
;;     (on 'small Peg) 
;;     (on 'medium Peg))
;;
;; Before encoding the component, go to a new line and leave two
;; spaces from the beginning. The encoding of every node begins from a
;; new line. The component name is capitalized.

(defun alpine-encode-component (component)
  (declare (type abs-component component) 
           (special *encoding*) (type string *encoding*))
  (format *encoding* "~%  (")
  (if (abs-component-name component)
    (format *encoding* "~A =" 
            (string-cap (abs-component-name component))))
  (dolist (node (abs-component-nodes component))
    (format *encoding* "~%    ")
    (alpine-encode-node node))
  (format *encoding* ")"))


;; Encode the abstraction node; that is, encode the name of the predicate
;; in the node and the types of its arguments; for example, (on 'small Peg).
;;
;; If the "args" field of the node is set to :any, then the node represents
;; the predicate with the specified name and any arguments. In this case,
;; encode only the predicate name (without brackets); for example, "on".
;;
;; If "args" is an integer, then the node represents the predicate
;; with the specified name and this number arguments of any type. In
;; this case, encode the predicate name and the number of arguments;
;; for example, "(on . 2)".

(defun alpine-encode-node (node)
  (declare (type abs-node node) 
           (special *encoding*) (type string *encoding*))
  (let ((args (abs-node-args node)))
    (cond
      ((eq args :any)
        (format *encoding* "~A" (string-downcase (abs-node-name node))))
      ((integerp args)
        (format *encoding* "(~A . ~D)" 
          (string-downcase (abs-node-name node)) args))
      (t 
        (format *encoding* "(~A" (string-downcase (abs-node-name node)))
        (dolist (arg-type args)
          (format *encoding* " ")
          (alpine-encode-arg arg-type))
        (format *encoding* ")")))))


;; Encode the type of an argument in an abstraction node.
;; The argument type must be a prodigy object, a simple type
;; (the type itself, not its name), or the disjcuntion of simple types.

(defun alpine-encode-arg (arg-type)
  (declare (special *encoding*) (type string *encoding*))
  (cond 
    ((prodigy-object-p arg-type) (alpine-encode-object arg-type))
    ((type-p arg-type) (alpine-encode-simple-type arg-type))
    ((and (listp arg-type) (eq (car arg-type) 'user::or)) ;; type disjunction?
      (alpine-encode-or-type arg-type))
    (t 
      (error "alpine-encode-arg: `~S' is not object, simple type, ~
         or type disjunction." arg-type))))


;; Encode a prodigy object; the object must be of a finite type.
;; The object-name encoding is in lower-case letters, preceded by '.
;; For example, "'small".

(defun alpine-encode-object (object)
  (declare (type prodigy-object object)
           (special *encoding*) (type string *encoding*))
  (format *encoding* "'~A" (string-downcase (prodigy-object-name object))))


;; Encode a type disjunction; the disjunction must be a list, where the
;; first element is 'user::or and the other elements are simple types
;; (the types themselves, not their names).

(defun alpine-encode-or-type (or-type)
  (declare (special *encoding*) (type string *encoding*))
  (unless (and (listp or-type) (eq (car or-type) 'user::or))
    (error "alpine-encode-or-type: `~S' is not a type disjunction." or-type))
  (format *encoding* "(or")
  (dolist (simple-type (cdr or-type))
    (format *encoding* " ")
    (alpine-encode-simple-type simple-type))
  (format *encoding* ")"))


;; Encode a simple type; the type must be finite.
;; The type-name encoding begins with a captial letter; for example, "Disk".
;;
;; simple-type   the type to be encoded (the type itself, not its name)

(defun alpine-encode-simple-type (simple-type)
  (declare (type type simple-type) 
           (special *encoding*) (type string *encoding*))
  (format *encoding* "~A" (string-cap (type-name simple-type))))
