/*
 **     Copyright (C) 1995  Rick Romero and Carnegie Mellon University
 **
 **     This program is free software; you can redistribute it and/or modify
 **     it under the terms of the GNU General Public License as published by
 **     the Free Software Foundation; either version 1, or (at your option)
 **     any later version.
 **     
 **     This program is distributed in the hope that it will be useful,
 **     but WITHOUT ANY WARRANTY; without even the implied warranty of
 **     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 **     GNU General Public License for more details.
 **     
 **     You should have received a copy of the GNU General Public License
 **     along with this program; if not, write to the Free Software
 **     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef _RATLIB_H_
#define _RATLIB_H_

#include "icompat.h"
#include "dkdtree.h"

typedef dkd_vector RATVector;
typedef DKDTree_t RATTree_t;
typedef DKDTree_p RATTree;
typedef DKDLeaf_p RATRegion;
typedef DKDObjList_p RATObjList;
typedef DKDObjList_t RATObjList_t;


typedef struct RATRegionList {
  RATRegion region;
  struct RATRegionList *next;
} RATRegionList_t,*RATRegionList;

typedef struct RATQualifiedOperation {
  int (*qualifier)(void *qualification,void *obj);
  void *qualification;
  int (*operation)(void *operand,void *obj);
  void *operand;
} RATQualifiedOperation;


extern DKDLeaf_t RATGlobalTmpRegion_t;


#ifndef myABS
#define myABS(x) (((x)<0) ? (-(x)) : (x))
#endif

#define RATComputeTransform(vec,x,y,w,h) {\
					    (vec)->location[0] = (w)+(x); \
					    (vec)->location[1] = (h)+(y); \
					    (vec)->location[2] = (w)-(x); \
					    (vec)->location[3] = (h)-(y); }

#define RATComputeTempTransform(x,y,w,h) \
{\
   RATGlobalTmpRegion_t.location[0] = (w)+(x); \
   RATGlobalTmpRegion_t.location[1] = (h)+(y); \
   RATGlobalTmpRegion_t.location[2] = (w)-(x); \
   RATGlobalTmpRegion_t.location[3] = (h)-(y); \
}


/*
 ** Region **
 */
RATRegion RATCreateRegion(int32, int32, int32, int32, void *);
#define   RATCreateBoundingRegion(x,y,w,h,obj)     \
     RATCreateRegion((x),(y),(w),(h),(obj))
#define   RATCreateIntersectionRegion(x,y,w,h,obj) \
     RATCreateRegion((x),(y),-(w),-(h),(obj))
#define   RATCreatePointRegion(x,y,obj)            \
     RATCreateRegion((x),(y),0,0,(obj))

/*void     *RATFreeRegion(RATRegion region);*/
#define   RATGetRegionObjList(region)  (&((region)->obj_list))
#define   RATListRegion(list)          ((list)->region)
/*
 ** Region List **
 */
RATRegionList RATListDestructiveRest(RATRegionList list);
#define RATListRest(list)   (list->next)
#define RATListFirst(list)  (list->region)
#define RATListObject(list) (list->object)


/**** Tree Contrustction/Destruction ****/
#define RATCreateTree()      CreateDKDTree(DKD_DIMENSIONS)
void RATDestroyTree(RATTree tree);


/*
 ** Access functions **
 */
#define RAT_NO_INLINE 1
#if RAT_NO_INLINE
RATRegion RATNearestMatch(RATTree, int32, int32, int32, int32);
void      RATInsert(RATTree, int32, int32, int32, int32, void *);
int       RATDelete(RATTree t, int32, int32, int32, int32, RATObjList);
#else

/*  Inlined Access Functions
 **
 ** Define the inlined versions of these commands
 ** they use global temp variables & don't do any sanity checking
 */
#define RATNearestMatch(t,x,y,w,h) {\
				    RATComputeTempTransform((x),(y),(w),(h));\
				    DKDAccess(t,&RATGlobalTmpRegion_t);\
				  }

#define RATInsert(t,x,y,w,h,o)  {\
				   dkd_vector vec; \
				   vec[0] = (x)+(w); \
				   vec[1] = (y)+(h); \
				   vec[2] = (w)-(x); \
				   vec[3] = (h)-(y); \
				   DKDInsert(t,vec,o); \
			       }

#define RATDelete(t,x,y,w,h,olist) {\
				      dkd_vector vec; \
				      vec[0] = (x)+(w); \
				      vec[1] = (y)+(h); \
				      vec[2] = (w)-(x); \
				      vec[3] = (h)-(y); \
				      DKDDelete(t,vec,(olist)); \
				  }

#endif


RATRegionList RATContainedWithin(RATTree, int32, int32, int32, int32);
RATRegionList RATIntersectsWith(RATTree, int32, int32, int32, int32);

/***** Tree Walkers *******/
int RATQualifiedTrue(void *,void *);

void RATBoundedWalk(RATTree tree,RATRegion bounds,RATQualifiedOperation *op);
void RATTotalWalk(RATTree tree,RATQualifiedOperation *op);



/*
 ** Useful stuff
 */
#define RATRegionInvert(reg,x,y,w,h) \
{ \
    (x) = (((reg)->location[0] - (reg)->location[2]) >> 1); \
    (y) = (((reg)->location[1] - (reg)->location[3]) >> 1); \
    (w) = (((reg)->location[0] + (reg)->location[2]) >> 1); \
    (h) = (((reg)->location[1] + (reg)->location[3]) >> 1); \
}
     
#endif
