/*
 **     Copyright (C) 1995  Rick Romero and Carnegie Mellon University
 **
 **     This program is free software; you can redistribute it and/or modify
 **     it under the terms of the GNU General Public License as published by
 **     the Free Software Foundation; either version 1, or (at your option)
 **     any later version.
 **     
 **     This program is distributed in the hope that it will be useful,
 **     but WITHOUT ANY WARRANTY; without even the implied warranty of
 **     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 **     GNU General Public License for more details.
 **     
 **     You should have received a copy of the GNU General Public License
 **     along with this program; if not, write to the Free Software
 **     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */



#include <stdio.h>

#include <ratlib.h>



#ifdef DEBUG
#define dfprintf(x) fprintf x
#else
#define dfprintf(x)
#endif


#define CastAsRegion(x) ((RATRegion)(x))
#define CastAsDKDNode(x)   ((DKDNode_p)(x))



DKDLeaf_t RATGlobalTmpRegion_t;


/*************
  Region
 *************/
RATRegion RATCreateRegion(int32 x,int32 y,int32 w,int32 h,void *object)
{
  RATRegion region;

  if ((region=(DKDLeaf_p)malloc(sizeof(DKDLeaf_t)))==NULL) {
    fprintf(stderr,"_RATCreateRegion: Out of memory\n");
    return NULL;
  }
  RATComputeTransform(region,x,y,w,h);
  region->obj_list.object = object;
  region->obj_list.next = NULL;
  return region;
}

/*void *RATFreeRegion(RATRegion region)
{
  RATObjList objs;

  if (region==NULL)
    obj=NULL;
  else
    objs=region->obj_list;

  free(region);
  return obj;
}*/

/*****************
  Region List 
 *****************/
RATRegionList RATListDestructiveRest(RATRegionList list)
{
  RATRegionList lp;

  lp=RATListRest(list);
  free(list);
  return lp;
}
/*********************
  Tree Destruction
 *********************/
void RATDestroyTree(RATTree tree)
{
  DKDDestroyTree(tree);
}

typedef struct IVec {
    int32   left;
    int32   right;
} IVec[DKD_DIMENSIONS];

static int RAT_CheckSplay(SplayNode_p node,int32 dim,IVec intvec)
{
  int i;
  RATRegion region;
  DKDNode_p dkdnode;

  if (node==NULL)
    return 0;

  if (RAT_CheckSplay(node->left,dim,intvec)) {
    dfprintf((stderr,"left-"));
    return 1;
  }

  if (RAT_CheckSplay(node->right,dim,intvec)) {
    dfprintf((stderr,"right-"));
    return 1;
  }

  if (node->obj==NULL)
    return 0;

  if (dim==1) {
    region=CastAsRegion(node->obj);
    for (i=1;i<DKD_DIMENSIONS;i++) {
      if (region->location[i] < intvec[i].left ||
          region->location[i] > intvec[i].right) {
        dfprintf((stderr,"Failure on dim [%d] for: [%d %d %d %d]\n",
		  i,region->location[0],region->location[1],
		  region->location[2],region->location[3]));
        return 1;
      }
    }
  } else {
    dkdnode=CastAsDKDNode(node->obj);
    if (dkdnode->stree!=NULL && dkdnode->stree->root!=NULL) {
      intvec[dim-1].left=dkdnode->left;
      intvec[dim-1].right=dkdnode->right;
      if (RAT_CheckSplay(dkdnode->stree->root,dim-1,intvec)) {
        dfprintf((stderr,"NODE Dim=[%D] L,R=[%D,%D]\n",
		  dim-1,dkdnode->left,dkdnode->right));
        return 1;
      }
    }
  }
  
  return 0;
}
        
int RATCheckIntegrity(RATTree tree)
{
  IVec intvec;

  if (tree==NULL || tree->stree==NULL || tree->stree->root==NULL)
    return 0;

  if (RAT_CheckSplay(tree->stree->root,tree->dimensions,intvec)) {
    dfprintf((stderr,"Top Of Tree\N"));
    return 1;
  }

  return 0;
}

/*********************
  Access Functions
 *********************/

#if RAT_NO_INLINE
RATRegion RATNearestMatch(RATTree t,int32 x, int32 y, int32 w, int32 h)
{
  RATComputeTempTransform(x,y,w,h);
  return DKDAccess(t, &RATGlobalTmpRegion_t);
}

void RATInsert(RATTree t, int32 x, int32 y, int32 w, int32 h, void *obj)
{
  RATVector vec;

  vec[0] = x+w;
  vec[1] = y+h;
  vec[2] = w-x;
  vec[3] = h-y;

  DKDInsert(t, vec, obj);
}

int RATDelete(RATTree t, int32 x, int32 y, int32 w, int32 h,
	      RATObjList objs)
{
  RATVector vec;

  vec[0] = x+w;
  vec[1] = y+h;
  vec[2] = w-x;
  vec[3] = h-y;

  
  return DKDDelete(t, vec, objs);
}
#endif

/*************************
   Tree Walkers 
 *************************/

#define GO_LEFT  (1)
#define GO_BOTH  (0)
#define GO_RIGHT (-1)

#define LIST_INCR(lt) (lt= &((*lt)->next))
#define LIST_END(lt)  {for (;(*lt)!=NULL;LIST_INCR(lt));}

static int32 RAT_Acceptable(RATRegion node,RATRegion refer,int dir)
{
  int32 i,accept;

  if (dir==GO_BOTH)
    return 1;

  if (node==NULL)
    return 0;

  if (refer==NULL)
    return 1;

  for (i=0,accept=1;i<DKD_DIMENSIONS;i++)
    accept &= ((node->location[i]-refer->location[i])*dir) <= 0;

  return accept;
}

static RATRegionList RAT_ListWalkDKDNode(DKDNode_p dkdnode,RATRegion region,int32 dir);

static RATRegionList RAT_ListWalkSplayNode(SplayNode_p node,RATRegion region,int32 dir,int32 dim)
{
  RATRegionList list,*lt;

  if (node==NULL) {
    dfprintf((stderr, "up.\n"));
    return NULL;
  }

  list=NULL;
  lt = &list;

  dfprintf((stderr, "LWSplay Left."));
  (*lt) = RAT_ListWalkSplayNode(node->left,region,dir,dim);
  LIST_END(lt);

  if (dim==1) {
    if (RAT_Acceptable(node->obj,region,dir)) {
      (*lt)=(RATRegionList)malloc(sizeof(RATRegionList_t));
      (*lt)->region=node->obj;
      dfprintf((stderr, "Adding: [%d %d %d %d]\n",
		(*lt)->region->location[0], (*lt)->region->location[1], 
		(*lt)->region->location[2], (*lt)->region->location[3]));
      (*lt)->next=NULL;
      LIST_END(lt);
    }
  } else {
    dfprintf((stderr, "Going down to node dim %d, left=%d, right=%d\n",
	      CastAsDKDNode(node->obj)->dimensions,
	      CastAsDKDNode(node->obj)->left,
	      CastAsDKDNode(node->obj)->right));
    (*lt)=RAT_ListWalkDKDNode(node->obj,region,dir);
    LIST_END(lt);
  }

  dfprintf((stderr, "LWSplay Right."));
  (*lt)=RAT_ListWalkSplayNode(node->right,region,dir,dim);

  dfprintf((stderr, "UP.\N"));
  return list;
}

static RATRegionList RAT_ListWalkDKDNode(DKDNode_p dkdnode,RATRegion region,int32 dir)
{
  RATRegionList list,*lt;
  DKDNode_t sorter;

  list=NULL;
  lt = &list;

  if (dkdnode==NULL || dkdnode->stree==NULL || dkdnode->stree->root==NULL)
    return NULL;

  if (dir != GO_BOTH) {
    if (dkdnode->dimensions == 1) {
      SplayAccess(dkdnode->stree,region);
    } else {
      sorter.left=sorter.right=region->location[dkdnode->dimensions-1];
      SplayAccess(dkdnode->stree,&sorter);
    }
  }

  if (dir==GO_RIGHT || dir==GO_BOTH) {
    (*lt) = RAT_ListWalkSplayNode(dkdnode->stree->root->right,region,dir,dkdnode->dimensions);
    LIST_END(lt);
  }
 
  if (dkdnode->dimensions==1) {
    if (RAT_Acceptable(dkdnode->stree->root->obj,region,dir)) {
      (*lt) = (RATRegionList)malloc(sizeof(RATRegionList_t));
      (*lt)->region = dkdnode->stree->root->obj;
      (*lt)->next = NULL;
      LIST_END(lt);
    }
  } else {
    (*lt) = RAT_ListWalkDKDNode(dkdnode->stree->root->obj,region,dir);
    LIST_END(lt);
  }

  if (dir==GO_LEFT || dir==GO_BOTH)  {
    (*lt) = RAT_ListWalkSplayNode(dkdnode->stree->root->left,region,dir,dkdnode->dimensions);
  }

  return list;
}
    

RATRegionList RATContainedWithin(RATTree tree, int32 x, int32 y,
				 int32 w, int32 h)
{

  if (tree==NULL || tree->stree==NULL) 
    return NULL;

  RATComputeTempTransform(x,y,w,h);
  return RAT_ListWalkDKDNode(CastAsDKDNode(tree), &RATGlobalTmpRegion_t,
			     GO_LEFT);
}

RATRegionList RATIntersectsWith(RATTree tree, int32 x, int32 y, int32 w,
				int32 h)
{
  if (tree==NULL || tree->stree==NULL)
    return NULL;

  RATComputeTempTransform(x,y,-w,-h);
  return RAT_ListWalkDKDNode(CastAsDKDNode(tree), &RATGlobalTmpRegion_t,
			     GO_RIGHT);
}

/************************
  Functional Tree Walkers
 *************************/
int RATQualifiedTrue(void *obj1,void *obj2)
{
  return 1;
}


static void RAT_FuncWalkDKDNode(DKDNode_p dkdnode,RATRegion region,
                                int32 dir,RATQualifiedOperation *op);

static void RAT_FuncWalkSplayNode(SplayNode_p node,RATRegion region,int32 dir,
                                  RATQualifiedOperation *op,int32 dim)
{
  RATObjList list;

  if (node==NULL) 
    return;

  RAT_FuncWalkSplayNode(node->left,region,dir,op,dim);

  if (dim==1) {
    if (RAT_Acceptable(node->obj,region,dir)) {
      list = &(CastAsRegion(node->obj)->obj_list);
      while(list) {
        if (op->qualifier(op->qualification,list->object))
	  op->operation(op->operand,list->object);
	list = list->next;
      }
    }
  } else {
    RAT_FuncWalkDKDNode(node->obj,region,dir,op);
  }

  RAT_FuncWalkSplayNode(node->right,region,dir,op,dim);
}

static void RAT_FuncWalkDKDNode(DKDNode_p dkdnode,RATRegion region,
                                int32 dir,RATQualifiedOperation *op)
{
  DKDNode_t sorter;
  RATObjList list;

  if (dkdnode==NULL || dkdnode->stree==NULL || dkdnode->stree->root==NULL)
    return;

  if (dir != GO_BOTH) {
    if (dkdnode->dimensions == 1) 
      SplayAccess(dkdnode->stree,region);
    else {
      sorter.left=sorter.right=region->location[dkdnode->dimensions-1];
      SplayAccess(dkdnode->stree,&sorter);
    }
  }

  if (dir==GO_RIGHT || dir==GO_BOTH) 
    RAT_FuncWalkSplayNode(dkdnode->stree->root->right,region,dir,op,dkdnode->dimensions);
 
  if (dkdnode->dimensions==1) {
    if (RAT_Acceptable(dkdnode->stree->root->obj,region,dir)) {
      list = &(CastAsRegion(dkdnode->stree->root->obj)->obj_list);
      while(list) {
        if (op->qualifier(op->qualification,list->object))
	  op->operation(op->operand,list->object);
	list = list->next;
      }
    }
  } else 
    RAT_FuncWalkDKDNode(dkdnode->stree->root->obj,region,dir,op);

  if (dir==GO_LEFT || dir==GO_BOTH)  
    RAT_FuncWalkSplayNode(dkdnode->stree->root->left,region,dir,op,dkdnode->dimensions);

}

void RATBoundedWalk(RATTree tree,RATRegion bounds,RATQualifiedOperation *op)
{
  RAT_FuncWalkDKDNode(CastAsDKDNode(tree),bounds,GO_LEFT,op);
}

void RATTotalWalk(RATTree tree,RATQualifiedOperation *op)
{
  RAT_FuncWalkDKDNode(CastAsDKDNode(tree),NULL,GO_BOTH,op);
}

