/* 
 * Mach Operating System
 * Copyright (c) 1991,1990,1989 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS 
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 * 
 * Carnegie Mellon requests users of this software to return to
 * 
 *  Software Distribution Coordinator  or  Software.Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 * 
 * any improvements or extensions that they make and grant Carnegie the
 * rights to redistribute these changes.
 */
/*
 * HISTORY
 * $Log:	cdp.c,v $
 * Revision 2.2  93/05/31  16:02:44  mrt
 * 	RCS-ed.
 * 	[93/05/08  12:09:48  af]
 * 
 *	Added +1 to cdrom_play_track_sec for Toshiba drive 
 *	for RISC/6000. from Dick Orgass
 * 	[93/05/08            af]
 * 
 */
/* cdmain.c -- a quickie command line interface to cd player
 *
 * by Chris Newman
 */

#include <stdio.h>
#include <ctype.h>
#include "cdrom_callb.h"
#include "debug.h"

void usage()
{
    fprintf(stderr, "usage: cdp [command]\n");
    fprintf(stderr, "  cdp info            display cd info\n");
    fprintf(stderr, "  cdp times           display cd playing times\n");
    fprintf(stderr, "  cdp play            play (or continue playing) the cd\n");
    fprintf(stderr, "  cdp #[:min.sec]     play from track # [starting at min.sec]\n");
    fprintf(stderr, "  cdp #-#             play a range of tracks\n");
    fprintf(stderr, "  cdp # # [# ...]     play a list of tracks (will sleep between selections\n");
    fprintf(stderr, "  cdp volume #        set volume [0 - 255]\n");
    fprintf(stderr, "  cdp volume # #      set left & right channel volume separately\n");
    fprintf(stderr, "  cdp stop            stop (pause) the cd\n");
    fprintf(stderr, "  cdp resume          resume after stop\n");
    fprintf(stderr, "  cdp eject           eject the cd\n");
    fprintf(stderr, "  [commands may be abbreviated to one letter]\n");
    exit(1);
}

void doopen()
{
    int track;
    
    if ((track = cdrom_open()) < 0) {
	fprintf(stderr, "cdrom_open() failed\n");
	exit(1);
    }
}

static char *cdstat;
static int cdistat;
void cdinfo()
{
    switch (cdistat = cdrom_status()) {
	case CDROM_INVALID:
	    cdstat = "No CD in CD player";
	    return;
	case CDROM_PLAYING:
	    cdstat = "CD is currently playing";
	    break;
	case CDROM_PAUSED:
	    cdstat = "CD player is paused";
	    break;
	case CDROM_COMPLETED:
	    cdstat = "CD is finished";
	    break;
	case CDROM_ERROR:
	    cdstat = "CD player error occured";
	    break;
	default:
	    cdstat = "CD player has no status";
	    break;
    }
    if (cdrom_get_times() < 0) {
	fprintf(stderr, "cdrom_get_times failed\n");
    }
}

int cdplay(str)
    char *str;
{
    int track = 0, sec = 0, etrack = 0, min;

    cdinfo();
    while (isdigit(*str)) {
	track = track * 10 + (*str++ - '0');
    }
if (!debug)
    if (track < cdi.cdi_mintrack || track > cdi.cdi_maxtrack) {
	fprintf(stderr, "only tracks %d to %d are valid\n",
		cdi.cdi_mintrack, cdi.cdi_maxtrack);
	exit(1);
    }
    if (*str == '.' || *str == ':') {
	++str;
	while (isdigit(*str)) {
	    sec = sec * 10 + (*str++ - '0');
	}
	if (*str == '.') {
	    min = sec;
	    ++str;
	    sec = 0;
	    while (isdigit(*str)) {
		sec = sec * 10 + (*str++ - '0');
	    }
	    sec += min * 60;
	}
	/* The +1 is for funny Toshiba drive for RISC/6000. */
	cdrom_play_track_sec(track, sec, cdi.cdi_maxtrack+1, 0);
    } else if (*str == '-') {
	++str;
	while (isdigit(*str)) {
	    etrack = etrack * 10 + (*str++ - '0');
	}
	if (etrack < track) etrack = track;
	cdrom_play_track(track, etrack);
	track = etrack;
    } else {
	cdrom_play_track(track, cdi.cdi_maxtrack);
    }

    return (track);
}

void cdplaylist(ptr)
    char **ptr;
{
    int track, curtrack, togo = 0;

    for (;;) {
	track = cdplay(*ptr);
	if (*++ptr == (char *) NULL) exit(0);
	sleep(2);
	do {
	    if (togo > 0) {
		printf("cdp: sleeping for %d:%02d\n", togo / 60, togo % 60);
		sleep(togo);
	    }
	    curtrack = cdrom_get_curtrack();
	    if (curtrack > track) break;
	    togo = cdi.cdi_times[track] - (int) cdi.cdi_times[curtrack - 1] - 1;
	    togo -= (int) cdi.cdi_dur;
	} while (togo > 1);
    }
}

main(argc, argv)
    int argc;
    char **argv;
{
    int track, dur, lvol, rvol;
    
    if (argc < 2) usage();
more:
    switch (*argv[1]) {
	case 'i':
	    if (argc > 2) usage();
	    doopen();
	    cdinfo();
	    printf("%s\n", cdstat);
	    if (cdistat == CDROM_INVALID) break;
	    printf("tracks %d to %d, currently at %d\n", cdi.cdi_mintrack, cdi.cdi_maxtrack, cdi.cdi_curtrack);
	    if (cdi.cdi_curtrack >= cdi.cdi_mintrack && cdi.cdi_curtrack <= cdi.cdi_maxtrack) {
		dur = cdi.cdi_times[cdi.cdi_curtrack - cdi.cdi_mintrack + 1]
		    - cdi.cdi_times[cdi.cdi_curtrack - cdi.cdi_mintrack];
		printf("track %d duration: %d.%02d\n", cdi.cdi_curtrack, dur / 60, dur % 60);
	    }
	    printf("%d.%02d played so far\n", cdi.cdi_dur / 60, cdi.cdi_dur % 60);
	    break;
	case 't':
	    if (argc > 2) usage();
	    doopen();
	    cdinfo();
	    for (track = cdi.cdi_mintrack; track <= cdi.cdi_maxtrack; ++track) {
		dur = (unsigned short) cdi.cdi_times[track] - (unsigned short) cdi.cdi_times[track - 1];
		printf("track %2d: %d.%02d\n", track, dur / 60, dur % 60);
	    }
	    dur = cdi.cdi_times[cdi.cdi_maxtrack];
	    printf("total: %d:%02d.%02d\n", dur / (60 * 60), (dur / 60) % 60, dur % 60);
	    break;
	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9':
	    doopen();
	    if (argc > 2) cdplaylist(argv + 1);
	    cdplay(argv[1]);
	    break;
	case 'p':
	    if (argc > 3) usage();
	    doopen();
	    if (argc == 3) {
		cdplay(argv[2]);
	    } else {
		if (cdrom_status() == CDROM_PAUSED) {
		    cdrom_resume();
		} else {
		    cdinfo();
		    cdrom_play_track(cdi.cdi_mintrack, cdi.cdi_maxtrack);
		}
	    }
	    break;
	case 'v':
	    if (argc < 3 || argc > 4) usage();
	    doopen();
	    rvol = lvol = atoi(argv[2]);
	    if (argc == 4) rvol = atoi(argv[3]);
	    cdrom_volume(lvol, rvol);
	    break;
	case 's':
	    if (argc > 2) usage();
	    doopen();
	    cdrom_stop();
	    break;
	case 'r':
	    if (argc > 2) usage();
	    doopen();
	    cdrom_resume();
	    break;
	case 'e':
	    if (argc > 2) usage();
	    doopen();
	    cdrom_eject();
	    break;
	case 'd':
	    debug = 1;
	    argv++, argc--;
	    goto more;
	default:
	    usage();
    }
    cdrom_close();
    exit(0);
}
