(*

	FoxNet: The Fox Project's Communication Protocol Implementation Effort
	Edoardo Biagioni (esb@cs.cmu.edu)
	Brian Milnes (Brian.Milnes@cs.cmu.edu)
	Nick Haines  (Nick.Haines@cs.cmu.edu)
	Fox Project
	School of Computer Science
	Carnegie Mellon University
	Pittsburgh, Pa 15139-3891

		i.	Abstract

	This file implments a simulated network wire. 


		ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	1.	functor Wire


		1.	functor Wire
*)

functor Wire (structure B: FOX_BASIS
	      val debug_level: int ref option) : WIRE =
 struct
  structure Trace = Trace (structure V = B.V
			   val debug_level = debug_level
			   val module_name = "wire.fun"
			   val makestring = fn _ => NONE)

  type key = Word48.word

  type value = Word_Array.T -> unit

  val mask = Word48.fromInt 0xffffff
  val byte_mask = Word48.fromInt 0xff
  fun hash addr = Word.fromLargeWord (Word48.toLargeWord
				      (Word48.andb (addr, mask)))

  fun eq_key (k1, k2: key) = k1 = k2

  val handlers = ref ((B.Store.new (hash, eq_key)): (key, value) B.Store.T)

  val current = ref (NONE: Word_Array.T option)

  fun packet_to_key packet =
       let val p48 = Word48_Array.to packet
       in Word48_Array.U_Big.F.head p48
       end

  fun makestring_key key =
       let val makes = Int.toString o Word48.toInt 
	   fun mask n = Word48.andb (n, byte_mask)
	   val a0 = makes (mask (Word48.>> (key, 0w40)))
	   val a1 = makes (mask (Word48.>> (key, 0w32)))
	   val a2 = makes (mask (Word48.>> (key, 0w24)))
	   val a3 = makes (mask (Word48.>> (key, 0w16)))
	   val a4 = makes (mask (Word48.>> (key,  0w8)))
	   val a5 = makes (mask (Word48.>> (key,  0w0)))
       in a0 ^ ":" ^ a1 ^ ":" ^ a2 ^ ":" ^ a3 ^ ":" ^ a4 ^ ":" ^ a5
       end

  fun makestring_packet (_, 0) = ""
    | makestring_packet (NONE, count) = ""
    | makestring_packet (SOME (head, rest), count) =
       Word8.toString head ^ "." ^
       makestring_packet (Word_Array.W8.U_Big.F.next rest, count - 1)

  fun string_key (k, v) = makestring_key k

  val bcast_key = Word48.- (Word48.fromInt 0, Word48.fromInt 1)


  fun deliver (key, handler, packet) =
       (Trace.debug_print (fn _ => "delivering packet to handler for " ^
			   makestring_key key);
	((handler packet)
	 handle x =>
	         if B.V.Control.exnName x = "Receive" then ()
		 else Trace.print_handled (x, SOME "handler")))

  fun deliver_map packet (key, handler) =
       (deliver (key, handler, packet);
	handler)
  
  fun busy_wait ms =
       case ! current of
          NONE => 
	   (B.Scheduler.fork_limited_time busy_wait;
	    ())
        | SOME p =>
           (current := NONE;
	    B.Scheduler.fork_limited_time busy_wait;
	    Trace.trace_print (fn _ => "busy_wait (" ^ Int.toString ms ^
			       "), dispatching packet " ^
			       makestring_packet
			       (Word_Array.W8.U_Big.F.next (Word_Array.to8 p),
				100));
	    let val key = packet_to_key p
	    in if key = bcast_key then
	        (Trace.debug_constant_string "broadcast, calling all handlers";
		 B.Store.map (deliver_map p) (! handlers);
		 ())
	       else
		case B.Store.look (! handlers, key) of
		   NONE =>
		    Trace.debug_print (fn _ =>
				       "no handler for packet with key " ^
				       makestring_key key ^
				       "\nhandlers are " ^
				       B.Store.makestring (! handlers,
							   string_key, ", "))
		 | SOME (new_table, f) =>
		    (handlers := new_table;
		     deliver (key, f, p));
	       Trace.debug_print (fn _ => "busy_wait done")
	    end)

  fun register (p, f) =
       let val key = packet_to_key p
           val h = ! handlers
       in B.Scheduler.fork_limited_time busy_wait;
          Trace.debug_print (fn _ => "registering key " ^ makestring_key key);
          case B.Store.look (h, key) of
             NONE => (handlers := B.Store.add (h, key, f); true)
           | SOME _ =>
	      (Trace.local_print ("key " ^ makestring_key key ^
				  " already registered");
	       false)
       end

  fun unregister p =
       let val key = packet_to_key p
           val h = ! handlers
       in Trace.debug_print (fn _ => "un-registering key " ^
			     makestring_key key);
          case B.Store.look (h, key) of
             SOME _ => (handlers := B.Store.remove (h, key); true)
           | NONE => false
       end

  fun send p =
       let val _ = Trace.debug_print
	               (fn _ => "in send, packet size " ^
			Word.toString
			 (Word_Array.W8.U_Big.F.length (Word_Array.to8 p)))
	   val key = packet_to_key p
           val h = ! handlers
       in case ! current of
	     NONE => ()
	   | SOME _ => 
	      Trace.trace_print (fn _ => "overwriting packet on the wire");
          current := SOME p;
   (* let other threads, particularly the receiver thread, be scheduled. *)
          Trace.debug_constant_string "sleeping in wire 10 ms";
          B.Scheduler.sleep 10
       end

 end (* struct *)
