(*

	FoxNet: The Fox Project's Communication Protocol Implementation Effort
	Edoardo Biagioni (esb@cs.cmu.edu)
	Ken Cline    (Ken.Cline@cs.cmu.edu)
	Nick Haines  (Nick.Haines@cs.cmu.edu)
	Brian Milnes (Brian.Milnes@cs.cmu.edu)
	Fox Project
	School of Computer Science
	Carnegie Mellon University
	Pittsburgh, Pa 15139-3891

		i.	Abstract

	packetfilter.fun: an interface to the OSF1 packetfilter pseudo-device.



		ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	1.	functor Packet_Filter
	2.	types and exceptions
	3.	module Trace
	4.	c functions
	5.	internal functions raise_fun and handle_fun
	6.	fun pfopen
	7.	fun close
	8.	fun get_ethernet_address
	9.	fun set_filter
	10.	fun readi.
	11.	fun write
	12.	fun select.
	13.	fun writev


	1.	functor Packet_Filter
*)

functor Packet_Filter (structure B: FOX_BASIS
		       val debug_level: int ref option): PACKET_FILTER =
 struct

(*
	2.	types and exceptions
*)

  type T = Posix.FileSys.file_desc
  type filter = Word_Array.T

  exception Packet_Filter

(*
	3.	module Trace
*)

  fun makestring_exn Packet_Filter = SOME "Packet Filter"
    | makestring_exn (System.Unsafe.Assembly.SysErr (str,NONE)) =
        SOME ("SysErr (" ^ str ^ ")")
    | makestring_exn (System.Unsafe.Assembly.SysErr (str,SOME int)) =
        SOME ("SysErr (" ^ str ^ ", " ^ Int.toString int ^ ")")
    | makestring_exn _ = NONE

  structure Trace = Trace (structure V = B.V
			   val debug_level = debug_level
			   val module_name = "packetfilter.fun"
			   val makestring = makestring_exn)

(*
	4.	c functions
*)

  val c_function = System.Unsafe.CInterface.c_function

  val c_pfopen: string * int -> int = c_function "FoxNet" "pfopen"

  val c_pfget_address: int -> Word8Array.array = 
       c_function "FoxNet" "pfget_address"

  val c_pfset_filter: int * Word8Array.array -> unit = 
       c_function "FoxNet" "pfset_filter"

  (* the last argument to select tells us if there is a timeout,
     and, if so, what the timeout is in seconds and microseconds. *)
  val c_select: int * int list * int list * int list * (int * int) option
                -> int = c_function "FoxNet" "select"

  val c_writev : int * (Word8Array.array*int*int) list -> int =
                 c_function "FoxNet" "writev";

(*
	5.	internal functions raise_fun and handle_fun
*)

  fun handle_fun (x, s) =
       (Trace.print_handled (x, SOME s);
	Trace.print_raise (Packet_Filter, SOME s))

(*
	6.	fun pfopen

	The second argument to c_function "pfopen" defines the packet
	filter bits to set.  In this case,
	ENNONEXCL | ENCOPYALL = 0x30.
	For details, see /usr/include/net/pfilt.h.
*)

  fun pfopen interface =
       let val pf = ((c_pfopen (interface, 0x30))
		     handle x => handle_fun (x, "pfopen (" ^ interface ^ ")"))
       in if pf < 0 then Trace.print_raise (Packet_Filter, SOME "pfopen")
	  else Posix.FileSys.wordToFD (Word32.fromInt pf)
	  before Trace.debug_print (fn _ => "pfopen returning fd = " ^
				    Int.toString pf)
       end

(*
	7.	fun close
*)

  fun close pf =
       (Trace.debug_print
	(fn _ => ("closing fd " ^
		  Word32.fmt StringCvt.DEC (Posix.FileSys.fdToWord pf)));
  (* Close raises SysErr for packetfilters on OSF1 1.3/2.0 even
     though it otherwise works properly.  We ignore this exception. *)
	((Posix.IO.close pf) handle x => ()))

(*
	8.	fun get_ethernet_address
*)

  fun makestring_ethernet buffer =
       let val b0 = Word8Array.sub (buffer, 0)
	   val b1 = Word8Array.sub (buffer, 1)
	   val b2 = Word8Array.sub (buffer, 2)
	   val b3 = Word8Array.sub (buffer, 3)
	   val b4 = Word8Array.sub (buffer, 4)
	   val b5 = Word8Array.sub (buffer, 5)
	   val ts = Word8.toString
       in ts b0^":"^ts b1^":"^ts b2^":"^ts b3^":"^ts b4^":"^ts b5
       end

  fun get_ethernet_address pf =
       let val int_pf = Word32.toInt (Posix.FileSys.fdToWord pf)
	   val buffer = ((c_pfget_address int_pf)
			 handle x => handle_fun (x, "get_ethernet_address"))
	   fun create_array n =
	        if n >= Word8Array.length buffer then NONE
		else SOME (Word8Array.sub (buffer, n), n + 1)
       in Trace.debug_print (fn _ => "ethernet address (fd = " ^
			     Int.toString int_pf ^ ") is " ^
			     makestring_ethernet buffer);
	  Word_Array.from8 (Word_Array.W8.U_Big.F.new create_array 0)
       end

(*
	9.	fun set_filter
*)

  fun set_filter (pf, filter) = ()

(*
	10.	fun readi.
	Read from packetfilter starting at position i of buf.

  fun readi (pf,a,n,i) =
       ((System.Unsafe.SysIO.readi (pf, a, i, n))
        handle x => handle_fun (x, "readi"))

*)

  fun readi (pf, buf, count, i) =
       let val (data, first, last) = Word_Array.expose buf
	   val size = Word.toInt (last - first) + 1
       in ((Posix.IO.readArr (pf, {buf = data, i = i + Word.toInt first,
				   sz = SOME (min (count, size))}))
	   handle x => handle_fun (x, "readi"))
       end

(*
	11.	fun write
*)

  fun write (pf, buf, n) =
       let val (data, first, last) = Word_Array.expose buf
	   val size = Word.toInt (last - first) + 1
       in ((Posix.IO.writeArr (pf, {buf = data, i = Word.toInt first,
				    sz = SOME (min (n, size))}))
	   handle x => handle_fun (x, "write"));
	  ()
       end

(*
	12.	fun select.
	Unix select packet filter for reading with timeout ms milliseconds.
	Returns false if it times out, true if the file descriptor is ready.
*)

  fun select (pf, ms) =
       let val sec = ms quot 1000
	   val usec = 1000 * (ms rem 1000)
	   val int_pf = Word32.toInt (Posix.FileSys.fdToWord pf)
	   val nfds = int_pf + 1
	   val result = ((c_select (nfds, [int_pf], [], [], SOME (sec, usec)))
			 handle x => handle_fun (x, "select"))
       in result > 0
       end (* let *)

(*
	13.	fun writev
*)

  fun writev (pf, bufferList) = 
     let val pf' = Word32.toInt (Posix.FileSys.fdToWord pf)
        fun convert wordArray = 
           let val ( buffer, first, last ) = Word_Array.expose wordArray
           in
              ( buffer, Word.toInt first, Word.toInt (last - first + 0w1) )
           end
     in
        c_writev (pf', map convert bufferList)
     end

end

