(*

        FoxNet: The Fox Project's Communication Protocol Implementation Effort
        Edo Biagioni (esb@cs.cmu.edu)
        Fox Project
        School of Computer Science
        Carnegie Mellon University
        Pittsburgh, Pa 15139-3891

	i.	Abstract
	arp.fun: a generic implementation of the Address Resolution Protocol


	ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	iii.	Overview
	1.	functor Arp
	2.	protocol types
	3.	protocol state
	4.	functions cache_add and cache_lookup
	5.	function request
	6.	function reply
	7.	function arp_connect
	8.	function listen_lower
	9.	function arp_listen
	10.	function try_listen
	11.	function arp_data
	12.	function start_lower_session
	13.	functions create_session and close_session
	14.	function session


	iii.	Overview

The trick about Arp is to make it into a virtual protocol, that is, a
protocol that participates in connection setup and teardown but not in
data delivery.  To do this, we define a 1-1 correspondence between Arp
connections and lower-protocol connections.  On connection setup, we
do our part of the work, then set up a connection using the lower
protocol, giving it the handlers that are handed in to Arp.

	1.	functor Arp
*)

functor Arp (structure Lower: PROTOCOL
	     structure Hardware_Address: EXTERN_KEY
	     structure Protocol_Address: ARP_PROTOCOL_EXTERN
	       sharing type Hardware_Address.extern_in
		          = Protocol_Address.regular_extern_in
			  = Lower.Incoming.T
		   and type Hardware_Address.extern_out
		          = Protocol_Address.extern_out = Lower.Outgoing.T
		   and type Hardware_Address.cursor
		          = Protocol_Address.cursor = Word.word
	     val null_hardware_address: Hardware_Address.T
	     val local_hardware_address: Lower.session_extension
	                               -> Hardware_Address.T
	     val arp_protocol_number: Word16.word
	     val broadcast_address: Word16.word -> Lower.Address.T
	     val broadcast_pattern: Word16.word -> Lower.Pattern.T
	     val lower_address: Hardware_Address.T * Word16.word
	                      -> Lower.Address.T
	     val lower_pattern: Hardware_Address.T * Word16.word
	                      -> Lower.Pattern.T
	     val min_max_packet: Lower.session_extension
	                       -> (Word.word * Word.word)
	     val hardware_type: Word16.word
	     val arp_timeout: int	(* milliseconds *)
	     val arp_resend: int	(* count *)
	     structure B: FOX_BASIS
	     structure Trace: TRACE) (* : ADDRESS_RESOLUTION_PROTOCOL *) =
 struct

  val max_cache_size = 64 (* expect about 50% utilization *)

  structure Host = Protocol_Address
  structure Protocol =
   struct
    type T = Word16.word
    val equal: (T * T -> bool) = op=
    val hash = Word.fromLargeWord o Word16.toLargeWord
    val makestring = Integer.toString o Word16.toInt
   end

  structure Arp_Address: ARP_ADDRESS =
   struct
    type host = Host.T
    type protocol = Word16.word
    datatype address = Specific of {self: host, peer: host, protocol: protocol}
                     | Broadcast of protocol
    type T = address
    fun makestring (Specific {self, peer, protocol}) =
         "self = " ^ Protocol_Address.makestring self ^
         ", peer = " ^ Protocol_Address.makestring peer ^
	 ", protocol = " ^ Integer.toString (Word16.toInt protocol)
      | makestring (Broadcast protocol) =
	 "broadcast " ^ Integer.toString (Word16.toInt protocol)
    fun equal (Specific {self = s1, peer = p1, protocol = pr1},
	       Specific {self = s2, peer = p2, protocol = pr2}) =
         Host.equal (s1, s2) andalso Host.equal (p1, p2) andalso pr1 = pr2
      | equal (Broadcast pr1, Broadcast pr2) = pr1 = pr2
      | equal _ = false
    fun hash (Specific {self, peer, protocol}) =
         Host.hash self + Host.hash peer +
	 Word.fromLargeWord (Word16.toLargeWord protocol)
      | hash (Broadcast protocol) =
	 Word.fromLargeWord (Word16.toLargeWord protocol)
   end (* struct *)
  structure Address = Arp_Address

  structure Connection_Key = Lower.Connection_Key

  structure Arp_Pattern: ARP_PATTERN =
   struct
    type host = Host.T
    type protocol = Word16.word
    datatype pattern = Specific of {self: host, protocol: protocol}
                     | Broadcast of protocol
    type T = pattern
    fun makestring (Specific {self, protocol}) =
         Protocol_Address.makestring self ^ "/" ^
	 Integer.toString (Word16.toInt protocol)
      | makestring (Broadcast protocol) =
	 "broadcast " ^ Integer.toString (Word16.toInt protocol)
    fun equal (Specific {self = s1, protocol = p1},
	       Specific {self = s2, protocol = p2}) =
         Host.equal (s1, s2) andalso p1 = p2
      | equal (Broadcast pr1, Broadcast pr2) = pr1 = pr2
      | equal _ = false
    fun hash (Specific {self, protocol}) =
         Host.hash self + Word.fromLargeWord (Word16.toLargeWord protocol)
      | hash (Broadcast protocol) =
	 Word.fromLargeWord (Word16.toLargeWord protocol)
   end (* struct *)
  structure Pattern = Arp_Pattern

  structure Setup = Lower.Setup
  structure Incoming = Lower.Incoming
  structure Outgoing = Lower.Outgoing
  structure Status = Lower.Status
  structure Count = Lower.Count
  structure X = Lower.X

  exception Already_Open of Connection_Key.T

  type connection_extension = Lower.connection_extension
  type listen_extension = unit

  type arp_session_extension = {maximum_packet_size: Word.word,
				minimum_packet_size: Word.word}
  type session_extension = arp_session_extension

  structure Header =
    Arp_Header (structure Hardware_Address = Hardware_Address
		structure Protocol_Address = Protocol_Address
		structure In = Lower.Incoming
		structure Out = Lower.Outgoing
		structure B = B
		val null_hardware_address = null_hardware_address)

(*
	2.	protocol types
*)

  local			(* type connection = Lower.connection *)
   structure S:
    sig
     datatype connection = C of {send: Outgoing.T -> unit,
				 abort: unit -> unit,
				 extension: connection_extension}
    end = Lower
  in
   open S
  end

  datatype listen = L of {stop: unit -> unit, extension: listen_extension}

  datatype handler = H of Connection_Key.T
                           -> {connection_handler: connection -> unit,
			       data_handler: connection * Incoming.T -> unit,
			       status_handler: connection * Status.T -> unit}

  datatype session =
      S of {connect: Address.T * handler -> unit,
	    listen: Pattern.T * handler * Count.T -> listen,
	    extension: session_extension}

(*
	3.	protocol state
*)

  datatype synch = Waiting | Ready
  datatype state =
      Arp of {count: int ref,
	      queries: (Protocol_Address.T,
			Hardware_Address.T option B.Pipe.T) B.Store.T ref,
	      listens: (Pattern.T, handler * Count.T * synch) B.Store.T ref,
	      lower_max_size: Word.word,
	      lower_min_size: Word.word,
	      lower_listen: Lower.Pattern.T * Lower.handler * Lower.Count.T
	                  -> Lower.listen,
	      lower_connect: Lower.Address.T * Lower.handler -> unit,
	      complete: unit B.Pipe.T,
	      completed_close: unit B.Pipe.T,
	      lower_send: Lower.Outgoing.T -> unit,
	      lower_stop: unit -> unit,
	      local_address: Hardware_Address.T,
	      cache: Header.T list ref}

  val state = ref (NONE: state option)

  val session_sem = B.Semaphore.new ()

(*
	4.	functions cache_add and cache_lookup
*)

  fun cache_add (cache, reply) =
       let fun purge list =
	        if List.length list > max_cache_size then
		 List.take (list, max_cache_size div 2)
		else list
       in cache := purge (reply :: (!cache))
       end

  fun cache_lookup (cache, Header.Request {protocol_number, sender_hardware,
					   sender_protocol,
					   receiver_protocol, ...}) =
        let exception Found of Header.T * Hardware_Address.T * int
            exception Impossible
	    fun lookup (reply as
		        Header.Reply {protocol_number = protocol_number',
				      sender_hardware = sender_hardware',
				      sender_protocol = sender_protocol',
				      receiver_hardware = receiver_hardware',
				      receiver_protocol = receiver_protocol',
				      ...}, n) =
	         if protocol_number = protocol_number' andalso
		    Hardware_Address.equal (sender_hardware,
					    sender_hardware') andalso
		    Protocol_Address.equal (sender_protocol,
					    sender_protocol') andalso
		    Protocol_Address.equal (receiver_protocol,
					    receiver_protocol') then
		  raise Found (reply, receiver_hardware', n)
		 else n + 1
	    | lookup _ = raise Impossible
        in (foldl lookup 0 (! cache); NONE)
           handle Found (reply, hardware, n) =>
	    (Trace.trace_print (fn () => "cache - hit!");
	     if n + n > max_cache_size then cache_add (cache, reply) else ();
	     SOME hardware)
        end
    | cache_lookup _ = let exception Impossible in raise Impossible end

(*
	5.	function request
*)

  fun word_max (a, b: Word.word) = if a > b then a else b

  fun network_request ({self, peer, protocol}, local_address,
		       send, count, time, queries, lower_min_size, cache) =
       (let val req_data = Header.Request {hardware_type = hardware_type,
					   protocol_number = protocol,
					   sender_hardware = local_address,
					   sender_protocol = self,
					   receiver_protocol = peer}
	    val packet_size = word_max (Header.size req_data, lower_min_size)
	    val packet = Lower.Outgoing.uninitialized packet_size
	    val _ = (Header.marshal (packet, req_data) 0w0)
		     handle x =>
		             (Trace.local_print
			      ("allocated packet of size " ^
			       Word.toString packet_size ^
			       ", lower_min_size " ^
			       Word.toString lower_min_size ^
			       ", Header.size " ^
			       Word.toString (Header.size req_data) ^
			       ", address is " ^ 
			       Arp_Address.makestring
			       (Arp_Address.Specific {self = self,
						      peer = peer,
						      protocol = protocol}));
			      Trace.print_raise_again (x, SOME "marshal"))
	    val receive = B.Pipe.new ()
	    fun fail () = B.Pipe.enqueue (receive, NONE)
	    fun timeout time () =
	         (B.Scheduler.sleep time;
		  fail ())
	    fun loop (0, _) = NONE
	      | loop (n, wait_time) =
	         (B.Scheduler.fork (timeout wait_time);
		  Trace.trace_print
		    (fn _ =>
		     Protocol_Address.makestring self ^
		     " sending request for address " ^
		     Protocol_Address.makestring peer ^ "/" ^
		     (Integer.toString (Word16.toInt protocol)));
		  send packet handle x => Trace.print_handled
		                            (x, SOME "lower send");
		  case B.Pipe.dequeue receive of
		     SOME response =>
		       (cache_add
			 (cache,
			  Header.Reply {hardware_type = hardware_type,
					protocol_number = protocol,
					sender_hardware = local_address,
					sender_protocol = self,
					receiver_hardware = response,
					receiver_protocol = peer});
			SOME response)
		   | NONE => loop (n - 1, wait_time + wait_time))
        in queries := B.Store.add (! queries, peer, receive);
	   loop (count, time)
	   before queries := B.Store.remove (! queries, peer)
        end)
	 handle x => Trace.print_raise_again (x, SOME "request")

  fun request (args as ({self, peer, protocol}, local_address,
			_, _, _, _, _, cache)) =
	 case cache_lookup (cache,
			    Header.Request {hardware_type = hardware_type,
					    protocol_number = protocol,
					    sender_hardware = local_address,
					    sender_protocol = self,
					    receiver_protocol = peer}) of
	   SOME response => SOME response
	 | NONE => network_request args
(*
	6.	function reply
*)

  fun reply (send, hardware_type, protocol_number, sender_hardware,
	     sender_protocol, receiver_hardware, receiver_protocol,
	     lower_min_size) =
       let val reply_data =
	         Header.Reply {hardware_type = hardware_type,
			       protocol_number = protocol_number,
			       sender_hardware = sender_hardware,
			       sender_protocol = sender_protocol,
			       receiver_hardware = receiver_hardware,
			       receiver_protocol = receiver_protocol}
	   val packet_size = word_max (Header.size reply_data, lower_min_size)
	   val packet = Lower.Outgoing.uninitialized packet_size
       in Trace.trace_print (fn _ =>
			     "sending reply for address " ^
			     Protocol_Address.makestring sender_protocol ^
			     " to " ^
			     Protocol_Address.makestring receiver_protocol);
          Header.marshal (packet, reply_data) 0w0;
          send packet
       end

(*
	7.	function arp_connect
*)

  fun arp_connect (Arp {lower_send, queries, lower_min_size,
			local_address, cache, ...}, lower_connect)
                  (addr as (Address.Specific address), H handler) =
       (case request (address, local_address, lower_send, arp_resend,
		      arp_timeout, queries, lower_min_size, cache) of
	   NONE =>
	    let val failure = "unable to resolve " ^ Address.makestring addr
	    in Trace.print_raise (X.Connection failure, NONE)
	    end
	 | SOME lower_peer =>
	    let val {self, peer, protocol} = address
	        val lower = lower_address (lower_peer, protocol)
	    in Trace.trace_print (fn _ => "connecting to lower " ^
				  Lower.Address.makestring lower ^
				  " for higher address " ^
				  Address.makestring addr);
	       ((lower_connect (lower, Lower.H handler))
		handle Lower.Already_Open lower_key =>
		        raise (Already_Open lower_key))
	    end)
	 (* no ARPing for broadcast addresses. *)
    | arp_connect (Arp {lower_send, queries, lower_min_size, ...},
		   lower_connect)
                  (addr as (Address.Broadcast protocol), H handler) =
       ((Trace.trace_print (fn _ => "connecting to lower " ^
			    Lower.Address.makestring
			       (broadcast_address protocol) ^
			    " for higher address " ^
			    Address.makestring addr);
         lower_connect (broadcast_address protocol, Lower.H handler))
	handle Lower.Already_Open lower_key =>
	        (Trace.local_print ("got already open on lower key " ^
				    Lower.Connection_Key.makestring lower_key ^
				    " for higher address " ^
				    Address.makestring addr);
		 raise X.Connection "unknown connection is open"))

(*
	8.	function listen_lower
*)

  fun listen_lower (lower_listen, protocol, handler, max) =
       let val lower_pattern = broadcast_pattern protocol
	   val lower_handler = Lower.H handler
	   val Lower.L {stop, extension} =
	         lower_listen (lower_pattern, lower_handler, max)
       in stop
       end

(*
	9.	function arp_listen

	An Arp_pattern always specifies that broadcast packets
	will be accepted.
*)

  fun arp_listen (Arp {listens, ...}, lower_listen)
                 (pattern as (Pattern.Specific {protocol, self}),
		  H handler, max) =
       (Trace.trace_print (fn _ => "arp-listen specific " ^
			   Pattern.makestring pattern);
	case B.Store.look (! listens, pattern) of
	   NONE =>
	    let val lower_stop = listen_lower (lower_listen, protocol,
					       handler, max)
		fun arp_stop () =
		     (listens := B.Store.remove (! listens, pattern);
		      lower_stop ())
	    in listens := B.Store.add (! listens, pattern,
				       (H handler, max, Ready));
	       L {stop = arp_stop, extension = ()}
	    end
	 | SOME _ =>
	    Trace.print_raise (X.Listen ("already listening for pattern " ^
					 Pattern.makestring pattern),
			       SOME "arp_listen"))
    | arp_listen (_, lower_listen)
                 (pattern as (Pattern.Broadcast protocol), H handler, max) =
       (Trace.trace_print (fn _ => "arp-listen broadcast " ^
			   Pattern.makestring pattern);
	L {stop = listen_lower (lower_listen, protocol, handler, max),
	   extension = ()})
       handle _ =>
	       Trace.print_raise (X.Listen ("already listening for " ^
					    Pattern.makestring pattern),
				  SOME "arp_listen")

(*
	10.	function try_listen

	For an incoming request, try to start a corresponding listen, and
	return whether we started one.
*)

  fun try_listen (self, peer, protocol, peer_lower, listens, lower_listen) =
       let val pattern = Arp_Pattern.Specific {self = self,
					       protocol = protocol}
       in case B.Store.look (! listens, pattern) of
	     SOME (_, (H upper_handler, max, Ready)) =>
	      let val (new_max, valid, remove) =
	                case max of
			   Count.Unlimited => (max, true, false)
			 | Count.Maximum n =>
			    (Count.Maximum (n - 1), n >= 1, n <= 1)
			 | Count.Incremental f =>
			    (case f () of
			        Count.Continue => (max, true, false)
			      | Count.Done =>     (max, false, true))
		  val listen_pattern = lower_pattern (peer_lower, protocol)
		  fun call_handler key =
		       (Trace.trace_constant_string "listen handler called";
			if remove then
			 listens := B.Store.remove (! listens, pattern)
			else
			 listens := B.Store.add (! listens, pattern,
						 (H upper_handler, new_max,
						  Ready));
			 Trace.debug_constant_string "calling upper handler";
			upper_handler key)
		  val lower_handler = Lower.H call_handler
	      in if valid then		(* listen and reply *)
	          (listens := B.Store.add (! listens, pattern,
					   (H upper_handler, new_max,
					    Waiting));
		   ((lower_listen (listen_pattern, lower_handler,
				   Lower.Count.Maximum 1);
		     ())
	(* we might already be listening, in which case, reply anyway. *)
		    handle _ => ());
		   true)
		 else
		  (if remove then
		    listens := B.Store.remove (! listens, pattern)
		   else ();
		   false)
	      end
	   | SOME _ => true		(* waiting for connection, so reply *)
	   | NONE => false		(* we're not listening, ignore *)
       end

(*
	11.	function arp_data
*)

  fun arp_data (queries, listens, lower_listen, received, local_address,
		lower_min_size) (Lower.C {send, ...}, packet) =
       ((case Header.unmarshal (packet, 0w0) of
	    (Header.Request {hardware_type, protocol_number, sender_hardware,
			     sender_protocol, receiver_protocol}, _) =>
	     if try_listen (receiver_protocol, sender_protocol,
			    protocol_number, sender_hardware, listens,
			    lower_listen) then
	      (Trace.trace_print
	           (fn _ =>
		    "replying to request for " ^
		    Protocol_Address.makestring receiver_protocol ^
		    ", protocol " ^
		    Integer.toString (Word16.toInt protocol_number));
	       reply (send, hardware_type, protocol_number,
		      local_address, receiver_protocol,
		      sender_hardware, sender_protocol, lower_min_size))
	     else			(* not for us, do not reply *)
	      Trace.debug_print
	       (fn _ => "discarding request for " ^
		Protocol_Address.makestring receiver_protocol ^
		(if B.Store.empty (! listens) then
		  (", lower address is " ^
		   Hardware_Address.makestring local_address)
		 else
		  (", local address(es) are " ^
		   B.Store.makestring (! listens,
				       fn (key, _) => Pattern.makestring key,
				       ", "))))
	  | (Header.Reply {hardware_type, protocol_number, sender_hardware,
			   sender_protocol, receiver_hardware,
			   receiver_protocol}, _) =>
	     case B.Store.look (! queries, sender_protocol) of
	        SOME (_, pipe) =>
		 (Trace.trace_print
		    (fn _ => "received reply from " ^
		     Protocol_Address.makestring sender_protocol ^
		     " for " ^
		     Protocol_Address.makestring receiver_protocol);
		  B.Pipe.enqueue (pipe, SOME sender_hardware))
	      | NONE =>
		 Trace.trace_print
		    (fn _ => "discarding reply from " ^
		     Protocol_Address.makestring sender_protocol ^
		     " for " ^
		     Protocol_Address.makestring receiver_protocol))
	handle Header.Extern =>
	        Trace.local_print ("illegal arp packet " ^
				   Lower.Incoming.makestring_max
				   (packet, 0w40));
        case received of
	   NONE => ()
	 | SOME pipe =>
	    (B.Pipe.enqueue (pipe, ());
	     ()))

(*
	12.	function start_lower_session
*)

  fun start_lower_session (complete_session, completed_close, setup) =
       let val initialized = B.Pipe.new ()
	   fun wait_session argument =
	        (B.Pipe.enqueue (initialized, argument);
		 B.Pipe.dequeue complete_session;
		 ())
	   fun lower_thread () =
	        (Lower.session (setup, wait_session);
		 B.Pipe.enqueue (completed_close, ()))
       in B.Scheduler.fork lower_thread;
	  B.Pipe.dequeue initialized
       end

(*
	13.	functions create_session and close_session

	The broadcast_handler will keep the connection open until
	something is enqueued on "complete", at which point it will
	signal "complete_session" to allow the completion of the
	session handler.  The specific handler
	will close the connection as soon as a packet (presumed
	to be an ARP reply) is received.
*)

  fun create_session setup =
       (case ! state of
	   NONE =>
	    let val complete = B.Pipe.new ()
	        val complete_session = B.Pipe.new ()
	        val completed_close = B.Pipe.new ()
	        val lower_send_pipe = B.Pipe.new ()
	        val queries = ref (B.Store.new (Protocol_Address.hash,
					        Protocol_Address.equal))
	        val cache = ref []
	        val listens = ref (B.Store.new (Pattern.hash, Pattern.equal))
	        val lower_session =
	             start_lower_session (complete_session, completed_close,
					  setup)
	        val (Lower.S {connect = lower_connect, listen = lower_listen,
			      extension}) = lower_session
		val local_address = local_hardware_address extension
		val (min_packet, max_packet) = min_max_packet extension
		fun c_handler (Lower.C {send, ...}) =
		     (B.Pipe.enqueue (lower_send_pipe, send);
		      B.Pipe.dequeue complete;
		      B.Pipe.enqueue (complete_session, ());
		      ())
	        fun broadcast_handler key =
		     (Trace.trace_print (fn _ =>
					 "broadcast handler called on key " ^
					 Lower.Connection_Key.makestring key);
	              {connection_handler = c_handler,
		       data_handler = arp_data (queries, listens, lower_listen,
					        NONE, local_address,
					        min_packet),
		       status_handler = (fn _ => ())})
	        fun specific_handler key =
	             let val pipe = B.Pipe.new ()
		         fun connection _ = B.Pipe.dequeue pipe
		     in Trace.debug_print (fn _ =>
					   "specific handler called on key " ^
					   Lower.Connection_Key.makestring
					   key);
		        {connection_handler = connection,
			 data_handler = arp_data (queries, listens,
						  lower_listen, SOME pipe,
						  local_address, min_packet),
			 status_handler = (fn _ => ())}
		     end
		fun connect_thread () = 
		     (lower_connect (broadcast_address arp_protocol_number,
				     Lower.H broadcast_handler);
		      ())
		val Lower.L {stop, ...} =
		     lower_listen (broadcast_pattern arp_protocol_number,
				   Lower.H specific_handler,
				   Lower.Count.Unlimited)
		val _ = B.Scheduler.fork connect_thread
	        val new_state = Arp {count = ref 1, queries = queries,
				     lower_connect = lower_connect,
				     lower_listen = lower_listen,
				     lower_min_size = min_packet,
				     lower_max_size = max_packet,
				     listens = listens, complete = complete,
				     completed_close = completed_close,
				     lower_send =
				       B.Pipe.dequeue lower_send_pipe,
				     lower_stop = stop,
				     local_address = local_address,
				     cache = cache}
            in state := SOME new_state;
	       new_state
            end
	 | SOME (s as (Arp {count, ...})) =>
	    (count := ! count + 1;
	     s))

  fun close_session () =
       (case ! state of
	   SOME (Arp {count, lower_stop, complete, completed_close, ...}) =>
	    (count := ! count - 1;
	     Trace.trace_print (fn _ => "session reference count is now " ^
				Int.toString (! count));
	     if ! count = 0 then
	      (state := NONE;
	       lower_stop ();
	       B.Pipe.enqueue (complete, ());
	       B.Pipe.dequeue completed_close;
	       ())
	     else ())
	 | NONE => ())

(*
	14.	function session
*)

  fun session (setup, session_fun) =
       let val state_value =
		B.Semaphore.with_lock (session_sem, create_session, setup) 
	   val Arp {lower_max_size, lower_min_size, lower_listen,
		    lower_connect, ...} = state_value
	   val our_extension = {maximum_packet_size = lower_max_size,
				minimum_packet_size = lower_min_size}
	   datatype 'a result = R of 'a | X of exn
	   val session = (S {connect = arp_connect (state_value,
						    lower_connect),
			     listen = arp_listen (state_value, lower_listen),
			     extension = our_extension})
	    val result = ((R (session_fun session)) handle x => X x)
       in B.Semaphore.with_lock (session_sem, close_session, ());
	  case result of
	     R value => value
	   | X x =>
	      Trace.print_raise_again (x, SOME "arp session")
       end

 end (* struct *)
