(*

	FoxNet: The Fox Project's Communication Protocol Implementation Effort
	Edo Biagioni (esb@cs.cmu.edu)
	Fox Project
	School of Computer Science
	Carnegie Mellon University
	Pittsburgh, Pa 15139-3891

	i.	Abstract

	event.fun: FoxNet event manager package

	ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	1.	functor Event_Queue
	2.	function new
	3.	function size
	4.	function empty
	5.	function clear
	6.	function wait
	7.	function select
	8.	function signal
	9.	function wait_timeout
	10.	functor Simple_Event_Queue


	1.	functor Event_Queue
*)

functor Event_Queue (structure Scheduler: COROUTINE
		     structure Store: STORE
		     structure Debug: DEBUG
		     structure V: VENDOR
		     val debug_level: int ref option): EVENT_QUEUE =
 struct
  structure Trace = Trace (structure V = V
			   val debug_level = debug_level
			   val module_name = "event.fun"
			   val makestring = fn _ => NONE)

  datatype ('a, '1b, 'c) wait =
      Wait of 'a * '1b Scheduler.suspension
    | Select of 'a * '1b Scheduler.suspension * 'c list

  (* the identifier for the store is a word ref.  The word is
     used for hashing, and so different entries may have the same
     value; the ref is used for uniqueness, and a new one is
     allocated for each entry. *)
  datatype ('1a, '1b) T =
      Q of (word ref, ('1a, '1b, ('1a, '1b) T) wait) Store.T ref

(*
	2.	function new
*)

  val hash_randomizer = ref 0w0
  val hash_range = 0w10000

  fun hash n = ! n
  fun equal (a: word ref, b) = a = b

  fun new () = Q (ref (Store.new (hash, equal)))

(*
	3.	function size
*)

  fun size (Q queue) = Store.size (! queue)

(*
	4.	function empty
*)

  fun empty (Q queue) = Store.empty (! queue)

(*
	5.	function clear
*)

  fun clear (Q queue) = queue := Store.new (hash, equal)

(*
	6.	function wait
*)

  fun wait {queue = Q queue, event, while_waiting} =
       let val id = ref (! hash_randomizer)
	   fun queue_suspension s =
                (queue := Store.add (! queue, id, Wait (event, s));
	         ((while_waiting ())
	          handle x =>
		          Trace.print_handled (x, SOME "wait/while_waiting")))
       in hash_randomizer := (! hash_randomizer + 0w1) mod hash_range;
	  Scheduler.suspend queue_suspension
       end

(*
	7.	function select
*)

  fun select {queues, while_waiting} =
       let val id = ref (! hash_randomizer)
	   val queue_list = map (fn (a, b) => a) queues
	   fun queue_suspension s (Q queue, event) =
                queue := Store.add (! queue, id, Select (event, s, queue_list))
	   fun suspend_all s =
	        (map (queue_suspension s) queues;
	         ((while_waiting ())
	          handle x =>
		          Trace.print_handled (x,
					       SOME "select/while_waiting")))
       in hash_randomizer := (! hash_randomizer + 0w1) mod hash_range;
          Scheduler.suspend suspend_all
       end

(*
	8.	function signal
*)

  local
   fun safe_match (match, v) =
        (match v)
	handle x =>
	       (Trace.print_handled (x, SOME "match");
		false)

   fun remove_id id (Q queue) = queue := Store.remove (! queue, id)

   fun find_match _ (_, SOME x) = SOME x
     | find_match (match, queue) ((id, Wait (event, s)), NONE) =
        if safe_match (match, event) then
	 (queue := Store.remove (! queue, id);
	  SOME (event, s))
	else NONE
     | find_match (match, _) ((id, Select (event, s, queue_list)), NONE) =
        if safe_match (match, event) then
	 (map (remove_id id) queue_list;
	  SOME (event, s))
	else NONE

  in
   fun signal {queue = Q queue, match, value} =
        case Store.fold (find_match (match, queue)) (! queue) NONE of
           NONE =>
	    (Trace.debug_print (fn _ => "no threads waiting for signal");
	     NONE)
	 | SOME (event, suspension) =>
	    (Scheduler.resume (suspension, value);
	     SOME event)
  end (* local *)

(*
	9.	function wait_timeout

	This implementation uses two queues.  The first queue is used
	to accept the result of signaling, if any.  The second queue
	is used to transmit a result option, namely NONE if called by
	the timeout or SOME result otherwise.  The timer cleans up the
	first queue if it expires, otherwise everything happens as
	usual and the timer is garbage collected when it expires.
*)

  fun wait_timeout {queue = Q queue, event, while_waiting, timeout} =
       let val new_queue = new ()
           val id = ref (! hash_randomizer)
	   fun queue_suspension s =
                (queue := Store.add (! queue, id, Wait (event, s));
	         ((while_waiting ())
	          handle x =>
		          Trace.print_handled
			    (x, SOME "wait_timeout/while_waiting")))
	   fun wait_for_data () =
                (hash_randomizer := (! hash_randomizer + 0w1) mod hash_range;
		 signal {queue = new_queue, match = fn _ => true,
			 value = SOME (Scheduler.suspend queue_suspension)};
		 ())
	   fun timer () =
	        (Scheduler.sleep timeout;
		 queue := Store.remove (! queue, id);
		 signal {queue = new_queue, match = fn _ => true,
			 value = NONE};
		 ())
	   fun start_thread () =
	        (Scheduler.fork timer;
		 wait_for_data ())
       in wait {queue = new_queue, event = (), while_waiting = start_thread}
       end

 end (*struct*)

(*
	10.	functor Simple_Event_Queue
*)

functor Simple_Event_Queue (structure Event_Queue: EVENT_QUEUE):
                           SIMPLE_EVENT_QUEUE =
 struct
  type T = (unit, unit) Event_Queue.T

  val new = Event_Queue.new
  val clear = Event_Queue.clear
  val size = Event_Queue.size
  val empty = Event_Queue.empty

  fun wait (queue, while_waiting) =
       Event_Queue.wait {queue = queue, event = (),
			 while_waiting = while_waiting}

  fun map_select queue = (queue, ())

  fun select (queues, while_waiting) =
       Event_Queue.select {queues = map map_select queues,
			   while_waiting = while_waiting}

  fun wait_timeout (queue, while_waiting, timeout) =
       case Event_Queue.wait_timeout {queue = queue, event = (),
				      while_waiting = while_waiting,
				      timeout = timeout} of
	  NONE => false
	| SOME () => true

  fun signal queue =
       case Event_Queue.signal {queue = queue, match = fn _ => true,
				value = ()} of
	  NONE => false
	| SOME () => true

 end (*struct*)






