(*

	FoxNet: The Fox Project's Communication Protocol Implementation Effort
	Edo Biagioni (esb@cs.cmu.edu)
	Fox Project
	School of Computer Science
	Carnegie Mellon University
	Pittsburgh, Pa 15139-3891

	i.	Abstract

	coro.fun: coroutine and communication package for the FoxNet

	ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	1.	Functor Coroutine
	2.	internal function call_n
	3.	internal function gettime
	4.	local implementation of Fifo
	5.	types and state
	6.	internal function null_thread
	7.	function reset
	8.	function fork
	9.	function fork_limited_time
	10.	internal function coroutine_exn
	11.	error functions internal to exit
	12.	internal functions call_ready and call_limited
	13.	internal function wakeup
	14.	function exit
	15.	function sleep
	16.	function suspend
	17.	function resume


	1.	Functor Coroutine
*)

functor Coroutine (structure Debug: DEBUG
		   structure V: VENDOR): COROUTINE =
 struct
  structure Trace = Trace (structure V = V
			   val debug_level = NONE
			   val module_name = "coro.fun"
			   val makestring = fn _ => NONE)
  val local_print = Trace.local_print

  exception No_Ready_Thread
  exception No_Such_Suspension

  datatype spec = Normal of unit -> unit
                | Limited_Time of int -> unit

  datatype '1a suspension = Suspension of '1a V.Control.cont * int ref

(*
	2.	internal function call_n

	Call the function at most once every count calls.
*)

   fun call_n (count, f) =
        let val counter = ref 0
	    val last = ref NONE
	    exception Implementation_Error
	    fun return_last () = 
	         case ! last of
		    NONE => raise Implementation_Error
		  | SOME v => v
	    fun do_call arg =
	         (if ! counter = 0 then
		   (counter := count;
		    last := SOME (f arg))
		  else
		   counter := ! counter - 1;
		  return_last ())
	in do_call
	end

(*
	3.	internal function gettime

	Call gettimeofday at most once every N calls.  This lets us
	run faster, at the cost of possibly inaccurate timings.  For
	accurate timings, set max_calls to 1.

*)

  val gettime = V.Time.now

  local			(* local to all the functions in coroutine. *)

(*
    val max_calls = 1 (* 100 *)
    val ms_in_s_div_max_calls = 1000 div max_calls
    val us_in_ms_times_max_calls = 1000 * max_calls
   local				(* local to gettime. *)

    val time = ref (V.Time.gettimeofday ())
    val count = ref max_calls			(* update on first call. *)
   in
    fun getcount () =
         (if ! count >= max_calls then
	   (time := V.Time.gettimeofday ();
	    count := 1)
	  else
	   count := ! count + 1;
	  ! time)

    val gettime = if max_calls <= 1 then V.Time.gettimeofday else getcount

   end
*)

(*
	4.	local implementation of Fifo

	This implementation is included here so the compiler can
	in-line it, for speed.
*)

   type 'a fifo = 'a list * 'a list

   fun fifo_new () = ([], [])

   fun fifo_size (fwd, back) = length fwd + length back

   fun fifo_add ((fwd, back), elt) = (fwd, elt :: back)

   fun fifo_next (h :: r, back) = SOME ((r, back), h)
     | fifo_next ([], []) = NONE
     | fifo_next ([], [single]) = SOME (([], []), single)
     | fifo_next ([], back) =
        case rev back of h :: rest => SOME ((rest, []), h) | _ => NONE

(*
	5.	types and state
*)

   datatype coroutine_type = Regular_Coroutine
                           | Limited_Time_Coroutine

   type ready = (unit -> unit)
   type limited = (int -> unit)

   type sleep = V.Time.time * unit V.Control.cont

   fun sleep_less ((t1, _), (t2, _)) = V.Time.< (t1, t2)

   structure Sleep = Priority_Queue (type key = sleep
				     val less = sleep_less)

   (* incarnation numbers increase monotonically from 1. An incarnation
      of zero is reserved as being invalid. *)
   datatype state = S of {incarnation: int ref,
			  ready_queue: ready fifo ref,
			  limited_ready_queue: limited fifo ref,
			  (* the first value in the sleep queue is cached. *)
			  sleep_queue: Sleep.T,
			  current: coroutine_type ref}

   val state = S {incarnation = ref 1,
		  ready_queue = ref (fifo_new ()),
		  limited_ready_queue = ref (fifo_new ()),
		  sleep_queue = Sleep.new (),
		  current = ref Regular_Coroutine}

   val S {incarnation, ready_queue, limited_ready_queue, sleep_queue,
	  current} = state

   fun makestring_state () =
        "generation " ^ Integer.toString (! incarnation) ^
	", ready " ^ Integer.toString (fifo_size (! ready_queue)) ^
	", limited " ^
	Integer.toString (fifo_size (! limited_ready_queue)) ^
	", sleeping " ^ Integer.toString (Sleep.size sleep_queue)

(*
	6.	internal function null_thread

	null_thread is called when the coroutine system has no ready
	threads and waits for sleeping threads to wake up.
	null_thread returns right away so we can check the clock
	and see if it's time to wake someone up.
*)

   fun null_thread n = ()

  in (* local *)
(*
	7.	function reset
*)

   fun reset () =
        (incarnation := ! incarnation + 1;
	 ready_queue := fifo_new ();
	 limited_ready_queue := fifo_new ();
	 while Sleep.size sleep_queue > 0 do Sleep.pop sleep_queue;
	 current := Regular_Coroutine)

(*
	8.	function fork

	Enqueue the function f.
*)

   fun fork f =
(* in-line the call to fifo_add, for speed. The original is:
	ready_queue := fifo_add (! ready_queue, f)
 *)
        let val (front, back) = ! ready_queue
	in ready_queue := (front, f :: back)
	end

(*
	9.	function fork_limited_time

	Enqueue the function f in the limited_ready_queue.
*)

   fun fork_limited_time f =
(* in-line the call to fifo_add, for speed. The original is:
	limited_ready_queue := fifo_add (! limited_ready_queue, f)
 *)
        let val (front, back) = ! limited_ready_queue
	in limited_ready_queue := (front, f :: back)
	end

(*
	10.	internal function coroutine_exn
*)

   local

    fun coroutine_exn x =
         (Trace.print_handled (x, SOME "thread");
	  case x of			(* avoid infinite loops. *)
	     No_Ready_Thread => Trace.print_raise (x, NONE)
	   | No_Such_Suspension => Trace.print_raise (x, NONE)
	   | _ => ())

(*
	11.	error functions internal to exit
*)

    fun no_ready_thread () =
         Trace.print_raise (No_Ready_Thread,
			    SOME ("exit (" ^ makestring_state () ^ ")"))

(*
	12.	internal functions call_ready and call_limited

	These functions helps us alternate between calling the next
	limited thread and calling the next ready thread:
        - If the current thread is limited, we call the next ready thread;
        - If the current thread is ready, we call the next limited thread;
	- If there are only limited or only ready threads we execute those.
	- Otherwise there are no ready or limited threads, we return Idle
*)

    fun call_ready sleep_time =
	 case fifo_next (! ready_queue) of
	    SOME (new_ready, f) =>
	     (ready_queue := new_ready;
	      current := Regular_Coroutine;
	      f ())
	  | NONE =>
	     (case fifo_next (! limited_ready_queue) of
	         SOME (new_limited, f) =>
		  (limited_ready_queue := new_limited;
		   current := Limited_Time_Coroutine;
		   f sleep_time)
	       | NONE =>
		  if sleep_time > 0 then null_thread sleep_time
		  else
		   case Sleep.first sleep_queue of
		    NONE => no_ready_thread ()
		  | _ => null_thread sleep_time)

    fun call_limited sleep_time =
	 case fifo_next (! limited_ready_queue) of
	    SOME (new_limited, f) =>
	     (limited_ready_queue := new_limited;
	      current := Limited_Time_Coroutine;
	      f (case ! ready_queue of ([], []) => sleep_time | _ => 0))
	  | NONE =>
	     (case fifo_next (! ready_queue) of
	         SOME (new_ready, f) =>
		  (ready_queue := new_ready;
		   current := Regular_Coroutine;
		   f ())
	       | NONE =>
		  if sleep_time > 0 then null_thread sleep_time
		  else
		   case Sleep.first sleep_queue of
		    NONE => no_ready_thread ()
		  | _ => null_thread sleep_time)

(*
 
	13.	internal function wakeup

	Wake up the first sleeping processes if it should be awake now.
	Return the interval from now until the next process will be
	ready to wake up, or zero if there are no sleeping processes.
*)

    fun wakeup () =
         case Sleep.first sleep_queue of
	    NONE => 0
	  | SOME (t, c) =>
	     if V.Time.< (gettime (), t) then 10
	      (* V.Time.deltams (t, now)  --- expensive operation *)
	     else
	      (ready_queue := fifo_add (! ready_queue,
					fn _ => V.Control.throw c ());
	       Sleep.pop sleep_queue;
	       0)

(*
	14.	function exit

	Discard the current thread and start the next one if any,
	after waking up any sleeping threads.

	Note that both wakeup and call_ready may modify all the queues,
	so do not trust any cached values.
*)
   in (* local *)
    val exit_cont: unit V.Control.cont option ref  = ref NONE

    (* schedule_once_and_for_all sets exit_cont to a continuation which
       invokes some thread and handles any exceptions raised.  Throwing
       to this continuation eliminates the problem of increasing handle
       stacks that occurs with a recursive implementation of exit. *)
    fun schedule_once_and_for_all () =
         (V.Control.callcc (fn c => (exit_cont := SOME c));
	  ((case ! current of
	       Limited_Time_Coroutine => call_ready (wakeup ())
	     | _ => call_limited (wakeup ()))
           handle x => coroutine_exn x);
          exit ())

    and exit () =
      (case (! exit_cont) of
	  NONE => schedule_once_and_for_all ()
        | SOME c =>  V.Control.throw c  ())

   end (* local *)

(*
	15.	function sleep

	suspend this thread for at least the specified number of
	milliseconds; immediately begin executing any coroutines
	on the ready queues.
*)

   fun sleep ms =
        let fun add_sleep c =
		 (Sleep.add (sleep_queue, (V.Time.addms (gettime (), ms), c));
		  exit ())
        in V.Control.callcc add_sleep
        end

(*
	16.	function suspend

	Note that the incarnation is copied into the suspension,
	so we can invalidate the suspension when we resume it.
*)

   fun suspend f =
        V.Control.callcc (fn cc =>
			  (f (Suspension (cc, ref (! incarnation)));
			   exit ()))

(*
	17.	function resume
*)

   fun resume (Suspension (_, ref 0), _) =
        Trace.print_raise (No_Such_Suspension, SOME "resume")
     | resume (Suspension (cc, valid), value) =
        if ! valid = ! incarnation then
	 (valid := 0;
	  fork (fn _ => V.Control.throw cc value);
	  ())
	else
	 Trace.print_raise (No_Such_Suspension,
			    SOME "resuming resumed suspension")

  end (* local *)

 end (*struct*)
