(*

	FoxNet: The Fox Project's Communication Protocol Implementation Effort
	Edoardo Biagioni (Edoardo.Biagioni@cs.cmu.edu)
	Brian Milnes (Brian.Milnes@cs.cmu.edu)
	Ken Cline (Kenneth.Cline@cs.cmu.edu)
        Chris Stone (stone+@cmu.edu)
	Fox Project
	School of Computer Science
	Carnegie Mellon University
	Pittsburgh, Pa 15213-3891

		i.	Abstract

        Features new in the 2.0 release:
          --- correctly serves compressed files
          --- can redirect URLs starting with ~ to the "people" subdirectory
              (that is, ...cs.cmu.edu/~cstone/foo is equivalent to the URL
                        ...cs.cmu.edu/people/cstone/foo)
          --- Server actions are more easily reconfigurable
              (it's very easy to define aliases such as ~ above, or
               to allow documents to be served from a directory tree
               rooted somewhere else than /afs/cs/project/fox/mosaic)          



		ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	1.	"server-side includes" (sort of)
	iii.	RCS Log
	2.	functor Httpd
	3.	translation routines
	4.	Act upon requests
	5.	"server-side includes" (sort of)
	6.	parsing functions
	7.	HTTP/HTML output
	8.	request logging
	9.	packet-parsing
	10.	internal function process_request
	11.	function install
	12.	functions for reporting statistics
	13.	inetd_service definition
                2.      translation routines
		1.	"server-side includes" (sort of)
                4.      parsing functions
		5.      HTTP/HTML output
                6.      loggin functions
                7.      internal function process_request
		8.      function install
                9.      function uninstall
                10.     function uptime

		iii.	RCS Log
	
$Log: httpd.fun,v $
Revision 1.15  1996/06/07  20:24:00  cline
added inetd service interface

Revision 1.14  1996/05/16  20:01:07  cstone
Sends 32K chunks

Revision 1.13  1996/05/13  16:14:25  cstone
Added server statistics

Revision 1.12  1996/03/02  18:17:10  cstone
Updated to new foxnet

Revision 1.11  1995/11/12  16:51:26  esb
changed an error message.

Revision 1.10  1995/10/17  00:01:00  cstone
Enabled logging and checkpointing

Revision 1.9  1995/10/05  18:53:56  cstone
Added comments

Revision 1.8  1995/10/04  18:57:24  cstone
Changed lstat to stat

Revision 1.7  1995/10/04  13:50:04  cstone
Fixed cgi-bin prefix.

Revision 1.6  1995/10/03  22:20:27  cstone
*** empty log message ***

Revision 1.5  1995/10/03  21:40:38  cstone
Fixed redirections

Revision 1.4  1995/09/29  15:27:20  cstone
New-protocol-signature httpd

Revision 1.3  1995/03/08  20:22:45  cstone
version for public access.

Revision 1.1  1995/01/25  22:43:25  cstone
Initial revision

 
		2.	functor Httpd
*)

functor Httpd (structure B : FOX_BASIS
               structure Lower : PROTOCOL
               val hostname : string
	       val server_pattern : Lower.Pattern.T
	       val setup : Lower.Setup.T
               val default_document : string
               val log_file : string
               val checkpoint_file : string
	       val external_fun : (string * string) list -> string
              ) : HTTPD =

 struct
 
  structure Pipe = B.Pipe
  val server_starttime     = ref (B.V.Time.zeroTime)
  val server_bytesout      = ref (Word64.fromInt 0)
  val server_long_version  = "FoxNet Httpd, Version 2.0, May 2, 1996"
  val server_short_version = "FoxNet-Httpd/2.0"
  val server_accesses      = ref 0


(*
-----------------------------------------------------------------------

		3.	translation routines
*)

  val explode = B.V.String.explode
  val implode = B.V.String.implode
  val ord     = B.V.Char.ord
  val chr     = B.V.Char.chr
  val ordof   = B.V.String.ordof
  val size    = B.V.String.length
  val tokenize = B.V.String.tokens B.V.Char.isSpace

  val ord_zero   = ord #"0"
  val ord_lowera = ord #"a"
  val ord_uppera = ord #"A"

  (* Convert a character representing a hexidecimal digit to its
     integer equivalent *)
  fun hexchar_to_dec c =
     if (c >= #"0" andalso c <= #"9") then
 	ord c - ord_zero
     else if (c >= #"A" andalso c <= #"F") then
        10 + (ord c) - ord_uppera
     else if (c >= #"a" andalso c <= #"f") then
        10 + (ord c) - ord_lowera
     else
        0

  (* Translate escaped character sequences (% followed by
     a two-hex-digit character code) to a single character. 
     
     Reference: RFC 1738  Uniform Resource Locators, Section 2.2
   *)
  fun unescape' [] = []
    | unescape' (#"%" :: d1 :: d2 :: xs) =
        chr (16 * (hexchar_to_dec d1) + (hexchar_to_dec d2)) :: unescape' xs
    | unescape' (x::xs) = x :: (unescape' xs)

  val unescape = implode o unescape' o explode

  (* Though lines in an HTTP request are defined as being terminated
     by a carriage-return,line-feed sequence, a tolerant server (such
     as this one) should accept a broader range of endline sequences.
     Here we turn all recognized sequences into a simple LF ("\n", chr 10).
     We recognize :  CRLF, LF, or CR.

     Reference:  HTTP 1.0 Internet-Draft 
                 http://www.ics.uci.edu/pub/ietf/http/ *)
  fun fix_endlines (#"\^M" :: #"\n" :: xs) = #"\n" :: (fix_endlines xs)
    | fix_endlines (#"\^M" :: xs)          = #"\n" :: (fix_endlines xs)
    | fix_endlines (x :: xs)               = x :: (fix_endlines xs)
    | fix_endlines []                      = []

  (* Form data has an additional translation: spaces are replaced by
     +'s and true +'s are escaped via the %dd mechanism. The following
     function undoes the space-to-plus translation, and must be done
     before the escaped pluses are unescaped.

     Reference: http://www.w3.org/hypertext/WWW/MarkUp/html-spec/
                       html-spec_8.html#SEC88

                or the full HTML 2.0 document:

                http://www.w3.org/hypertext/WWW/MarkUp/html-spec/
                       html-spec_toc.html

   *)
  fun plus_to_space []           = []
    | plus_to_space (#"+" :: xs) = #" " :: (plus_to_space xs)
    | plus_to_space (x :: xs)    = x  :: (plus_to_space xs)

(*
-----------------------------------------------------------------------
		4.	Act upon requests
*)

  datatype action = 
    SERVE | RETRY | REDIRECT | EXEC_CGI 

  (* for now, 
        $ = name without the prefix, 
       $$ = full name
      $$$ = full name before unencoding
   *)
  val default_prefix_actions =
    [("/~",       (SERVE,"/afs/cs.cmu.edu/project/fox/mosaic/people/$")),
     ("/afs/cs/", (RETRY,"/afs/cs.cmu.edu/$")),
     ("/afs/cs.cmu.edu/project/fox/mosaic/cgi-bin/", (EXEC_CGI, "$")),
     ("/afs/cs.cmu.edu/project/fox/mosaic/", (SERVE, "$$")),
     ("/afs/cs.cmu.edu/", (REDIRECT, "http://www.cs.cmu.edu$$$")),
     ("/",        (RETRY, "/afs/cs.cmu.edu/project/fox/mosaic/$"))]

  val prefix_actions = ref default_prefix_actions

  fun insert_file (long_suffix, full_name, encoded_name) =
    let
      fun loop [] = []
        | loop (#"$" :: #"$" :: #"$" :: cs) = (explode encoded_name) @ cs
        | loop (#"$" :: #"$" :: cs) = (explode full_name) @ cs
        | loop (#"$" :: cs) = (explode long_suffix) @ cs
        | loop (c :: cs) = c :: (loop cs)
    in
      implode o loop o explode
    end

  fun prefix_of s t =
    let fun loop [] ys      = SOME (implode ys)
	  | loop _  []      = NONE
	  | loop (x::xs) (y::ys) = 
	    if (x <> y) then NONE else loop xs ys
    in
       loop (explode s) (explode t)
    end

  (* forbid * and /../ from appearing in filenames *)
  val insecure =
      let fun insecure' (#"*" :: xs) = true
            | insecure' (#"/" :: #"." :: #"." :: #"/" :: xs) = true
            | insecure' (x :: xs) = insecure' xs
            | insecure' [] = false
      in
         insecure' o explode
      end

  exception FileNotFound of string

  (* We assume that we've already split out the _filename_ from the
     params and searchpath parts of the URL.  Or else we assume there
     are none. *)
  fun respond {serve, redirect, exec} encoded_name = 
    let
      val full_name = unescape encoded_name

      fun loop full_name [] = (print "could not find an action for";
			       print full_name;
			       print "\n";
			       raise (FileNotFound full_name))
        
	| loop full_name ((prefix, (action, str)) :: xs) = 
  	     (case (prefix_of prefix full_name) of
		NONE => loop full_name xs
	      | SOME long_suffix => 
		  let
		    val action_string = 
		      insert_file (long_suffix, full_name, encoded_name) str
		  in
		    case action of
		      RETRY => 
			loop action_string (! prefix_actions)
		    | SERVE =>
			serve action_string
		    | REDIRECT =>
			redirect action_string
		    | EXEC_CGI =>
			exec action_string
		  end)
    in
      loop full_name (! prefix_actions)
    end

(*
-----------------------------------------------------------------------
		5.	"server-side includes" (sort of)
*)


  type imported_fun = unit -> string
  type 'a env       = (string * 'a) list ref
  val  exp_funs     = ref [] : imported_fun env

  exception HTTPD_Lookup
  fun empty_env () = ref [] : imported_fun env
  fun lookup (env: 'a env) n = 
    let fun lookup' [] n = raise HTTPD_Lookup
          | lookup' ((x,f)::xs) n = if (x = n) then f else lookup' xs n
    in
      lookup' (!env) n
    end
  fun extend env n f = env := (n,f)::(!env)

  fun exec (name:string) : string = (lookup exp_funs name) ()

  val install_fn = extend exp_funs

  local 
     structure Link_Parser = Link_Parser(structure V=B.V)
     open Link_Parser
     open Parser
       infix  2 -- ##
       infixr 3 &&
       infix  2 wth suchthat return guard
       infixr 1 ||
     open Parsing_Utils
       infixr 4 cor 
       infixr 4 cand 
       infixr 3 &-&
   in
     fun replace_execs s =
        (case (B.V.String.index ("#",s,0)) of
           NONE => s
         | SOME _ => 
             (case (parse (repeat any
                          && Parser.string (explode "<!--#exec")
                          &-& Parser.string (explode "cmd")
                          &-& (literal #"=")
                          &-& string
                          &-& Parser.string (explode "-->")
                          && uptoeol)
                     (Position.markstream (Input.readstring s))) of
                 NONE => s
               | SOME (lst1, (_, (_, (_, (cmd, (_, lst2)))))) =>
                   (replace_execs (implode lst1)) ^
                   ((exec (implode cmd)) handle HTTPD_Lookup => "???") ^
                   (implode lst2)))
   end

(*
-----------------------------------------------------------------------
		6.	parsing functions
*)

  infix X
  fun f X g = fn (x,y) => (f x, g y)

  (* extract the extension from a filename *)
  fun split_extension str =
      let val dot = #"."
	  fun ext_list [] ext = ([], ext)
	    | ext_list (x::xs) ext = 
	         if (x = dot) then (xs, ext) 
                 else ext_list xs (x::ext)

	  val base_and_ext = ext_list (rev (explode str)) []
      in 
	  ((implode o rev) X implode) base_and_ext
      end

  fun extensions str =
    (case split_extension str of
       ("", ext) => [ext]
     | (base, ext) => ext :: (extensions base))

  fun should_parse filename =
     (case (split_extension filename) of
         (_,"shtml") => true
       | _           => false)

  (* Reference:  HTTP/1.0 Internet-Draft v3.0, September 4, 1995 *)
  val default_compress_extensions =
    [("Z",  "x-compress"),
     ("gz", "x-gzip")]

  val compress_extensions = ref default_compress_extensions

  (* Reference: RFC 1700 Assigned Numbers (or its successor RFC).
     also ftp://ftp.isi.edu/in-notes/iana/assignments/media-types
     for an up-to-date listing *)
  val default_file_extensions =
    [("ps",    "application/postscript"),
     ("dvi",   "application/x-dvi"),
     ("pdf",   "application/pdf"),
     ("txt",   "text/plain"),
     ("html",  "text/html"),
     ("htm",   "text/html"),
     ("shtml", "text/html"),
     ("gif",   "image/gif"),
     ("xbm",   "image/x-xbm"),
     ("jpe",   "image/jpeg"),
     ("jpg",   "image/jpeg"),
     ("jpeg",  "image/jpeg"),
     ("tif",   "image/tiff"),
     ("tiff",  "image/tiff"),
     ("mpeg",  "video/mpeg")]

  val file_extensions = ref default_file_extensions

  fun filename_to_filetype filename = 
    let
      fun loop [] ext = "text/plain"
	| loop ((e,m)::es) ext = if (e = ext) then m else loop es ext

      fun loop' [] ext = NONE
	| loop' ((e,c)::es) ext = if (e = ext) then SOME c else loop' es ext

      val extensions = extensions filename
    in
      case loop' (! compress_extensions) (hd extensions) of
        SOME compress =>
	  (SOME compress, loop (! file_extensions) (hd (tl extensions)))
      | NONE =>
	  (NONE, loop (! file_extensions) (hd extensions))
    end
	  
  val html_type = (NONE, "text/html")

   (* Split input from a form into key/value string pairs *)
   val split_input =
      let fun splitoff_pair l []           = [rev l]
            | splitoff_pair l (#"&" :: xs) = (rev l) :: (splitoff_pair [] xs)
            | splitoff_pair l (x :: xs)    = splitoff_pair (x::l) xs

          fun splitoff_key l []           = (rev l, [])
            | splitoff_key l (#"=" :: xs) = (rev l, xs)
            | splitoff_key l (x :: xs)    = splitoff_key (x::l) xs

          val decode_list = 
	    implode o fix_endlines o unescape' o plus_to_space
          val decode_pair = decode_list X decode_list
      in
          ((map decode_pair) o 
           (map (splitoff_key [])) o 
           (splitoff_pair []) o 
           explode)
      end

  (* Returns filename & searchpath (neither decoded) *)
  fun parse_filename filename =
    let 
      val implode_pair = implode X implode
      fun loop [] prefix = implode_pair (rev prefix, [])
        | loop (#"?"::spath) prefix = implode_pair (rev prefix, spath)
        | loop (c :: cs) prefix = loop cs (c::prefix)
    in
      loop (explode filename) []
    end

  exception ContentLengthMissing
  fun get_content_length get_line =
    let
      fun skip_headers () =
	  (case (tokenize (get_line ())) of
	       [] => ()
	     | _ => skip_headers ())
	       
      fun loop () = 
	  (case (tokenize (B.V.String.to_lower (get_line ()))) of
	       [] => raise ContentLengthMissing
	     | ["content-length:", len_string] => 
		   (case (explode len_string) of
			(#":" :: xs) => B.V.String.string_to_int (implode xs)
		      | _ => B.V.String.string_to_int len_string)
	     | ["content-length", ":", len_string] =>
			B.V.String.string_to_int len_string
	     | _ => loop ())
    in
	(loop ()) before (skip_headers ())
    end


(*
-----------------------------------------------------------------------
		7.	HTTP/HTML output
*)

  val CRLF = "\n"

  fun http_begin_header (number, msg) = 
      let val code = (Int.toString number) ^ " " ^ msg
      in
	  "HTTP/1.0 " ^ code ^ CRLF ^
          "Server: " ^ server_short_version ^ CRLF ^
	  "Date: " ^ B.V.Time.toDate (B.V.Time.now ()) ^ CRLF
      end

  fun http_content (NONE, mimetype) = "Content-type: " ^ mimetype ^ CRLF
    | http_content (SOME encoding, mimetype) =
      "Content-type: " ^ mimetype ^ CRLF ^
      "Content-encoding: " ^ encoding ^ CRLF

  fun http_uri uri = "URI: <" ^ uri ^ ">" ^ CRLF ^
                     "Location: " ^ uri ^ CRLF

  fun http_end_header () = CRLF

  fun http_ok_header filename_opt =
    ((http_begin_header (200, "OK")) ^
     (case filename_opt of
	SOME filename => http_content (filename_to_filetype filename)
      | NONE => http_content html_type) ^
     (http_end_header ()))

  fun html_message title bodytext =
      "<HTML><HEAD><TITLE>" ^ title ^ "</TITLE><P></HEAD>\n" ^
      "<BODY><H1>" ^ title ^ "</H1>\n" ^
      bodytext ^ "<P></BODY></HTML>\n"

  fun html_redirect_body new_url =
    (html_message "301 Moved Permanently" 
     ("Moved Permanently.  The file can be reached via <P><A href=\"" ^
      new_url ^ "\">" ^ new_url ^ "</A>."))

  fun http_redirect_header new_url =
    ((http_begin_header(301, "Moved Permanently")) ^ 
     (http_content html_type) ^
     (http_uri new_url) ^
     (http_end_header ()))

  fun html_badrequest_body url_name =
    (html_message "400 Bad Request"
     ("There is a problem with your request, so that the server " ^
      "cannot determine the proper response."))

  fun http_badrequest_header () = 
    ((http_begin_header(400, "Bad Request")) ^
     (http_content html_type) ^
     (http_end_header ()))

  fun html_insecure_body url_name =
    (html_message "403 Forbidden"
     ("The URL " ^ url_name ^ "is not allowed by this server.")) 
     
  fun http_insecure_header () =
    ((http_begin_header(403, "Forbidden")) ^
     (http_content html_type) ^
     (http_end_header ()))
     
  fun html_missing_body url_name =
    (html_message "404 Not Found" 
     ("The server could not find this URL:<P>" ^ url_name))

  fun http_missing_header () =
    ((http_begin_header(404, "Not Found")) ^
     (http_content html_type) ^
     (http_end_header ()))

  fun html_unimplemented_body () = 
    (html_message "501 Not Implemented" "Sorry")

  fun http_unimplemented_header () =
    ((http_begin_header (501, "Not Implemented")) ^
     (http_content html_type) ^
     (http_end_header ()))

(*
-----------------------------------------------------------------------
		8.	request logging
*)

  (* Function to save the number of bytes served *)
  fun checkpoint s =
     let val dest = open_out checkpoint_file
     in
	 output (dest, s^"\n");
	 close_out dest
     end
       handle _ => ()

  (* Function to restore the number of bytes served *)
  fun read_checkpoint () =
     let val src = open_in checkpoint_file
	 val data = input_line src
	 val _ = close_in src
         val ord0 = ord #"0"
         val ord9 = ord #"9"

	 fun toWord (nil, a) = a
	   | toWord (d::ds, a) =
	     let val ascii = ord d
	     in
		 if ascii >= ord0 andalso ascii <= ord9 then
		     toWord
		     (ds,
		      Word64.+ (Word64.* (Word64.fromInt 10, a),
				Word64.fromInt (ascii - ord0)))
		 else
		     a
	     end
     in
	 toWord (explode data, Word64.fromInt 0)
     end
       handle _ => Word64.fromInt 0

  fun write_to_log s =
     let val dest = open_append log_file
     in
         output (dest, s);
         close_out dest;
         B.V.Print.print s
     end
       handle _ => ()

  fun log (log_entry, checkpoint_data) =
     (write_to_log log_entry;
      checkpoint checkpoint_data)

  fun log_header () = write_to_log (server_long_version ^ "\n")

(*
-----------------------------------------------------------------------
		9.	packet-parsing
*)

  structure ESL = Extern_String_Length(structure In = Lower.Incoming
                                       structure Out = Lower.Incoming
                                       structure V = B.V)
  val w0 = Word.fromInt 0

  fun incoming_to_charlist pkt = 
    explode (#1 ((ESL.unmarshal ((pkt, Lower.Incoming.size pkt), w0))))
  
  fun string_to_outgoing s =
    (server_bytesout := 
           Word64.+ (!server_bytesout, Word64.fromInt (String.size s));
     Lower.Outgoing.new
     (Word_Array.from8
      (Word_Array.W8.U_Big.F.tabulate 
	(fn w => Word8.fromInt (ord (ordof (s,Word.toInt w))), 
         Word.fromInt (size s)))))

(*
-----------------------------------------------------------------------
		10.	internal function process_request
*)


  fun file_exists filename =
    (close_in (open_in filename); true)
    handle _ => false

  local
     open Posix.FileSys
  in
     fun is_file_readable s = 
       file_exists s andalso
       (isReg (ST.fileType (stat s)))
      handle _ => false
  end

  fun error s = B.V.Print.print ("httpd.fun:  " ^ s ^ "\n")

  local
    fun send_file send filename =
      let 
        val src = open_in filename

	fun loop () = 
	  let val next = input (src, 32 * 1024)
	  in
	      if (next <> "") then (send next; loop()) else ()
	  end
      
	fun parse_loop n str =
	    if (end_of_stream src) then
		if (n <= 0) then () else send str
	    else 
		if (n >= 8192) then
		    (send str; parse_loop 0 "")
		else
		    let val line = input_line src
		    in
			parse_loop (n + B.V.String.length line)
			(str ^ (replace_execs line))
		    end
      in
	((if (should_parse filename) then (parse_loop 0 "") else (loop ()))
	 handle _ => error ("Send of file " ^ filename ^ " aborted.\n"));
	((close_in src) handle _ => ())
      end

  in
    fun serve_file9 send redirect filename = 
      if (insecure filename) then
	send (html_insecure_body filename)
      else if (is_file_readable filename) then
	send_file send filename
      else if (is_file_readable (filename ^ default_document)) then
	send_file send (filename ^ default_document)
      else if (is_file_readable (filename ^ "/" ^ default_document)) then
        redirect ("http://" ^ hostname ^ filename ^ "/")
      else
	send (html_missing_body filename)
    
    fun serve_file send redirect filename =
      if (insecure filename) then
	(send (http_insecure_header ());
	 send (html_insecure_body filename))
      else if (is_file_readable filename) then
	(send (http_ok_header (SOME filename));
	 send_file send filename)
      else if (is_file_readable (filename ^ default_document)) then
	serve_file send redirect (filename ^ default_document)
      else if (is_file_readable (filename ^ "/" ^ default_document)) then
        redirect ("http://" ^ hostname ^ filename ^ "/")
      else
	(send (http_missing_header ());
	 send (html_missing_body filename))

    fun redirect9 send new_url =
      send (html_redirect_body new_url)

    fun redirect send new_url =
      (send (http_redirect_header new_url);
       send (html_redirect_body new_url))

    fun head_serve send filename =
      if (insecure filename) then
	send (http_insecure_header ())
      else if (is_file_readable filename) then
	send (http_ok_header (SOME filename))
      else
	send (http_missing_header ())

    fun head_redirect send new_url =
      send (http_redirect_header new_url)

    fun bad_request9 send filename = 
      send (html_badrequest_body filename)

    fun bad_request send filename =
      (send (http_badrequest_header ());
       send (html_badrequest_body filename))

    fun exec send args f fname =
      (send (http_ok_header NONE);
       send (f args))
  end
	
  val unescape_pair = unescape X unescape

  fun process_request (connection_name, Lower.C {send, abort, ...}, 
		       quit_pipe, get_line, get_chars) =
    (server_accesses := !server_accesses + 1;
     case (tokenize(get_line())) of
       ["GET", filename'] => 
	 let val (filename, searchpath) = 
	   unescape_pair (parse_filename filename')
	     val send = send o string_to_outgoing
	 in
	   log (("GET9 " ^ filename ^ " >>> " ^ connection_name ^ "\n"),
		Word64.toString (!server_bytesout));
	   respond {serve = serve_file9 send (redirect9 send), 
		    redirect = redirect9 send,
		    exec = bad_request9 send} filename'
	 end
       
     | ["GET", filename', "HTTP/1.0"] =>
	 let val (filename, searchpath) = 
	   unescape_pair (parse_filename filename')
	     val send = send o string_to_outgoing
	 in
	   log (("GET  " ^ filename ^ " >>> " ^ connection_name ^ "\n"),
		Word64.toString (!server_bytesout));
	   respond {serve = serve_file send (redirect send),
		    redirect = redirect send,
		    exec = bad_request send} filename'
	 end

     | ["HEAD", filename', "HTTP/1.0"] =>
	 let val (filename, searchpath) = 
	   unescape_pair (parse_filename filename')
	     val send = send o string_to_outgoing
	 in
	   log (("HEAD " ^ filename ^ " >>> " ^ connection_name ^ "\n"),
		Word64.toString (!server_bytesout));
	   respond {serve = head_serve send, 
		    redirect = head_redirect send,
		    exec = bad_request send} filename
	 end

     | ["POST", filename', "HTTP/1.0"] =>
	 let 
	   val (filename, searchpath) = 
	     unescape_pair (parse_filename filename')
           val len     = get_content_length get_line
	   val args    = split_input (get_chars len)
	   val send : (string -> unit) = send o string_to_outgoing
	 in
	   log (("POST " ^ filename ^ " >>> " ^ connection_name ^ "\n"),
		Word64.toString (!server_bytesout));
	   respond {serve = bad_request send,
		    redirect = redirect send, 
		    exec = exec send args external_fun} filename
	 end

     | ["QUIT"] => B.Pipe.enqueue (quit_pipe, ())

     | _ => 
	 let
	     val send = send o string_to_outgoing
	 in
	   send (http_unimplemented_header ());
	   send (html_unimplemented_body ())
	 end)

  exception NoOutputAvailable
  
  fun httpd_listen_handler quit_pipe connection_key = 
    let
      val _ = print "httpd_listen_handler entered\n"
      val packet_pipe = B.Pipe.new () : char list option B.Pipe.T
      val char_stream = ref [] : char list ref
    
      fun get_line () =
	let
	  fun loop (#"\^M" :: #"\n" :: cs) accum = (char_stream := cs;
						    implode (rev accum))
            | loop (#"\n" :: cs) accum           = (char_stream := cs;
						    implode (rev accum))
	    | loop (c :: cs) accum = loop cs (c :: accum)
	    | loop nil       nil = 
	      (case B.Pipe.dequeue packet_pipe of
		 NONE => raise NoOutputAvailable
	       | SOME p => loop p nil)
            | loop nil       accum =
	      (case B.Pipe.dequeue packet_pipe of
		 NONE => (char_stream := nil;
			  B.Pipe.enqueue (packet_pipe, NONE);
			  implode (rev accum))
	       | SOME p => loop p nil)

	  val line = loop (! char_stream) []
	in
	  print (line ^ "\n");
	  line
	end

      fun get_chars (n : int) =
	let
	  fun loop _ 0 accum = implode (rev accum)
            | loop (c::cs) n accum = loop cs (n - 1) (c :: accum)
	    | loop nil n accum = 
	      (case B.Pipe.dequeue packet_pipe of
		 NONE => raise NoOutputAvailable
	       | SOME p => loop p n accum)
	in
	  loop (! char_stream) n []
	end

      fun data_handler (_, packet) = 
	B.Pipe.enqueue (packet_pipe, SOME (incoming_to_charlist packet))

      fun status_handler _ =
	B.Pipe.enqueue (packet_pipe, NONE)

      fun connection_handler connection =
	process_request (Lower.Connection_Key.makestring connection_key, 
			 connection, quit_pipe, get_line, get_chars)
    in
      {connection_handler = connection_handler,
       data_handler = data_handler,
       status_handler = status_handler}
    end

  fun httpd_session_fun (Lower.S{connect, listen, extension}) =
    let
      val quit_pipe = B.Pipe.new () : unit B.Pipe.T
      val _ = print "httpd_session_fun entered\n"
    in
      listen (server_pattern, Lower.H (httpd_listen_handler quit_pipe),
		    Lower.Count.Unlimited);
      B.Pipe.dequeue quit_pipe
    end
      handle _ => (error "exception caught in httpd_session_fun"; ())

(*
-----------------------------------------------------------------------
		11.	function install
*)

   fun run_httpd () = 
       (server_starttime := B.V.Time.now ();
        log_header ();
	server_bytesout := read_checkpoint ();
	Lower.session (setup, httpd_session_fun));

(*
-----------------------------------------------------------------------
		12.	functions for reporting statistics
*)

   fun uptime () = 
      B.V.Time.toSeconds(B.V.Time.- (B.V.Time.now (), !server_starttime))

   fun bytesout () = !server_bytesout

   fun accesses () = !server_accesses

(*
		13.	inetd_service definition
*)
   structure Transport = Lower

   local
    val first_time = ref true

    fun handler arguments =
         (if ! first_time then
	   (first_time := false;
	    server_starttime := B.V.Time.now ();
	    log_header ();
	    server_bytesout := read_checkpoint ())
	  else ();
	  httpd_listen_handler (B.Pipe.new ()) arguments)
   in
    val inetd_handler = Lower.H handler

   end
 end
