(*

	FoxNet: The Fox Project's Communication Protocol Implementation Effort
	Edo Biagioni (esb@cs.cmu.edu)
	Fox Project
	School of Computer Science
	Carnegie Mellon University
	Pittsburgh, Pa 15139-3891

	i.	Abstract

	store.sml: functor Store which, given a key type, a value type,
	and a hash function on the keys, produces a STORE structure.

	This implementation uses tries, that is, trees where successively
	MORE significant bits of the key's hash value are used to index
	successive layers. Each node of a tree is either internal, storing
	two subtrees, or terminal, storing either one key-value pair, or a
	list of key-value pairs. A list is needed if two different keys
	have the same hash value.

	ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	1.	functor Store
	2.	datatypes ('key, 'value) node and ('key, 'value) T
	3.	internal function safe_hash
	4.	debugging function check_tree
	5.	function new
	6.	function remove
	7.	function remove_selected
	8.	function add
	9.	function look
	10.	function find
	11.	function size
	12.	function empty
	13.	function map
	14.	function fold
	15.	function makestring


	1.	functor Store
*)

functor Store (structure V: VENDOR): STORE =
 struct

  exception Not_Present_In_Store

  exception Illegal_Store_Empty_Tlist
  exception Illegal_Store_Single_Tlist
  exception Illegal_Store_Same_Hash_Call
  exception Broken_Invariants

  fun makestring_exn Not_Present_In_Store =
       SOME "key not present in store for find operation"
    | makestring_exn Illegal_Store_Empty_Tlist =
       SOME ("unexpected illegal value: Tlist has empty list " ^
	     "(implementation error)")
    | makestring_exn Illegal_Store_Single_Tlist =
       SOME ("unexpected illegal value: Tlist has single element list " ^
	     "(implementation error)")
    | makestring_exn Illegal_Store_Same_Hash_Call =
       SOME ("unexpected illegal value: hash values should differ " ^
	     "(implementation error)")
    | makestring_exn Broken_Invariants =
       SOME "unexpected illegal value"
    | makestring_exn _ = NONE

  structure Trace = Trace (structure V = V
			   val debug_level = NONE
			   val module_name = "store.fun"
			   val makestring = makestring_exn)

(*
	2.	datatypes ('key, 'value) node and ('key, 'value) T
*)

  datatype ('key, 'value) node =
       Empty
     | Internal of ('key, 'value) node * ('key, 'value) node
     | Terminal of word * 'key * 'value
     | Tlist of word * ('key * 'value) list

  datatype ('key, 'value) T =
   T of ('key * 'value) option * ('key -> word) *
        ('key * 'key -> bool) * (('key, 'value) node)

(*
	3.	internal function safe_hash
*)

  fun safe_hash (hash, x) =
       (hash x handle e => Trace.print_raise_again (e, SOME "hash"))

(*
	4.	debugging function check_tree

	To use this for debugging, insert a call to check_tree to check
	every tree value returned; specifically, on functions "new",
	"remove", "remove_selected", "add", "look", and "map".
*)

  fun indent (0, s) = s ^ "\n"
    | indent (n, s) = " " ^ indent (n - 1, s)
  fun makestring_debug (d, Empty) = indent (d, "Empty")
    | makestring_debug (d, Terminal (hash, key, value)) =
       indent (d, Word.toString hash)
    | makestring_debug (d, Tlist (hash, values)) =
       indent (d, Word.toString hash ^ " x" ^
	       Integer.toString (length values))
    | makestring_debug (d, Internal (left, right)) =
       indent (d, "+") ^ makestring_debug (d + 1, left) ^
       makestring_debug (d + 1, right)

  local
   fun check_hash (index, depth, hash) =
        let val mask = Word.- (Word.<< (0w1, depth), 0w1)
        in Word.andb (index, mask) = Word.andb (hash, mask)
        end

   fun check_invariants (index, depth, Empty) = true
     | check_invariants (index, depth, Terminal (hash, _, _)) =
        check_hash (index, depth, hash)
     | check_invariants (index, depth, Tlist (hash, list)) =
        check_hash (index, depth, hash) andalso V.List.length list > 1
     | check_invariants (index, depth, Internal (left, right)) =
        check_invariants (index, Word.+ (depth, 0w1), left) andalso
        check_invariants (Word.orb (index, Word.<< (0w1, depth)),
			  Word.+ (depth, 0w1), right)

  in
   fun check_tree (string, tree) =
        if check_invariants (0w0, 0w0, tree) then tree
	else
	 (Trace.local_print ("tree returned by " ^ string ^
			     " breaks invariants:\n" ^
			     makestring_debug (0, tree));
	  Trace.print_raise (Broken_Invariants, SOME "check_tree"))
  end

(*
	5.	function new
*)

  fun new (hash, eq) = T (NONE, hash, eq, Empty)

(*
	6.	function remove
*)

  local

   fun remove_list (_, _, []) = []
     | remove_list (key, eq, (k, v) :: rest) =
        if eq (key, k) then rest
        else (k, v) :: remove_list (key, eq, rest)

   fun remove_tree (key, shifthed_hash, eq, Empty) = Empty
     | remove_tree (key, shifthed_hash, eq, Internal (left, right)) =
        let val even = Word.andb(shifthed_hash, 0w1) = 0w0
	    val new_shift = Word.>> (shifthed_hash, 0w1)
	    val new_tree =
	         if even then (remove_tree (key, new_shift, eq, left), right)
		 else         (left, remove_tree (key, new_shift, eq, right))
        in case new_tree of
	      (Empty, Empty) => Empty
	    | _ => Internal new_tree
        end
     | remove_tree (key, _, eq, (node as Terminal (_, k, _))) =
        if eq (k, key) then Empty else node
     | remove_tree (_, _, _, Tlist (_, [])) =
        Trace.print_raise (Illegal_Store_Empty_Tlist, SOME "remove_tree")
     | remove_tree (_, _, _, Tlist (_, (h :: []))) =
        Trace.print_raise (Illegal_Store_Single_Tlist, SOME "remove_tree")
     | remove_tree (key, _, eq, Tlist (hash_value, l)) =
        case remove_list (key, eq, l) of
	   [] => Empty
	 | (single_key, single_value) :: [] =>
	    Terminal (hash_value, single_key, single_value)
	 | new_list => Tlist (hash_value, new_list)

  in

   fun remove (T (cache, hash, eq, table), key) =
        T (NONE, hash, eq,
	   remove_tree (key, safe_hash (hash, key), eq, table))
  end (* local *)

(*
	7.	function remove_selected
*)

  local
   fun remove_list_selected (_, []) = []
     | remove_list_selected (f, head :: rest) =
       if f head then remove_list_selected (f, rest)
       else head :: remove_list_selected (f, rest)

   fun remove_tree_selected (f, Empty) = Empty
     | remove_tree_selected (f, Internal (left, right)) =
        (case (remove_tree_selected (f, left),
	       remove_tree_selected (f, right)) of
	    (Empty, Empty) => Empty
	  | pair => Internal pair)
     | remove_tree_selected (f, node as (Terminal (hash, key, value))) =
        if f (key, value) then Empty else node
     | remove_tree_selected (f, Tlist (hash, list)) =
        (case remove_list_selected (f, list) of
	    [] => Empty
	  | (key, value) :: [] => Terminal (hash, key, value)
	  | new_list => Tlist (hash, new_list))

  in

   fun remove_selected (T (cache, hash, eq, table), f) =
	T (NONE, hash, eq, remove_tree_selected (f, table))

  end (* local *)

(*
	8.	function add
*)

  local

   fun add_list (eq, pair, []) = pair :: []
     | add_list (eq, new_pair as (new_k, _), (old_pair as (k, _)) :: rest) =
       if eq (new_k, k) then new_pair :: rest
       else old_pair :: (add_list (eq, new_pair, rest))

   (* new_node takes a terminal node, its hash value, the shifted
      hash of the node to be added, the current depth in the tree
      (0 for a root node), and the parameters of the node to be added,
      and returns a new tree containing both the terminal node and
      the new node *)

   fun new_node (node, node_hash, shifted_hash, depth, hash, key, value, eq) =
    let val shifted_node = Word.>> (node_hash, depth)
	val node_bit = Word.andb (shifted_node, 0w1) = 0w1
	val hash_bit = Word.andb (shifted_hash, 0w1) = 0w1
    in if shifted_node = shifted_hash then
        (Trace.local_print ("shifted_node = " ^ Word.toString shifted_node ^
			    ", shifted_hash = " ^ Word.toString shifted_hash ^
			    ", node_hash = " ^ Word.toString node_hash ^
			    ", hash = " ^ Word.toString hash);
	 Trace.print_raise (Illegal_Store_Same_Hash_Call, SOME "new_node"))
       else
        case (node_bit, hash_bit) of
           (false, false) =>		(* both zero: create new left child *)
	    Internal (add_tree (node, Word.>> (shifted_hash, 0w1),
				depth + 0w1, hash, key, value, eq), Empty)
         | (true, true) =>		(* both one: create new right child *)
	    Internal (Empty, add_tree (node, Word.>> (shifted_hash, 0w1),
				       depth + 0w1, hash, key, value, eq))
	 | (true, false) =>		(* one, zero: split *)
	    Internal (Terminal (hash, key, value), node)
	 | (false, true) =>		(* zero, one: split *)
	    Internal (node, Terminal (hash, key, value))
    end

   and add_tree (Empty, _, _, hash, key, value, _) =
        Terminal (hash, key, value)
     | add_tree (Internal (left, right),
		 shifted_hash, depth, hash, key, value, eq) =
        if Word.andb (shifted_hash, 0w1) = 0w0 then
	 Internal (add_tree (left, Word.>> (shifted_hash, 0w1), depth + 0w1,
			     hash, key, value, eq), right)
        else
	 Internal (left, add_tree (right, Word.>> (shifted_hash, 0w1),
				   depth + 0w1, hash, key, value, eq))
      | add_tree ((node as Terminal (h, k, v)), shifted_hash, depth,
		  hash, key, value, eq) =
	 if eq (key, k) then Terminal (hash, key, value)
	 else if hash = h then
	  Tlist (hash, [(key, value), (k, v)])
         else
	  new_node (node, h, shifted_hash, depth, hash, key, value, eq)
      | add_tree ((n as Tlist (_, [])), _, _, _, _, _, _) =
	 Trace.print_raise (Illegal_Store_Empty_Tlist, SOME "add_tree")
      | add_tree ((n as Tlist (_, (_ :: []))), _, _, _, _, _, _) =
	 Trace.print_raise (Illegal_Store_Single_Tlist, SOME "add_tree")
      | add_tree ((node as Tlist (h, values as (_ :: _))), shifted_hash,
		  depth, hash, key, value, eq) =
	 if hash = h then Tlist (h, add_list (eq, (key, value), values))
         else new_node (node, h, shifted_hash, depth, hash, key, value, eq)

  in

   fun add (T (cache, hash, eq, table), key, value) =
        ((let val hash_value = safe_hash (hash, key)
              val new_tree = add_tree (table, hash_value, 0w0,
				       hash_value, key, value, eq)
          in T (SOME (key, value), hash, eq, new_tree)
	  end)
	   handle x => (Trace.local_print ("hash " ^
					   Word.toString (safe_hash
							  (hash, key)) ^
					   ", tree\n" ^
					   makestring_debug (0, table));
			Trace.print_raise_again (x, SOME "add")))

  end (* local *)

(*
	9.	function look
*)

  local
   fun look_list (_, _, []) = NONE
     | look_list (eq, key, ((k, v) :: rest)) =
       if eq (k, key) then SOME v else look_list (eq, key, rest)

   fun look_tree (_, _, _, _, Empty) = NONE
     | look_tree (eq, shifted_hash, hash, key, Internal (left, right)) =
        if Word.andb (shifted_hash, 0w1) = 0w0 then
	 look_tree (eq, Word.>> (shifted_hash, 0w1), hash, key, left)
        else
	 look_tree (eq, Word.>> (shifted_hash, 0w1), hash, key, right)
     | look_tree (eq, shifted_hash, hash, key, Terminal (_, k, v)) =
        if eq (key, k) then SOME v else NONE
     | look_tree (_, _, _, _, Tlist (_, [])) =
	Trace.print_raise (Illegal_Store_Empty_Tlist, SOME "look_tree")
     | look_tree (_, _, _, _, Tlist (_, _ :: [])) =
	Trace.print_raise (Illegal_Store_Single_Tlist, SOME "look_tree")
     | look_tree (eq, shifted_hash, hash, key, Tlist (h, l)) =
        if hash = h then look_list (eq, key, l)
        else NONE

   fun look_table (key, hash, eq, table) =
    let val hash_key = safe_hash (hash, key)
    in look_tree (eq, hash_key, hash_key, key, table)
    end

  in
   fun look (T (NONE, hash, eq, table), key) =
        let val res = look_table (key, hash, eq, table)
        in case res of
            SOME v => (SOME (T (SOME (key, v), hash, eq, table), v))
          | NONE => NONE
        end
     | look (store as (T ((SOME (cache_key, value)), hash, eq, table)), key) =
        if eq (cache_key, key) then SOME (store, value)
	else
	 let val res = look_table (key, hash, eq, table)
	 in case res of
	       SOME v => SOME (T (SOME (key, v), hash, eq, table), v)
	     | NONE => NONE
	 end
  end (* local *)

(*
	10.	function find
*)

  fun find (store, key) =
       case look (store, key) of
          NONE => raise Not_Present_In_Store
	| SOME v => v

(*
	11.	function size
*)

  fun size (T (_, _, _, table)) =
       let fun lsize Empty = 0
	     | lsize (Internal (left, right)) = (lsize left) + (lsize right)
	     | lsize (Terminal _) = 1
	     | lsize (Tlist (_, l)) = length l
       in lsize table
       end

(*
	12.	function empty
*)

  fun empty (T (_, _, _, Empty)) = true
    | empty _ = false

(*
	13.	function map
*)

  fun map f (T (cache, hash, eq, table)) =
       let fun lmap Empty = Empty
	     | lmap (Internal (left, right)) = Internal (lmap left, lmap right)
	     | lmap (Terminal (hash, key, value)) =
	        Terminal (hash, key, f (key, value))
             | lmap (Tlist (h, l)) =
	        Tlist (h, (List.map (fn (p as (k, v)) => (k, f p)) l))
       in T (NONE, hash, eq, lmap table)
       end

(*
	14.	function fold
*)

  fun fold f (T (_, hash, eq, table)) init = 
       let fun flatten Empty = []
	     | flatten (Internal (left, right)) =
		(flatten left) @ (flatten right)
	     | flatten (Terminal (hash, key, value)) = (key, value) :: []
	     | flatten (Tlist (hash, l)) = l
       in V.List.fold f (flatten table) init
       end

(*
	15.	function makestring
*)

  fun makestring (store, elstring, separator) =
       let fun foldstring (element, "") = elstring element
	     | foldstring (element, prev) =
	        (elstring element) ^ separator ^ prev
       in fold foldstring store ""
       end

 end (* struct *)
