(*

	FoxNet: The Fox Project's Communication Protocol Implementation Effort
	Edo Biagioni (esb@cs.cmu.edu)
	Fox Project
	School of Computer Science
	Carnegie Mellon University
	Pittsburgh, Pa 15139-3891

	i.	Abstract

	Care and feeding of tcp's retransmit queue.

	Page numbers in the comments refer to page numbers in RFC 793,
	the definition of TCP [793]. Other commonly-referred to documents are:
	[1122] RFC 1122, Requirements for Internet Hosts (1989)
	[VJ88] Van Jacobson, Congestion Avoidance and Control,
	       ACM SIGCOMM-88, August 1988, pp 314-329
	[KP87] Karn and Partridge, Improving Round-Trip Time Estimates
	       in Reliable Transport Protocols, ACM SIGCOMM-87,
	       August 1987, pp 2-7


	ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	1.	functor Tcp_Retransmit
	2.	function retransmit
	3.	internal function b4less
	4.	internal function recompute_srtt
	5.	internal function recompute_cwnd
	6.	function acknowledge


	1.	functor Tcp_Retransmit
*)

functor Tcp_Retransmit (structure Tcp_Tcb: TCP_TCB
			structure B: FOX_BASIS): TCP_RETRANSMIT =
 struct

  structure Tcb = Tcp_Tcb

  val one32 = Word32.fromInt 1
  val zero32 = Word32.fromInt 0

(*
	2.	function retransmit
*)

  fun retransmit (tcb as (Tcp_Tcb.Tcb {resend, to_do, timers, ...}),
		  out_seg, timeout) =
       let val send_time = Tcp_Tcb.current_time ()
	   val resent = false
	   val record = (out_seg, send_time, resent)
       in if Tcp_Tcb.D.empty (! resend) then
	   (#start_resend timers) timeout
	  else ();
	  resend := Tcp_Tcb.D.add_to_back (! resend, record);
	  tcb
       end

(*
	3.	internal function b4less
*)

  local

   val signed_min_b4 =
        (case Word32.fromString "0x80000000" of
	   SOME w => w
	 | NONE => let exception Bug in raise Bug end)

   fun b4less (less, greater) =
        Word32.> (Word32.- (less, greater), signed_min_b4)

(*
	4.	internal function recompute_srtt

	In Jacobson's description, M is the time since sent (we use
        delta), SA is the scaled average (srtt), SD is the scaled
        deviation (srtd), rto is the retransmit timeout (srto).
*)

   fun recompute_srtt (start, tcb) =
        let val Tcp_Tcb.Tcb {srtt, srtd, to_do, srto, ...} = tcb
	    val deltams = Word32.- (Tcp_Tcb.current_time (), start)
	    val old_srtt = ! srtt
	    val old_rtt = Word32.>> (old_srtt, 0w3) (* old_srtt quot 8 *)
	    val centered_delta = deltams - old_rtt
	    val abs_delta = if centered_delta > signed_min_b4 then
	                     Word32.- (zero32, centered_delta)
			    else centered_delta
	    val new_srtt = old_srtt + centered_delta
	    val old_srtd = ! srtd
	    val new_srtd = Word32.- (Word32.+ (old_srtd, abs_delta),
				     Word32.>> (old_srtd, 0w2))
	                                     (* old_srtd quot 4 *)
	    val new_rtt = Word32.>> (new_srtt, 0w3) (* new_srtt quot 8 *)
(* the second computation for new_srto, below, is the standard.  However,
   preliminary tests show this one to have better throughput. *)
	    val new_srto = Word32.+ (Word32.>> (new_srtt, 0w2), new_srtd)
(*
	    val new_srto = Word32.>> (Word32.+ (Word32.>> (new_srtt, 0w2),
						new_srtd), 0w1)
*)
	    val log = Tcp_Tcb.Log_Event
	               (Tcp_Tcb.Log.Round_Trip_Time_Change
			{new_rtt = new_rtt, old_rtt = old_rtt,
			 new_rto = new_srto, old_rto = ! srto})
	    val toString = Word32.toString
	in srtt := new_srtt;
	   srtd := new_srtd;
	   srto := new_srto;
	   to_do := Tcp_Tcb.Q.add (! to_do, log)
	end

(*
	5.	internal function recompute_cwnd

	Note that all quantities are multiplied by mss compared to
        Jacobson's description (appendix B of Congestion Avoidance and
        Control, SIGCOMM 88). This gives us sizes in bytes rather than
        in units of MSS, and also helps the integer division produce
        the desired results.  For example, the last case of delta is
        really 1/cwnd, since mss^2/cwnd = mss/(cwnd[*mss]/mss) =
        mss(1/cwnd).
*)

   fun recompute_cwnd (Tcp_Tcb.Tcb {cwnd, ssthresh, mss, to_do, ...}) =
        let val int_mss = Word16.toInt (! mss)
	    val int_cwnd = ! cwnd
	    val delta = if int_cwnd < ! ssthresh then int_mss
			else (int_mss * int_mss) quot int_cwnd
	    val new_cwnd = int_cwnd + delta
	    val log_data = {new_size = new_cwnd, old_size = int_cwnd}
	    val log = Tcp_Tcb.Log.Congestion_Window_Change log_data
	in cwnd := (if new_cwnd < 0xffffff then new_cwnd else 0xffffff);
	   to_do := Tcp_Tcb.Q.add (! to_do, Tcp_Tcb.Log_Event log)
	end

(*
	6.	function acknowledge
*)

   val probe = Tcp_Tcb.Probe_Window

   fun seq_len (Tcp_Tcb.Seg {seq, len, syn_flag, fin_flag, options, ...}) =
        let val correction = if syn_flag orelse fin_flag then one32
			     else zero32
	in (seq, Word32.+ (len, correction))
	end

  in (* local *)

   fun acknowledge (state, tcb, ack, send_time) =
       let val Tcp_Tcb.Tcb {to_do, resend, ...} = tcb
	   val first = Tcp_Tcb.D.first (! resend)
	   val found_one = case first of NONE => false | _ => true
(* ack_loop takes acked segments off the resend queue.  If the resend
   queue has been emptied, clear the resend and window timers, and
   probe the window if there are segments waiting to be sent.

   The start time is taken to be either the time since sending the
   segment (which does not correspond to RFC 1323, p. 36, which
   says to use the elapsed time since the first segment in the
   retransmission queue) or the time in the timestamp option. *)
	   fun ack_loop NONE =
	        if found_one andalso Tcp_Tcb.D.empty (! resend) then
		 let val Tcp_Tcb.Tcb {timers, queued, ...} = tcb
		 in (#stop_resend timers) ();
		    (#stop_window timers) ();
		    if not (Tcp_Tcb.D.empty (! queued)) then
		     to_do := Tcp_Tcb.Q.add (! to_do, probe)
		    else ()
		 end
		else ()
	     | ack_loop (SOME (new_queue, (segment, start, resent))) =
	        let val start_time = case send_time of SOME x => x | _ => start
		    val (seq, len) =
		          case segment of
			     Tcp_Tcb.Fast_Out {seq, len, ...} => (seq, len)
			   | Tcp_Tcb.Fast_Empty {seq, ...} => (seq, zero32)
			   | Tcp_Tcb.Timestamp_Out {seq, len, ...} =>
			      (seq, len)
			   | Tcp_Tcb.Timestamp_Empty {seq, ...} =>
			      (seq, zero32)
			   | Tcp_Tcb.Out_Seg {seg, data} => seq_len seg
		    (* val last_seq = Word32.+ (seq, len) *)
		    val delta = Word32.- (ack, Word32.+ (seq, len))
		in (* in-line b4less if b4less (ack, last_seq) then *)
		   if Word32.> (delta, signed_min_b4) then
		    ()			(* this segment is not acked yet. *)
		   else
		    (if not resent andalso delta = zero32 then
		      (* Karn's algorithm [1122 p. 95] *)
		      (recompute_srtt (start_time, tcb);
		       (* Jacobson's slow start algorithm [1122 p. 90] *)
		       recompute_cwnd tcb)
		     else ();
		     resend := new_queue;
		     ack_loop (Tcp_Tcb.D.first new_queue))
		end
       in ack_loop first;
	  if found_one then
	   to_do := Tcp_Tcb.Q.add (! to_do, Tcp_Tcb.Complete_Send ack)
	  else ();
	  tcb
       end

  end (* local *)
 end (* struct *)
