(*

        FoxNet: The Fox Project's Communication Protocol Implementation Effort
        Edo Biagioni (esb@cs.cmu.edu)
	Ken Cline (Kenneth.Cline@cs.cmu.edu)
        Brian Milnes (milnes@cs.cmu.edu)
        Nick Haines (nickh@cs.cmu.edu)
        Fox Project
        School of Computer Science
        Carnegie Mellon University
        Pittsburgh, Pa 15139-3891

		i.	Abstract
	tcp.sig: signature for protocols in the TCP stack


		ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	1.	signature TCP_STATUS
	2.	signature TCP_PROTOCOL
	3.	connection management
	4.	session statistics
	5.	sharing


	1.	signature TCP_STATUS

	TCP returns a status to the status handler whenever: urgent
	data is received, the connection is closed by the peer, or the
	connection is reset by the peer.  Urgent data is only
	delivered to the status handler, which can pass it to the data
	handler if so desired.
*)

signature TCP_STATUS =
 sig
  include PRINTABLE
  type incoming
  datatype tcp_status =
      Urgent_Data of incoming
    | Connection_Closing
    | Connection_Reset
  sharing type T = tcp_status
 end (* sig *)

(*
	2.	signature TCP_PROTOCOL

	A TCP protocol uses lower-layer addresses together with local
	and remote port numbers as TCP addresses.  It is also possible
	to leave the local port unspecified if the remote port and
	peer are specified.

	The lower_layer_address will typically be an IP address (e.g.,
	a ubyte4), but should not be specified in this signature. This
	should make it easier to use TCP directly over ethernet or ATM.
*)

signature TCP_PROTOCOL =
 sig
  include TRANSPORT_PROTOCOL

  structure Tcp_Status: TCP_STATUS

(*
	3.	connection management

	Tcp windows:

	TCP does flow control to avoid swamping the receiver. In order
	to do flow control properly, the TCP must know when the user
	has finished processing the data it received. TCP can either
	assume that this is when the data handler returns, or it can
	let some handler explicitly signal how much data has been
	consumed.  Note that in the explicit case, if the handler
	never signals that data has been consumed, TCP will assume the
	receiver is swamped and refuse to accept additional data.
	Since this is undesirable, the default is to assume the data
	has been consumed whenever the data handler (or status
	handler, for urgent data) returns.  If the handler will
	consume the data long before it returns, or if the handler
	may not return, the handler should explicitly signal when
	the data has been consumed.

	In the default case, "add_to_window" has no effect.

	Send_Immediately:

	Some implementations of TCP try to accumulate small packets
	before sending them, so they can be sent more efficiently.
	RFC 1122, p. 98 specifies that a TCP MUST allow this algorithm
	to be disabled on an individual connection.  Send_Immediately
	is off by default and can only be turned on.

	Logging and statistics:

	This is useful debugging/tuning/optimizing information, but
	is made optional since keeping track of this information may
	slow down the protocol implementation.
*)

  structure Log: TCP_LOG

  datatype tcp_connection_extension =
      Tcp_Connection_Extension of
        {urgent_send: Outgoing.T -> unit,

	 automatic_window_update: unit -> bool,
	 update_automatically: unit -> unit,
	 update_explicitly: unit -> unit,
	 add_to_window: int -> unit,

	 send_immediately: unit -> unit,

	 start_logging: unit -> unit,
	 get_log: unit -> (Log.time * Log.tcp_event) list,
	 start_stats: unit -> unit,
	 get_stats: unit -> Log.tcp_stats option}

(*
	4.	session statistics
*)

  datatype tcp_session_extension =
      Tcp_Session_Extension of
        {bytes_sent: unit -> Word64.word,
	 bytes_received: unit -> Word64.word,
	 failed_sends: unit -> Word64.word,
	 packets_rejected: unit -> Word64.word}

(*
	5.	sharing
*)

  sharing Status = Tcp_Status
      and type Tcp_Status.incoming = Incoming.T
      and type session_extension = tcp_session_extension
      and type connection_extension = tcp_connection_extension
      and type additional_listen_extension = unit

 end (* sig *)
