(*

	FoxNet: The Fox Project's Communication Protocol Implementation Effort
	Edo Biagioni (esb@cs.cmu.edu)
	Nick Haines (nickh@cs.cmu.edu)
	Brian Milnes (milnes@cs.cmu.edu)
	Fox Project
	School of Computer Science
	Carnegie Mellon University
	Pittsburgh, Pa 15139-3891

	i.	Abstract

	eth.fun: An ethernet protocol functor, and a test Functor.

	ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	1.	functor types
	2.	structure Eth_Number
	3.	structure Eth_Protocol
	4.	structure Eth_Address
	5.	structure Eth_Pattern
	6.	other sub-structures
	7.	extension types
	8.	structure Eth_Header
	9.	functions for structure Connection
	10.	structure Connection
	11.	export objects declared in signature

	1.	functor types
*)

functor Ethernet (structure Device: DEVICE_PROTOCOL
		    sharing type Device.Incoming.T = Device.Outgoing.T
                  structure B: FOX_BASIS
	          val debug_level: int ref option): ETHERNET_PROTOCOL =
 struct
  structure Trace = Trace (structure V = B.V
			   val debug_level = debug_level
			   val module_name = "eth.fun"
			   val makestring = Device.X.makestring)

(*
	2.	structure Eth_Number
*)

  structure Eth_Number: ETH_NUMBER =
   struct
    type T = Word48.word

    fun new n = n

    fun convert n = n

    fun equal (n1: T, n2: T) = n1 = n2

    fun hash n = Word.fromLargeWord (Word48.toLargeWord (Word48.>> (n, 0w20)))

    fun makestring address =
         let fun makestring_one byte = Word48.toString byte
	     val mask = Word48.fromInt 0xff
	     fun make_byte shift =
	          Word48.andb (Word48.>> (address, shift), mask)
	     val b0 = makestring_one (make_byte 0w40)
	     val b1 = makestring_one (make_byte 0w32)
	     val b2 = makestring_one (make_byte 0w24)
	     val b3 = makestring_one (make_byte 0w16)
	     val b4 = makestring_one (make_byte  0w8)
	     val b5 = makestring_one (make_byte  0w0)
	 in b0 ^ ":" ^ b1 ^ ":" ^ b2 ^ ":" ^ b3 ^ ":" ^ b4 ^ ":" ^ b5
	 end
   end

(*
	3.	structure Eth_Protocol
*)

  structure Eth_Protocol: KEY =
   struct
    type T = Word16.word
    val makestring = Integer.toString o Word16.toInt
    fun equal (a: T, b) = a = b
    val hash = Word.fromLargeWord o Word16.toLargeWord
   end

(*
	4.	structure Eth_Address
*)

  structure Eth_Address: ETH_ADDRESS =
   struct
    type eth_number = Eth_Number.T
    type eth_protocol = Eth_Protocol.T
    datatype address = Address of {eth: eth_number, proto: eth_protocol}
    type T = address
    fun makestring (Address {eth, proto}) =
         Eth_Number.makestring eth ^ "^" ^ Eth_Protocol.makestring proto
    fun hash (Address {eth, proto}) = Eth_Number.hash eth
    fun equal (a: T, b) = a = b
   end

(*
	5.	structure Eth_Pattern
*)

  structure Eth_Pattern: ETH_PATTERN =
   struct
    type eth_number = Eth_Number.T
    type eth_protocol = Eth_Protocol.T
    datatype pattern = Complete of {eth: eth_number, proto: eth_protocol}
                     | Partial of {proto: eth_protocol}
    type T = pattern
    fun makestring (Complete address) = Eth_Address.makestring
					  (Eth_Address.Address address)
      | makestring (Partial {proto}) = Integer.toString (Word16.toInt proto)
    fun hash (Complete address) = Eth_Address.hash
				    (Eth_Address.Address address)
      | hash (Partial {proto}) = Word.fromLargeWord (Word16.toLargeWord proto)
    fun equal (a: T, b) = a = b
   end

(*
	6.	other sub-structures
*)

   structure Unit =
    struct
     type T = unit
     fun makestring () = "()"
     fun hash () = 0
     fun equal ((), ()) = true
    end

(*
	7.	extension types
*)

  datatype eth_connection_extension = 
    Eth_Connection_Extension of
      {connection_address: Eth_Address.T}

  datatype eth_session_extension =
    Eth_Session_Extension of
      {local_address: Eth_Number.T,
       packets_sent: unit -> Word64.word,
       packets_received: unit -> Word64.word,
       failed_sends: unit -> Word64.word,
       packets_rejected: unit -> Word64.word,
       minimum_packet_size: Word.word,
       maximum_packet_size: Word.word}

(*
	8.	structure Eth_Header
*)

  structure Eth_Header = Eth_Header (structure In = Device.Incoming
				     structure Out = Device.Outgoing
				     structure B = B)

(*
	9.	functions for structure Connection
*)

  fun lower_setup setup = setup

  val min_eth_payload = 0w46
  val max_eth_payload = 0w1500

  datatype protocol_state =
      PS of Word48.word *
            Word64.word ref * Word64.word ref *
	    Word64.word ref * Word64.word ref

  fun init_proto (setup,
		  Device.S {extension = Device.Dev_Session_Extension
			                  {local_address = a, ...}, ...}, _) =
       let val a48 = Word48_Array.to a
	   val local_address = Word48_Array.U_Big.F.head a48
	   val zero64 = Word64.fromInt 0
	   val packets_sent = ref zero64
	   val packets_received = ref zero64
	   val failed_sends = ref zero64
	   val packets_rejected = ref zero64
	   val extension = Eth_Session_Extension
	                   {local_address = local_address,
			    packets_sent = fn _ => ! packets_sent,
			    packets_received = fn _ => ! packets_received,
			    failed_sends = fn _ => ! failed_sends,
			    packets_rejected = fn _ => ! packets_rejected,
			    minimum_packet_size = min_eth_payload,
			    maximum_packet_size = max_eth_payload}
	   val state = (local_address,
			packets_sent, packets_received,
			failed_sends, packets_rejected)
       in (PS state, extension)
       end

  fun fin_proto _ = ()

  fun resolve _ = SOME ()

  fun make_key (_, address, _, _) = address

  fun map_pattern (_, pattern, _) = SOME ((), ())

  fun match (_, Eth_Pattern.Complete address1, _, address2) =
       Eth_Address.equal
         (Eth_Address.Address address1, address2)
    | match (_, Eth_Pattern.Partial {proto = proto1}, _,
	     Eth_Address.Address {eth, proto = proto2}) =
       proto1 = proto2

  type connection_state = protocol_state

  fun init_connection (state, address, _) =
       (state, Eth_Connection_Extension {connection_address = address})

  fun fin_connection _ = ()

  val header_size = Eth_Header.size {self = Word48.fromInt 0,
				     peer = Word48.fromInt 0,
				     proto = Word16.fromInt 0}

  val one64 = Word64.fromInt 1

  fun send (Eth_Address.Address {eth, proto},
	    PS (local_address, packets_sent, _, failed_sends, _)) =
       let val header = {self = local_address, peer = eth, proto = proto}
	   val bytes = Device.Outgoing.uninitialized header_size
	   val _ = Eth_Header.marshal (bytes, header) 0w0
	   fun send_packet packet = 
	        let val packet_size = Device.Outgoing.size packet
		in if packet_size > max_eth_payload then
		    (failed_sends := Word64.+ (! failed_sends, one64);
		     Trace.print_raise (Device.X.Send "packet too large",
					SOME "send"))
		   else if packet_size < min_eth_payload then
		    (failed_sends := Word64.+ (! failed_sends, one64);
		     Trace.print_raise (Device.X.Send "packet too small",
					SOME "send"))
		   else
		    (packets_sent := Word64.+ (! packets_sent, one64);
		     [Device.Outgoing.join (bytes, packet)])
		end
       in
	  send_packet
       end

  val broadcast_eth = Word48.notb (Word48.fromInt 0)
  fun identify (_, PS (_, _, _, _, packets_rejected)) packet =
       (let val ({self, peer, proto}, _) = Eth_Header.unmarshal (packet, 0w0)
	in [Eth_Address.Address {eth = peer, proto = proto},
	    Eth_Address.Address {eth = broadcast_eth, proto = proto}]
	end)
	 handle Eth_Header.Extern =>
	         (packets_rejected := Word64.+ (! packets_rejected, one64);
		  Trace.local_print "unable to unmarshal packet, discarding";
		  [])
	      | x =>
	         (packets_rejected := Word64.+ (! packets_rejected, one64);
		  Trace.print_raise_again (x, SOME "identify"))

  fun receive (_, PS (_, _, packets_received, _, _)) packet =
       let val (_, new_packet) = Device.Incoming.split (packet, header_size)
       in packets_received := Word64.+ (! packets_received, one64);
	  SOME new_packet
       end

  fun undelivered _ _ = ()

  fun lower_status (_, lower_key) status =
       Trace.local_print ("received status " ^ Device.Status.makestring status)

(*
	10.	structure Connection
*)

  structure Connection =
      Connection (structure Lower = Device
		  structure Setup = Device.Setup
		  structure Address = Eth_Address
		  structure Pattern = Eth_Pattern
		  structure Connection_Key = Eth_Address
		  structure Incoming = Device.Incoming
		  structure Outgoing = Device.Outgoing
		  structure Status = Unit
		  structure Count = Device.Count
		  structure X = Device.X
		  type connection_extension = eth_connection_extension
		  type listen_extension = unit
		  type session_extension = eth_session_extension
		  type connection_state = connection_state
		  type protocol_state = protocol_state
		  val lower_setup = lower_setup
		  val init_proto = init_proto
		  val fin_proto = fin_proto
		  val resolve = resolve
		  val make_key = make_key
		  val map_pattern = map_pattern
		  val match = match
		  val init_connection = init_connection
		  val fin_connection = fin_connection
		  val send = send
		  val identify = identify
		  val receive = receive
		  val undelivered = undelivered
		  val lower_status = lower_status
		  structure B = B
		  val module_name = "eth.fun"
		  val debug_level = debug_level)

(*
	11.	export objects declared in signature
*)

  open Connection

 end (* struct *)

