(*

	FoxNet: The Fox Project's Communication Protocol Implementation Effort
	Edoardo Biagioni (Edoardo.Biagioni@cs.cmu.edu)
	Brian Milnes (Brian.Milnes@cs.cmu.edu)
	Ken Cline (Kenneth.Cline@cs.cmu.edu)
        Nick Haines (Nick.Haines@cs.cmu.edu)
	Fox Project
	School of Computer Science
	Carnegie Mellon University
	Pittsburgh, Pa 15139-3891

		i.	Abstract

	Install DNS for host_id name lookup (parsing).


		ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	1.	functor Dns_Lookup
	2.	type declarations
	3.	sub-structures DnsM and Cache
	4.	internal function lookup_session
	5.	function lookup
	6.	function inverse_lookup


	1.	functor Dns_Lookup
*)

functor Dns_Lookup (structure B: FOX_BASIS
		    structure Host_Id: TRANSPORT_HOST_ID
		    structure Dns: DNS_PROTOCOL
		      sharing type Host_Id.T = Dns.Message.internet_address
		    val debug_level: int ref option): DNS_LOOKUP =
 struct

  structure Trace = Trace (structure V = B.V
			   val debug_level = NONE
			   val makestring = fn _ => NONE
			   val module_name = "dnslookup.fun")

(*
	2.	type declarations
*)

  type transport_setup = Dns.Setup.T

  type host_id = Host_Id.T

  type setup = {domain_list: string list,
		servers: host_id list,
		transport: transport_setup}

(*
	3.	sub-structures DnsM and Cache
*)

  structure DnsM = Dns.Message

  structure Cache = Dns_Cache (structure B = B
			       structure DnsM = DnsM
			       val debug_level = debug_level)

(*
	4.	internal function lookup_session
*)

  fun lookup_session (setup: transport_setup, queries, server, extract_result,
		      result_pipe) () =
       let val query_list = ref queries
	   exception No_Queries
	   fun pop_query () =
	        case ! query_list of
		   [] => raise No_Queries
		 | head :: rest => head before query_list := rest
	   fun handler key =
	        let val done_pipe = B.Pipe.new ()
		    val done = ref false
		    fun timeout () =
		         (B.Scheduler.sleep 30000;
			  if ! done then ()
			  else
			   (Trace.debug_constant_string "timeout";
			    B.Pipe.enqueue (done_pipe, NONE)))
		    fun connection_handler (Dns.C {send, ...}) =
		         (send (pop_query ());
			  B.Scheduler.fork timeout;
			  B.Pipe.enqueue (result_pipe,
					  B.Pipe.dequeue done_pipe);
			  done := true)
		    fun data_handler (Dns.C {send, ...}, message) =
	                 (Cache.add message;
			  case extract_result message of
	                    SOME x =>
			     B.Pipe.enqueue (done_pipe, SOME x)
	                  | NONE =>
			     ((send (pop_query ()))
			      handle No_Queries =>
			              B.Pipe.enqueue (done_pipe, NONE)
				   | x =>
			              Trace.print_raise_again
				        (x, SOME "data_handler")))
		    fun status_handler _ = B.Pipe.enqueue (done_pipe, NONE)
		in {connection_handler = connection_handler,
		    data_handler = data_handler,
		    status_handler = status_handler}
		end

	   fun run_session (Dns.S {connect, ...}) =
		connect (server, Dns.H handler)

	   fun get_question (DnsM.Message {question = [q], ...}) = q
	     | get_question _ =
	         let exception Impossible in raise Impossible end
	   fun try_cache (q :: rest, try_server) =
	        (case Cache.lookup (get_question q) of
		    SOME m =>
		     (case extract_result m of
		         SOME r =>
			  B.Pipe.enqueue (result_pipe, SOME r)
		       | NONE =>
			  try_cache (rest, try_server))
		  | NONE =>
		     try_cache (rest, try_server))
	     | try_cache ([], try_server) = try_server ()
       in try_cache (queries, fn () => Dns.session (setup, run_session))
       end

(*
	5.	function lookup
*)

    fun lookup {domain_list, servers = [], transport} name =
	 (case Host_Id.parse name of
	     NONE =>
	      (Trace.local_print "empty servers list, DNS query fails";
	       NONE)
	   | SOME ip => SOME ip)
      | lookup {domain_list, servers = (server :: _), transport} name =
	 let val result_pipe = B.Pipe.new ()
	     fun append_domain "." = name
	       | append_domain d = name^"."^d
	     val name_list = map append_domain domain_list
	     fun build_query name =
		  let val header =
		           DnsM.Header {query = true, opcode = DnsM.Query,
					rcode = DnsM.No_Error, aa = false,
					tc = false, rd = true, ra = false}
		      val question =
		           DnsM.Question {name = DnsM.Domain_Name.parse name,
					  rr_qtype = DnsM.A_Q,
					  rr_class = DnsM.IN}
		  in DnsM.Message {header = header, question = [question],
				   answer = [], authority = [],
				   additional = []}
		  end
	     val queries = map build_query name_list

	     fun extract_host_id (DnsM.Message {answer, ...}) =
		  let fun loop [] = NONE
			| loop (DnsM.RR {rr_type = DnsM.A a, ...} :: _) =
			   SOME a
		        | loop (_::rest) = loop rest
		  in loop answer
		  end
	       | extract_host_id (DnsM.Parse_Error e_message) =
		  (Trace.local_print ("Parse error from DNS server: " ^
				      e_message);
		   NONE)

	 in case Host_Id.parse name of
	       SOME ip => SOME ip
	     | NONE =>
		(B.Scheduler.fork (lookup_session (transport, queries,
						   server,
						   extract_host_id,
						   result_pipe));
		 B.Pipe.dequeue result_pipe)
	 end

(*
	6.	function inverse_lookup
*)

    fun inverse_lookup {domain_list, servers = [], transport} host_id = NONE
      | inverse_lookup {domain_list, servers = (server :: _),
			transport} host_id =
         let val result_pipe = B.Pipe.new ()
	     val query_header =
	          DnsM.Header
	            {query = true, opcode = DnsM.Query, rcode = DnsM.No_Error,
		     aa = false, tc = false, rd = true, ra = false} 
	     val qname =
		  DnsM.Domain_Name.invert
		    (DnsM.Domain_Name.parse (Host_Id.makestring host_id))
	     val question =
	          DnsM.Question {name = qname, rr_qtype = DnsM.PTR_Q,
				 rr_class = DnsM.IN}
	     val query =
	          DnsM.Message {header = query_header, question = [question],
				answer = [], authority = [], additional = []}
	     fun extract_host_name (DnsM.Message {answer, ...}) =
                  let fun loop [] = NONE
		        | loop (DnsM.RR {rr_type=DnsM.PTR addr, ...} :: _) =
		           SOME (DnsM.Domain_Name.makestring addr)
		        | loop (_::rest) = loop rest
		  in loop answer
		  end
	       | extract_host_name (DnsM.Parse_Error e_message) =
		  (Trace.local_print ("Parse error from DNS server: " ^
				      e_message);
		   NONE)
	 in B.Scheduler.fork (lookup_session (transport, [query], server,
					      extract_host_name, result_pipe));
	    B.Pipe.dequeue result_pipe
	 end

 end
