(*

	FoxNet: The Fox Project's Communication Protocol Implementation Effort
	Edoardo Biagioni (esb@cs.cmu.edu)
	Ken Cline    (Ken.Cline@cs.cmu.edu)
	Nick Haines  (Nick.Haines@cs.cmu.edu)
	Brian Milnes (Brian.Milnes@cs.cmu.edu)
	Fox Project
	School of Computer Science
	Carnegie Mellon University
	Pittsburgh, Pa 15139-3891

		i.	Abstract

	ethdev.fun: an implementation of a general device signature to
	send and receive binary ethernet packets through the OSF1
	packetfilter pseudo-device.



		ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	1.	functor Ethernet_Device
	2.	structure Trace
	3.	internal functions raise_fun and handle_fun
	4.	structure External
	5.	functor types, exceptions and state
	6.	makestrings
	7.	management of read buffers
	8.	internal function receive_timeout
	9.	function send
	10.	function session


	1.	functor Ethernet_Device
*)

functor Ethernet_Device (structure Packet_Filter: PACKET_FILTER
			 structure B: FOX_BASIS
			 val debug_level: int ref option): RAW_DEVICE =
 struct

(*
	2.	structure Trace
*)

  structure Trace = Trace (structure V = B.V
			   val debug_level = debug_level
			   val module_name = "ethdev.fun"
			   val makestring = fn _ => NONE)
  val do_prints = Trace.debug_on ()
  val do_traces = Trace.trace_on ()

(*
	3.	internal functions raise_fun and handle_fun

	Defined so we can consistently handle and report exceptions.
*)

  fun raise_fun (s, x) = Trace.print_raise (x, SOME s)

  fun handle_fun (x, s, f) =
       (if B.V.Control.exnName x <> "Receive" orelse Trace.trace_on () then
	 Trace.print_handled (x, SOME s)
	else ();
        f ())

(*
		4.	structure External
*)

  structure External = Protocol_External (structure B = B
					  val debug_level = debug_level)

(*
	5.	functor types, exceptions and state
*)

  exception Session_Already_Open

  structure Setup =
   struct
    type T = string
    fun makestring interface_name = "interface: " ^ interface_name
    fun equal (n0: T, n1) = (n0 = n1)
    fun hash _ = 0w0
   end (* struct *)

  type session =
        {send: External.T -> unit,
         local_address: Word_Array.T,
         packets_sent: unit -> Word64.word,
         packets_received: unit -> Word64.word,
         read_timeouts: unit -> Word64.word,
         failed_sends: unit -> Word64.word,
         packets_rejected: unit -> Word64.word}

(*
		6.	makestrings
*)

  fun makestring_header h =
       let fun address_string offset =
	     let
	       fun s i = Word8.toString (Word_Array.W8.Big.F.nth (h, i+offset))
	     in
	       s 0w0^"."^s 0w1^"."^s 0w2^"."^s 0w3^"."^s 0w4^"."^s 0w5
	     end
	   val to_string = address_string 0w0
	   val from_string = address_string 0w6
	   val proto_string = (Word16.toString
			       (Word_Array.W16.Big.F.nth
				(Word_Array.to16 h, 0w6)))
       in
	 "[ " ^ to_string ^ " -> " ^ from_string ^ " : " ^ proto_string ^ " ]"
       end (* let *)

  fun print_header (data, len, s) =
       Trace.local_print (s ^ " " ^ makestring_header data ^
			  "(" ^ Word.toString len ^ ")")

  fun debug_print_packet (data, length, s) =
       if Word_Array.W8.U_Big.F.length data < 0w14 then
        Trace.local_print ("error in debug_print (" ^ s ^ "), array length " ^
			   Word.toString (Word_Array.W8.U_Big.F.length data) ^
			   " minimum is 14.")
       else
	let fun tostring i = (Word8.toString
			      (Word_Array.W8.Big.F.nth (data, i)))
	    fun format_data (i, 1) = tostring i
	      | format_data (i, n) = (tostring i ^ " " ^
				      format_data (i+0w1, n-1))
	    val banner = "\nethdev.fun: " ^ s ^ " packet "
	    val header_s = makestring_header data
	    val length_s = "(" ^ Word.toString length ^ ")\n"
	    val data_s = if length>0w45 then format_data (0w14, 45) ^ " ..."
			 else format_data (0w14, 45)
	    val packet_string = banner ^ header_s ^ length_s ^ data_s
	in Trace.local_print packet_string
	end

   fun trace_packet (data, s) =
        print_header (data, Word_Array.W8.U_Big.F.length data, s)
   fun debug_packet (data, s) =
        debug_print_packet (data, Word_Array.W8.U_Big.F.length data, s)
   val log_packet = if do_prints then debug_packet
                    else if do_traces then trace_packet
		    else fn _ => ()

(*
	7.	management of read buffers

	Buffers are aligned on a word+2 boundary, to make the ethernet
	data aligned on a word boundary.
*)

  functor Buffers (val buffer_size: int
		   val max_buffers: int) =
   struct
    local
     structure W = Word_Array.W8.U_Big.F
     val zero = Word8.fromInt 0
     val real_buffer_size = Word.fromInt buffer_size + 0w2
     fun new_buffer () =
          Word_Array.from8 (W.seek (W.create_uninitialized real_buffer_size, 0w2))
     val cache = ref [new_buffer ()]
     val present = ref 1
    in

     fun get () =
          case ! cache of
	     [] => new_buffer ()
	   | head :: rest =>
	      (cache := rest;
	       present := ! present - 1;
	       head)

     fun recycle b =	(* also add a buffer if ! present < max_buffer. *)
          (cache := b :: (! cache);
	   present := ! present + 1;
	   if ! present < max_buffers then
	    (cache := (new_buffer ()) :: (! cache);
	     present := ! present + 1)
	   else ())

    end (* local *)
   end (* struct *)

(*
	8.	internal function receive_timeout

	Receive_timeout is a polling function for the packetfilter
	pseudo-device.  We install it in the scheduler, and it is
	called whenever the scheduler has no active threads and
	whenever the scheduler switches threads. The numerical
	parameter is the number of milli-seconds (ms) that the
	function should take.
*)

  val one64 = Word64.fromInt 1

  local

   val buffer_size = 1650		(* don't know the exact lower limit:
					   should be 1546 for ethernet, but
					   there are mach headers and such *)

   (* max buffers is the number of buffers we try to build up to
      during idle times. If you make it too large you needlessly
      take up memory space and g.c. too often; if you make it too
      small, it will take longer (on average) to receive a packet
      that is ready to be received. *)
   val max_buffers = 10

   structure Buffers = Buffers (val buffer_size = buffer_size
				val max_buffers = max_buffers)

  in (* local *)

   fun receive_timeout (fd, self, handler, packets_received,
			read_timeouts) ms =
        (B.Scheduler.fork_limited_time (! self);
	 if Packet_Filter.select (fd, ms) then
	  let val buffer = Buffers.get ()
	      val read_args = (fd, buffer, buffer_size, 0)
	      val bytes_read = Packet_Filter.readi read_args
	  in log_packet (Word_Array.to8 buffer, "received");
	     packets_received := Word64.+ (! packets_received, one64);
	     (((! handler) (External.new buffer))
	      handle x => handle_fun (x, "packet handler", fn _ => ()))
	  end
	 else read_timeouts := Word64.+ (! read_timeouts, one64))
  end (* local *)

(*
	9.	function send
*)

  fun send (fd, closed, packets_sent, failed_sends) packet =
     let val bufferList = rev (External.fold (packet, op :: ,  []))
     in (( (*log_packet (Word_Array.to8 buffer, "sending ");*)
          if !closed then
             Trace.local_print "ignoring attempt to send on closed device"
          else
             (Packet_Filter.writev (fd, bufferList);
             packets_sent := Word64.+ (! packets_sent, one64);
             ()))
         handle x =>
            (Trace.print_handled (x, SOME "send");
             failed_sends := Word64.+ (! failed_sends, one64)))
     end

(*
		10.	function session
*)

  val zero64 = Word64.fromInt 0
  datatype 'a result = Result of 'a | Exception of exn
  fun session (interface_name, f) =
       case ((let val fd = Packet_Filter.pfopen interface_name
	          val addr = Packet_Filter.get_ethernet_address fd
		  fun dummy_receive _ = ()
		  fun dummy_handler _ = ()
		  val sent = ref zero64
		  val received = ref zero64
		  val failed = ref zero64
		  val timeouts = ref zero64
		  val closed = ref false
		  val session_arg = {send = send (fd, closed, sent, failed),
				     local_address = addr,
				     packets_sent = fn _ => ! sent,
				     packets_received = fn _ => ! received,
				     read_timeouts = fn _ => ! timeouts,
				     failed_sends = fn _ => ! failed,
				     packets_rejected = fn _ => zero64}
		  val (session_handler, data_handler) = f session_arg
		  val handler: (External.T -> unit) ref = ref data_handler
(* The value "receive" is initialized to "ref dummy_receive" since
immediately initializing it to its final value has problems of
recursive definition (ML supports recursive definition, but only for
functions, not refs).  *)
		  val receive: (int -> unit) ref = ref dummy_receive
		  val _ = receive := receive_timeout (fd, receive, handler,
						      received, timeouts);
		  val _ = B.Scheduler.fork_limited_time (! receive)
		  val result = ((Result (session_handler ()))
				handle x => Exception x)
(* reset "receive" to keep from forking in the limited queue, and reset
   "handler" and "closed" to prevent reception and sending by the higher
   protocol. *)
	      in receive := dummy_receive;
	         handler := dummy_handler;
		 closed := true;
	         Packet_Filter.close fd;
		 result
	      end)
	       handle x =>
		       Trace.print_raise_again (x, SOME "session")) of
	  Result x => x
	| Exception x => raise x

 end (* struct *)
