(*

	FoxNet: The Fox Project's Communication Protocol Implementation Effort
	Brian Milnes (Brian.Milnes@cs.cmu.edu)
	Nick Haines  (Nick.Haines@cs.cmu.edu)
	Edoardo Biagioni (esb@cs.cmu.edu)
	Fox Project
	School of Computer Science
	Carnegie Mellon University
	Pittsburgh, Pa 15139-3891

		i.	Abstract

	This file, sim.fun, provides a simulated ethernet driver with
	a possibly large number of simulated hosts.  It requires
	wire.fun to work.

		ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	1.	functor Ethernet_Device_Simulator


		1.	functor Ethernet_Device_Simulator
*)

functor Ethernet_Device_Simulator (val local_address: Word48.word
				   structure Wire: WIRE
				   structure B: FOX_BASIS
				   val debug_level: int ref option
				   val name: string): RAW_DEVICE =
 struct

  structure Trace = Trace (structure V = B.V
			   val debug_level = debug_level
			   val module_name = "sim.fun (" ^ name ^ ")"
			   val makestring = fn _ => NONE)

  structure External = Protocol_External (structure B = B
					  val debug_level = debug_level)

  fun send packet =
       (Trace.debug_print (fn _ => "sending packet of size " ^
			   Word.toString (External.size packet) ^
			   ", packet is (first 60 bytes) " ^
			   External.makestring_max (packet, 0w60));
	if External.size packet < 0w60 then
	 Trace.local_print ("packet " ^
			    External.makestring_max (packet, 0w60) ^
			    " in ethernet simulator has size " ^
			    Word.toString (External.size packet) ^
			    ", required minimum size is 60")
	else
	 let val size = External.size packet
	     val data = External.sub (packet, {start = 0w0, length = size})
	 in Trace.debug_print (fn _ => "sending packet of size " ^
			       Word.toString size ^
			       ", data size " ^
			       Word.toString
			          (Word_Array.W8.U_Big.F.length
				   (Word_Array.to8 data)));
	    Wire.send data
	 end)

  structure Setup =
   struct
    type T = string
    fun makestring s = s
    fun equal (a: T, b) = a = b
    fun hash _ = 0w0
   end (* struct *)

  type session =
        {send: External.T -> unit,
         local_address: Word_Array.T,
         packets_sent: unit -> Word64.word,
         packets_received: unit -> Word64.word,
         read_timeouts: unit -> Word64.word,
         failed_sends: unit -> Word64.word,
         packets_rejected: unit -> Word64.word}

  exception Session_Already_Open

  fun session (_, handler) =
       let val address = Word48_Array.from
	                    (Word48_Array.Big.F.create (local_address, 0w1))
	   fun zero64 () = Word64.fromInt 0
	   val session_value = {send = send,
				local_address = address,
				packets_sent = zero64,
				packets_received = zero64,
				read_timeouts = zero64,
				packets_rejected = zero64,
				failed_sends = zero64}
	   val (connection, receive) = handler session_value
       in ((Wire.register (address, receive o External.new))
	   handle x => (Trace.print_handled (x, NONE);
			Trace.print_raise (Session_Already_Open, NONE)));
	  ((connection ()
	    before Wire.unregister address)
	   handle x => (Wire.unregister address;
			Trace.print_raise_again (x, NONE)))
       end

 end (* struct *)
