(*

        FoxNet: The Fox Project's Communication Protocol Implementation Effort
        Edo Biagioni (esb@cs.cmu.edu)
        Fox Project
        School of Computer Science
        Carnegie Mellon University
        Pittsburgh, Pa 15139-3891

	i.	Abstract
	arpeth.fun: ARP for ethernet


	ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	1.	functor Arp_Eth
	2.	internal structures
	3.	internal types and values
	4.	structure Arp
	5.	exported types and values


	1.	functor Arp_Eth
*)

functor Arp_Eth (structure Eth: ETHERNET_PROTOCOL
		   sharing type Eth.Incoming.T = Eth.Outgoing.T
		 val arp_protocol_number: Word16.word
		 structure B: FOX_BASIS
	         val debug_level: int ref option):
                ADDRESS_RESOLUTION_PROTOCOL =
 struct
  structure Trace = Trace (structure V = B.V
			   val debug_level = debug_level
			   val module_name = "arp.fun(eth)"
			   val makestring = Eth.X.makestring)

(*
	2.	internal structures
*)

  structure Hardware_Address =
      Protocol_Extern48_Big (structure In = Eth.Incoming
			     structure Out = Eth.Outgoing
			     structure B = B)

  structure Protocol_Address: ARP_PROTOCOL_EXTERN =
   struct
    type T = Word_Array.T
    type regular_extern_in = Eth.Incoming.T
    type protocol_extern_in = regular_extern_in * Word.word
    type extern_in = protocol_extern_in
    type extern_out = Eth.Outgoing.T
    type cursor = Word.word
    exception Extern
    val size = Word_Array.W8.U_Big.F.length o Word_Array.to8
    fun marshal (outgoing, data) cursor =
	 (Eth.Outgoing.update (outgoing, cursor, data);
	  size data + cursor)
    fun unmarshal ((incoming, length), position) =
         (Eth.Incoming.sub (incoming, {start = position, length = length}),
	  position + length)
    fun equal (array1, array2) =
         case (Word_Array.W8.U_Big.F.next (Word_Array.to8 array1),
	       Word_Array.W8.U_Big.F.next (Word_Array.to8 array2)) of
	    (NONE, NONE) => true
	  | (SOME (f1, r1), SOME (f2, r2)) =>
	     f1 = f2 andalso equal (Word_Array.from8 r1, Word_Array.from8 r2)
	  | _ => false
    fun hash array =
         case Word_Array.W8.U_Big.F.next (Word_Array.to8 array) of
	    NONE => 0w0
	  | SOME (first, rest) =>
	     Word.fromLargeWord (Word8.toLargeWord first) +
	     hash (Word_Array.from8 rest)
    fun makestring array =
         case Word_Array.W8.U_Big.F.next (Word_Array.to8 array) of
	    NONE => ""
	  | SOME (first, rest) =>
	     Word8.toString first ^ "." ^
	     makestring (Word_Array.from8 rest)
   end

(*
	3.	internal types and values
*)

  val null_hardware_address = Word48.fromInt 0

  fun local_hardware_address
    (Eth.Eth_Session_Extension {local_address, packets_sent, packets_received,
				failed_sends, packets_rejected,
				minimum_packet_size, maximum_packet_size}) =
       local_address

  fun min_max_packet
    (Eth.Eth_Session_Extension {local_address, packets_sent, packets_received,
				failed_sends, packets_rejected,
				minimum_packet_size, maximum_packet_size}) =
       (minimum_packet_size, maximum_packet_size)

  fun broadcast_address protocol =
    Eth.Eth_Address.Address {eth = Word48.notb (Word48.fromInt 0),
			     proto = protocol}

  fun broadcast_pattern protocol = 
    Eth.Eth_Pattern.Partial {proto = protocol}

  fun lower_address (eth_number, eth_protocol) =
    Eth.Eth_Address.Address {eth = eth_number, proto = eth_protocol}

  fun lower_pattern (eth_number, eth_protocol) =
    Eth.Eth_Pattern.Complete {eth = eth_number, proto = eth_protocol}

  val hardware_type = Word16.fromInt 1

  val arp_timeout = if Trace.debug_on () then 1000 else 100
  val arp_resend = 5

  val lower_header_size = 14

(*
	4.	structure Arp
*)

  structure Arp: ADDRESS_RESOLUTION_PROTOCOL =
       Arp (structure Lower = Eth
	    structure Hardware_Address = Hardware_Address
	    structure Protocol_Address = Protocol_Address
	    val null_hardware_address = null_hardware_address
	    val local_hardware_address = local_hardware_address
	    val arp_protocol_number = arp_protocol_number
	    val broadcast_address = broadcast_address
	    val broadcast_pattern = broadcast_pattern
	    val lower_address = lower_address
	    val lower_pattern = lower_pattern
	    val min_max_packet = min_max_packet
	    val hardware_type = hardware_type
	    val arp_timeout = arp_timeout
	    val arp_resend = arp_resend
	    structure B = B
	    structure Trace = Trace) 

(*
	5.	exported types and values
*)

  open Arp

 end (* struct *)
