(*

	FoxNet: The Fox Project's Communication Protocol Implementation Effort
	Brian Milnes (Brian.Milnes@cs.cmu.edu)
	Nick Haines  (Nick.Haines@cs.cmu.edu)
	Edoardo Biagioni <Edoardo.Biagioni@cs.cmu.edu>
	Fox Project
	School of Computer Science
	Carnegie Mellon University
	Pittsburgh, Pa 15139-3891

		i.	Abstract

	event.sig: signature for an event synchronization module.
	An event manager has two major operations: wait and signal. A
	thread doing a wait is suspended until another thread calls a
	matching signal.

		ii.	Table of Contents

	i.	Abstract
	ii.	Table of Contents
	1.	signature SIMPLE_EVENT_QUEUE
	2.	signature EVENT_QUEUE


		1.	signature SIMPLE_EVENT_QUEUE
*)

signature SIMPLE_EVENT_QUEUE =
 sig

  type T

  val new: unit -> T

  val clear: T -> unit

  val size: T -> int
  val empty: T -> bool

(* the parameters to "wait" are the queue, the event ID, and a
   "post-processing" closure that will be called only after the
   wait has been queued.
 *)
  val wait: T * (unit -> unit) -> unit

  val select: T list * (unit -> unit) -> unit

(* returns true if it returns as a result of a signal, false otherwise *)
  val wait_timeout: T * (unit -> unit) * int -> bool

(* the parameters to "signal" are the queue, a function to determine
   whether an event matches this signal, and a value to be returned
   by the corresponding "wait".

   returns true if it found a waiting thread, false otherwise.
 *)
  val signal: T -> bool

 end

(*
		2.	signature EVENT_QUEUE
*)

signature EVENT_QUEUE =
 sig

(* an event queue can be used to suspend a thread (wait) until the
   awaited event is signaled (signal). The 'a represents the type
   of the event, the '1b the type of the value waited for. *)
  eqtype ('1a, '1b) T

  val new: unit -> ('1a, '1b) T

  val clear: ('1a, '1b) T -> unit

  val size: ('1a, '1b) T -> int
  val empty: ('1a, '1b) T -> bool

(* the parameters to "wait" are the queue, the event ID, and a
   "post-processing" closure that will be called only after the
   wait has been queued.
 *)
  val wait: {queue: ('1a, '1b) T, event: '1a, while_waiting: unit -> unit}
          -> '1b

  val select: {queues: (('1a, '1b) T * '1a) list,
	       while_waiting: unit -> unit}
            -> '1b

  val wait_timeout: {queue: ('1a, '1b) T, event: '1a,
		     while_waiting: unit -> unit, timeout: int} -> '1b option

(* the parameters to "signal" are the queue, a function to determine
   whether an event matches this signal, and a value to be returned
   by the corresponding "wait".

   If there is no matching "wait", this function has no effect, other
   than perhaps calling the match function multiple times. The
   value returned is NONE.

   If there are multiple matching "wait"s, only the oldest is woken up
   and given the value. Signal awakens the "event" coroutine and
   immediately returns (SOME event). Callers should not depend on the
   event coroutine being called immediately as implementations may or
   may not choose to do this.
 *)
  val signal: {queue: ('1a, '1b) T, match: '1a -> bool, value: '1b}
            -> '1a option

 end
