(* Sample usage:
    structure TH = Build_Httpd(val hostname = (*SOME "lair.foxnet.cs.cmu.edu"*)
                                              NONE
                              val log_file = "/tmp/httpd.log"
                              val checkpoint_file = "/tmp/httpd.ckpt"
                              val port = 80);
    TH.Httpd.run_httpd ();
*)
     

functor Build_Httpd (val hostname : string option
                     val log_file : string
                     val checkpoint_file : string
                     val port     : int) =
struct

 local

   structure B = Fox_Basis
   structure Base = Build_Eth_Dev (structure B = Fox_Basis
				   val debug_level = NONE)

   structure Init = Initialize (structure Device = Base.Raw
				structure V = Fox_Basis.V
(* modify this function to avoid using the system files. *)
				val hostname_to_ip = fn _ => NONE
				val interface_number = 0)

   structure Stack = Build_Dns (structure Device = Base.Dev
				structure B = Fox_Basis
				val tcp_window_size = 32 * 1024
				val tcp_user_timeout = Word32.fromInt 300000
				val eth_debug_level = NONE
				val arp_debug_level = NONE
				val ip_debug_level = NONE
				val tcp_debug_level = NONE
				val udp_debug_level = NONE
				val dns_debug_level = NONE)

   val network_setup = Init.Ip.setup ()
   val transport_setup = Stack.Tcp.Transport_Setup.Setup (network_setup)

   structure Tcp = Stack.Tcp
   structure Ip = Stack.Ip
   structure Eth = Stack.Eth

 in

   structure Httpd = Httpd (structure B = Fox_Basis
			    structure Lower = Tcp
			    val server_pattern = 
				Tcp.Transport_Pattern.Local_Specified 
				{local_port = Word16.fromInt 80}
			    val hostname = 
				(case hostname of
				     NONE => Init.Ip.local_hostname ()
				   | SOME h => h) 
			    val setup = transport_setup
			    val default_document = "index.html"
			    val log_file = log_file
			    val checkpoint_file = checkpoint_file
			    val external_fun = fn _ => "")
       

   structure Bignum_Message = Bignum_Message(structure Ints = W64
                                             structure B = B)   

   
   fun bytes_out _ = Bignum_Message.message 
                         Bignum_Message.Capitalize
                          (Httpd.bytesout ())

   fun eth_acc () =
      let val pkts = 
        Eth.session 
          (Init.Dev.interface (),
	   fn (Eth.S{extension=Eth.Eth_Session_Extension{packets_received,
							 ...},...}) => 
              packets_received ())
      in
	  Word64.toString pkts
      end

   fun eth_sent () =
      let val pkts = 
        Eth.session 
          (Init.Dev.interface (),
	   fn (Eth.S{extension=Eth.Eth_Session_Extension{packets_sent,
							 ...},...}) => 
	   packets_sent())
      in
        Word64.toString pkts
      end

   fun ip_rcv () =
      let val pkts = 
        Ip.session (transport_setup,
		    fn (Ip.S{extension = Ip.Session_Extension{packets_received,
							      ...},...}) => 
                       packets_received())
      in
        Word64.toString pkts
      end

   fun ip_sent () =
      let val pkts = 
        Ip.session (transport_setup,
		    fn (Ip.S{extension = Ip.Session_Extension{packets_sent,
							      ...},...}) => 
                       packets_sent())
      in
        Word64.toString pkts
      end

   fun days_up   _ = Int.toString (Httpd.uptime () div 86400)
   fun hrs_up    _ = Int.toString ((Httpd.uptime () mod 86400) div 3600)
   fun mins_up   _ = Int.toString ((Httpd.uptime () mod 3600) div 60)
   fun accesses  _ = Int.toString (Httpd.accesses())

   val _ = Httpd.install_fn "bytes_out" bytes_out;
   val _ = Httpd.install_fn "days_up"   days_up;
   val _ = Httpd.install_fn "hrs_up"    hrs_up;
   val _ = Httpd.install_fn "mins_up"   mins_up;
   val _ = Httpd.install_fn "accesses"  accesses;
   val _ = Httpd.install_fn "eth_acc"   eth_acc;
   val _ = Httpd.install_fn "eth_sent"  eth_sent;
   val _ = Httpd.install_fn "ip_rcv"    ip_rcv;
   val _ = Httpd.install_fn "ip_sent"   ip_sent; end (* local *)

end (* struct *)
