(* crc.sml
 *
 * COPYRIGHT (c) 1994 AT&T Bell Laboratories.
 *)

signature CRC =
sig
  type crc
  val new : unit -> crc
  val start : string -> crc
  val append : crc -> int -> unit
  val extract: crc -> string
end

abstraction CRC : CRC = 
  struct

    val wtoi = Word.toInt
    val itow = Word.fromInt

  (* 128-bit CRC.  
   * The call `append crc c' corresponds to eight steps of a shift register
   * circuit, shifting in one bit of character c from the left in each step.
   * See Figure 2.16 in Bertsekas and Gallager: Data Networks (1987), 
   * or Figure 3-32 in Siewiorek and Swarz: The Theory and Practice 
   * of Reliable System Design (Digital Press, 1982). 
   *)

    type crc = int array * int ref

  (* The prime generator polynomial is 1 + x + x^2 + x^7 + x^128.
   * Reversing the low coefficient bits we have 1110.0001 = 225
   *)
    val poly = 0w225

    val table = let
	  fun init n = let
	        fun f (0w0,_,r) = wtoi r
		  | f (i,p,r) = if Word.andb(i, 0w1) <> 0w0
		      then f(Word.>>(i,0w1), p+p, Word.xorb(p,r))
		      else f(Word.>>(i,0w1), p+p, r)
	        in
		  f(itow n, poly, 0w0)
	        end
	  in
	    Vector.tabulate(256, init)
	  end

    fun new() = (Array.array(16, 0), ref 0)

    fun start s = (Array.fromList(map Char.ord (explode s)), ref 0)

    fun extract(a,ref i) =
        let fun f(j,s) = if j=i then Char.chr(Array.sub(a, j)) :: s
	                 else f((j+1) mod 16, Char.chr(Array.sub(a, j)) :: s)
         in implode(f((i+1) mod 16, nil))
        end

    fun append (a,r as ref i) c =
       let val rmbyte = Array.sub(a, i)   (* rightmost byte *)
           val hilo = Vector.sub(table, rmbyte)
           val hi = Word.>>(itow hilo, 0w8)
           val lo = Word.andb(itow hilo, 0w255)
           val i' = (i+15) mod 16         (* leftmost byte  *)
        in Char.chr c;
           Array.update(a,i, wtoi (Word.xorb(itow c, hi)));
           Array.update(a,i',wtoi (Word.xorb(itow (Array.sub(a, i')), lo)));
           r := (i+1) mod 16              
       end

  end (* abstraction CRC *)
