(* Copyright 1994 by AT&T Bell Laboratories *)
(* mtderiv.sml *)

signature MTDERIV = 
  sig
    val mtderivDec: Modules.env * Absyn.dec -> Absyn.dec
  end

structure MTDeriv : MTDERIV = 
struct

local
  open Array List Types Variables Access BasicTypes TypesUtil Unify Absyn
     Overload ErrorMsg PrettyPrint PPUtil PPType PPAbsyn Modules
  infix 9 sub
  infix -->

in 

val printDepth = Control.Print.printDepth
val anyErrors = ref false
val err = ErrorMsg.errorNoFile(ErrorMsg.defaultConsumer(),anyErrors)

exception AntiUnify
fun antiUnifyTy(type1,type2) =
    let val type1 = headReduceType(prune type1)
	val type2 = headReduceType(prune type2)
     in
	(case (type1, type2) 
          of (VARty var1,VARty var2) =>
		if eqTyvar(var1,var2) then ()
                else raise AntiUnify
	   | (CONty(tycon1,args1),CONty(tycon2,args2)) =>
		if eqTycon(tycon1,tycon2) then
		    List2.app2 antiUnifyTy (args1,args2)
		else raise AntiUnify 
	   | _ => raise AntiUnify 
        )
    end


fun groundTy(VARty (ref (INSTANTIATED ty))) = groundTy ty
  | groundTy(VARty (ref _)) = true
  | groundTy(CONty(_,l)) = 
      let fun h(t,b) = b andalso (groundTy t) 
       in foldr h true l
      end
  | groundTy(POLYty{tyfun=TYFUN{body,...},...}) = groundTy body
  | groundTy(IBOUND _) = false
  | groundTy _ = true

fun refNewDcon(DATACON{name,const,rep,typ,sign,orig}) = 
     DATACON{name=name,const=const,rep=rep,typ=refPatType,sign=sign,orig=orig}

val sortFields = Sort.sort (fn ((LABEL{number=n1,...},_),
				(LABEL{number=n2,...},_)) => n1>n2)
fun map2 f nil = (nil,nil)
  | map2 f (hd::tl) = let val (x,y) = f(hd)
                          val (xl,yl) = map2 f tl
		       in (x::xl,y::yl)
                      end

val say = Control.Print.say

exception NotThere

fun mtderivDec (env,dec) = let

(* the tyvar set *)
val tvsets : (tyvar list ref) = ref ([] : tyvar list)

fun addtvs l = 
  let val z = !tvsets
   in (tvsets := (l@z))
  end

fun bounded (v : tyvar) = 
  let fun h(a::r) = if eqTyvar(a,v) then true else h r
        | h [] = false
   in h(!tvsets)
  end

fun renameTy (typ,menv) = 
  let fun bind(z as (ref k)) = 
        let val new = VARty (ref k)
         in (menv := (z,new)::(!menv)); new
        end
      fun lookup tv =
        let fun find [] = bind(tv)
              | find((tv',ty)::rest) = 
                  if eqTyvar(tv,tv') then ty else find rest
         in find(!menv)
	end
      fun h ty = 
        (case ty 
          of VARty (ref (INSTANTIATED t)) => h t
           | VARty z => if (bounded z) then lookup z else ty
           | CONty(c,l) => CONty(c,map h l)
           | _ => ty)
   in h typ
  end

(* the environment that holds the instances for each polymorphic types *)
exception INSTANCEMAP
type info = ty list option
val InstTable : info Intmap.intmap = Intmap.new(32,INSTANCEMAP)
val getTable = Intmap.map InstTable
val addTable = Intmap.add InstTable
fun addlist(v,t) = 
     ((case (getTable v) 
        of NONE => ()
         | SOME l => (addTable(v,SOME(t::l)))) 
      handle _ => addTable(v,SOME[t]))
fun markvis(v) = addTable(v,NONE)
fun infer(v) = 
  ( (* say ("Inferring the lvar "^(makestring(v))^" #################\n"); *)
    case (getTable v)
    of SOME (a::r) => 
         let val menv = ref([]: (tyvar * ty) list)
             val ren = fn t => renameTy(t,menv)
             val a' = headReduceType(a)
             fun h([],a') = SOME a'
               | h(b::z,a') = 
                   (antiUnifyTy(ren b,a'); h(z,a')) handle _ => NONE

          in (* say ("Someting "^(makestring(v))^" #################\n"); *)
             if groundTy a' then h(r,ren a') else NONE
         end
     | _ => NONE) handle _ => NONE

(* pretty-printing for debugging purpose *)
val ppType = PPType.ppType env
val ppPat = PPAbsyn.ppPat env
val ppExp = PPAbsyn.ppExp(env,NONE)
val ppRule = PPAbsyn.ppRule(env,NONE)
val ppVB = PPAbsyn.ppVB(env,NONE)
val ppRVB = PPAbsyn.ppRVB(env,NONE)

fun generalizeTy(VALvar{typ,path,...}, userbound: tyvar list,
		 occ:occ, loc) : unit =
  let val should_complain = ref true
      fun complain() = 
        if !should_complain  (* don't complain again *)
	then (should_complain := false;
	      err loc COMPLAIN "nongeneric weak type variable"
                (fn ppstrm =>
		     (add_newline ppstrm; ppSymPath ppstrm path;
		      add_string ppstrm " : "; ppType ppstrm (!typ))))
        else ()
      val index = ref 0  (* counts no of type variables bound *)
      fun next() = let val i = !index in index := i+1; i end
      val sign = ref([]: {weakness:int,eq:bool} list)
      val uenv = array(length userbound, UNDEFty)
      fun pos(tv,tv'::rest) = if eqTyvar(tv,tv') then 0 else pos(tv,rest)+1
        | pos(_,[]) = raise NotThere
      val menv = ref([]: (tyvar*ty) list)
      fun lookup tv =
        let fun find [] = raise NotThere
              | find((tv',ty)::rest) = 
                  if eqTyvar(tv,tv') then ty else find rest
         in find(!menv)
	end
      fun bind(b as (_,ty)) = (menv := b::(!menv); ty)
      fun gen(ty) =     
	case ty
	 of VARty(ref(INSTANTIATED ty)) => gen ty
	  | VARty(tv as ref(OPEN{depth,weakness,eq,kind})) =>
	      (case kind
		of FLEX[(lab,_)] =>
 		     (err loc COMPLAIN 
	                 ("unresolved flex record\n\
			  \(can't tell what fields there are besides #"
			  ^Symbol.name lab ^ ")\n") nullErrorBody;
		      WILDCARDty)
		 | FLEX _ =>
                     (err loc COMPLAIN 
		         ("unresolved flex record"
                          ^"(need to know the names of ALL the fields\n\
                          \in this context)")
                         (fn ppstrm =>
			      (PPType.resetPPType();
			       add_newline ppstrm;
			       add_string ppstrm "type: ";
                               ppType ppstrm ty));
                      WILDCARDty)
		 | META =>
                     if depth > lamdepth occ
		     then (if weakness > generalize_point occ
                           then lookup tv handle NotThere =>
                             (sign := {weakness=weakness,eq=eq} :: !sign;
			      bind(tv,IBOUND (next()) ))
                           else (if toplevel occ then complain() else (); ty))
		     else ty (* raise SHARE *)
		 | UBOUND name =>
		     (let val i = pos(tv,userbound)
		       in if depth > lamdepth occ then 
                            (case (uenv sub i)
			      of UNDEFty =>
			           let val weakness = 
				         if weakness > generalize_point occ
				         then weakness
				         else (complain(); 
                                               generalize_point occ+1)
				       val new = IBOUND(next())
				    in update(uenv,i,new);
				       sign := ({weakness=weakness,eq=eq}
					        :: !sign);
				       new
			           end
			       | ty => ty)  (* raise SHARE *)
		          else (err loc COMPLAIN
			          ("explicit type variable cannot be \
                                    \generalized at scoping declaration: " 
                                   ^(tyvar_printname tv)) nullErrorBody;
			        tv := INSTANTIATED WILDCARDty;
			        WILDCARDty)
		      end handle NotThere => ty))
          | CONty(tyc,args) => CONty(tyc, map gen args) (*shareMap*)
          | WILDCARDty => WILDCARDty
          | _ => (err loc COMPLAIN "generalizeTy -- bad arg" nullErrorBody;
		  (* conjecture that this error is impossible now *)
		  WILDCARDty)
      val ty = gen(!typ)
      val _ = addtvs (map #1 (!menv))
   in typ := POLYty{sign = rev(!sign), abs = abscount occ,
                    tyfun = TYFUN{arity=(!index),body=ty}}
      (* always produce a POLYty, even when no variables are generalized.
	 this is to save the abs value for use in instantiateType below. *)
  end
  | generalizeTy _ = impossible "typecheck generlizeTy 121"
  
fun generalizePat(pat: pat, userbound: tyvar list, occ: occ, loc) =
  let fun gen p = case p 
         of VARpat v => generalizeTy(v,userbound,occ,loc)
       	  | RECORDpat{fields,...} => app (gen o #2) fields
	  | APPpat(_,_,arg) => gen arg
	  | CONSTRAINTpat(pat,_) => gen pat
	  | LAYEREDpat(varPat,pat) => (gen varPat; gen pat)
	  | _ => ()
   in gen pat
  end

fun applyType(ratorTy: ty, randTy: ty) : ty =
  let val resultType = mkMETAty()
   in unifyTy(ratorTy, (randTy --> resultType));
      resultType
  end

fun patType(pat: pat, depth, loc, vis) : pat * ty =
 case pat
  of WILDpat => (pat,mkRefMETAty depth)
   | VARpat(VALvar{typ,access=LVAR v,...}) =>
       if vis then (typ := mkRefMETAty depth; (pat,!typ))
       else (case (infer v)
              of NONE => 
                   (case (!typ) 
                     of POLYty _ => (typ := mkRefMETAty depth; (pat,!typ))
                      | UNDEFty => (typ := mkRefMETAty depth; (pat,!typ))
                      | _ => (pat,!typ))
               | SOME t => (typ := t; (pat, t)))
   | VARpat(VALvar{typ, ...}) => 
       (case (!typ) 
         of POLYty _ => (typ := mkRefMETAty depth; (pat,!typ))
          | UNDEFty => (typ := mkRefMETAty depth; (pat,!typ))
          | _ => (pat,!typ))
   | INTpat (_,ty) => (pat,ty)
   | WORDpat (_,ty) => (pat,ty)
   | REALpat _ => (pat,realTy)
   | STRINGpat _ => (pat,stringTy)
   | CHARpat _ => (pat,charTy)
   | CONpat(dcon as DATACON{typ,...},_) => 
       let val ty = applyPoly(typ,Root)
        in case typ of POLYty _ => (CONpat(dcon,SOME ty),ty)
                     | _ => (CONpat(dcon,NONE),ty)
       end
   | RECORDpat{fields,flex,typ} =>
       (* fields assumed already sorted by label *)
       let fun g(lab,pat') = 
             let val (npat,nty) = patType(pat',depth,loc,vis)
              in ((lab,npat), (lab,nty))
             end
           val (fields',labtys) = map2 g fields
           val npat = RECORDpat{fields=fields',flex=flex,typ=typ}
        in if flex
           then let val ty = VARty(mkTyvar(mkFLEX(labtys,depth)))
                 in typ := ty; (npat,ty)
                end
           else (npat,recordTy(labtys))
       end
   | VECTORpat(pats,_) => 
       (let val (npats,ntys) = 
                 map2 (fn pat => patType(pat,depth,loc,vis)) pats
            val nty = foldr (fn (a,b) => (unifyTy(a,b); b)) (mkRefMETAty depth) ntys
         in (VECTORpat(npats,nty), CONty(vectorTycon,[nty]))
        end handle Unify(mode) => 
              (err loc COMPLAIN "vector pattern type failure" nullErrorBody;
               (pat,WILDCARDty)))
   | ORpat(p1, p2) => 
       let val (p1, ty1) = patType(p1,depth,loc,vis)
	   val (p2, ty2) = patType(p2,depth,loc,vis)
        in (unifyTy (ty1, ty2))
	     handle (Unify mode) => 
	       (err loc COMPLAIN ("or-patterns don't agree (" ^ mode ^ ")")
		  (fn ppstrm =>
   		    (PPType.resetPPType();
   		     add_newline ppstrm; add_string ppstrm "expected: ";
		     ppType ppstrm ty1; add_newline ppstrm;
   		     add_string ppstrm "found:    ";
		     ppType ppstrm ty2; add_newline ppstrm;
   		     add_string ppstrm "in pattern:"; add_break ppstrm (1,2);
   		     ppPat ppstrm (pat,!printDepth))));
	   (ORpat(p1, p2), ty1)
       end
   | APPpat(dcon as DATACON{typ,rep,...},_,arg) =>
       let val (argpat,argty) = patType(arg,depth,loc,vis)
           val (ty1,ndcon) = 
             case rep 
              of Access.REF => (refPatType,refNewDcon dcon)
               | _ => (typ,dcon)
           val ty2 = applyPoly(ty1,Root)
           val npat = case typ of POLYty _ => APPpat(ndcon,SOME ty2,argpat)
                                | _ => APPpat(ndcon,NONE,argpat)
        in (npat,applyType(ty2,argty)) 
             handle Unify(mode) =>
	      (err loc COMPLAIN
                 ("constructor and argument don't agree in pattern ("^mode^")")
		 (fn ppstrm =>
		   (PPType.resetPPType();
		    add_newline ppstrm;
		    add_string ppstrm "constructor: ";
		    ppType ppstrm typ; add_newline ppstrm;
		    add_string ppstrm "argument:    ";
		    ppType ppstrm argty; add_newline ppstrm;
		    add_string ppstrm "in pattern:"; add_break ppstrm (1,2);
		    ppPat ppstrm (pat,!printDepth)));
	       (pat,WILDCARDty))
       end
   | CONSTRAINTpat(pat',ty) => 
       let val (npat,patTy) = patType(pat',depth,loc,vis)
        in (unifyTy(patTy, ty); 
            (CONSTRAINTpat(npat,ty),ty))
	       handle Unify(mode) =>
	       (err loc COMPLAIN
	          ("pattern and constraint don't agree (" ^ mode ^ ")")
		  (fn ppstrm =>
		     (PPType.resetPPType();
		      add_newline ppstrm;
		      add_string ppstrm "pattern:    ";
		      ppType ppstrm patTy; add_newline ppstrm;
		      add_string ppstrm "constraint: ";
		      ppType ppstrm ty; add_newline ppstrm;
		      add_string ppstrm "in pattern:"; add_break ppstrm (1,2);
		      ppPat ppstrm (pat,!printDepth)));
	        (pat,WILDCARDty))
       end
   | LAYEREDpat(vpat as VARpat(VALvar{typ,...}),pat') =>
       let val (npat,patTy) = patType(pat',depth,loc,vis)
           val _ = (typ := patTy)
        in (LAYEREDpat(vpat,npat),patTy)
       end
   | LAYEREDpat(cpat as CONSTRAINTpat(VARpat(VALvar{typ,...}),ty),pat') =>
       let val (npat,patTy) = patType(pat',depth,loc,vis)
	in (unifyTy(patTy,ty); typ := ty; 
           (LAYEREDpat(cpat,npat),ty))
	       handle Unify(mode) =>
	         (err loc COMPLAIN
		    ("pattern and constraint don't agree (" ^ mode ^ ")")
		    (fn ppstrm =>
		     (PPType.resetPPType();
		      add_newline ppstrm;
		      add_string ppstrm "pattern:    ";
		      ppType ppstrm patTy; add_newline ppstrm;
		      add_string ppstrm "constraint: ";
		      ppType ppstrm ty; add_newline ppstrm;
		      add_string ppstrm "in pattern:"; add_break ppstrm (1,2);
		      ppPat ppstrm (pat,!printDepth)));
                  (pat,WILDCARDty))
       end
   | p => impossible "patType -- unexpected pattern"

fun expType(exp: exp, occ: occ, loc, vis) : exp * ty =
 case exp
  of VARexp(r as ref(VALvar{typ,access,...}),_) => 
       let val ty = instantiateType(!typ,occ)
           val _ = case access of (LVAR x) => addlist(x,ty)
                                | _ => ()
	in (VARexp(r, case !typ of POLYty _ => SOME ty | _ => NONE), ty)
       end
   | CONexp(dcon as DATACON{typ,...},_) => 
       let val ty = applyPoly(typ,occ)
        in case typ 
            of POLYty _ => (CONexp(dcon,SOME ty),ty)
             | _ => (CONexp(dcon,NONE),ty)
       end
   | INTexp (_,ty) => (exp,ty)
   | WORDexp (_,ty) => (exp,ty)
   | REALexp _ => (exp,realTy)
   | STRINGexp _ => (exp,stringTy)
   | CHARexp _ => (exp,charTy)
   | RECORDexp fields =>
       let fun h(l as LABEL{name,...},exp') = 
             let val (nexp,nty) = expType(exp',occ,loc,vis)
              in ((l,nexp),(l,nty))
             end
           fun extract(LABEL{name,...},t) = (name,t)
           val (fields',tfields) = map2 h fields
           val rty = map extract (sortFields tfields)
        in (RECORDexp fields',recordTy(rty))
       end
   | VECTORexp(exps,_) =>
       (let val (exps',nty) = map2 (fn e => expType(e,occ,loc,vis)) exps
            val vty = foldr (fn (a,b) => (unifyTy(a,b); b)) (mkMETAty()) nty
         in (VECTORexp(exps',vty), CONty(vectorTycon,[vty]))
        end handle Unify(mode) =>
 	      (err loc COMPLAIN ("vector expression type failure (" ^mode^ ")")
	       nullErrorBody;
	       (exp,WILDCARDty)))
   | SEQexp exps => 
       let fun scan nil = (nil,unitTy)
             | scan [e] = 
                 let val (e',ety) = expType(e,occ,loc,vis)
                  in ([e'],ety)
                 end
             | scan (e::rest) = 
                 let val (e',_) = expType(e,occ,loc,vis)
                     val (el',ety) = scan rest
                  in (e'::el',ety)
                 end
            val (exps',expty) = scan exps
        in (SEQexp exps',expty)
       end
   | APPexp(rator, rand) =>
       let val (rator',ratorTy) = expType(rator,Rator occ,loc,vis)
           val (rand',randTy) = expType(rand,Rand occ,loc,vis)
           val exp' = APPexp(rator',rand')
        in (exp',applyType(ratorTy,randTy))
             handle Unify(mode) => 
	       let val ratorTy = prune ratorTy
	           val reducedRatorTy = headReduceType ratorTy
                in PPType.resetPPType();
                   if isArrowType(reducedRatorTy)
		   then (err loc COMPLAIN
			   ("operator and operand don't agree (" ^ mode ^ ")")
			   (fn ppstrm =>
			    (add_newline ppstrm;
			     add_string ppstrm "operator domain: ";
			     ppType ppstrm (domain reducedRatorTy);
			     add_newline ppstrm;
			     add_string ppstrm "operand:         ";
			     ppType ppstrm randTy; add_newline ppstrm;
			     add_string ppstrm "in expression:";
			     add_break ppstrm (1,2);
			     ppExp ppstrm (exp,!printDepth)));
			 (exp,WILDCARDty))
		   else (err loc COMPLAIN "operator is not a function"
			   (fn ppstrm =>
			    (add_newline ppstrm;
			     add_string ppstrm "operator: ";
			     ppType ppstrm (ratorTy); add_newline ppstrm;
			     add_string ppstrm "in expression:";
			     add_break ppstrm (1,2);
			     ppExp ppstrm (exp,!printDepth)));
			 (exp,WILDCARDty))
               end
       end
   | CONSTRAINTexp(e,ty) =>
       let val (e',ety) = expType(e,occ,loc,vis)
	in (unifyTy(ety, ty); (CONSTRAINTexp(e',ty),ty))
	     handle Unify(mode) =>
	       (err loc COMPLAIN
	          ("expression and constraint don't agree (" ^ mode ^ ")")
		  (fn ppstrm =>
		      (PPType.resetPPType();
		       add_newline ppstrm;
		       add_string ppstrm "expression: ";
		       ppType ppstrm ety; add_newline ppstrm;
		       add_string ppstrm "constraint: ";
		       ppType ppstrm ty; add_newline ppstrm;
		       add_string ppstrm "in expression:"; 
                       add_break ppstrm (1,2);
		       ppExp ppstrm (e,!printDepth)));
		(exp,WILDCARDty))
       end
   | HANDLEexp(e,HANDLER h) =>
       let val (e',ety) = expType(e,occ,loc,vis)
	   and (h',hty) = expType(h,occ,loc,vis)
           val exp' = HANDLEexp(e',HANDLER h')
        in (unifyTy(hty, exnTy --> ety); (exp',ety))
	     handle Unify(mode) =>
	       let val hty = prune hty
	        in PPType.resetPPType();
		   if ((unifyTy(domain hty,exnTy); false) 
                         handle Unify _ => true)
                   then (err loc COMPLAIN "handler domain is not exn"
			  (fn ppstrm =>
			     (add_newline ppstrm;
			      add_string ppstrm "handler domain: ";
			      ppType ppstrm (domain hty); add_newline ppstrm;
			      add_string ppstrm "in expression:";
			      add_break ppstrm (1,2);
			      ppExp ppstrm (exp,!printDepth))))
                   else (err loc COMPLAIN
			  ("expression and handler don't agree (" ^ mode ^ ")")
			  (fn ppstrm => 
			     (add_newline ppstrm;
			      add_string ppstrm "body:          ";
			      ppType ppstrm ety; add_newline ppstrm;
			      add_string ppstrm "handler range: ";
			      ppType ppstrm (range hty); add_newline ppstrm;
			      add_string ppstrm "in expression:";
			      add_break ppstrm (1,2);
			      ppExp ppstrm (exp,!printDepth))));
		   (exp,WILDCARDty)
	       end
       end
   | RAISEexp(e,_) =>
       let val (e',ety) = expType(e,occ,loc,vis)
           val newty = mkMETAty()
	in unifyTy(ety,exnTy)
	     handle Unify(mode) =>
	       (err loc COMPLAIN "argument of raise is not an exception"
		 (fn ppstrm =>
		   (PPType.resetPPType();
		    add_newline ppstrm; add_string ppstrm "raised: ";
		    ppType ppstrm ety; add_newline ppstrm;
		    add_string ppstrm "in expression:"; add_break ppstrm (1,2);
		    ppExp ppstrm (exp,!printDepth))));
           (RAISEexp(e',newty),newty)
       end
   | LETexp(d,e) => 
       let val (e',ety) = expType(e,occ,loc,vis)
           val d' = decType0(d,LetDef(occ),loc,false)
        in (LETexp(d',e'),ety)
       end
   | CASEexp(e,rules) =>
       let val (e',ety) = expType(e,occ,loc,vis)
           val (rules',_,rty) = matchType(rules,Rator occ,loc,vis)
           val exp' = CASEexp(e',rules')
        in (exp',applyType(rty,ety))
	     handle Unify(mode) => 
	       (err loc COMPLAIN
		 ("case object and rules don't agree (" ^ mode ^ ")")
		 (fn ppstrm =>
	          (PPType.resetPPType();
		   add_newline ppstrm; add_string ppstrm "rule domain: ";
		   ppType ppstrm (domain rty); add_newline ppstrm;
		   add_string ppstrm "object:      ";
		   ppType ppstrm ety; add_newline ppstrm;
		   add_string ppstrm "in expression:"; add_break ppstrm (1,2);
		   ppExp ppstrm (exp,!printDepth)));
	        (exp,WILDCARDty))
       end (* this causes case to behave differently from let, i.e.
	      bound variables do not have generic types *)
   | FNexp(rules,_) => 
       let val (rules',ty,rty) = matchType(rules,occ,loc,vis)
        in (FNexp(rules',ty),rty)
       end
   | MARKexp(e,region) => 
       let val (e',et) = expType(e,occ,region,vis)
        in (MARKexp(e',region),et)
       end
   | _ => impossible "Typecheck.exptype -- bad expression"

and ruleType(RULE(pat,exp),occ,loc,vis) =  
  let val occ = Abstr occ
      val (pat',pty) = patType(pat,lamdepth occ,loc,vis)
      val (exp',ety) = expType(exp,occ,loc,vis)
   in (RULE(pat',exp'),pty,pty --> ety)
  end

and matchType(l,occ,loc,vis) =
 case l
  of [] => impossible "empty rule list in typecheck.matchType"
   | [rule] => 
       let val (rule0,argt,rty) = ruleType(rule,occ,loc,vis)
        in ([rule0],argt,rty)
       end
   | rule::rest =>
       let val (rule0,argt,rty) = ruleType(rule,occ,loc,vis)
           fun checkrule rule' =
             let val (rule1,argt',rty') = ruleType(rule',occ,loc,vis)
              in unifyTy(rty, rty')
                   handle Unify(mode) =>
                     (err loc COMPLAIN ("rules don't agree (" ^ mode ^ ")")
			(fn ppstrm =>
   			   (PPType.resetPPType();
   			    add_newline ppstrm; add_string ppstrm "expected: ";
			    ppType ppstrm rty; add_newline ppstrm;
   			    add_string ppstrm "found:    ";
			    ppType ppstrm rty'; add_newline ppstrm;
   			    add_string ppstrm "rule:"; add_break ppstrm (1,2);
   			    ppRule ppstrm (rule',!printDepth))));
                 rule1
             end
        in (rule0::(map checkrule rest),argt,rty)
       end

and decType0(decl,occ,loc,vis) : dec =
 case decl
  of VALdec(vbs) =>
       let fun vbType(vb as VB{pat, exp, tyvars=(tv as (ref tyvars))},z) =
	     let val (pat',pty) = patType(pat,infinity,loc,vis)
		 val (exp',ety) = expType(exp,occ,loc,vis)
              in unifyTy(pty,ety) handle Unify(mode) =>
 	           (err loc COMPLAIN
                      ("pattern and expression in val dec don't agree ("
			  ^ mode ^ ")")
                      (fn ppstrm =>
  		          (PPType.resetPPType();
 		           add_newline ppstrm; 
                           add_string ppstrm "pattern:    ";
			   ppType ppstrm pty; add_newline ppstrm;
			   add_string ppstrm "expression: ";
			   ppType ppstrm ety; add_newline ppstrm;
			   add_string ppstrm "in declaration:";
			   add_break ppstrm (1,2);
			   ppVB ppstrm (vb,!printDepth))));
                 generalizePat(pat,tyvars,occ,loc);
                 (VB{pat=pat',exp=exp',tyvars=tv})::z
             end
         
        in VALdec(foldr vbType [] vbs)
       end
   | VALRECdec(rvbs) =>
       let fun allVis(RVB{var=VALvar{typ,access=LVAR x,...},...}::r) =
                 ((case (getTable x) 
                    of NONE => true
                     | _ => allVis r) handle _ => allVis r)
             | allVis [] = false
             | allVis _ = impossible "mtderiv.sml 782"
           val vis = if vis then true else allVis(rvbs)

           fun setType(RVB{var=VALvar{typ,access=LVAR x,...},resultty,...}) =
                 if vis then 
                   (case resultty 
                     of SOME ty => typ := ty
                      | _ => typ := mkRefMETAty(1 + lamdepth occ))
                 else (case (infer x, resultty)
                        of (SOME ty,_) => typ := ty
                         | (_,SOME ty) => typ := ty
                         | _ => typ := mkRefMETAty(1 + lamdepth occ))
             | setType _  = impossible "typecheck.783"

           fun rvbType(rvb as RVB{var=v as VALvar{typ,...},
		       exp,resultty,tyvars},z) =
 	         let val (exp',ety) = expType(exp,Abstr(Rator occ),loc,vis)
		  in unifyTy(!typ, ety) handle Unify(mode) =>
                       (err loc COMPLAIN
			 ("pattern and expression in val rec dec don't agree ("
			 ^ mode ^ ")")
			 (fn ppstrm =>
		             (PPType.resetPPType();
		              add_newline ppstrm; 
                              add_string ppstrm "pattern:    ";
			      ppType ppstrm (!typ); add_newline ppstrm;
		              add_string ppstrm "expression: ";
			      ppType ppstrm ety; add_newline ppstrm;
		              add_string ppstrm "in declaration:";
			      add_break ppstrm (1,2);
		              ppRVB ppstrm (rvb,!printDepth))));
                     (RVB{var=v,exp=exp',resultty=resultty,tyvars=tyvars})::z
		 end
             | rvbType _ = impossible "typecheck.786"

           val _ = (app setType rvbs)
           val rvbs' = foldr rvbType [] rvbs
 	   fun genType(RVB{var,tyvars = ref tyvars,...}) = 
                    generalizeTy(var,tyvars,occ,loc)
           val _ = (app genType rvbs)
        in VALRECdec rvbs'
       end
   | LOCALdec(decIn,decOut) =>
       let val decOut' = decType0(decOut,occ,loc,vis)
           val decIn' = decType0(decIn,LetDef occ,loc,false)
        in LOCALdec(decIn',decOut')
       end
   | SEQdec(decls) => 
       let fun h(decl,z) = (decType0(decl,occ,loc,vis))::z
        in SEQdec(foldr h [] decls)
       end
   | ABSTYPEdec{abstycs,withtycs,body} => 
       let fun makeAbstract(GENtyc{stamp,arity,eq,path,kind}) =
	         (kind := ABStyc(GENtyc{stamp=stamp,arity=arity,path=path,
				        eq=eq,kind=ref(!kind)});
                  eq := NO)
             | makeAbstract _ = impossible "typecheck.718"
           val body'= decType0(body,occ,loc,vis) (* not sure *)
        in app makeAbstract abstycs;
           ABSTYPEdec{abstycs=abstycs,withtycs=withtycs,body=body'}
       end
   | MARKdec(dec,region) => MARKdec(decType0(dec,occ,region,vis),region)
   | STRdec strbs => 
       let fun h(strb,z) = (strbType(occ,loc,strb))::z
        in STRdec(foldr h [] strbs)
       end
   | ABSdec strbs => 
       let fun h(strb,z) = (strbType(occ,loc,strb))::z
        in ABSdec(foldr h [] strbs)
       end
   | FCTdec fctbs => 
       let fun h(fctb,z) = (fctbType(occ,loc,fctb))::z
        in FCTdec(foldr h [] fctbs)
       end
   |_ => decl

and strbType(occ,loc,STRB{strvar,def,thin,abslty,constraint}) =
      STRB{strvar=strvar,def=strexpType(occ,loc,def,thin),thin=thin,
           abslty=abslty,constraint=constraint}

and fctbType(occ,loc,FCTB{fctvar,def}) =
      let fun fctexpType(v as VARfct _) = v
            | fctexpType(FCTfct{param,def,thin,constraint}) =
    	        FCTfct{param=param,def=strexpType(occ,loc,def,thin), 
                       thin=thin,constraint=constraint}
            | fctexpType(LETfct(dec,e)) =
                LETfct(decType0(dec,LetDef occ,loc,false),fctexpType e)
       in FCTB{fctvar=fctvar,def=fctexpType def}
      end

and strexpType(occ,loc,STRUCTstr{body,str,locations},SOME(x,ll)) = 
      let fun g(VALtrans(PATH(j,LVAR y),_,_)) = 
                if (x=y) then (case (nth(locations,j)) 
                                of VALtrans(LVAR z,_,_) => markvis(z)
                                 | _ => ())
                else () 
            | g _ = ()

          fun h(d,z) = (decType0(d,occ,loc,false))::z
       in app g ll;
          STRUCTstr{body=foldr h [] body,str=str,locations=locations}
      end
  | strexpType(occ,loc,STRUCTstr{body,str,locations},NONE) = 
      let fun h(d,z) = (decType0(d,occ,loc,true))::z
       in STRUCTstr{body=foldr h [] body,str=str,locations=locations}
      end
  | strexpType(occ,loc,APPstr{oper,instancelty,argexp,argthin,str},_) =
      APPstr{oper=oper,instancelty=instancelty,
             argexp=strexpType(occ,loc,argexp,argthin),
             argthin=argthin,str=str}
  | strexpType(occ,loc,LETstr(dec,e),thin) =
      let val e' = strexpType(occ,loc,e,thin)
          val dec' = decType0(dec,LetDef occ,loc,false)
       in LETstr(dec',e')
      end
  | strexpType(occ,loc,MARKstr(e,region),thin) = 
      MARKstr(strexpType(occ,region,e,thin),region)
  | strexpType(occ,loc,v as VARstr _,_) = v

val dec' = decType0(dec,Root,(0,0),true)

in dec'
end

end (* local *)

end (* structure MTDeriv *)
