(* translate.sml
 *
 * Copyright 1989 by AT&T Bell Laboratories
 *)

signature TRANSLATE = sig
   val transDec : StaticEnv.staticEnv (* coreEnv *) ->
                  StaticEnv.staticEnv (* diagnostic Env *) ->
		  (Source.region -> ErrorMsg.complainer) ->
                  (Source.region -> string) -> Absyn.dec -> Lambda.lexp ->
		  { genLambda: Lambda.lexp option list -> Lambda.lexp,
		    importPids: PersStamps.persstamp list }
   val newIdLexp: unit -> Lambda.lexp
end

structure Translate : TRANSLATE =
struct

open Access Absyn Types Lambda Modules Variables BasicTypes 
     ElabUtil Nonrec ErrorMsg Unboxed Transtypes TransBinding PrettyPrint
structure LT = LambdaType
val arrowLty = LT.arrowLty

 fun newIdLexp () = let
     val v = mkLvar ()
 in
     FN (v, LT.injBOXED, VAR v)
 end
  
(**************************************************************************
 * CONSTANTS AND UTILITY FUNCTIONS.                                       *
 **************************************************************************)
val unitLexp = RECORD[]
fun ident x = x

val error = ErrorMsg.impossible	

(* old-style fold for cases where it is partially applied *)
fun fold f l init = foldr f init l

val predTy = LT.injARROW(LT.injRECORD[LT.injINT,LT.injINT],LT.inj LT.BOOL)
val lengthTy = LT.injARROW(LT.injBOXED,LT.injINT)
val subscriptTy = LT.injARROW(LT.injRECORD[LT.injBOXED,LT.injINT],LT.inj LT.RBOXED)

val elemgtr = (fn ((LABEL{number=x,...},_),(LABEL{number=y,...},_))=> x>y);
val sorted = Sort.sorted elemgtr
val sortrec = Sort.sort elemgtr

fun unwrapOp1(lt,NONE) = ident
  | unwrapOp1(lt,SOME tt) = unwrapOp(lt,(transTyLty tt))

fun unwrapOp2(tt,NONE) = ident
  | unwrapOp2(tt,SOME tt') = unwrapOp(transTyLty(tt),transTyLty(tt'))

fun mergety(t1,NONE) = t1
  | mergety(t1,SOME t2) = t2

fun getEqualElem (CONty(_,[CONty(_,[t,_]),_])) = t
  | getEqualElem _ = IBOUND 0

(* generate the lambda code for ":=" using the given update primop.
 * ":=" ==> "fn x => update(#1 x, 0, #2 x)"
 *)
val updateTy = LT.injARROW(LT.injRECORD[LT.injBOXED,LT.injINT,LT.inj LT.RBOXED],LT.injINT)
val assignTy = LT.injARROW(LT.injRECORD[LT.injBOXED,LT.inj LT.RBOXED],LT.injINT)
val icmplty = LT.injARROW(LT.injRECORD[LT.injINT,LT.injINT],LT.inj LT.BOOL)
val intoplty = LT.injARROW(LT.injRECORD[LT.injINT,LT.injINT],LT.injINT)
val ineglty = LT.injARROW(LT.injINT,LT.injINT)


fun sum(i,j) = (i + j) mod 65537
fun hashlty t = case LT.out t
   of LT.INT => 0
    | LT.BOOL => 1
    | LT.REAL => 2
    | LT.BOXED => 3
    | LT.RBOXED => 4
    | LT.SRCONT => 5
    | LT.CONT t => (hashlty t)+6
    | LT.ARROW(t1,t2) => (sum(hashlty t1,hashlty t2))+7
    | LT.RECORD l => foldr sum 8 (map hashlty l)
    | LT.SRECORD l => foldr sum 9 (map hashlty l)
    | LT.GREC _ => 10
    | _ => 11

structure Map = PersMap

exception NoCore

fun transDec coreEnv env err errorMatch rootdec exportLexp =
let 

 local val region = ref(0,0)
      val markexn = PRIM(P.MARKEXN,
			 LT.injARROW(LT.injRECORD[LT.injBOXED,LT.injBOXED],
				     LT.injBOXED))
  in fun withRegion loc f x = let val r = !region
                         in (region := loc; f x before region:=r)
	                    handle e => (region := r; raise e)
                        end
      fun Raise(x,t) = 
	   RAISE(if !Control.trackExn 
		      then APP(markexn,RECORD[x,STRING(errorMatch(!region))])
			       
		      else x,
		 t)
      fun complain s = err (!region) s
  end


 val (special_unwrapOp,special_wrapOp, final) = specialWrapperGen true

 fun coreCON id =
     let val CONbind v = ModuleUtil.lookVARCON 
        (coreEnv,SymPath.SPATH [Symbol.strSymbol "Core", Symbol.varSymbol id],
	  fn _ => fn s => fn _ => raise NoCore)
      in v
     end
  handle NoCore => (Control.Print.say "WARNING: no Core access\n";
                      (* this happens when compiling boot/dummy.sml,
                         boot/assembly.sig, and boot/core.sml *)
		    bogusEXN)

 fun coreAccess (id) = 
     (case ModuleUtil.lookVARCON 
        (coreEnv,SymPath.SPATH [Symbol.strSymbol "Core", Symbol.varSymbol id],
	      fn _ => fn s => fn _ => raise NoCore)
      of VARbind(VALvar{access,...}) => access
       | _ => NO_ACCESS)
    handle NoCore => (Control.Print.say "WARNING: no Core access\n";
                      (* this happens when compiling boot/dummy.sml,
                         boot/assembly.sig, and boot/core.sml *)
		      LVAR 0)


 type pid = PersStamps.persstamp

 val persmap = ref (Map.empty : (lvar * LT.lty) Map.map)
 fun trans_pid(pid,lt) =
   (let val (v,tt) = Map.lookup (!persmap) pid
        val _ = (persmap := Map.add(Map.delete(pid,!persmap),
                                    pid,(v,LT.mergeLty(lt,tt))))
     in v
    end handle Map.MapF => 
	 (let val v = mkLvar()
	   in (persmap := Map.add(!persmap,pid,(v,lt));
	      v)
	  end))

 local val EE = Intset.new()
 in 
    val extern_var = Intset.mem EE 
    val add_extern = Intset.add EE

    fun extern(EXTERN _) = true
      | extern(PATH(_,p)) = extern p
      | extern(LVAR v) = extern_var v
      | extern _ = false 
 end

 local open Intmap
       exception VARMAP
       val varmap : LT.lty intmap = new(32,VARMAP)
  in fun map_var v = (map varmap v) handle _ => LT.inj(LT.GREC[])
     fun trans_var(v,lt) =
       if extern_var v then  
         (let val tt = map varmap v
              val _ = add varmap (v,LT.mergeLty(lt,tt))
           in v 
          end handle VARMAP =>
                (add varmap (v,lt); v))
       else v
 end (* local *)

 fun untypedTranslateAccess(PATH(x,a),t) = 
       SELECT(x, untypedTranslateAccess(a,LT.inj(LT.GREC[(x,t)])))
   | untypedTranslateAccess(LVAR v,t) = VAR(trans_var(v,t))
   | untypedTranslateAccess(EXTERN pid,t) = VAR(trans_pid(pid,t))
   | untypedTranslateAccess _ = error "translate.untypedTranslateAccess"


 exception HASHTABLE
 type key = int
 type info = (key * int list * (LT.lty * LT.lty) option * lvar) 
 val hashtable : info list Intmap.intmap = Intmap.new(32,HASHTABLE)    

 fun hashkey(l,inst) = 
   let val key = foldr (op +) 0 l
    in case inst 
         of NONE => ~key
          | SOME(_,lt) => (hashlty(lt) + key)
   end

 fun buildheader v = let
     val info = Intmap.map hashtable v
     fun h ((_, l, inst, w), hdr) = let
	 val le = foldl (fn (k, e) => SELECT (k, e)) (VAR v) l
	 val b = case inst of
	     NONE => le 
	   | SOME p => (unwrapOp p le)
     in
	 fn e => hdr (APP (FN (w, LT.BOGUS, e), b))
     end
 in
     foldr h ident info
 end handle _ => ident

 fun lookbind(v,l,inst) = 
   let val info = (Intmap.map hashtable v) handle _ => []
       val key = hashkey(l,inst)

       fun eqinst(NONE,NONE) = true 
         | eqinst(SOME(_,t1),SOME(_,t2)) = LT.equivLty(t1,t2)
         | eqinst _ = false

       fun h [] = (let val u = mkLvar()
                    in Intmap.add hashtable (v,(key,l,inst,u)::info); u
                   end)
         | h((k',l',inst',w)::r) = 
             if (k'=key) then 
                   (if (l'=l) andalso eqinst(inst',inst) then w else h r)
             else h r    
    in h info 
   end

 fun bindvar(v,[],NONE) = v
   | bindvar(v,l,inst) = (lookbind(v,l,inst))

 fun bindpid(pid,l,lt,inst) =
   (let val (v,tt) = Map.lookup (!persmap) pid
        val _ = (persmap := Map.add(Map.delete(pid,!persmap),
                                    pid,(v,LT.mergeLty(lt,tt))))
     in bindvar(v,l,inst)
    end handle Map.MapF => 
	 (let val v = mkLvar()
	   in (persmap := Map.add(!persmap,pid,(v,lt));
	       bindvar(v,l,inst))
	  end))

 fun transpath0(p,t1,t2) = 
   let fun h(LVAR v,l) = 
             (case t2 
               of NONE => bindvar(v,l,NONE)
                | SOME t2' => 
                   (let val lt2 = transTyLty t2'
                        val lt1 = case t1 of NONE => LT.injBOXED
                                           | SOME t1' => transTyLty t1'
                     in bindvar(v,l,SOME(lt1,lt2))
                    end))
         | h(EXTERN pid,l) = 
             (let val (lt1,inst) = 
                   case (t1,t2) 
                    of (SOME t1',NONE) => (transTyLty t1',NONE)
                     | (SOME t1',SOME t2') => 
                          (let val lt1 = transTyLty t1'
                            in (lt1,SOME(lt1,transTyLty t2'))
                           end)
                     | (NONE,_) => (LT.injBOXED,NONE)

                  val lt = foldr (fn (k,b) => LT.inj(LT.GREC[(k,b)])) lt1 l
               in bindpid(pid,l,lt,inst)
              end)
         | h(PATH(x,p),l) = h(p,x::l)
         | h _ = impossible "translate.transpath on unexpected access"
    in h(p,[])
   end

 fun transpath p = VAR(transpath0 p)

 val subscriptTy = LT.injARROW(LT.injRECORD[LT.injBOXED,LT.injINT],LT.inj(LT.RBOXED))

 local val lt = LT.injARROW(LT.injRECORD([LT.injBOXED,LT.injBOXED]),LT.inj LT.BOOL)
       val z =
            {polyequal = untypedTranslateAccess(coreAccess "polyequal",lt),
	     stringequal = untypedTranslateAccess(coreAccess "stringequal",lt)}
  in val equal = Equal.equal z env
 end

 val LESSU = P.CMP{oper=P.LTU,kind=P.INT 31}

 fun translateAccess (acc as LVAR _,t1,t2) = transpath(acc,SOME t1,t2)
   | translateAccess (acc as PATH _, t1,t2) = transpath(acc,SOME t1,t2)
   | translateAccess (acc as EXTERN _, t1,t2) = transpath(acc,SOME t1,t2)
   | translateAccess (INLINE prim,t1,t2) =
     (case prim
       of P.POLYEQL => equal (getEqualElem (mergety(t1,t2)))
        | P.POLYNEQ => 
              let val t = mergety(t1,t2)
               in composeNOT(equal (getEqualElem t), transTyLty t)
              end
        | P.INLSUBSCRIPTV => 
             let val lty = transTyLty(mergety(t1,t2))
		 val p = mkLvar() and a = mkLvar() and i = mkLvar()
		 val vp = VAR p and va = VAR a and vi = VAR i
		 val argt = LT.injRECORD[LT.injBOXED,LT.injINT]
		 val (_,res) = arrowLty(lty)
		 val header = unwrapOp(subscriptTy,lty)
	     in FN(p,argt,
		   LET(a,SELECT(0,vp),
		       LET(i,SELECT(1,vp),
			   COND(APP(PRIM(LESSU,predTy),
				    RECORD[vi,APP(PRIM(P.LENGTH,lengthTy), va)]),
				APP(header(PRIM(P.SUBSCRIPT,subscriptTy)),
				    RECORD[va,vi]),
				Raise(conToLexp(coreCON "Subscript"),res)))))
	     end
        | P.INLSUBSCRIPT => 
             let val lty = transTyLty(mergety(t1,t2))
		 val p = mkLvar() and a = mkLvar() and i = mkLvar()
		 val vp = VAR p and va = VAR a and vi = VAR i
		 val argt = LT.injRECORD[LT.injBOXED,LT.injINT]
		 val (_,res) = arrowLty(lty)
		 val header = unwrapOp(subscriptTy,lty)
	     in FN(p,argt,
		   LET(a,SELECT(0,vp),
		       LET(i,SELECT(1,vp),
			   COND(APP(PRIM(LESSU,predTy),
				    RECORD[vi,APP(PRIM(P.LENGTH,lengthTy), va)]),
				APP(header(PRIM(P.SUBSCRIPT,subscriptTy)),
				    RECORD[va,vi]),
				Raise(conToLexp(coreCON "Subscript"),res)))))
		 
             end
        | P.INLUPDATE => 
             let val ty = mergety(t1,t2)
		 val lty = transTyLty ty
                 val header = unwrapOp(updateTy,lty)
                 val oper = header(PRIM(unboxedUpdate ty, updateTy))
		 val t = mkLvar() and a = mkLvar() and i = mkLvar() and v = mkLvar()
		 val vt = VAR t and va = VAR a and vi = VAR i and vv = VAR v
		 val (argt,res) = arrowLty(lty)
	     in FN(t,argt,
		   LET(a,SELECT(0,vt),
		       LET(i,SELECT(1,vt),
			   LET(v,SELECT(2,vt),
			       COND(APP(PRIM(LESSU,predTy),
					RECORD[vi,APP(PRIM(P.LENGTH,lengthTy),va)]),
				    APP(oper,RECORD[va,vi,vv]),
				    Raise(conToLexp(coreCON "Subscript"),res))))))
	     end

	| P.NUMSUBSCRIPT{kind,checked=true,immutable} =>
	  let val lty = transTyLty(mergety(t1,t2))
	      val p = mkLvar() and a = mkLvar() and i = mkLvar()
	      val vp = VAR p   and va = VAR a   and vi = VAR i
	      val LT.ARROW(argt,randt) = LT.out lty
	      val oper = P.NUMSUBSCRIPT{kind=kind,checked=false,immutable=immutable}
	   in FN(p,argt,
		LET(a,SELECT(0,vp),
		 LET(i,SELECT(1,vp),
		  COND(APP(PRIM(LESSU,predTy), 
			   RECORD[vi,APP(PRIM(P.LENGTH,lengthTy),va)]),
		       APP(PRIM(oper,lty),RECORD[va,vi]),
		       Raise(conToLexp(coreCON "Subscript"),randt)))))
	  end


	| P.NUMUPDATE{kind,checked=true} =>
	  let val lty = transTyLty(mergety(t1,t2))
	      val p=mkLvar() and a=mkLvar() and i=mkLvar() and v=mkLvar()
	      val vp=VAR p   and va=VAR a   and vi=VAR i   and vv=VAR v
	      val LT.ARROW(argt,randt) = LT.out lty
	      val oper = P.NUMUPDATE{kind=kind,checked=false}
	   in  FN(p,argt,
		LET(a,SELECT(0,vp),
		 LET(i,SELECT(1,vp),
		  LET(v,SELECT(2,vp),
		   COND(APP(PRIM(LESSU,predTy), 
			    RECORD[vi,APP(PRIM(P.LENGTH,lengthTy),va)]),
		        APP(PRIM(oper,lty),RECORD[va,vi,vv]),
			Raise(conToLexp(coreCON "Subscript"),randt))))))
	  end


        | P.INLDIV =>  
             let val a = mkLvar() and b = mkLvar() and z = mkLvar()
              in FN(z,LT.injRECORD[LT.injINT,LT.injINT],
                   LET(a,SELECT(0,VAR z),LET(b,SELECT(1,VAR z),
                      COND(APP(PRIM(P.IGE,icmplty),RECORD[VAR b,INT 0]),
                           COND(APP(PRIM(P.IGE,icmplty),RECORD[VAR a,INT 0]),
                                DIV(VAR a,VAR b),
                                SUB(DIV(ADD(VAR a,INT 1),VAR b),INT 1)),
                           COND(APP(PRIM(P.IGT,icmplty),RECORD[VAR a,INT 0]),
                                SUB(DIV(SUB(VAR a,INT 1),VAR b),INT 1),
                                DIV(VAR a,VAR b))))))
             end 
        | P.INLMOD =>
             let val a = mkLvar() and b = mkLvar() and z = mkLvar()
              in FN(z,LT.injRECORD[LT.injINT,LT.injINT],
                   LET(a,SELECT(0,VAR z),LET(b,SELECT(1,VAR z),
                      COND(APP(PRIM(P.IGE,icmplty),RECORD[VAR b,INT 0]),
                           COND(APP(PRIM(P.IGE,icmplty),RECORD[VAR a,INT 0]),
                                SUB(VAR a,MUL(DIV(VAR a,VAR b),VAR b)),
                                ADD(SUB(VAR a,MUL(DIV(ADD(VAR a,INT 1),VAR b),VAR b)),VAR b)),
                           COND(APP(PRIM(P.IGT,icmplty),RECORD[VAR a,INT 0]),
                                ADD(SUB(VAR a,MUL(DIV(SUB(VAR a,INT 1),VAR b),VAR b)),VAR b),
                                COND(APP(PRIM(P.IEQL,icmplty),RECORD[VAR a,INT ~1073741824]),
                                     COND(APP(PRIM(P.IEQL,icmplty),RECORD[VAR b,INT 0]),
                                          INT 0,
                                          SUB(VAR a,MUL(DIV(VAR a,VAR b),VAR b))),
                                     SUB(VAR a,MUL(DIV(VAR a,VAR b),VAR b))))))))
             end
        | P.INLREM =>
             let val a = mkLvar() and b = mkLvar() and z = mkLvar()
              in FN(z,LT.injRECORD[LT.injINT,LT.injINT],
                   LET(a,SELECT(0,VAR z),
                     LET(b,SELECT(1,VAR z),
                         SUB(VAR a,MUL(DIV(VAR a,VAR b),VAR b)))))
             end
        | P.INLMIN =>
             let val x = mkLvar() and y = mkLvar() and z = mkLvar()
              in FN(z,LT.injRECORD[LT.injINT,LT.injINT],
                       LET(x,SELECT(0,VAR z),
                          LET(y,SELECT(1,VAR z),
                            COND(APP(PRIM(P.ILT,icmplty),RECORD[VAR x,VAR y]),
                                 VAR x, VAR y))))
             end
        | P.INLMAX =>
             let val x = mkLvar() and y = mkLvar() and z = mkLvar()
              in FN(z,LT.injRECORD[LT.injINT,LT.injINT],
                       LET(x,SELECT(0,VAR z),
                          LET(y,SELECT(1,VAR z),
                            COND(APP(PRIM(P.IGT,icmplty),RECORD[VAR x,VAR y]),
                                 VAR x, VAR y))))
             end
        | P.INLABS =>
             let val x = mkLvar()
              in FN(x,LT.injINT,COND(APP(PRIM(P.IGT,icmplty),RECORD[VAR x,INT 0]),
                                 VAR x,APP(PRIM(P.INEG,ineglty),VAR x)))
             end
        | P.INLNOT =>
             let val x = mkLvar()
              in FN(x,LT.inj LT.BOOL,COND(VAR x,CON(transDcon falseDcon,unitLexp),
                                        CON(transDcon trueDcon,unitLexp)))
             end 
        | P.INLCOMPOSE =>
             let val (arglt,reslt) = arrowLty(transTyLty(mergety(t1,t2)))
                 val (at,_) = arrowLty(reslt)
                 val x = mkLvar() and z = mkLvar() and f = mkLvar() and g = mkLvar()
              in FN(z,arglt,LET(f,SELECT(0,VAR z),
                             LET(g,SELECT(1,VAR z),
                               FN(x,at,APP(VAR f,APP(VAR g,VAR x))))))
             end                  
        | P.INLBEFORE =>
             let val (arglt,_) = arrowLty(transTyLty(mergety(t1,t2)))
                 val x = mkLvar()
              in FN(x,arglt,SELECT(0,VAR x))
             end
        | P.ASSIGN => 
	     let val ty = mergety(t1,t2)
		 val x = mkLvar()
		 val varX = VAR x
		 val argty = LT.injRECORD([LT.injBOXED,LT.injBOXED])
              in unwrapOp(assignTy,transTyLty ty)
		   (FN(x,argty,APP(PRIM (unboxedAssign ty,updateTy),
	              RECORD[SELECT(0,varX),INT 0,SELECT(1, varX)])))

	     end
        | P.UPDATE => 
	     let val ty = mergety(t1,t2)
		 val x = mkLvar()
		 val varX = VAR x
		 val argty = LT.injRECORD([LT.injBOXED,LT.injINT,LT.injBOXED])
              in unwrapOp(updateTy,transTyLty ty)
		   (FN(x,argty,APP(PRIM (unboxedUpdate ty,updateTy),
	              RECORD[SELECT(0,varX),SELECT(1, varX),SELECT(2, varX)])))
	     end
        | P.CAST =>
             let val lt = LT.injARROW(LT.inj LT.RBOXED, LT.inj LT.RBOXED)
                 val header = unwrapOp1(lt,t2)
              in header(PRIM(P.CAST,lt))
             end
        | i =>          (* may need rearrangement here *)
             let val lt = transTyLty(t1)
                 val header = unwrapOp1(lt,t2)
              in header(PRIM(i,lt))
             end)
     
   | translateAccess _ = error "translate.translateAccess"

 and transac acc = LVAR(transpath0(acc,NONE,NONE))

 and transrep(VARIABLE acc) = VARIABLE(transac acc)
   | transrep(VARIABLEc acc) = VARIABLEc(transac acc)
   | transrep r = r

 and transDcon(DATACON{name,rep,typ,orig=SOME t,...}) = 
                 (name,transrep rep,transTyLty t)
   | transDcon(DATACON{name,rep,typ,orig=NONE,...}) = 
                 (name,transrep rep,transTyLty typ)

 and conToLexp (d as DATACON{const=true,...}) = CON'(transDcon d,unitLexp)
   | conToLexp (d as DATACON{const=false,typ=ty,orig,name=name,rep=rep,...}) =
              let val v = mkLvar() 
                  val t = case orig of NONE => transTyLty ty
                                     | SOME z => transTyLty z
                  val (t1,t2) = arrowLty(t) 
               in FN(v,t1,CON'((name,transrep rep,t),VAR v)) 
              end

 and COND(a,b,c) =
              SWITCH(a,boolsign,
                    [(DATAcon(transDcon trueDcon),b),
                     (DATAcon(transDcon falseDcon),c)],NONE)
 and LET(v,e,b) = APP(FN(v,LT.BOGUS,b),e)

 and DIV(b,c) = APP(PRIM(P.IDIV,intoplty),RECORD[b,c])
 and SUB(b,c) = APP(PRIM(P.ISUB,intoplty),RECORD[b,c])
 and ADD(b,c) = APP(PRIM(P.IADD,intoplty),RECORD[b,c])
 and MUL(b,c) = APP(PRIM(P.IMUL,intoplty),RECORD[b,c])

 and composeNOT (eq,t) =  
    let val v = mkLvar()
        val (argt,_) = arrowLty(t)
     in FN(v,argt,COND(APP(eq, VAR v),
		       CON(transDcon falseDcon,unitLexp),
		       CON(transDcon trueDcon,unitLexp)))
   end

 fun transBoolDcon(DATACON{name,rep,typ,...}) = (name,rep,LT.inj LT.BOOL)
 val trueDcon' = transBoolDcon trueDcon 
 val falseDcon' = transBoolDcon falseDcon

 fun patvars (VARpat (VALvar{access=LVAR w,...})) = [w]
   | patvars (LAYEREDpat (v,p)) = (patvars(v))@(patvars(p))
   | patvars (RECORDpat{fields,...}) = 
         List.concat (map (patvars o #2) fields)
   | patvars (VECTORpat(pats,_)) = List.concat (map patvars pats)
   | patvars (ORpat(p1, _)) = patvars p1
   | patvars (APPpat(_,_,p)) = patvars p
   | patvars (CONSTRAINTpat (p,_)) = patvars p
   | patvars _ = []

(* The function fill : Error.pos2 -> Absyn.pat -> Absyn.pat expands the
 * flexible record pattern. 
 *)
	  
  fun fill (CONSTRAINTpat (p,t)) = fill p
    | fill (LAYEREDpat (p,q)) = LAYEREDpat(fill p, fill q)
    | fill (RECORDpat {fields,flex=false,typ}) =
	 RECORDpat{fields= map (fn(lab,p)=>(lab,fill p)) fields,
		   typ=typ,flex=false}
    | fill (pat as RECORDpat {fields,flex=true,typ}) =
	   let exception DontBother
	       val fields' = map (fn (l,p) => (l, fill p)) fields
	       fun find (t as CONty(RECORDtyc labels, _)) = 
				  (typ := t; labels)
		 | find _ = (complain COMPLAIN "unresolved flexible record"
			       (fn ppstrm =>
				(add_newline ppstrm;
				 add_string ppstrm "pattern: ";
				 PPAbsyn.ppPat env ppstrm
				  (pat,!Control.Print.printDepth)));
			     raise DontBother)
	       fun merge (a as ((id,p)::r), lab::s) =
		   if Symbol.eq(id,lab) then (id,p) :: merge(r,s)
		   else (lab,WILDpat) :: merge(a,s)
		 | merge (nil, lab::s) = (lab,WILDpat) :: merge(nil,s)
		 | merge (nil,nil) = nil
		 | merge _ = error "merge in translate"
	   in RECORDpat{fields=merge(fields',
				  find(TypesUtil.headReduceType (!typ))),
			flex=false,typ=typ}
	       handle DontBother => WILDpat
	   end
    | fill (VECTORpat(pats,ty)) = VECTORpat(map fill pats, ty)
    | fill (ORpat(p1, p2)) = ORpat(fill p1, fill p2)
    | fill (CONpat(DATACON{name,const,typ,orig,rep,sign},ty)) = 
	 CONpat(DATACON{name=name,const=const,typ=typ,sign=sign,
			orig=orig,rep=transrep rep},
		ty)
    | fill (APPpat(DATACON{name,const,typ,orig,rep,sign},ty,pat)) = 
	 APPpat(DATACON{name=name,const=const,typ=typ,sign=sign,
			orig=orig,rep=transrep rep},
		ty,fill pat)
    | fill pat = pat
	  
(**************************************************************************
 * STRUCTURE AND FUCTOR THINING FUNCTIONS.                                *
 **************************************************************************)
 fun thinStr(e,NONE) = e
   | thinStr(e,SOME(v,locs)) = 
       APP(FN(v,LT.BOGUS,SRECORD(map transLoc locs)), e)

 and thinFct(e,_,NONE,NONE) = e
   | thinFct(e,ty,SOME (v1,locs1),SOME (v2,locs2)) =
        FN(v1,ty,APP(FN(v2,LT.BOGUS,SRECORD(map transLoc locs2)),
                     APP(e,SRECORD(map transLoc locs1))))
   | thinFct(e,ty,NONE,SOME(v,locs)) =
        let val w = mkLvar ()
         in FN(w,ty,APP(FN(v,LT.BOGUS,SRECORD(map transLoc locs)), 
                        APP(e,VAR w)))
	end
   | thinFct (e,ty,SOME(v1,locs1),NONE) =
        FN(v1,ty,APP(e,SRECORD(map transLoc locs1)))

(**************************************************************************
 *               transLoc : Modules.trans -> Lambda.lexp                  *
 **************************************************************************)
 and transLoc(VALtrans(acc as INLINE _,t1,t2)) = translateAccess(acc,t1,t2)
   | transLoc(STRtrans(acc,lt)) = untypedTranslateAccess(acc,lt)
   | transLoc(VALtrans(acc,t1,t2)) = 
          let val lt = transTyLty t1
              val header = unwrapOp1(lt,t2)
           in header(untypedTranslateAccess(acc,lt))
          end
   | transLoc(THINtrans(acc,v,locs)) =
          let val bb = extern acc
              val _ = if bb then add_extern v else ()
              val ee = map transLoc locs
              val be = untypedTranslateAccess(acc,map_var v)
           in APP(FN(v,LT.BOGUS,RECORD ee),be)
          end
   | transLoc(FCTtrans(acc,lt,(thin1,thin2,fctWrapper))) =
          let val header = case fctWrapper of NONE => ident
                                            | SOME p => special_unwrapOp p
              val (argt,_) = arrowLty lt
           in thinFct(header(untypedTranslateAccess(acc,lt)),
                      argt,thin1,thin2)
          end
   | transLoc(CONtrans(d as DATACON{const=true,typ=t1,...},t2)) = 
          let val lt1 = transTyLty(t1)
              val header = unwrapOp1(lt1,t2)
           in header(CON'(transDcon d,unitLexp))
          end
   | transLoc(CONtrans(d as DATACON{const=false,typ=t1,...},t2)) =
          let val v = mkLvar()
              val lt1 = transTyLty(t1)
              val (argt,_) = arrowLty(lt1)
              val header = unwrapOp1(lt1,t2)
           in header(FN(v,argt,CON'(transDcon d, VAR v)))
          end
     (*** The above code is not worried about the "orig" field in 
          DATACON because "sigmatch" has already taken care of it ***)


(**************************************************************************
 *       transStr : Absyn.strexp -> Lambda.lexp          *
 **************************************************************************)
 and transStr sdec =
   case sdec
    of VARstr(STRvar{access,binding,...}) => 
        if extern access then 
             untypedTranslateAccess(access,transStrLty binding)
        else transpath(access,NONE,NONE)
     | STRUCTstr{body,locations,...} =>
	 makedec (SEQdec body) (SRECORD(map transLoc locations))
     | APPstr{oper=FCTvar{access,binding,...},
              instancelty=lt,argexp,argthin,...} =>
         let val tt = transFctLty(binding)
             val header = special_unwrapOp(tt,lt)
          in APP(header(untypedTranslateAccess(access,tt)), 
	         thinStr(transStr argexp, argthin))
         end
     | LETstr(d,body) => makedec d (transStr body)
     | MARKstr(body,region) => withRegion region transStr body


(**************************************************************************
 *   makedec : Absyn.dec -> Lambda.lexp -> Lambda.lexp   *
 **************************************************************************) 
 and makedec dec =
   case dec
   of VALdec vbl => 
      fold (fn (VB{pat=VARpat(VALvar{access=INLINE(_),...}),...},b) => b
	     | (VB{pat=CONSTRAINTpat(VARpat(VALvar{access=INLINE _,...}),_),
		   exp=_,...},b) => b
	     | (VB{pat=VARpat(VALvar{access=LVAR v,...}),
                   exp,...},b) => 
                 let val b = buildheader v b 
                  in APP(FN(v,LT.BOGUS,b),translate exp)
                 end
	     | (VB{pat=CONSTRAINTpat(VARpat(VALvar{access=LVAR v,...}),_),
                   exp,...},b) => 
                 let val b = buildheader v b 
                  in APP(FN(v,LT.BOGUS,b),translate exp)
                 end
 	     | (VB{pat,exp=CASEexp(e,l),...},b) => 
                  let val ee = translate e
                      val le = MC.bindCompile env 
                                 (transrules l,LT.BOGUS,complain)
                      val npat = fill pat
                      val vars = patvars pat
                      val b = (foldr (op o) ident (map buildheader vars)) b
                      val fe = MC.bindCompile env 
                                ([(npat,b),(WILDpat,unitLexp)],LT.BOGUS,complain)
                   in APP(fe,APP(le,ee))
                  end
             | (VB{pat,exp,...},b) =>
                  let val ee = translate exp
                      val npat = fill pat
                      val vars = patvars pat
                      val b = (foldr (op o) ident (map buildheader vars)) b
                      val fe = MC.bindCompile env 
                                ([(npat,b),(WILDpat,unitLexp)],LT.BOGUS,complain)
                   in APP(fe,ee)
                  end)
	  vbl

    | a as VALRECdec rvbl =>
       (makedec (nonrec a) handle Isrec =>
 	(fn e => 
          (let val (vlist,tlist,elist,header) = 
                foldr (fn (RVB{var=VALvar{access=LVAR v,typ=ref ty,...},
                              exp,...}, (vlist,tlist,elist,header)) =>
                            let val ee = translate exp
                                val vt = transTyLty ty
                                val insthd = buildheader(v)
                             in (v::vlist,vt::tlist,ee::elist,insthd o header)
                            end
	               | _ => error "#73 in translate") 
                 (nil,nil,nil,ident) rvbl

            in FIX((vlist,tlist,elist,header(e)))
           end)))

    | LOCALdec(localdec,visibledec) =>
        makedec (SEQdec[localdec,visibledec])

    | EXCEPTIONdec ebl =>
      fold(fn (EBgen{exn=DATACON{rep=VARIABLE(LVAR v),typ,...},
                     ident,...},lexp) =>
                 let val lt = transTyLty typ
                  in APP(FN(v,LT.BOGUS,lexp),EXNF(translate ident, lt))
                 end
	    | (EBgen{exn=DATACON{rep=VARIABLEc(LVAR v),...},ident,...},lexp) =>
 		 APP(FN(v,LT.BOGUS,lexp),EXNC(translate ident))
	    | (EBdef{exn=DATACON{rep=VARIABLE(LVAR v),...},
		     edef=DATACON{rep=VARIABLE(acc),...}}, lexp) => 
                 APP(FN(v,LT.BOGUS,lexp),transpath(acc,NONE,NONE))
	    | (EBdef{exn=DATACON{rep=VARIABLEc(LVAR v),...},
	             edef=DATACON{rep=VARIABLEc(acc),...}}, lexp) => 
                 APP(FN(v,LT.BOGUS,lexp),transpath(acc,NONE,NONE))
	    | _ => error "in makedec EXCEPTIONdec")
        ebl
 
     | SEQdec decl =>
        let fun f(a::r) = (makedec a) o (f r) 
              | f nil = ident
         in f decl 
        end
     | DATATYPEdec _ => ident
     | ABSTYPEdec{body,...} => makedec body
     | TYPEdec _ => ident
     | STRdec sbl =>
         fold(fn (STRB{strvar=STRvar{access=LVAR v,...},abslty,def,thin,...},
                  lexp) =>
                   let val header = case abslty of NONE => ident
                                                 | SOME p => special_wrapOp p
                       val insthd = buildheader(v)
                    in APP(FN(v,LT.BOGUS,insthd(lexp)),
                           header(thinStr(transStr def, thin)))
                   end
 	       | _ => error "makedec(STRdec) in translate")
 	   sbl
     | ABSdec sbl => makedec (STRdec sbl)
     | FCTdec fbl =>
         let fun transFct(FCTfct{param=STRvar{access=LVAR p,...},
 				 def,thin,...},argt) =
                   let val body = thinStr(transStr def,thin)
                       val hdr = buildheader p
  		    in FN(p,argt,hdr(body))
                   end
 	       | transFct(VARfct{fctThin=(thinIn,thinOut,fctWrap), 
                                 def=FCTvar{access,binding,...},...},argt) =
                   let val header = 
                         case fctWrap of NONE => ident
                                       | SOME p => special_unwrapOp p
                       val tt = transFctLty binding
 		    in thinFct(header(untypedTranslateAccess(access,tt)),
                               argt,thinIn,thinOut)
                   end
 	       | transFct(LETfct(dec,body),argt) =
 		   makedec dec (transFct(body,argt))
               | transFct _ = error "transFct in translate"
 
 	  in fold (fn (FCTB{fctvar = FCTvar{access=LVAR v,binding,...},
 			    def,...}, lexp) =>
   		        let val (argt,_) = arrowLty(transFctLty(binding))
                            val hdr = buildheader(v)
 		         in APP(FN(v,LT.BOGUS,hdr(lexp)),transFct(def,argt)) 
                        end
 		   | _ => error "makedec(FCTdec) in translate")
 	      fbl
 	 end
     | SIGdec _ => ident
     | FSIGdec _ => ident 
     | OPENdec _ => ident
     | FIXdec _ => ident
     | OVLDdec _ => ident
     | MARKdec(dec,region) => withRegion region makedec dec 

 and transrules rules = 
       let fun f (RULE(p,e)) = (fill p, translate e) 
        in map f rules 
       end


(**************************************************************************
 *        translate : Source.region -> Absyn.exp -> Lambda.lexp           *
 **************************************************************************) 
 and translate exp =
   case exp 
    of INTexp (s,t) =>
        ((if TypesUtil.equalType (t, intTy) then
	   INT (StringToNum.int s)
	 else error "translate INTexp")
           handle Overflow =>
	     (complain COMPLAIN "int constant too large" nullErrorBody;
	      INT 0))
     | WORDexp(s, t) =>
        ((if TypesUtil.equalType (t, wordTy) then
           WORD (StringToNum.word s)
	 else if TypesUtil.equalType (t, word8Ty) then
	   WORD (StringToNum.word8  s)
	 else if TypesUtil.equalType (t, word32Ty) then
	   WORD32 (StringToNum.word32 s)
	 else error "translate WORDexp")
           handle Overflow =>
	     (complain COMPLAIN "word constant too large" nullErrorBody;
	      INT 0))
     | REALexp r => REAL r
     | STRINGexp s => STRING s
(** NOTE: the following won't work for cross compiling to multi-byte characters **)
     | CHARexp s => INT(Char.ord(String.sub(s, 0)))
     | RECORDexp nil => INT 0
     | RECORDexp l =>
	 if sorted l
	 then let val lexpl = map (fn (_,e) => translate e) l
               in RECORD lexpl
              end
	 else let val vars = map (fn (l,e) => (l,(translate e,mkLvar()))) l
		  fun bind ((_,(e,v)),x) = APP(FN(v,LT.BOGUS,x),e)
                  val bexp = map (fn (_,(_,v)) => VAR v) (sortrec vars)
	       in foldr bind (RECORD bexp) vars
	      end
     | VECTORexp(nil,_) => untypedTranslateAccess(coreAccess "vector0",LT.injBOXED)
     | VECTORexp(l,ty) => 
	 let val t = transTyLty(ty)
             val vars = map (fn e => (e,mkLvar())) l
             fun bind ((e,v),x) = 
               let val header = wrapOp(LT.inj LT.RBOXED,t)
                in APP(FN(v,LT.BOGUS,x), header(translate e))
               end
          in foldr bind (VECTOR(map (fn(_,v)=>VAR v) vars)) vars
         end 
     | SEQexp [e] => translate e
     | SEQexp (e::r) => 
         let val le = translate e
             val re = translate (SEQexp r)
          in APP(FN(mkLvar(),LT.BOGUS,re),le)
         end
     | APPexp(e1 as CONexp(dcon as DATACON{rep=CONSTANT _,...},_), e2) =>
         APP(translate e1, translate e2)
     | APPexp(CONexp(dcon as DATACON{name,rep,typ,orig=NONE,...},NONE), e) => 
         let val le = translate e
             val lt = transTyLty(typ)  
          in CON'((name,transrep rep,lt),le)
         end        
     | MARKexp(e,region) => withRegion region translate e
     | CONexp(dcon as DATACON{const=false,typ=ty,name,rep,orig,...},tt) =>
         let val v = mkLvar()
             val lt = transTyLty(ty)
             val header = unwrapOp1(lt,tt)
             val (lt1,header1) = 
                  (case orig 
                    of SOME z => 
                          (let val zz = transTyLty(z)
                            in (zz, header o (unwrapOp(zz,lt)))
                           end)
                     | NONE => (lt, header))
             val (t1,_) = arrowLty(lt1)
          in header1(FN(v,t1,CON'((name,transrep rep,lt1),VAR v)))
         end
     | CONexp (dcon as DATACON{const=true,typ=ty,name,rep,orig,...},tt) => 
         let val lt = transTyLty(ty) 
             val header = unwrapOp1(lt,tt)
             val (lt1,header1) = 
                  (case orig 
                    of SOME z =>
                          (let val zz = transTyLty(z)
                            in (zz, header o (unwrapOp(zz,lt)))
                           end)
                     | NONE => (lt, header))
          in header1(CON'((name,transrep rep,lt1), unitLexp))
         end
     | VARexp (ref(VALvar{access,typ=ref t1,...}), t2) => 
	      translateAccess(access,t1,t2)
     | VARexp (ref(OVLDvar{name,...}),_) =>
	 error("unresolved overloading: "^Symbol.name name)
     | VARexp (ref ERRORvar,_) =>
         error ("error variable in translate")
     | APPexp (f,a) => 
         APP(translate f, translate a)
     | CONSTRAINTexp (e,_) => translate e
     | HANDLEexp (e,HANDLER(FNexp(l,ty))) =>
         let val ee = translate e
             val le = transrules l
             val lt = transTyLty ty 
             val re = MC.matchCompileHandler env (le, lt, complain) 
         in HANDLE(ee,re) end
     | RAISEexp(e,ty) => 
         let val t = transTyLty ty
          in Raise(translate e,t)
         end
     | FNexp(l,ty) => 
         let val lt = transTyLty ty
             val le = transrules l
          in MC.matchCompile env (le, lt, complain)
	 end
     | CASEexp (e,l) => 
         let val ee = translate e
             val le = transrules l
          in APP(MC.matchCompile env (le, LT.BOGUS, complain),ee)
	 end	
     | LETexp (d,e) => makedec d (translate e)
     | x => ErrorMsg.impossibleWithBody "untranslateable expression"
	      (fn ppstrm =>
	       (add_string ppstrm "expression: ";
		PPAbsyn.ppExp (env,NONE) ppstrm (x,!Control.Print.printDepth)))


 val base = exportLexp

 local val dupacc = coreAccess "dupstring"
 in val dupstring = 
      case dupacc 
       of PATH _ => let val lt = LT.injARROW(LT.injBOXED,LT.injBOXED)
                        val e = untypedTranslateAccess(dupacc,lt)
                     in fn x => APP(e, x)
                    end
        | _ (* a hack, special-case for boot/core.sml *) => (fn le => le)
 end

 val bodyLexp =  Literals.liftlits (dupstring, makedec rootdec base)

 (* `close' over all free (EXTERN) variables [`inlining' version]
  *   - make sure that all operations on various imparative data structures
  *     are carried out NOW and not later when the result function is
  *     called. *)
 fun close lexp: (lexp option list -> lexp) * pid list = let

     (* all the free variables together with their associated pid and
      * lambdatype (as they have been collected during translation *)
     val l: (pid * (lvar * LT.lty)) list = Map.members (!persmap)

     (* the name of the `main' argument *)
     val imports = mkLvar ()

     (* alpha-convert lambda-expression *)
     val copy = Lambda.copy mkLvar

     fun constr (l1 :: ln, (_, (lvar, lt)) :: rest, i, lexp) =
	 let
	     val insthd = buildheader lvar
	     val body = insthd lexp
	     val argsel = APP (PRIM (P.SUBSCRIPTV, subscriptTy),
			       RECORD [VAR imports, INT i])

	     val l1 = NONE		(* no inlining just yet!!! *)
					(* delete this line later! *)

	     val inl =
		 case l1 of
		     NONE => argsel
		   | SOME lambda_i => APP (copy lambda_i, argsel)
	     val fullarg = UNWRAP (lt, inl)
	     val letexp = APP (FN (lvar, LT.BOGUS, body), fullarg)
	 in
	     constr (ln, rest, i + 1, letexp)
	 end
       | constr ([], [], _, lexp) = FN (imports, LT.injBOXED, final lexp)

     fun genLambda inl = constr (inl, l, 0, lexp)
 in
     (genLambda, map #1 l)
 end

 (* the old version of `close':
 fun close lexp : (lexp * pid list) = let
     val l: (pid * (lvar * LT.lty)) list = Map.members (!persmap)
     val imports = mkLvar ()

     fun selectv (n,le) = APP (PRIM (P.SUBSCRIPTV, subscriptTy),
			       RECORD [le, INT n])
     fun f (i, (pid, (lvar, lt)) :: rest, lexp) =
	 let
	     val insthd = buildheader (lvar)
	 in
	     f (i + 1, rest, APP (FN (lvar, LT.BOGUS, insthd (lexp)),
				  UNWRAP (lt, selectv (i, VAR imports))))
	 end
       | f (i, nil, lexp) = lexp
 in
     (FN (imports, LT.injBOXED, final (f (0, l, lexp))), map #1 l)
 end
 *)

 val (genLambda, pids) = close bodyLexp

in
    { genLambda = genLambda, importPids = pids }
end

end (* structure Translate *)
