(* Copyright 1989 by AT&T Bell Laboratories *)
(* transbinding.sml *)

signature TRANSBINDING = sig 
  val transStrLty  : Modules.Structure -> LambdaType.lty
  val transFctLty  : Modules.Functor -> LambdaType.lty
  val transStrLty0 : Modules.Structure * Modules.Structure -> LambdaType.lty
  val transFctLty0 : Modules.Functor * Modules.Structure -> LambdaType.lty
end

structure TransBinding : TRANSBINDING =
struct 

  open Modules Variables Types Transtypes Lambda 
  structure LT = LambdaType

val error = ErrorMsg.impossible

local fun path(Access.PATH(i,_)) = (Control.Print.say "funny 22 in transbinding.sml"; i)
        | path(Access.SLOT i) = i
	| path _ = ErrorMsg.impossible "transbinding.sml 333"
      fun getSlot(VARbind(VALvar{access,...})) = path access
	| getSlot(CONbind(DATACON{rep=Access.VARIABLE(access),...})) = path access
	| getSlot(CONbind(DATACON{rep=Access.VARIABLEc(access),...})) = path access
	| getSlot(STRbind(STRvar{access,...})) = path access
	| getSlot(FCTbind(FCTvar{access,...})) = path access
	| getSlot _ = ~1
      fun gt ((_,b1),(_,b2)) = getSlot b1 > getSlot b2
  in val sortSlots = Sort.sort gt
end

fun makeSRECORD [] = LT.injINT
  | makeSRECORD tt = LT.inj(LT.SRECORD tt)

fun transBindingLty binding = 
  case binding
   of VARbind(VALvar{typ,...}) => [transTyLty(!typ)]
    | CONbind(DATACON{typ,rep=Access.VARIABLE _,...}) => [LT.injBOXED]
                                           (***>> [transTyLty(typ)] <<***)
    | CONbind(DATACON{typ,rep=Access.VARIABLEc _,...}) => [LT.injBOXED]
                                           (***>> [transTyLty(typ)] <<***)
    | CONbind(DATACON _) =>  []
    | SIGbind(SIGvar{binding,...}) => []
    | STRbind(STRvar{binding,name,...}) => 
           if Extern.hidden name then [] else [transStrLty(binding)]
    | FSIGbind(FSIGvar{binding,...}) => []
    | FCTbind(FCTvar{binding,...}) => [transFctLty(binding)]
    | _ => []

and transStrLty str =
  case str 
   of SIMPLE{env,id=ModuleId.INSTANCEid _,...} => transEnvLty(env,sortSlots)
    | SIMPLE{env,...} => transEnvLty(env,fn x => x)
    | INSTANCE{sign,subStrs,subFcts,types,...} => transSigLty0(sign,str)
    | APPLY{res,...} => transStrLty(res)
    | ERROR_STR => LT.BOGUS
    | STR_FORMAL _ => error "transStrLty on STR_FORMAL"
    | STR_OPEN _ => error "transStrLty on STR_OPEN"
    | STR_ABSFB _ => error "transStrLty on STR_ABSFB"
    | STR_EXTERN _ => error "transStrLty on STR_EXTERN"

and transFctLty fct = 
  case fct 
   of FCT{lambdaty,...} => lambdaty
    | FCT_INSTANCE{lambdaty,...} => lambdaty
    | ERROR_FCT => LT.BOGUS
    | FCT_FORMAL _ => error "transFctLty on FCT_FORMAL"
    | FCT_OPEN _ => error "transFctLty on FCT_OPEN"
    | FCT_ABSFB _ => error "transFctLty on FCT_ABSFB"
    | FCT_EXTERN _ => error "transFctLty on FCT_EXTERN"

and transEnvLty(env, sort) = 
  let val revEnv = sort(ModuleUtil.sortEnvBindings env)
      val tyLst = map (transBindingLty o #2) revEnv
   in makeSRECORD(List.concat tyLst)
  end

and transBindingLty0 str binding = 
  let val mapty = transTyLty o (ModuleUtil.transType str)
   in case binding
       of VARbind(VALvar{typ,...}) => [mapty(!typ)]
        | CONbind(DATACON{typ,rep=Access.VARIABLE _,...}) => [LT.injBOXED]
                                                (***>>  [mapty(typ)] <<***)
        | CONbind(DATACON{typ,rep=Access.VARIABLEc _,...}) => [LT.injBOXED]
                                                (***>>  [mapty(typ)] <<***)
        | SIGbind(SIGvar{binding,...}) => []
        | STRbind(STRvar{binding,name,...}) => 
              if Extern.hidden name then [] else [transStrLty0(binding,str)]
        | FSIGbind(FSIGvar{binding,...}) => []
        | FCTbind(FCTvar{binding,...}) => [transFctLty0(binding,str)]
        | _ => []
  end

and transSigLty0(sign,str) =
  case sign 
    of SIG{env,symbols,...} =>
        let fun lookup s = Env.look(!env,s)
            val tyLst = map ((transBindingLty0 str) o lookup) (!symbols)
         in makeSRECORD(List.concat tyLst)	  
        end
     | FULL_SIG => error "transSigLty on FULL_SIG"
     | _ => error "transSigLty on wrong objects"

and transStrLty0(str,str1 as INSTANCE{subStrs,...}) =
    (case str 
      of SIMPLE{env,id=ModuleId.INSTANCEid _,...} =>
          let val revEnv = ModuleUtil.sortEnvBindings env
	      val _ = Control.Print.say "That's strange, I thought we'd never get here.\n"
	      val revEnv = sortSlots revEnv
              val tyLst = map ((transBindingLty0 str1) o #2) revEnv
           in makeSRECORD(List.concat tyLst)	  
          end
       | SIMPLE{env,...} => 
          let val revEnv = ModuleUtil.sortEnvBindings env
              val tyLst = map ((transBindingLty0 str1) o #2) revEnv
           in makeSRECORD(List.concat tyLst)	  
          end
       | INSTANCE{sign,...} => transSigLty0(sign,str)
       | STR_OPEN{spec=spec as FULL_SIG,pos=[i],...} => 
          let val newstr = Array.sub(subStrs,i)
           in transStrLty0(newstr,str1)
          end
       | STR_OPEN{spec=spec as FULL_SIG,pos=i::r,path} => 
          let val newstr = Array.sub(subStrs,i)
           in transStrLty0(STR_OPEN{spec=FULL_SIG,pos=r,path=path},newstr)
          end
       | STR_OPEN{spec,...} => transSigLty0(spec,str1)
       | STR_FORMAL{pos,...} => 
          let val newstr = Array.sub(subStrs,pos)
           in transStrLty0(newstr,str1)
          end
       | APPLY{res,...} => transStrLty0(res,str1)
       | STR_ABSFB _ => error "transStrLty0 on STR_ABSFB objects"
       | ERROR_STR => LT.BOGUS
       | _ => error "transStrLty0 on ERROR object")

  | transStrLty0 _ = error "transStrLty0 on non-INSTANCE backgrounds"   

and transFctLty0(fct,str as INSTANCE{subFcts,...}) = 
    (case fct 
      of FCT{lambdaty,...} => lambdaty
       | FCT_INSTANCE{lambdaty,...} => lambdaty
       | FCT_FORMAL{pos,...} =>
          let val newfct = Array.sub(subFcts,pos)
           in transFctLty0(newfct,str)
          end
       | FCT_OPEN{spec=spec as FULL_FSIG,pos=[i],...} => 
          let val newfct = Array.sub(subFcts,i)
           in transFctLty0(newfct,str)
          end
       | FCT_OPEN{spec=spec as FULL_FSIG,pos=i::r,path} => 
          let val newfct = Array.sub(subFcts,i)
           in transFctLty0(FCT_OPEN{spec=FULL_FSIG,pos=r,path=path},str)
          end
       | ERROR_FCT => LT.BOGUS
       | FCT_OPEN _ => error "transFctLty on FCT_OPEN" 
                               (* transFsigLty0(spec,str) *)
       | _ => error "transFctLty on wrong object")

  | transFctLty0 _ = error "transFctLty0 on non-INSTANCE backgrounds"   

(****************************************************************************
 *  Turn off all effects if !Control.CG.representations is false            * 
 ****************************************************************************)
val rep_flag = ref true (* Control.CG.representations  *)
val bogus1 = LT.BOGUS
val bogus2 = LT.injARROW(LT.BOGUS,LT.BOGUS)

val transStrLty = fn x => if !rep_flag then transStrLty x else bogus1
val transFctLty = fn x => if !rep_flag then transFctLty x else bogus2
val transStrLty0 = fn x => if !rep_flag then transStrLty0 x else bogus1
val transFctLty0 = fn x => if !rep_flag then transFctLty0 x else bogus2


end (* structure TransBinding *)
