(* profile.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

structure Profile : sig

    val msPerSample : int

    val reset : unit -> unit          (* reset profiling counts to zero *)

    val report : TextIO.outstream -> unit
	  (* print profiling report to stream *)
    val reportAll : TextIO.outstream -> unit
	  (* print profiling report to stream; DON'T suppress zero entries*)
    val reportData: unit -> {name: string, count: int, time: Time.time} list
	  (* Return the unformatted data for a report *)

    val runTimeIndex : int
    val gcIndex : int
    val otherIndex : int
    val compileIndex : int

  end = struct

    structure A = Array

  (** NOTE: at some point, we should be able to set this. **)
    val msPerSample = 10

    datatype compunit = UNIT of {
	base: int,
	size: int,
	counts: int A.array,
	names: string
      }
			   
    val runTimeIndex = 0
    val gcIndex = 1
    val otherIndex = 2
    val compileIndex = 3
    val numPredefIndices = 4

    val current : int ref = System.Unsafe.Assembly.profCurrent
    val times = ref(A.array(numPredefIndices, 0)) (** FIX ME **)
    val _ = (
	  current := otherIndex;
	  times := A.array(numPredefIndices, 0))

    fun increase n = if n <= A.length(!times)
	  then ()
	  else let val new = A.array(n+n, 0)
	    in
	      A.copy{di=0, dst=new, len=NONE, si=0, src = !times};
	      times := new
	    end

    val units = ref [UNIT{
	    base = 0,
	    size = numPredefIndices,
	    counts = A.array(numPredefIndices, 0),
	    names = "\
		\Run-time System\n\
		\Garbage Collection\n\
		\Other\n\
		\Compilation\n\
	      \"
	  }];

    fun newlines s = 
	let val n = size s
	    fun f(i,count) = if i=n then count 
			     else if substring(s,i,1)="\n"
				      then f(i+1,count+1)
				      else f(i+1,count)
         in f(0,0)
	end
					     

    fun register names =
	let val ref (list as UNIT{base,size,...}::_) = units
	    val count = newlines names
	    val a = A.array(count,0)
	    val b = base+size
	 in increase(b+count);
	     units := UNIT{base=b,size=count,counts=a,names=names}::list;
	    (b,a,current)
        end

    val _ =  System.Unsafe.profile_register := register;

    fun reset() = let
	  fun zero a = A.modify (fn _ => 0) a
	  in
	    zero (!times);
	    app (fn UNIT{counts,...}=> zero counts) (!units)
	  end
    
    datatype entry = ENTRY of {name: string, count: int, time: int}

    val splitlines = String.tokens (fn #"\n" => true | _ => false)

    fun join(entries, base, _, counts, times, nil) = entries
      | join(entries, base, n, counts, times, line::lines) =
          join (ENTRY{
	      name = line, count= A.sub(counts, n), time = A.sub(times, base+n)
	    }::entries, base, n+1, counts, times, lines)

    fun batch(UNIT{base,size,counts,names}) =
	  join(nil, base, 0, counts, !times, splitlines names)

    fun log10 0 = 0
      | log10 i = 1 + log10(i div 10)

    fun field (st, w) = StringCvt.padLeft #" " w st

    fun decimal(st,w) =
	(substring(st,0,size st - w) handle Substring => "") ^ "." ^
	      let val st' = "00000000000" ^ st
	       in substring(st',size st' - w,w)
	      end

    fun muldiv(i,j,k) =
	  (i*j div k) 
	     handle Overflow => muldiv(i,j div 2, k div 2)

    fun decfield(n,j,k,w1,w2) = 
	  field(decimal(Int.toString (muldiv(n,j,k)),w1)
		  handle Div => "",w2)

    fun getBigList() = let
	  val biglist = List.concat (List.map batch (!units))
          fun compare (
		ENTRY{time=a,count=ca,name=na,...},
		ENTRY{time=b,count=cb,name=nb,...}
	      ) = a<b orelse a=b andalso (ca<cb orelse ca=cb andalso na>nb)
	  in
	    Sort.sort compare biglist
	  end

    fun reportData() =
	  map (fn ENTRY{name,count,time} => {
	      name=name, count=count, time=Time.fromMilliseconds(time*msPerSample)
	    }) (getBigList ())

    fun reportx suppress outstream = let
	  val biglist' = getBigList()
	  val tot = List.foldr (fn (ENTRY{time=a,...},b)=> a+b) 0 biglist'
	  val maxc = List.foldr (fn (ENTRY{count=a,...},b)=> max(a,b)) 0 biglist'
	  val digits_cum = log10 tot
          val digits_cnt = max(6,1+log10 maxc)
	  fun pr s = TextIO.output(outstream, s)
	  fun printlines (ENTRY{time,name,count}::rest, cum) =
		if suppress andalso count=0 andalso time=0 then ()
                else (
		  pr(decfield(time,10000,tot,2,6));
		  if (digits_cum > 4)
		    then pr(field(Int.toString(cum+time+50 div 100),7))
		    else pr(decfield(cum+time,1,1,2,7));
		  pr(field(Int.toString count,digits_cnt));
(*		  pr(decfield(time,50000,count,4,10)); *)
		  pr "  "; pr name; pr "\n";
		  printlines(rest,cum+time))
	    | printlines (nil, _) = ()

	 in pr(field("%time",6));
	    pr(field("cumsec",7));
	    pr(field("#call",digits_cnt));
(*	    pr(field("ms/call",10)); *)
	    pr("  name\n");
	    printlines(biglist',0);
	    flush_out outstream
        end

   val report = reportx true
   val reportAll = reportx false

end;
