(* Copyright 1994 by AT&T Bell Laboratories *)
(* print/ppast.sml *)

signature PPAST =
sig
  val ppExp     : PrettyPrint.ppstream -> Ast.exp * int -> unit
  val ppPat     : PrettyPrint.ppstream -> Ast.pat * int -> unit
  val ppRule    : PrettyPrint.ppstream -> Ast.rule * int   -> unit
  val ppVb      : PrettyPrint.ppstream -> Ast.vb * int -> unit
  val ppRvb     : PrettyPrint.ppstream -> Ast.rvb * int -> unit
  val ppDec     : PrettyPrint.ppstream -> Ast.dec * int -> unit
  val ppStrexp  : PrettyPrint.ppstream -> Ast.strexp * int -> unit
  val ppFctexp  : PrettyPrint.ppstream -> Ast.fctexp * int -> unit
  val ppSigexp  : PrettyPrint.ppstream -> Ast.sigexp * int -> unit
  val ppFsigexp : PrettyPrint.ppstream -> Ast.fsigexp * int -> unit
  val ppSpec    : PrettyPrint.ppstream -> Ast.spec * int -> unit
  val ppTy      : PrettyPrint.ppstream -> Ast.ty * int -> unit
end

structure PPAst: PPAST =
struct

open Ast

open Symbol PrettyPrint PPUtil

(* symbolic path (Modules.spath) *)
type path = symbol list

fun C f x y = f y x

(* PATTERN *)
fun ppPat ppstrm =
    let val ppsay = add_string ppstrm
	fun ppPat' (_,0) = ppsay "<pat>"
	  | ppPat' (VarPat p,_) = ppSymPath ppstrm p
	  | ppPat' (WILDpat,_) = ppsay "_"
	  | ppPat' (INTpat i,_) = ppsay (makestring i)
	  | ppPat' (REALpat r,_) = ppsay r
	  | ppPat' (STRINGpat s,_) = pp_mlstr ppstrm s
	  | ppPat' (CHARpat s,_) = (ppsay "#"; pp_mlstr ppstrm s)
	  | ppPat' (LayeredPat(v,p),d) =
	      (begin_block ppstrm CONSISTENT 0;
	       ppPat'(v,d); ppsay " as "; ppPat'(p,d-1);
	       end_block ppstrm)
		    (* Handle 0 length case specially to avoid {,...}: *)
	  | ppPat' (RecordPat{def=[],flexibility},_) =
	      if flex then ppsay "{...}"
	      else ppsay "()"
	  | ppPat' (r as RecordPat{def,flexibility},d) =
	      ppClosedSequence ppstrm
		{front=(C add_string "{"),
		 sep=(fn ppstrm => (add_string ppstrm ",";
				    add_break ppstrm (0,0))),
		 back=(fn ppstrm => if flexibility then add_string ppstrm ",...}"
				    else add_string ppstrm "}"),
		 pr=(fn ppstrm => fn (sym,pat) =>
		     (ppSym ppstrm sym; add_string ppstrm "=";
		      ppPat'(pat,d-1))),
		 style=INCONSISTENT}
		def
	  | ppPat' (TuplePat fields,d) =
	      ppClosedSequence ppstrm
		{front=(C add_string "("),
		 sep=(fn ppstrm => (add_string ppstrm ",";
				    add_break ppstrm (0,0))),
		 back=(C add_string ")"),
		 pr=(fn _ => fn pat => ppPat'(pat,d-1)),
		 style=INCONSISTENT}
		fields
	  | ppPat' (ListPat nil, d) = ppsay "[]"
	  | ppPat' (ListPat pats, d) = 
	      let fun pr _ pat = ppPat'(pat, d-1)
	       in ppClosedSequence ppstrm
		    {front=(C add_string "["),
		     sep=(fn ppstrm => (add_string ppstrm ",";
					add_break ppstrm (0,0))),
		     back=(C add_string "]"),
		     pr=pr,
		     style=INCONSISTENT}
		    pats
	      end
	  | ppPat' (VectorPat nil, d) = ppsay "#[]"
	  | ppPat' (VectorPat pats, d) = 
	      let fun pr _ pat = ppPat'(pat, d-1)
	       in ppClosedSequence ppstrm
		    {front=(C add_string "#["),
		     sep=(fn ppstrm => (add_string ppstrm ",";
					add_break ppstrm (0,0))),
		     back=(C add_string "]"),
		     pr=pr,
		     style=INCONSISTENT}
		    pats
	      end
	  | ppPat' (OrPat pats, d) =
	      ppClosedSequence ppstrm
		{front = (C add_string "("),
		 sep = fn ppstrm =>
			(add_break ppstrm (1,0); add_string ppstrm "| "),
		 back = (C add_string ")"),
		 pr = (fn _ => fn pat => ppPat'(pat, d-1))
		 style = INCONSISTENT} 
		pats
	  | ppPat' (APPpat(constructor,argument), d) =
	      (begin_block ppstrm CONSISTENT 0;
	       ppsay "(";
	       ppExp'(constructor,d-1);
	       ppExp'(argument,d-1);
	       ppsay ")";
	       end_block ppstrm)
	  | ppPat' (ConstraintPat{pattern,constraint},d) =
	     (begin_block ppstrm INCONSISTENT 0;
	      ppPat'(pattern,d-1); ppsay " :";
	      add_break ppstrm (1,2);
	      ppType ppstrm constraint;
	      end_block ppstrm)
	  | ppPat' (MarkPat(pat,_),d) = ppPat'(pat,d)
     in ppPat'
    end


(* EXPRESSIONS *)
fun ppExp ppstream =
    let val ppsay = add_string ppstrm
	fun ppExp'(_,0) = ppsay "<exp>"
	  | ppExp'(VarExp p,_) = ppsay (formatQid p)
	  | ppExp'(FnExp rules,d) =
	      (begin_block ppstrm CONSISTENT 0;
	       ppvlist ppstrm ("(fn ","  | ",
			       (fn ppstrm => fn r =>
				  ppRule ppstrm (r,d-1)),
			       rules);
	       ppsay ")";
	       end_block ppstrm)
          | ppExp'(AppExp{function,argument},d) =
	      (begin_block ppstrm CONSISTENT 0;
	       ppsay "(";
	       ppExp'(function,d-1);
	       ppExp'(argument,d-1);
	       ppsay ")";
	       end_block ppstrm)
	  | ppExp'(CaseExp{expr,rules},d) =
	      (begin_block ppstrm CONSISTENT 0;
	       ppsay "(case "; ppExp'(exp,d-1); nl_indent ppstrm 2;
	       ppvlist ppstrm ("of ","   | ",
		 (fn ppstrm => fn r => ppRule ppstrm (r,d-1)), rules);
	       ppsay ")";
	       end_block ppstrm)
	  | ppExp'(LetExp{dec,expr},d) =
	      (begin_block ppstrm CONSISTENT 0;
	       ppsay "let "; ppDec ppstrm (dec,d-1);
	       add_break ppstrm (1,0);
	       ppsay " in "; ppExp'(exp,d-1);
	       add_break ppstrm (1,0);
	       ppsay "end";
	       end_block ppstrm)
	  | ppExp'(SeqExp exps,d) =
	      ppClosedSequence ppstrm
	        {front=(C add_string "("),
		 sep=(fn ppstrm => (add_string ppstrm ";";
				    add_break ppstrm (1,0))),
		 back=(C add_string ")"),
		 pr=(fn _ => fn exp => ppExp'(exp,d-1)),
		 style=INCONSISTENT}
		exps
	  | ppExp'(IntExp i,_) = ppsay i
	  | ppExp'(WordExp w,_) = ppsay w
	  | ppExp'(RealExp r,_) = ppsay r
	  | ppExp'(StringExp s,_) = pp_mlstr ppstrm s
	  | ppExp'(CharExp s,_) = (ppsay "#"; pp_mlstr ppstrm s)
	  | ppExp'(RecordExp fields,d) =
	      ppClosedSequence ppstrm
		{front=(C add_string "{"),
		 sep=(fn ppstrm => (add_string ppstrm ",";
				    add_break ppstrm (0,0))),
		 back=(C add_string "}"),
		 pr=(fn ppstrm => fn (label,exp) =>
		     (ppSym ppstrm label; ppsay "=";
		      ppExp'(exp,d))),
		 style=INCONSISTENT}
		fields
	  | ppExp'(TupleExp fields,d) =
	      ppClosedSequence ppstrm
		{front=(C add_string "("),
		 sep=(fn ppstrm => (add_string ppstrm ",";
				    add_break ppstrm (0,0))),
		 back=(C add_string ")"),
		 pr=(fn _ => fn exp => ppExp'(exp,d-1)),
		 style=INCONSISTENT}
		fields
	  | ppExp'(ListExp elements,d) =
	      ppClosedSequence ppstrm
		{front=(C add_string "["),
		 sep=(fn ppstrm => (add_string ppstrm ",";
				    add_break ppstrm (0,0))),
		 back=(C add_string "]"),
		 pr=(fn _ => fn exp => ppExp'(exp,d-1)),
		 style=INCONSISTENT}
		fields
	  | ppExp'(VectorExp elements,d) =
	      ppClosedSequence ppstrm
		{front=(C add_string "#["),
		 sep=(fn ppstrm => (add_string ppstrm ",";
				    add_break ppstrm (0,0))),
		 back=(C add_string "]"),
		 pr=(fn _ => fn exp => ppExp'(exp,d-1)),
		 style=INCONSISTENT}
		fields
	  | ppExp'(SelectorExp sym,_) =  (* selector of a record field *)
	      (ppsay "#"; ppSym ppstrm sym)
          | ppExp'(ConstraintExp{expr,constraint},d) = (* type constraint *)
	      (begin_block ppstrm INCONSISTENT 0;
	       ppsay "("; ppExp'(expr,d); ppsay ":";
	       add_break ppstrm (1,2);
	       ppType(constraint,d); ppsay ")";
	       end_block ppstrm)
          | ppExp'(HandleExp{expr, rules},d) = (* exception handler *)
	     (begin_block ppstrm CONSISTENT 0;
	      ppExp'(exp,d-1); add_newline ppstrm; ppsay "handle ";
	      nl_indent ppstrm 2;
	      ppvlist ppstrm ("  ","| ",
		  (fn ppstrm => fn r => ppRule ppstrm (r,d-1)), rules);
	      end_block ppstrm)
          | ppExp'(RaiseExp exp,d) =  (* raise an exception *)
	      (begin_block ppstrm CONSISTENT 0;
	       ppsay "raise "; ppExp'(exp,d-1);
	       end_block ppstrm)
          | ppExp'(IfExp{test, thenCase, elseCase},d) = (* if expression *)
	      (begin_block ppstrm CONSISTENT 0;
	       ppsay "if "; ppExp'(test, d-1);
	       add_break ppstrm (1,0);
	       ppsay "then"; ppExp'(thenCase, d-1);
	       add_break ppstrm (1,0);
	       ppsay "else"; ppExp'(elseCase, d-1);
	       end_block ppstrm)
          | ppExp'(AndalsoExp(exp1,exp2),d) =	(* andalso (derived form) *)
	      (begin_block ppstrm INCONSISTENT 0;
	       ppsay "("; ppExp'(exp1,d-1);
	       add_break ppstrm (1,0);
	       ppsay "andalso"; 
	       add_break ppstrm (1,0);
	       ppExp'(exp2,d-1); ppsay ")";
	       end_block ppstrm)
          | ppExp'(OrelseExp(exp1,exp2),d) =	(* orelse (derived form) *)
	      (begin_block ppstrm INCONSISTENT 0;
	       ppsay "("; ppExp'(exp1,d-1);
	       add_break ppstrm (1,0);
	       ppsay "orelse"; 
	       add_break ppstrm (1,0);
	       ppExp'(exp2,d-1); ppsay ")";
	       end_block ppstrm)
          | ppExp'(WhileExp{test,expr},d) = (* while (derived form) *)
	      (begin_block ppstrm CONSISTENT 0;
	       ppsay "while"; ppExp'(test,d-1);
	       add_break ppstrm (1.0); ppsay "do";
	       ppExp'(expr,d-1);
	       end_block ppstrm)
     in ppExp'
    end

(* RULE for case functions and exception handler *)
and ppRule ppstrm (Rule{pat,exp},d) =
    if d>0
    then (begin_block ppstrm CONSISTENT 0;
	  ppPat ppstrm (pat,d-1);
	  add_string ppstrm " =>"; add_break ppstrm (1,2);
	  ppExp ppstrm (exp,d-1);
	  end_block ppstrm)
    else add_string ppstrm "<rule>"


(* STRUCTURE EXPRESSION *)

and ppStrexp ppstrm =
    let val ppsay = add_string ppstrm
	fun ppStrexp'(_,0) = ppsay "<strexp>"
	  | ppStrexp'(VarStr p,d) = ppsay (formatQid p)
	  | ppStrexp'(StructStr dec,d) =
	      (begin_block ppstrm CONSISTENT 0;
	       ppsay "struct"; nl_indent ppstrm 2;
	       ppDec ppstrm (dec,d-1);
	       ppsay "end";
	       end_block ppstrm)
	  | ppStrexp'(AppStr(path, args),d) =
	      (ppsay (formatQid p);
	       begin_block ppstrm INCONSISTENT 2;
	       List.app (fn (str,_) => 
			  ppsay"("; ppStrexp'(str,d-1); ppsay")")
	         args;
	       end_block ppstrm)
	  | ppStrexp'(LETstr(dec,body),d) =
	      (begin_block ppstrm CONSISTENT 0;
	       ppsay "let "; ppDec ppstrm (dec,d-1); add_newline ppstrm;
	       ppsay " in "; ppStrexp'(body,d-1); add_newline ppstrm;
	       ppsay "end";
	       end_block ppstrm)
	  | ppStrexp'(MARKstr(body,_),d) = ppStrexp'(body,d)
     in ppStrexp'
    end

(* FUNCTOR EXPRESSION *)
and ppFctexp ppstrm =
    let val ppsay = add_string ppstrm
	fun ppFctexp'(VarFct(p,constraint),d) =
	      (ppsay(formatQid p);
	       case constraint
		 of SOME(fsig) => (ppsay ":"; ppFsigexp ppstrm (fsig,d-1))
		  | NONE => ())
	  | ppFctexp'(FctFct{params,body,constraint},d) =
	      (begin_block ppstrm CONSISTENT 2;
	       ppsay "FUNCTOR";
	       begin_block ppstrm INCONSISTENT 2;
	       List.app (fn (pname,sign) => 
			  ppsay"(";
			  case pname of NONE => ppsay "_"
			     | SOME sym => ppSym ppstr sym;
			  ppsay ":"; ppSigexp ppstr (sign,d-1); ppsay")")
	         params;
	       end_block ppstrm;
	       case constraint
		 of SOME(fsig) => (ppsay ":"; ppFsigexp ppstrm (fsig,d-1))
		  | NONE => ()
	       ppsay " = ";
	       ppStrexp ppstrm (body,d-1);
	       end_block ppstrm)
	  | ppFctexp'(LetFct(dec,fctexp),d) =
	      (begin_block ppstrm CONSISTENT 0;
	       ppsay "let "; ppDec ppstrm (dec,d-1); add_newline ppstrm;
	       ppsay " in "; ppFctexp'(fctexp,d-1); add_newline ppstrm;
	       ppsay "end";
	       end_block ppstrm)
	  | ppFctexp'(AppFct(p,args,constraint),d) =
	      (ppsay (formatQid p);
	       begin_block ppstrm INCONSISTENT 2;
	       List.app (fn (str,_) => 
			  ppsay"("; ppStrexp'(str,d-1); ppsay")")
	         args;
	       end_block ppstrm)
	  | ppFctexp'(MarckFct(fct,_),d) = ppFctexp'(fct,d)
     in ppFctexp'
    end

(* SIGNATURE EXPRESSION *)
and ppSigexp ppstrm =
    let val ppsay = add_string ppstrm
	fun ppSigexp'(_,0) = ppsay "<sigexp>"
	  | ppSigexp'(VarSig sym,d) = ppSym ppstrm sym
	  | ppSigexp'(SigSig specs,d) = 
	      (begin_block ppstrm CONSISTENT 0;
	       ppsay "sig";
	       begin_block ppstrm CONSISTENT 0;
	       nl_indent ppstrm 2;
	       ppvlist ppstrm ("","",
			       (fn ppstrm => fn spec => ppSpec ppstrm (spec,d-1)),
			       specs);
	       end_block ppstrm;
	       ppsay "end";
	       end_block ppstrm)
	  | ppSigexp'(MarkSig(sigexp,_),d) = ppSigexp'(sigexp,d)
     in ppSigexp''
    end

(* FUNCTOR SIGNATURE EXPRESSION *)
and ppFsigexp ppstrm =
    let val ppsay = add_string ppstrm
	fun ppFsigexp'(_,0) = ppsay "<fsigexp>"
	  | ppFsigexp'(VarFSig sym,d) = ppSym ppstrm sym
	  | ppFsigexp'(FsigFsig{param,def},d) =
	      (begin_block ppstrm CONSISTENT 2;
	       ppsay "FUNSIG";
	       begin_block ppstrm INCONSISTENT 2;
	       List.app (fn (pname,sign) => 
			  ppsay"(";
			  case pname of NONE => ppsay "_"
			     | SOME sym => ppSym ppstr sym;
			  ppsay ":"; ppSigexp ppstr (sign,d-1); ppsay")")
	         param;
	       end_block ppstrm;
	       ppsay ":"; ppFsigexp ppstrm (fsig,d-1);
	       end_block ppstrm)
	  | ppFsigexp'(MarkFsig(fsigexp,_),d) = ppFsigexp'(fsigexp,d)
     in ppFsigexp'
    end

(* SPECIFICATION FOR SIGNATURE DEFINITIONS *)
and ppSpec ppstrm =
    let val ppsay = add_string ppstrm
	fun ppFsigexp'(_,0) = ppsay "<spec>"
	  | ppFsigexp'(StrSpec specs,d) = ppsay "<StrSpec>"
	  | ppFsigexp'(TycSpec _,d) = ppsay "<TycSpec>"
	  | ppFsigexp'(FctSpec _,d) = ppsay "<FctSpec>"
	  | ppFsigexp'(ValSpec _,d) = ppsay "<ValSpec>"
	  | ppFsigexp'(DataSpec _,d) = ppsay "<DataSpec>"
	  | ppFsigexp'(ExceSpec _,d) = ppsay "<ExceSpec>"
	  | ppFsigexp'(FixSpec _,d) = ppsay "<FixSpec>"
	  | ppFsigexp'(ShareSpec _,d) = ppsay "<ShareSpec>"
	  | ppFsigexp'(ShatySpec _,d) = ppsay "<ShatySpec>"
	  | ppFsigexp'(LoacalSpec _,d) = ppsay "<LocalSpec>"
	  | ppFsigexp'(IncludeSpec _,d) = ppsay "<IncludeSpec>"
	  | ppFsigexp'(OpenSpec _,d) = ppsay "<OpenSpec>"
	  | ppFsigexp'(MarkSpec _,d) = ppsay "<MarkSpec>"
     in ppFsigexp'
    end

(* DECLARATIONS (let and structure) *)
and ppDec ppstrm =
    let val ppsay = add_string ppstrm
	fun ppDec'(_,0) = ppsay "<dec>"
	  | ppDec'(ValDec _,d) = ppsay "<ValDec>"
	  | ppDec'(ValrecDec _,d) = ppsay "<ValrecDec>"
	  | ppDec'(FunDec _,d) = ppsay "<FunDec>"
	  | ppDec'(TypeDec tbs,d) = ppsay "<TypeDec>"
	      (begin_block ppstrm CONSISTENT 0;
	       ppvlist ppstrm ("type "," and ",
		(fn ppstrm => fn tb => ppTb ppstrm (tb,d)),
		 tbs);
	       end_block ppstrm)
	  | ppDec'(DatatypeDec _,d) = ppsay "<DatatypeDec>"
	  | ppDec'(AbstypeDec _,d) = ppsay "<AbstypeDec>"
	  | ppDec'(ExceptionDec _,d) = ppsay "<ExceptionDec>"
	  | ppDec'(StrDec strbs,d) = ppsay "<StrDec>"
	      (begin_block ppstrm CONSISTENT 0;
	       ppvlist ppstrm ("structure ","and ",
		(fn ppstrm => fn strb => ppStrb ppstrm (strb,d-1)),
		 strbs);
	       end_block ppstrm)
	  | ppDec'(AbsDec _,d) = ppsay "<AbsDec>"
	      (begin_block ppstrm CONSISTENT 0;
	       ppvlist ppstrm ("abstraction ","and ",
		(fn ppstrm => fn strb => ppStrb ppstrm (strb,d-1)),
		strbs);
	       end_block ppstrm)
	  | ppDec'(FctDec fctbs,d) = ppsay "<FctDec>"
	      (begin_block ppstrm CONSISTENT 0;
	       ppvlist ppstrm ("functor ","and ",
		(fn ppstrm => fn ftrb => ppFctb ppstrm (fctb,d-1)),
		strbs);
	       end_block ppstrm)
	  | ppDec'(SigDec sigbs,d) =
	      (begin_block ppstrm CONSISTENT 0;
	       ppvlist ppstrm ("signature ","and ",
		(fn ppstrm => fn sigb => ppSigb ppstrm (sigb,d-1)),
		 sigbs);
	       end_block ppstrm)
	  | ppDec'(FsigDec fsigbs,d) =
	      (begin_block ppstrm CONSISTENT 0;
	       ppvlist ppstrm ("signature ","and ",
		(fn ppstrm => fn fsigb => ppFsigb ppstrm (fsigb,d-1)),
		 fsigbs);
	       end_block ppstrm)
	  | ppDec'(LocalDec(dec1,dec2),d) = ppsay "<LocalDec>"
	      (begin_block ppstrm CONSISTENT 0;
	       ppsay "local "; ppDec ppstrm (dec1,d-1);
	       add_break ppstrm (1,0);
	       ppsay " in "; ppExp'(dec2,d-1);
	       add_break ppstrm (1,0);
	       ppsay "end";
	       end_block ppstrm)
	  | ppDec'(SeqDec decs,d) =
	      ppvlist ppstrm ("","",(fn ppstrm => fn dec => ppDec'(dec,d-1)),decs)
	  | ppDec'(OpenDec paths,d) =
	      (ppsay "open ";
	       ppSequence ppstrm
	         {sep=(fn ppstrm => add_string ppstrm " "),
		  pr=ppSympath,
		  style=INCONSISTENT})
	  | ppDec'(OvldDec _,d) = ppsay "<OvldDec>"
	  | ppDec'(FixDec _,d) = ppsay "<FixDec>"
	  | ppDec'(ImportDec _,d) = ppsay "<ImportDec>"
	  | ppDec'(MarkDec _,d) = ppsay "<MarkDec>"
     in ppDec'
    end

(* VALUE BINDINGS *)
and ppVb ppstrm = 
    fn (_,0) => add_string ppstrm "<vb>"
     | (Vb{pat,exp},d) =>
	 (begin_block ppstrm CONSISTENT 0;
	  ppPat ppstrm (pat,d-1); add_string ppstrm " =";
	  add_break ppstrm (1,2); ppExp ppstrm (exp,d-1);
	  end_block ppstrm)
     | (MarkVb(vb,_),d) => ppVb ppstrm (vb,d)

(* RECURSIVE VALUE BINDINGS *)
and ppRvb ppstrm =
    fn (_,0) => add_string ppstrm "<vb>"
     | (Rvb{pat,exp,resultty},d) =>
	 (begin_block ppstrm CONSISTENT 0;
	  ppPat ppstrm (pat,d-1); add_string ppstrm " =";
	  add_break ppstrm (1,2); ppExp ppstrm (exp,d-1);
	  end_block ppstrm)
     | (MarkRvb(vb,_),d) => ppRvb ppstrm (vb,d)

(* TYPE BINDING *)
and ppTb ppstrm =
    fn (_,0) => add_string ppstrm "<tb>"
     | (Tb{tyc,def,tyvars},d) =>
	 (case tyvars
	    of nil => ()
	     | _ => (ppTuple ppstrm ppTyvar tyvars);
	  ppSym ppstrm tyc;
	  add_string ppstrm " ="; add_break ppstrm (1,0);
	  ppType ppstrm (ty,d-1))
     | (MarkTb(tb,_),d) => ppTb ppstrm (tb,d)


(* STRUCTURE BINDING *)
and ppStrb ppstrm =
    fn (Strb{name,def,constraint},d) =>
	 (ppSym ppstrm name; 
	  case constraint
	    of NONE => ()
	     | SOME sigexp =>
		(ppsay ": ";
		 ppSigexp ppstrm (sigexp,d-1));
	  add_string ppstrm " = ";
	  add_break ppstrm (1,2);
	  ppStrexp ppstrm (def,d-1))
     | (MarkStrb((strb,_),d)) => ppStrb ppstrm (strb,d)

(* FUNCTOR BINDING *)
and ppFctb ppstrm =
    let val ppsay = add_string ppstrm
	fun ppFctb'(_,0) = ppsay "<fctb>"
          | ppFctb'(Fctb{name,def=MarkFct(fctexp,_)},d) =
	      ppFctb'(Fctb{name=name,def=fctexp},d)
          | ppFctb'(Fctb{name,def=FctFct{params,body,constraint}},d) =
  	      (begin_block ppstrm CONSISTENT 2;
	       ppsay "functor "; ppSym ppstrm name;
	       begin_block ppstrm INCONSISTENT 2;
	       List.app (fn (pname,sign) => 
			  ppsay"(";
			  case pname of NONE => ppsay "_"
			     | SOME sym => ppSym ppstr sym;
			  ppsay ":"; ppSigexp ppstr (sign,d-1); ppsay")")
	         params;
	       end_block ppstrm;
	       case constraint
		 of SOME(fsig) => (ppsay ":"; ppFsigexp ppstrm (fsig,d-1))
		  | NONE => ()
	       ppsay " = "; add_newline ppstrm;
	       ppStrexp ppstrm (body,d-1);
	       end_block ppstrm)
          | ppFctb'(Fctb{name,def},d) =
	      (begin_block ppstrm CONSISTENT 2;
	       ppsay "functor "; ppSym ppstrm name;
	       ppsay " = ";
	       ppFctexp ppstrm (def,d-1);
	       end_block ppstrm)
          | ppFctb'(MarkFctb(fctb,_),d) = ppFctb'(fctb,d)
     in ppFctb'
    end

(* SIGNATURE BINDING *)
and ppSigb ppstrm =
    fn (_,0) => add_string ppstrm "<sigb>"
     | (Sigb{name,def},d) =>
	 (begin_block ppstrm CONSISTENT 0;
	  ppSym ppstrm name; add_string ppstrm " =";
	  add_break ppstrm (1,2); ppSigexp ppstrm (def,d-1);
	  end_block ppstrm)
     | (MarkSigb(sigb,_),d) => ppSigb ppstrm (sigb,d)

(* FUNSIG BINDING *)
and ppFsigb ppstrm =
    fn (_,0) => add_string ppstrm "<fsigb>"
     | (Fsigb{name,def},d) =>
	 (begin_block ppstrm CONSISTENT 0;
	  ppSym ppstrm name; add_string ppstrm " =";
	  add_break ppstrm (1,2); ppFsigexp ppstrm (def,d-1);
	  end_block ppstrm)
     | (MarkFsigb(fsigb,_),d) => ppFsigb ppstrm (fsigb,d)

(* TYPE VARIABLE *)
and ppTyvar ppstrm =
    fn (Tyv name,_) => ppSym ppstrm name
     | (MarkTyv(tyvar,_),d) => ppTyvar ppstrm tyvar

(* TYPES *)
and ppTy ppstrm =
    let val ppsay = add_string ppstrm
        fun ppTy'(VarTy tyvar,d) = ppTyvar ppstrm tyvar
          | ppTy'(ConTy(path,args),d) =			 
	     (begin_block ppstrm INCONSISTENT 2;
	      case args
		of nil => ()
		 | _ =>
		   ppClosedSequence ppstrm
		     {front=(fn ppstrm => add_string ppstrm "("),
		      sep=(fn ppstrm => add_string ppstrm ","),
		      back=(fn ppstrm => add_string ppstrm ")"),
		      pr=(fn ppstrm => fn ty => ppTy ppstrm (ty,d-1)),
		      style=INCONSISTENT}
		     args;
	      ppSymPath ppstrm path;
	      end_block ppstrm)
	 | ppTy'(RecordTy fields,d) =
	      ppClosedSequence ppstrm
		{front=(C add_string "{"),
		 sep=(fn ppstrm => (add_string ppstrm ",";
				    add_break ppstrm (0,0))),
		 back=(fn ppstrm => add_string ppstrm "}"),
		 pr=(fn ppstrm => fn (sym,ty) =>
		     (ppSym ppstrm sym; add_string ppstrm ":";
		      ppTy'(ty,d-1))),
		 style=INCONSISTENT}
		def
	 | ppTy'(TupleTy fields,d) =
	      ppClosedSequence ppstrm
		{front=(C add_string "("),
		 sep=(fn ppstrm => (add_string ppstrm ",";
				    add_break ppstrm (0,0))),
		 back=(C add_string ")"),
		 pr=(fn _ => fn ty => ppTy'(ty,d-1)),
		 style=INCONSISTENT}
		fields
	 | ppTy'(MarkTy(ty,_),d) = ppTy(ty,d)
     in ppTy'
    end
 
end (* structure Ast *)
