(* Copyright 1993 by AT&T Bell Laboratories *)
(* modules.sig *)

signature MODULES =
sig

(* position in files *)
type strpos

(* static environments *)
type env

type thinning
type fctWrapper
type fctThinning
type absWrapper

datatype origin
  = STAMP_ORIGIN of Stamps.stamp
  | ABS_ORIGIN of Types.absfbpos
  | NULL_ORIGIN

(* BINDINGS *)

(* The following datatypes are used to describe objects stored in static
   environments.
   - Name is the symbol used to access them but it is not necessarily the key
   stored in the environment.
   - Binding is the value itself.
   - Access describes the path to follow to get the value of the component in
   the dynamic representation of the object. In a structure it is the slot
   where the object is stored. In the global environment, it is a PATH with
   the global lvar to look for *)

datatype fixityVar
    = FIXvar of {name: Symbol.symbol, binding: Fixity.fixity}

datatype signatureVar
    = SIGvar of {name: Symbol.symbol, binding: Signature}

and funsigVar 
    = FSIGvar of {name: Symbol.symbol, binding: FctSignature}

and structureVar
    = STRvar of {name: Symbol.symbol,
		 access: Access.access,	   
		 binding: Structure}

and functorVar
    = FCTvar of {name: Symbol.symbol,
		 access: Access.access,
		 binding: Functor}

(* all those types are grouped by binding *)

and binding
    = VARbind of Variables.var
    | CONbind of Types.datacon
    | TYCbind of Types.tycon
    | SIGbind of signatureVar
    | STRbind of structureVar
    | FSIGbind of funsigVar
    | FCTbind of functorVar
    | FIXbind of fixityVar

(* SIGNATURES *)

and Signature
    = SIG of
        {symbols : Symbol.symbol list ref,
	  (* the ordered list of symbols defined (in sig only) *)
	 env : env ref, (* specification of objects;
			  Andrew says, "why the hell does this need
			  to be a ref?" *)
	 stamp : Stamps.stamp, (* for fast equality test *)
	 name : Symbol.symbol option, (* used for printing only *)
	 kind : sigkind ref}
    | EXTERN_SIG of ModuleId.modId
    (* the signature is the full actual structure *)
    | FULL_SIG 
    (* bogus signature *)
    | ERROR_SIG

(* a structure obtained by coercion by a signature is of type instance.
   signatures contain enough information to build the skeleton of the
   instance *)

and sigkind 
  (* This is a toplevel signature. All sharing constraints have been pushed
     to that level and are expressed by reference to the root of the signature.
     strcount, fctcount and typecount are the sizes of the arrays needed to
     build an instance of the signature. Slotcount gives the size of the
     dynamic record for structures of this signature. *)
  = TOP of {
	(* specifies the shape of the skeleton of an instance *)
	strcount : int,
	fctcount : int, 
	typecount : int,
	slotcount : int,

	abbreviations : {internal : int, external : Types.tycon} list,

	(* specifies the constraints of sharing *)
	sConstraints : {internal: SymPath.path list,
			external: Structure option} list,
	tConstraints : {internal: SymPath.path list,
			external: Types.tycon option} list}
  (* The signature is embedded in a top signature. The instance structure will
     share the instance arrays with its embedding structure *)
  | EMBEDDED 
  (* the kind field is irrelevant in a signature of an instance. It is true
     when the signature is obtained while building the structure. But even
     then it is important to maintain the invariant that signature contains
     enough information to rebuild the structure because it is used by
     instantiate. *)
  | IRRELEVANT 

(* FUNCTOR SIGNATURE: note the parallel between this and fct and sig *)
and FctSignature
    (* a complete specification of a functor signature *)
    = FSIG of
	{name : Symbol.symbol option, (* its name (toplevel object) *)
	 paramName : Symbol.symbol,   (* name of the parameter (printing) *)
	 argument : Signature,	      (* signature of the argument *)
	 body : Signature}	      (* signature of the body *)
    | EXTERN_FSIG of ModuleId.modId
    (* the signature is irrelevant: look at the actual functor to get it *)
    | FULL_FSIG
    (* the signature is bogus *)
    | ERROR_FSIG


(* STRUCTURES *)
and Structure
    (* a simple raw structure defined at toplevel *)
    = SIMPLE of
	{stamp: Stamps.stamp,	(* stamp *)
	 env: env,		(* environment defining the bindings *)
	 path: InvPath.path,	(* (inv)path to the structure *)
	 id: ModuleId.modId}
    (* result of coercing the origin structure by a signature. *)
    | INSTANCE of
	{sign : Signature,		  (* signature describing the instance *)
	 subStrs : Structure Array.array, (* structure array *)
	 subFcts : Functor Array.array,	  (* functor array *)
	 types : Types.tycon Array.array, (* types array *)
	 origin : origin,		  (* origin - stamp or absfbpos *)
	 path : InvPath.path}		  (* (inv)path to the structure *)
    (* the result res of the application of functor fct to arg *)
    | APPLY of
	{fct: Functor,
	 arg: Structure,
	 res: Structure,
	 path: InvPath.path} 
    (* representation of a binding coming from an opened structure *)
    | STR_OPEN of {pos : int list, spec : Signature, path : SymPath.path}
    (* representation of a formal binding in a signature *)
    | STR_FORMAL of {pos : strpos, spec : Signature}
    (* abstracted binding in a functor body *)
    | STR_ABSFB of Types.absfbpos
    (* a stub for an external structure after factoring *)
    | STR_EXTERN of ModuleId.modId
    (* an error has been found *)
    | ERROR_STR

(* FUNCTORS *)

(* Invariants maintained in functors:
   - The parent structure must be an instance or ERROR_STR (which means that
   the functor is a toplevel one).
   - The body describes an INSTANCE structure. It is a recepee to build it.
	- str describes the root of the result
	- sequences describe basic operation. Objects resulting from these
	  are identified by the position of the basic operation is the sequence
	- strseq describe how to build a one level structure from other 
	  structures types or functors obtained by
	    - looking up in the parameter ABSFB(PARAM path)
	    - looking up in the result of another basic operation
		ABSFB(SEQ number) or ABSFB(SEQ(number,path))
	    - applying a functor to a structure (structure of the APPLY form)
	    - taking the object as is if it is not in any of the previous
	      categories
   - The result structures verifies certain properties depending on how it
     has been obtained.
	- if the specification is just a variable (a component of the parameter
	  or of the outside world). Then the body is just that structure
	- if the specification is a complete definition of a structure, then
	  this structure is an INSTANCE whose substructure stored in 0 is
	  the argument. The argument is a pair made of the Parent structure
	  and of the actual parameter. It is an instance. The way it is done
	  now leads that the parent is in slot 0 and the parameter in slot 1
	- if it is an application of another functor to an argument, then the
	  argument verifies the constraints as defined previously but it is
	  the result of the second application that is given as body
*)

and Functor
    (* raw functor *)
    = FCT of
	{stamp: Stamps.stamp, 	   (* stamp for identification and age *)
	 parent: Structure,	   (* parent structure *)
	 paramName: Symbol.symbol, (* formal name of the parameter *)
	 lambdaty: LambdaType.lty, (* lambdatype info of the functor *)
	 argument: Signature,	   (* signature coercing the argument *)
	 body: {strseq: Structure list,  (* sequence of orders to follow *)
		fctseq: Functor list,    (* to build a new instance *)
		tyseq: Types.tycon list, (* of the body *)
		str: Structure}}         (* a list of actions *) 
    (* as STR_FORMAL *)
    | FCT_FORMAL of {pos: strpos, spec: FctSignature}
    (* as STR_OPEN *)
    | FCT_OPEN of
        {pos : int list,
	 spec : FctSignature,
	 path : SymPath.path}
    (* as INSTANCE a new parent may be given and used in FctSignature *)
    | FCT_INSTANCE of
	{fsig : FctSignature,
	 fct : Functor,
	 parent : Structure,
	 lambdaty : LambdaType.lty}
    (* as STR_ABSFB *)
    | FCT_ABSFB of Types.absfbpos
    | FCT_EXTERN of ModuleId.modId
    (* as ERROR_STR *)
    | ERROR_FCT


(* THINNING *)

datatype trans 
     (* old position, used for val, exn, or unthinned str *)
   = VALtrans of Access.access * Types.ty * Types.ty option
     (* old str position, substr thinning *)
   | THINtrans of Access.access * Access.lvar * trans list
     (* functor body thinning, same as previous for the fields 
        idea: THINtrans translates into fn str => {strthined}
              FCTtrans translates into 
		fn f => fn a => case (f a) of str => {strthined}
	where strthined is describe by the trans list
      *)
   | FCTtrans of Access.access * LambdaType.lty * fctThinning
     (* constructor as value component *)
   | CONtrans of Types.datacon * Types.ty option
   | STRtrans of Access.access * LambdaType.lty 
       (* a hack for representation analysis *)

end (* signature MODULES *)
