(* Copyright 1991 by AT&T Bell Laboratories *)
(* instantiate.sml *)

(* This function constructs a dummy structure which satisfies all sharing
 * constraints (explicit or induced) of a given signature.  The resulting
 * structure is used as the dummy parameter of a functor while elaborating
 * and abstracting the functor body.
 *
 * The process of constructing the structure is essentially a unification
 * problem.  The algorithm used here is based on the Linear Unification
 * algorithm first presented in [1] which was subsequently corrected
 * and cleaned up in [2].
 *
 * This code was originally designed by Damien Doligez, Georges Gonthier, 
 * and Dave MacQueen.  Greg Morrisett, Dave MacQueen, and David Tarditi
 * modified the code to work with the current implementation of the
 * SML/NJ module system.
 *
 * The basic algorithm makes 2 passes.  The first pass builds a DAG in
 * a quasi-top down fashion which corresponds to the minimal structure 
 * needed to match the signature.  The second pass takes the DAG and
 * constructs the actualy dummy structure in a bottom-up fashion.
 * Pass 1 has a fairly complicated control structure.  The major 
 * invariant is that no node in the graph is expanded unless all 
 * of its ancestors have been expanded.  This insures that all sharing
 * constraints (explicit or derived) have reached the node at the
 * time of its expansion.  The second major invariant is that no
 * node is finalized until all members in its equivalence class have
 * been found.
 *
 * [1] Paterson, M.S., and Wegman, M.N., "Linear Unification", 
 *     J. Comp. Sys. Sci. 16,2 (April 1978), pp. 158-167.
 *
 * [2] de Champeaux, D., "About the Paterson-Wegman Linear Unification
 *     Algorithm", J. of Comp. Sys. Sci. 32, 1986, pp. 79-88.
 *)

signature INSTANTIATE =
sig
  val instantiate : 
      (InvPath.path * Stamps.scope * ErrorMsg.complainer) (* context *)
      * Modules.Signature     (* signature to be instantiated *)
      -> Modules.Structure
  val instantiate_argument : 
      (InvPath.path * Stamps.scope * ErrorMsg.complainer) (* context *)
      * Symbol.symbol         (* parameter name *)
      * Modules.Structure     (* parent structure *)
      * Modules.Signature     (* argument signature - to be instantiated *)
      -> Modules.Structure
  val debugging : bool ref
end

structure Instantiate : INSTANTIATE =
struct

open Symbol Modules Types ModuleUtil ErrorMsg TypesUtil
     Control Access Stamps PrintUtil Extern

structure SP = SymPath
structure IP = InvPath

val say = Control.Print.say

val debugging = ref false

fun debugmsg (msg: string) =
    if !debugging then (say msg; say "\n") else ()

(* debugging code *)
fun wrap fname f arg =
    if !debugging then
      let val _ = say (">> "^fname^"\n")
	  val result = f arg
       in say ("<< "^fname^"\n");
	  result
      end
    else f arg

val error_found = ref false

(* type of context triple passed among main functions *)
type context = IP.path * Stamps.scope * ErrorMsg.complainer

(* This datatype represents the continually changing DAG that is being 
 * constructed by instantiate.  We start off with just an Initial node.  
 * It is expanded into a Partial node whose children (subs and typs) are 
 * initialized to Initial nodes.  When all of the members of the nodes
 * equivalence class have been found, and converted to Partial nodes, 
 * the node is converted to either Final_top or Final_embed depending 
 * on its signature kind.  Finally, we recurse on the children of the
 * node.  
 *
 * Invariants:
 *
 *    The parent node is in a singleton equivalence class.
 *
 *    All nodes that are about to be explored are either Initial or Partial.
 *    (Exploring a Final node implies circularity.)
 *
 *    If a Final node's expanded field is true, then all of its children
 *    are Final with expanded field true.
 *
 *    The node (subs sub i) of a structure corresponds to the sub-structure
 *    (actual_subStrs sub i) to be built.  Similarly for (typs sub i) and 
 *    (actual_types sub i)
 *)
datatype instance
  = Final_top of
     {sign : Signature,
      actual_subStrs : Structure Array.array,
      actual_types : Types.tycon Array.array,
      origin_info : origin option ref,
      subs : instance Array.array,
      typs : type_ins Array.array,
      final_struct: Structure option ref,
      expanded : bool ref}
  | Final_embed of
     {sign : Signature,
      path : IP.path,
      origin_info : origin option ref,
      subs : instance Array.array,
      typs : type_ins Array.array,
      expanded : bool ref}
  | Partial of
     {signat : Signature,
      path : IP.path,
      subs : instance Array.array,
      typs : type_ins Array.array,
      depth : int,
      final_rep : instance option ref}
  | Initial of
     {signat : Signature,
      path : IP.path,
      parent_typs : type_ins Array.array,
      inherited : constraint list ref}
  | NULLinstance
  | ERRORinstance

(* These are the similar nodes for types. *)
and type_ins
  = tFinal of tycon
  | tPartial of
     {tycon : tycon,
      path : IP.path}
  | tInitial of
     {tycon : tycon,
      path : IP.path,
      inherited : constraint list ref}
  | tAbbrev of  (* predistributed type definition *)
     {def: tycon,
      subs : instance Array.array,
      typs : type_ins Array.array}
  | tNULLinstance
  | tERRORinstance

(* Some of the nodes of the structure are not actually constructed
 * since they were previously defined structures or types.  These
 * are labelled with EXTstr and EXTtyc respectively.  The pointers
 * to the nodes of the DAG are destructively updated, 
 * so they are represented using "slots" of an array.
 *)
and instrep
  = EXTstr of Structure
  | EXTtyc of tycon
  | SLOTstr of instance slot
  | SLOTtyc of type_ins slot
  | ABVtyc of {def: tycon,              (* type abbreviation - Cregut *)
	       subs : instance Array.array,
	       typs : type_ins Array.array}

(* A constraint is essentially a directed arc indicating that two
 * nodes are to be identified.  The constraint is always interpreted
 * relative to an instance node.  The my_path field is a symbolic
 * path (in regular order) indicating which subcomponent of the instance
 * is participating
 * in the sharing.  The other component is accessed by first finding
 * the instance node in the its_ancestor slot, and then following
 * the symbolic path its_path to the node.  By going through the
 * ancestor, we are able to insure that the ancestor is explored
 * before the actual component is.
 *)
withtype constraint =
    {my_path : SP.path,  (* regular symbolic path *)
     its_ancestor : instrep,
     its_path : SP.path} (* regular symbolic path *)

and 'a slot = {base : 'a Array.array, offset : int}

val emptyp : SP.path = SP.empty

fun get {base, offset} = Array.sub(base, offset)
fun set ({base, offset}, inst) = Array.update (base, offset, inst)
fun push(r,x) = (r := x::(!r))

(*********************)
(* Utility Functions *)
(*********************)	    
(* Retrieves all [formal] substructure components from a signature *)
fun getSubSigs (sign: Modules.Signature) : 
    (Symbol.symbol * Modules.strpos * Modules.Signature) list =
    case sign 
      of SIG {symbols,env,...} =>
	  let fun check_binding (s, prevSubSigs) =
	          case (nameSpace s)
		    of STRspace => 
			((case Env.look (!env,s)
			    of STRbind(STRvar{binding=STR_FORMAL{pos,spec,...},
					      ...}) =>
				(s,pos,spec) :: prevSubSigs
			     | _ => impossible "Modules.instantiate:getSubSigs 1")
                         handle Env.Unbound => impossible "Modules.getSubSigs 2")
		     | _ => prevSubSigs
	   in foldr check_binding [] (!symbols)
	  end
       | _ => []

(* Retrieves all [formal] type components from a signature *)
fun getTypes (sign: Modules.Signature) : 
    (Symbol.symbol * Modules.strpos * Types.tycon) list =
    case sign
      of SIG {symbols,env,...} =>
	  let fun check_binding (s, prevTypes) =
	          case (nameSpace s)
		    of TYCspace =>
			((case Env.look(!env,s)
			    of TYCbind(tyc as (FORMtyc{pos,spec,...})) =>
				(s,pos,tyc) :: prevTypes
			     | _ => impossible 
				   "Modules.instantiate:getTypes 2")
			 handle Env.Unbound => 
			   impossible "Modules.instantiate:getTypes 2")
		     | _ => prevTypes
	   in foldr check_binding [] (!symbols)
	  end
       | _ => []

 
fun pathName (path: IP.path) = 
    SP.makestring(ConvertPaths.invertIPath path)

fun strName (SIMPLE {path,...}) = pathName path
  | strName (INSTANCE {path,...}) = pathName path
  | strName _ = "?"


(* gets all the substructures of a structure *)
fun getSubStrs str =
    let val r = ref nil
     in case str
	  of (SIMPLE{env,...}) =>
	      Env.app 
		(fn (sym,STRbind(STRvar{binding=str,...})) => r := (sym,str) :: !r
		  | _ => ())
		env
	   | (str as INSTANCE{sign=SIG{env,...},subStrs,...}) =>
	      Env.app
		(fn (sym,STRbind(STRvar{binding=STR_FORMAL{pos,...},...})) =>
		       r := (sym,Array.sub(subStrs,pos)) :: !r
		  | (sym,STRbind(STRvar{binding=STR_OPEN{pos,...},...})) =>
		       r := (sym,transPosStr str pos ) :: !r
		  | _ => ())
		(!env)
	   | (INSTANCE{sign=ERROR_SIG,...}) => ()
	   | APPLY{res,...} => r:=getSubStrs res
	   | ERROR_STR => ()
	   | _ => impossible "getSubStrs";
        !r
    end

(* Creates instrep nodes for all of the substructures of a
 * presumably pre-defined (i.e. external) structure.
 *)
fun getStrSlots (str : Structure) : (symbol * instrep) list =
    let fun f (sym, str) = (sym, EXTstr ((* getOrigin *) str))
     in map f (getSubStrs str)
    end

fun getSubTyps str =
    let val r = ref nil
     in case str
	  of (SIMPLE{env,...}) =>
	      Env.app 
		(fn (sym,TYCbind tyc) => r := (sym,tyc) :: !r
		  | _ => ())
		env
	   | (str as INSTANCE{sign=SIG{env,...},types,...}) =>
	      Env.app
		(fn (sym,TYCbind (OPENFORMtyc{pos=(path,pos),...})) => 
		       r := (sym,transPosTycon str (path @ [pos])) :: !r
		   | (sym,TYCbind (FORMtyc{pos,...})) => 
		       r := (sym,Array.sub(types,pos)) :: !r
		   | _ => ()) 
		(!env)
	   | (INSTANCE{sign=ERROR_SIG,...}) => (error_found := true; ())
	   | APPLY{res,...} => r:=getSubTyps res
	   | ERROR_STR => (error_found := true; ())
	   | _ => impossible "Instantiate:getSubTyps";
        !r
    end
		   
        
fun getStrTSlots (str : Structure) : (symbol * instrep) list =
    let fun f (sym, tyc) = (sym, EXTtyc tyc)
     in map f (getSubTyps str)
    end

(* Finds all sub-signatures in a signature, and updates the corresponding
 * slots in the subs array so that they are Initial instances.
 *)
fun getSigSlots (sign, path: IP.path, subs, typs) =
    let fun newInstance (signat, path) =
            Initial {signat=signat, path=path, parent_typs=typs,
		     inherited=ref []}
        fun f (name, pos, signat) =
            let val sl = {base=subs, offset=pos}
             in set (sl, newInstance (signat, IP.extend(path,name)));
                (name, SLOTstr sl)
            end
     in map f (getSubSigs sign)
    end

(* Finds all types in a signature, and updates the corresponding slots
 * in the typs array so that they are tInitial instances.
 *   Checks to see if a slot has already been updated by distribution of
 * an abbreviation constraint in a parent signature.
 *)
fun getSigTSlots (sign, path: IP.path, typs) =
    let fun f (name, pos, tycon) =
            let val sl = {base=typs, offset=pos}
             in case get sl
		  of tNULLinstance =>
		      (debugmsg "<tNULLinstance converted to tInitial>";
		       set(sl, tInitial{tycon=tycon,
					path=IP.extend(path,name),
					inherited=ref []}))
		   | tAbbrev{def,subs,typs} =>
		      (debugmsg "<tAbbrev converted to tInitial>";
		       set(sl, tInitial{tycon=tycon,
					path=IP.extend(path,name),
					inherited=ref[{my_path=emptyp,
						       its_path=emptyp,
						       its_ancestor=
						         ABVtyc{def=def,subs=subs,
								typs=typs}}]}))
		   | _ => (); (* impossible "getSigTSlots"; *)
		      (* it's already been initialized because duplication
		         in the symbols list *)
                (name, SLOTtyc sl)
            end
     in map f (getTypes sign)
    end

(* debugging wrappers
val getSubSigs = wrap "getSubSigs" getSubSigs
val getTypes = wrap "getTypes" getTypes
val getSubStrs = wrap "getSubStrs" getSubStrs
val getStrSlots = wrap "getStrSlots" getStrSlots
val getSubTyps = wrap "getSubTyps" getSubTyps
val getStrTSlots = wrap "getStrTSlots" getStrTSlots
val getSigSlots = wrap "getSigSlots" getSigSlots
val getSigTSlots = wrap "getSigTSlots" getSigTSlots
*)

(***********************************************************************
 * This function merges a node into a class.  It does so by looking    *
 * up the elements the node requires in the "union" environment for    *
 * the class.  If a binding does not exist, then one is added to the   *
 * union.  If a binding does exist to some other node for the symbol,  *
 * then constraints are added to both nodes indicating that they are   *
 * equivalent.                                                         *
 ***********************************************************************)
fun merge (union, defined, err, path: IP.path) (sym, rep) =
    (case (Env.look (!union,sym), rep)
       of (extrep as EXTstr _, SLOTstr sl) =>
           (case get sl
	      of Initial {inherited, ...}
		   => (push(inherited,
			    {my_path=emptyp, its_ancestor=extrep, its_path=emptyp});
		       union := Env.bind (sym,rep,!union))
	       | ERRORinstance => ()
	       | _ => impossible "Instantiate:merge.1")
	| (SLOTstr sl, extrep as EXTstr _) =>
	   (case get sl
	      of Initial {inherited, ...}
		   => push(inherited,
			   {my_path=emptyp, its_ancestor=extrep, its_path=emptyp})
	       | ERRORinstance => (error_found := true)
	       | _ => impossible "Instantiate:merge.2")
	| (SLOTstr sl1, SLOTstr sl2) =>
	   (case (get sl1, get sl2)
	      of (Initial {inherited=inherited1, ...},
		  Initial {inherited=inherited2, ...})
		   => (push(inherited1,
			    {my_path=emptyp, its_ancestor=rep, its_path=emptyp});
		       push(inherited2,
			    {my_path=emptyp, its_ancestor=SLOTstr sl1, its_path=emptyp}))
	       | (ERRORinstance,_) => ()
	       | (_,ERRORinstance) => ()
	       | _ => impossible "Instantiate:merge.3")
	| (extrep as EXTtyc _, SLOTtyc sl) =>
	   (case get sl
	      of tInitial {inherited, ...}
		   => (push(inherited,
			    {my_path=emptyp, its_ancestor=extrep, its_path=emptyp});
		       union := Env.bind (sym,rep,!union))
	       | tERRORinstance => ()
	       | _ => impossible "Instantiate:merge.4")
	| (SLOTtyc sl, extrep as EXTtyc _) =>
	   (case get sl
	      of tInitial {inherited, ...} =>
		  push(inherited,{my_path=emptyp, its_ancestor=extrep, its_path=emptyp})
	       | tERRORinstance => ()
	       | _ => impossible "Instantiate:merge.5")
	| (SLOTtyc sl1, SLOTtyc sl2) =>
	   (case (get sl1, get sl2)
	      of (tInitial {inherited=inherited1, ...},
		  tInitial {inherited=inherited2, ...}) =>
		   (push(inherited1,{my_path=emptyp, its_ancestor=rep, its_path=emptyp});
		    push(inherited2,
			 {my_path=emptyp, its_ancestor=SLOTtyc sl1, its_path=emptyp}))
	       | (tERRORinstance,_) => ()
	       | (_,tERRORinstance) => ()
	       | _ => impossible "Instantiate:merge.6")
	| _ => impossible "Instantiate:merge.7")
    handle Env.Unbound =>
	   case defined
	     of SOME str =>
		 err WARN 
		     ("This signature cannot be matched : sharing "
			      ^(pathName path)^" = "^(strName str)
			      ^" : "^(strName str)^" has no "
			      ^(Symbol.name sym)^" component.")
		     nullErrorBody
	      | NONE => union := Env.bind (sym,rep,!union)

(* debugging wrappers
val merge = wrap "merge" merge
*)
(*************************************************************************
 * This function just checks to make sure that each of the bindings      *
 * that a class requires actually exits in the origin of some [external] *
 * structure.                                                            *
 *************************************************************************)
fun check_def (union,str,err,path) =
    let val env = case ((* getOrigin *) str) 
		    of (SIMPLE{env,...}) => env
		     | (INSTANCE{sign=SIG{env,...},...}) => !env
		     | ERROR_STR => (error_found := true; Env.empty)
		     | _ => impossible "instantiate:check_def"
	val path' = getStrPath str
	fun check (sym, binding) =
	    (if !debugging then
		(say "check: "; PrintUtil.printSym sym; say "\n")
	     else ();
	     Env.look (env,sym); ())
	    handle Env.Unbound =>
		err WARN 
		  ("This signature cannot be matched : sharing "
		   ^(pathName path')^" = "^(pathName path)^" : "
		   ^(pathName path')^" is also sharing with a structure with a "
		   ^(Symbol.name sym)^" component, and "^(pathName path)
		  ^" has no "^(Symbol.name sym)^" component.")
		  nullErrorBody
     in if !debugging then say "check_def: about to Env.app\n" else ();
	Env.app check (!union)
    end

(* debugging wrappers
val check_def = wrap "check_def" check_def
*)
(**************************************************************************
 * This function distributes the structure sharing constraints that a     *
 * signature has to the children of a corresponding node.  Note that this *
 * only deals with the explicit constraints.  Implied and inherited       *
 * constraints are propogated by merge and the constrain functions of     *
 * explore_class and explore_tclass.                                      *
 **************************************************************************)
fun distributeS 
      (sign as SIG {kind=ref(TOP{sConstraints,...}),...}, subs, err) =
    let exception DistributeS
	fun f (SP.SPATH(sym::path)) =
	      let val pos = (getSigPos sign sym)
	       in case Array.sub(subs, pos)
		    of Initial {inherited, ...} =>
			(SP.SPATH path, inherited, SLOTstr {base=subs, offset=pos})
		     | ERRORinstance => raise DistributeS
		     | _ => impossible "distributeS:f DD134"
	      end
          | f (SP.SPATH []) = impossible "distributeS:f DD148"
	fun dist {internal=(p::rest),external} = 
	      let val (p1, h1, ir1) = f p
		  fun g (p2, h2, ir2) =
		      (push(h1,{my_path=p1, its_path=p2, its_ancestor=ir2});
		       push(h2,{my_path=p2, its_path=p1, its_ancestor=ir1}))
	       in app (fn p' => g (f p')) rest;
		  case external
		    of NONE => ()
		     | SOME str =>
			 push(h1,
			      {my_path=p1,
			       its_ancestor=EXTstr ((* getOrigin *) str),
			       its_path=emptyp})
	      end
	  | dist {internal=[],...} = ()
     in (app dist sConstraints) handle DistributeS => ()
    end
  (* don't have any sharing constraints to *)
  (* distribute if we don't have a TOP sig *)
  | distributeS _ = () 

(***************************************************************************
 * This function distributes the type sharing constraints that a signature *
 * has to the children of the corresponding node.                          *
 * It also distributes the abbreviation definitions considered here as a   *
 * special kind of constraints						   *
 ***************************************************************************)
fun distributeT 
      (sign as SIG {kind=ref(TOP{tConstraints, abbreviations,...}),...},
       subs,typs,err) =
    let exception DistributeT
	fun f (SP.SPATH[sym]) =
              let val pos = getSigTPos sign sym
               in case Array.sub(typs, pos)
		    of tInitial {inherited, ...} =>
                        (SP.SPATH[], inherited, SLOTtyc {base=typs, offset=pos})
		     | tERRORinstance => raise DistributeT
		     | _ => impossible "distributeT:f DD144"
              end
          | f (SP.SPATH(sym::path)) =
              let val pos = (getSigPos sign sym)
               in case Array.sub(subs, pos)
		    of Initial {inherited, ...} =>
                        (SP.SPATH path, inherited, SLOTstr {base=subs, offset=pos})
		     | _ => impossible "distributeT:f DD146"
	      end
          | f _ = impossible "distributeT:f DD147"
	fun dist {internal=p::rest,external} = 
	    let val (p1,h1,ir1) = f p
		fun g (p2, h2, ir2) =
		    (push(h1,{my_path=p1, its_path=p2, its_ancestor=ir2});
		     push(h2,{my_path=p2, its_path=p1, its_ancestor=ir1}))
	     in app (fn p' => g (f p')) rest;
		case external
		  of NONE => ()
		   | SOME tyc =>
		       push(h1,{my_path=p1,its_ancestor=EXTtyc tyc,its_path=emptyp})
	    end
	  | dist {internal=[],...} = ()
	fun dist_abv {internal=pos, external=def} =
            case Array.sub(typs, pos)
              of tInitial {inherited, ...} =>
                   push(inherited, {my_path=emptyp,its_path=emptyp,
				    its_ancestor=ABVtyc{def=def,subs=subs,
							typs=typs}})
               | tNULLinstance =>
		  (* distributing into an embedded substructure signature
		     that hasn't been initialized by getSigTSlots *)
		  Array.update(typs,pos,tAbbrev{def=def,subs=subs,typs=typs})
	       | tERRORinstance => ()
               | tFinal _ => impossible "distributeT: dist_abv - tFinal"
               | tPartial _ => impossible "distributeT: dist_abv - tPartial"

     in (app dist tConstraints) handle DistributeT => ();
	 app dist_abv abbreviations
    end
  (* don't have any sharing constraints to *)
  (* distribute if we don't have a TOP sig *)
  | distributeT _ = () 

(* debugging wrappers
val distributeS = wrap "distributeS" distributeS
val distributeT = wrap "distributeT" distributeT
*)

exception ExploreInst of IP.path

(***************************************************************************
 * This is the main function of the algorithm.  Given an Initial node, 
 * it creates subs and typs arrays and converts the node to a Partial
 * node.  This step is skipped if the node has been explored previously
 * and is already Partial.  Then, we find all of the child nodes of
 * the node, create Initial instances of them, apply merge to them
 * resulting in a new union_components enviornment.  We then distribute
 * all of the sharing constraints to the new children and apply
 * the constrain function to our list of inherited constraints.  Constrain
 * in turn uses the constraints to track down other nodes that should
 * be placed in the same equivalence class.  If the ancestors of such
 * a node have not been explored yet, then they are explored first.
 * Once constrain is complete, class holds a list of instance nodes
 * (all Partial) that are equivalent and union_components holds an 
 * enviornment that maps the "union" of all components in the class
 * to instreps.  A class_origin is [lazily] built.  (The origin will
 * be available iff there is an external sharing constraint.)  Finally,
 * we apply the function finalize to each member of the class which
 * converts each of the Partial nodes to either Final_top or Final_embed
 * nodes.  
 *
 * This has been slightly modified so that if two slots in the class
 * have nodes that share the same signature, then the slots are made
 * to point to only one of the nodes.  Of course, the sharing constraints
 * for both must be propogated to the descendants.  
 ***************************************************************************)
fun explore_class (first_slot : instance slot, 
		   class_depth: int, 
		   err: severity -> string -> (PrettyPrint.ppstream -> unit)
		        -> unit)
                  : unit =
    let val union_components = ref (Env.empty : instrep Env.env)
        val class = ref ([] : instance slot list)
        val class_def = ref (NONE : Structure option)
	exception Error_Sig
	(* Convert the node from Initial to Partial.  Merge its components
	 * into the union.  Push down any sharing constraints it has in 
         * its signature.  Then apply constrain to each of the inherited
         * constraints.
         *)
        fun explore_inst (sl: instance slot) : unit =
            (case get sl
	       of ERRORinstance => ()
		| (Partial {depth, path, ...}) =>
		    if (depth = class_depth) then ()
		    else raise ExploreInst path
		| (Initial {signat, path, parent_typs, inherited}) =>
		    let fun find_same_sig [] = false
			  | find_same_sig (sl'::rest) =
			     (case (get sl')
				of (p as (Partial{signat=signat',
						  subs,typs,
						  path=path',...})) =>
				    (if eqSignature(signat,signat') then
					 (set (sl,p);
					  push(class,sl);
					  app (constrain (signat, subs,
							  typs, path))
					    (!inherited);
					  true)
				     else find_same_sig rest)
				 | ERRORinstance => false
				 | _ => impossible 
				          "Instantiate:find_same_sig")
		      in if not(find_same_sig(!class)) then 
			   let val subs = 
				 case signat
				   of SIG {kind=ref(TOP {strcount, ...}), ...} =>
				       Array.array (strcount, NULLinstance)
				    | SIG {kind=ref IRRELEVANT, ...} =>
				       impossible "can't instantiate IRRELEVANT"
				    | SIG {kind=ref EMBEDDED, ...} => #base sl
				    | ERROR_SIG => raise Error_Sig
				    | FULL_SIG =>  raise Error_Sig
			       val typs = 
			         case signat
				   of SIG {kind=ref(TOP {typecount,...}),...} =>
				       Array.array (typecount, tNULLinstance)
				    | SIG {kind=ref EMBEDDED, ...} => 
				       parent_typs
				    | _ => impossible
				      "Instantiate::explore_class.2"
			    in set (sl, Partial{signat=signat,
						path=path,subs=subs,
						typs=typs,
						final_rep = ref NONE,
						depth=class_depth});
			       push(class,sl);
			       app (merge (union_components, !class_def, 
					   err, path))
			           (getSigSlots (signat, path, subs, typs));
			       app (merge (union_components, !class_def, 
					   err, path))
			           (getSigTSlots (signat, path, typs));
			       distributeS (signat, subs, err);
			       distributeT (signat, subs, typs, err);
			       app (constrain (signat, subs, typs, path)) 
			           (!inherited)
			   end
			   handle Error_Sig => (error_found := true;
						set (sl,ERRORinstance))
			 else () 
		     end
		| _ => if (!error_found) then (set (sl,ERRORinstance))
		       else (impossible "Instantiate:explore_class.3"))

	(* Class shares with some external structure *)

        and constrain (signat, subs, _, path)
	              {my_path=SP.SPATH[], its_ancestor=EXTstr str,...} =
	    (case !class_def
	       of SOME str' =>
		   (debugmsg "<checking eqOrigin>";
		    if eqOrigin(str',str) then ()
		    else err COMPLAIN
			     ("Inconsistent defining constraints : "
			      ^(strName str')^" = "^(strName str))
			     nullErrorBody)
		| NONE =>
		   (class_def := SOME str;
		    app (merge (union_components, NONE, err, IP.IPATH[]))
		        (getStrSlots str);
		    app (merge (union_components, NONE, err, IP.IPATH[]))
			(getStrTSlots str);
		    check_def (union_components, str, err, path)))

	  (* Class shares with the structure in slot -- explore it *)
          | constrain (signat, subs, _, path)
	      {my_path=SP.SPATH[],its_ancestor=SLOTstr slot,its_path=SP.SPATH []} =
	    (debugmsg "<calling explore_inst to add member to equiv class>";
	     explore_inst slot handle (ExploreInst path') =>
		(err COMPLAIN
		  "Sharing constraint is also sharing with a substructure"
		  nullErrorBody;
	         set (slot, ERRORinstance)))

	  (* Class shares with another structure.  Make sure its ancestor
	   * has been explored.  Then push the constraint down a level.
	   *)
          | constrain (signat, subs', typs, path')
	              {my_path=SP.SPATH[],its_ancestor=SLOTstr slot,
		       its_path=SP.SPATH(sym::rest)} =
	    (case (get slot)
	       of Initial _ => 
	           (debugmsg "<Having to call explore class on an ancestor \
		             \of a node I'm equivalent to.>";
		    explore_class (slot, (class_depth+1), err)
	              handle (ExploreInst _) => 
			  impossible "Instantiate:explore_class.4")
		| ERRORinstance => ()
		| _ => ();
	     debugmsg "<finished exploring his ancestor>";
	     case (get slot)
	       of Final_top {sign, subs, ...} =>
		   (debugmsg "<calling constrain recursively>";
		    constrain (signat, subs', typs, path')
		      {my_path=SP.SPATH[], its_path=SP.SPATH rest,
		       its_ancestor=SLOTstr {base=subs,
					     offset=getSigPos sign sym}})
		| Final_embed {sign, subs, ...} =>
		   (debugmsg "<found Final_embed>";
		    constrain (signat, subs', typs, path')
		      {my_path=SP.SPATH[], its_path=SP.SPATH rest,
		       its_ancestor=SLOTstr {base=subs,
					     offset=getSigPos sign sym}})
	        | Partial _ =>
		   (err COMPLAIN
		      "Sharing constraint is also sharing with a substructure"
		      nullErrorBody;
		    set (slot,ERRORinstance))
	        | ERRORinstance => ()
	        | _ => impossible "Instantiate:explore_class.5")

	  (* One of the nodes children shares with someone.  Push the
	   * constraint down to the child now that we are explored.
	   *)
	  | constrain (signat, subs, typs, _) 
	              {my_path=SP.SPATH(sym::rest), its_ancestor, its_path} =
	    (case getSigPosGen signat sym
	       of TYCbind (FORMtyc {pos, ...}) =>
		   (case Array.sub(typs, pos)
		      of tInitial {inherited, ...} =>
			  push(inherited,{my_path=SP.SPATH[], 
					  its_ancestor=its_ancestor, 
					  its_path=its_path})
		       | _ => impossible "Instantiate:explore_class.6")
		| STRbind (STRvar {binding=STR_FORMAL {pos, ...}, ...}) =>
		   (case Array.sub(subs, pos)
		      of Initial {inherited, ...} =>
			  push(inherited,{my_path=SP.SPATH rest, 
					  its_ancestor=its_ancestor, 
					  its_path=its_path})
		       | _ => impossible "Instantiate:explore_class.7")
		| _ => impossible "Instantiate:explore_class.8")
	  | constrain _ _ = impossible "Instantiate:explore_class.9"


	(* Should find everyone in the equiv. class and convert them to 
	 * Partial nodes.  
	 *)
        val _ = explore_inst first_slot;

	val class_origin = 
	      ref (case !class_def
		     of SOME str => SOME(STAMP_ORIGIN(getStrStamp str))
		      | NONE => NONE)

	(* Converts all of the nodes in the class (which should be Partial)
	 * to Final nodes.  Note that nodes which share the same signature
	 * should share the same Final nodes.  So, they are memoized using
	 * the final_rep field of the Partial node.
	 *)
	fun finalize sl =
	    (case (get sl) 
	       of ERRORinstance => ()
	        | Partial {signat, path, subs, typs, final_rep, ...} =>
		   (case (!final_rep)
		      of SOME f => (set (sl, f))
		       | NONE =>
			  case signat
			    of SIG {env,
				    kind=ref(TOP{strcount,typecount,...}),...} =>
				let val strArray = Array.array (strcount, ERROR_STR)
				    val tycArray = Array.array (typecount, ERRORtyc)
				    val f = 
					Final_top {sign = signat,
						   origin_info=class_origin,
						   actual_subStrs = strArray,
						   actual_types = tycArray,
						   subs = subs,
						   typs = typs,
						   final_struct = ref NONE,
						   expanded = ref false}
				 in final_rep := SOME f;
				    set (sl, f)
				end
			     | SIG {env, kind=ref EMBEDDED, ...} =>
				(set (sl, 
				      Final_embed {sign=signat, path=path,
						   origin_info = class_origin,
						   subs=subs, 
						   typs=typs,
						   expanded=ref false}))
			     | _ => impossible "Instantiate:explore_class.12")
		| _ => impossible "Instantiate:explore_class.11")

     in app finalize (!class)
    end (* explore_class *)

(* debugging wrappers
val explore_class = wrap "explore_class" explore_class
*)

(*************************************************************************
 * This function deals with exploration of type nodes in the instance
 * graph and is similar to the explore_class function above.  It is
 * a bit simpler since it doesn't have to worry about "children" of
 * type nodes.  However, we must check that the arities of equivalenced
 * types are the same.  Also, if they have constructors, we must check
 * to see that they have the same constructor names.  We don't know how
 * to check that the types of the constructors are satisfiable -- this
 * involves a limited form of second-order unification.
 *************************************************************************)
fun explore_tclass (first_slot, make_stamp, err, argOption) =
    let val class = ref ([] : type_ins slot list)
        val class_def = ref (NONE : tycon option)
	fun substParam (rpath: IP.path, argOption: Symbol.symbol option)
	      : IP.path =
	    case argOption
	      of NONE => rpath
	       | SOME s =>
	         (case rpath
		    of IP.IPATH nil => impossible "substParam"
		     | IP.IPATH p =>
		         let val (first::rest) = rev p
			  in if Symbol.eq(first, parameterId)
			     then IP.IPATH(rev(s::rest))
			     else rpath
			 end)
        fun explore_inst sl =
            case get sl
	      of tPartial _ => ()
	       | tInitial {tycon, path, inherited} =>
                    (debugmsg "<setting tInitial to tPartial>";
		     set (sl, tPartial {tycon=tycon, path=path});
                     push(class,sl);
                     app constrain (!inherited))
	       | tERRORinstance => ()
	       | _ => impossible "Instantiate:explore_tclass.1"

	and expand_abbrev {def=DEFtyc{path,strict,tyfun=TYFUN{body,arity}},
			   subs,typs} =
	    let (* extracted and adapted from ModuleUtil *)
		fun transType ty =
		    case ty
		      of VARty _ => ty
		       | IBOUND _ => ty
		       | CONty (tc, tl) =>
			   mkCONty (transTycon tc, map transType tl)
		       | POLYty {sign, tyfun=TYFUN {arity, body}, abs} =>
			   POLYty{sign=sign,
				  tyfun=TYFUN{arity=arity,body=transType body},
				  abs=abs}
		       | UNDEFty => ty
		       | WILDCARDty => ty
		and transTycon (RELtyc{pos,...}) = 
		      access_type{path=pos,subs=subs,typs=typs}
		  | transTycon tyc = tyc
	     in debugmsg "<expanding an abbreviation>";
		DEFtyc{path=path,strict=strict,
		       tyfun=TYFUN{body=transType body,arity=arity}}
	    end
	  | expand_abbrev {def,...} = 
	      impossible "expand_abbrev : wrong kind of def"
	(* forces the definition of a class : used to retrieve the
	 * definition of an abbreviation. *)

	and access_type {path = ([],i), subs,typs} =
	     (case Array.sub(typs, i)
		of tInitial _  =>
		     explore_tclass ({base =typs,offset=i},make_stamp,err,
				     argOption)
		 | _ => ();
	      case Array.sub(typs, i)
		of tFinal tycon => tycon
		 | tPartial _ => 
		     (err COMPLAIN 
			"There is a circularity in type abbreviation."
			nullErrorBody;
		      ERRORtyc)
		 | _ => impossible "access_type.1")
	  | access_type {path = (j::p,i), subs,typs} =
	     (case Array.sub(subs, j)
		of Initial _ =>
		    (explore_class ({base=subs,offset=i}, 0, err)
		     handle ExploreInst _ => impossible "access_type.2")
			  | _ => ();
	      case Array.sub(subs, j)
		of Final_top {sign, typs,subs, ...} =>
		      access_type{subs=subs,typs=typs,path=(p,i)}
		 | Final_embed {sign, typs,subs, ...} =>
		      access_type{subs=subs,typs=typs,path=(p,i)}
		 | ERRORinstance => ERRORtyc
		 | _ => impossible "access_type.3")

        and constrain {my_path=SP.SPATH[], its_ancestor=EXTtyc tyc, ...} =
             (case !class_def
		of SOME tyc' =>
                    if equalTycon (tyc',tyc)
                    then ()
                    else let val s = concat
			           ["Inconsistent defining constraints : type ",
				    (Symbol.name (tycName tyc')), " = ",
				    (Symbol.name (tycName tyc))]
			  in err COMPLAIN s nullErrorBody
			 end
		 | NONE => class_def := SOME tyc)
          | constrain {my_path=SP.SPATH[], its_ancestor=SLOTtyc slot,
		       its_path=SP.SPATH[]} =
             explore_inst slot
	  | constrain {my_path=SP.SPATH[], its_ancestor=ABVtyc abbrev, ...} =
	     constrain {my_path=SP.SPATH[],its_path=SP.SPATH[],
			its_ancestor=EXTtyc(expand_abbrev abbrev)}
          | constrain {my_path=SP.SPATH[],its_ancestor=SLOTstr slot,
		       its_path=SP.SPATH(sym::rest)} =
             (case get slot
		of Initial _ =>
		    (explore_class (slot, 0, err)
		     handle ExploreInst _ => 
		       impossible "Instantiate:explore_tclass.2")
		 | _ => ();
	      case (get slot, rest)
		of (Final_top {sign, typs, ...}, []) =>
  		     constrain {my_path=SP.SPATH[], its_path=SP.SPATH[],
				its_ancestor=
				  SLOTtyc{base=typs,offset=getSigTPos sign sym}}
		 | (Final_top {sign, subs, ...}, _) =>
		     constrain {my_path=SP.SPATH[], its_path=SP.SPATH rest,
				its_ancestor=
				  SLOTstr{base=subs,offset=getSigPos sign sym}}
		 | (Final_embed {sign, typs, ...}, []) =>
		     constrain {my_path=SP.SPATH[], its_path=SP.SPATH[],
				its_ancestor=
				  SLOTtyc{base=typs,offset=getSigTPos sign sym}}
		 | (Final_embed {sign, subs, ...}, _) =>
		     constrain {my_path=SP.SPATH[], its_path=SP.SPATH rest,
				its_ancestor=
				  SLOTstr{base=subs,offset=getSigPos sign sym}}
		 | (ERRORinstance, _) => ()
		 | _ => impossible "Instantiate:explore_tclass.3")
          | constrain _ = 
	     impossible "Instantiate:explore_tclass:constrain.4"

        val _ = explore_inst first_slot

	exception GetProps

        fun getProps sl =
            case get sl
	      of (tPartial {tycon= FORMtyc{spec=GENtyc{arity,eq,kind,...},...},
			    path,...}) =>
		   (arity, eq, path, kind)
	       | tERRORinstance => raise GetProps
	       | tPartial{tycon,...} =>
		   ErrorMsg.impossibleWithBody
		     "Instantiate.getProps: wrong kind of tycon"
		     (fn ppstrm =>
		       (PPType.ppTycon Env.empty ppstrm tycon;
			PrettyPrint.add_newline ppstrm))
	       | _ => impossible "Instantiate:explore_tclass:getProps"

	exception GetPos

        fun getPos sl' =
            case get sl'
	      of (tPartial{tycon=FORMtyc {pos, ...},...}) => pos
	       | tERRORinstance => (raise GetPos)
	       | _ => impossible "Instantiate:explore_tclass:getPos"

        fun check_arity (ar1, ar2, path1: IP.path, path2: IP.path) =
            if ar1 = ar2 then ()
            else err COMPLAIN 
                     ("Inconsistent arities in sharing type "
                      ^(pathName path1)^" = "^(pathName path2)^" : "
                      ^(pathName path1)^" has arity "^(makestring ar1)^" and "
                      ^(pathName path2)^" has arity "^(makestring ar2)^".")
		     nullErrorBody

	val sortD = Sort.sort
	    (fn (DATACON{name=name1,...},DATACON{name=name2,...}) =>
	     Symbol.symbolGt(name1,name2))

	fun eqDataCons (DATACON{name=name1,...},DATACON{name=name2,...}) =
	    Symbol.eq(name1,name2)

	fun compareD ([], []) = true
	  | compareD (d1::r1, d2::r2) = 
	      eqDataCons(d1,d2) andalso compareD (r1,r2)
	  | compareD _ = false
		    
	val class_tycons = ref 
	    (case (!class_def)
	       of NONE => (NONE : datacon list option)
		| SOME (GENtyc {kind=ref (DATAtyc datacons),...}) =>
		   (SOME (sortD datacons))
		| SOME (DEFtyc {tyfun=TYFUN{body=ty,...},...}) =>
		   (case (headReduceType ty)
		      of CONty (GENtyc {kind = ref (DATAtyc datacons),...},_) =>
			  (SOME (sortD datacons))
		       | _ => NONE)
		| SOME _ => (NONE : datacon list option))

	fun check_kind (ref (DATAtyc datacons), path) = 
	    (case (!class_tycons)
	       of NONE => class_tycons := (SOME (sortD datacons))
		| SOME (dcs) => 
		   if (compareD (dcs,datacons)) then ()
		   else err COMPLAIN
			 ("Inconsistent constructors in sharing datatype "
			  ^(pathName path)^".")
			 nullErrorBody)
	  | check_kind _ = ()

        val finalize =
            case !class_def
	      of SOME (GENtyc {stamp, arity,eq=ref eq, kind=ref kind, path}) =>
                  (fn sl =>
		      let val (arity', _, path', kind') = getProps sl
			  val path'' = substParam(path',argOption)
			  val result = GENtyc {stamp=stamp, arity=arity,
					       eq=ref eq,
					       kind=ref kind,
					       path=path''}
		       in check_arity (arity, arity', path, path');
			  check_kind (kind',path');
			  set (sl, tFinal result)
		      end)
	       | SOME (DEFtyc {path, strict, tyfun as TYFUN {arity, ...}}) =>
                  (fn sl =>
		      (let val (arity', _, path', kind') = getProps sl
			    val path'' = substParam(path',argOption)
			    val result = DEFtyc {path=path'',
						 strict=strict,
						 tyfun=tyfun}
			in check_arity (arity, arity', path, path');
			   check_kind (kind',path');
			   set (sl, tFinal result)
		       end) handle GetProps => (error_found := true;
						set (sl,tERRORinstance)))
             | SOME tyc => 
		 if TypesUtil.eqTycon(tyc,BasicTypes.unitTycon) then
                  (fn sl =>
		      (let val (arity', _, path', kind') = getProps sl
			    val path'' = substParam(path',argOption)
			    val path = InvPath.IPATH[Symbol.tycSymbol "unit"]
			    val result = BasicTypes.unitTycon
			in check_arity (0, arity', path, path');
			   check_kind (kind',path');
			   set (sl, tFinal result)
		       end) handle GetProps => (error_found := true;
						set (sl,tERRORinstance)))
		 else impossible "Instantiate:explore_tclass.5"
             | NONE =>
                case !class
                  of (sl::rest) =>
                       (let val (arity, _, path, kind) = getProps sl
			    val stamp = make_stamp ()
			 in fn sl' =>
                              (let val (arity',ref eq',
					path',kind'' as ref kind') = getProps sl'
				   val path'' = substParam(path', argOption)
				   val pos = getPos sl'
				   val tyc = GENtyc {stamp=stamp, arity=arity,
						     eq=ref eq', path=path'',
						     kind=ref kind'}
			        in check_arity (arity, arity', path, path');
				   check_kind (kind'',path');
				   set (sl', tFinal tyc)
			       end
			       handle GetProps => 
				       (error_found := true;
				        set (sl',tERRORinstance))
				    | GetPos => 
				       (error_found := true;
					set (sl',tERRORinstance)))
			  end
			  handle GetProps => 
			          (error_found := true;
				   set (sl,tERRORinstance);
				   (fn _ => ())))
                   | [] => impossible "Instantiate:explore_tclass.6"
     in app finalize (!class)
    end (* explore_tclass *)

(* debugging wrapper
val explore_tclass = wrap "explore_tclass" explore_tclass
*)

fun sig_to_instance (sign, path, makeStamp, err,argOption) : instance =
    let val dummy = Array.array (1, Initial {signat=sign, path=path, 
				       inherited=ref [],
				       parent_typs=Array.fromList []})

	fun expand ERRORinstance = ()
	  | expand (Final_top {expanded=ref true,...}) = ()
	  | expand (Final_top {actual_subStrs,actual_types,sign,subs,
			       typs, expanded,...}) = 
	    (* We must expand the Final_top instance in a top-down 
	     * fashion.  So, we iterate through the bindings, updating
	     * the subs array appropriately.  As we encounter a
	     * sub signature or type, we recursively expand it.
	     *)
	    let fun expand_substr (subs, typs, actual_types) (sym,i,spec) =
		    (debugmsg("<Expanding substr " ^ symbolToString sym ^ ">");
		     (if Symbol.eq(sym,parentId)
		     then Array.update(subs,i,ERRORinstance) 
                     else case Array.sub(subs, i)
			    of Initial _ =>
				(debugmsg "<substr was Initial, exploring class>";
				 explore_class ({base=subs, offset=i},0,err)
				 handle ExploreInst _ =>
				   impossible "instantiate.2")
			     | Partial _ => (debugmsg "<substr already partial>")
			     | _ => (debugmsg "<substr already final>");
		     case Array.sub(subs, i)
		       of (inst as (Final_top _)) =>
			   (debugmsg "<Going into substrs of top>";
			    expand inst)
			| Final_embed {sign,...} =>
			   (debugmsg "<substr was embedded, going recursive>";
			    app (expand_substr (subs, typs, actual_types))
			                        (getSubSigs sign);
			    debugmsg "<Expanding types of substructure>";
			    app (expand_type (typs, actual_types))
			                        (getTypes sign))
			| ERRORinstance => ()
			| _ => impossible "instantiate.3"))

                and expand_type (typs,actual_types) (sym,i,_) =
                    (debugmsg("<Expanding type " ^ symbolToString sym ^ ">");
		     case Array.sub(typs, i)
		       of tInitial _ =>
			   explore_tclass ({base=typs, offset=i}, makeStamp, 
					   err,argOption)
			| _ => ();
		     debugmsg "<Plugging typ into actual_types>";
                     case Array.sub(typs, i)
		       of tFinal tycon => Array.update (actual_types, i, tycon)
			| tERRORinstance => Array.update (actual_types, i, ERRORtyc)
			| _ => impossible"instantiate.4")
             in expanded := true;
		debugmsg "<Expanding...>";
		case (getSubSigs sign)
		  of [] => (debugmsg "<No sub sigs>")
		   | l => (debugmsg "<app'ing expand_substr>";
			   app (expand_substr (subs,typs,actual_types)) l);
		case (getTypes sign)
		  of [] => (debugmsg "<No sub types>")
		   | l => (debugmsg "<app'ing expand_type>";
			   app (expand_type (typs,actual_types)) l)
            end
	  | expand _ = impossible "instantiate:expand"

     in explore_class({base=dummy,offset=0},0,err)
	 handle (ExploreInst _) => impossible "instantiate.1";
	expand(Array.sub(dummy, 0));
	Array.sub(dummy, 0)
    end

exception Get_Origin

fun get_origin_info instance =
    case instance
      of (Final_top {origin_info,...}) => origin_info
       | (Final_embed {origin_info,...}) => origin_info
       | ERRORinstance => raise Get_Origin
       | _ => impossible "Instantiate:get_origin_info"

(*************************************************************************
 * This function takes an instance graph (assuming that the root node is
 * a Final_top node) and creates a structure corresponding to the graph.
 * It does so by building the graph bottom up.  When an origin structure
 * is needed, we look at the node and determine if one is already built
 * or not (orgin_info is BUILT_STR).  If we need to build a SIMPLE origin,
 * then we are given an instrep env which we iterate over.  For each of
 * the elements, we recursively generate origin structures.
 * 
 * Structures that are constructed for Final_top nodes are memoized
 * (using the final_struct field) so that if two slots in the graph
 * point to the same node, they can use the same instance structure.
 ************************************************************************)
fun instance_to_structure
      (path: IP.path, make_stamp,
       instance as (Final_top{sign,actual_subStrs, actual_types,
			      subs,final_struct,...})) =
    (case (!final_struct)
       of SOME str => str
	| NONE =>
	    let val fct_array =
		    case sign
		      of SIG{kind=ref (TOP{fctcount,...}),...} =>
			  Array.array (fctcount,ERROR_FCT)
		       | _ => Array.fromList []


		(* Gets the origin of an instance -- builds one if one is not
		 * already built. 
		 *)
		fun get_origin instance : origin = 
		    let val origin_info = get_origin_info instance
		     in case (!origin_info)
			  of SOME(origin) => origin
			   | NONE =>
			       let val origin = STAMP_ORIGIN(make_stamp())
			        in origin_info := SOME(origin);
				   origin
			       end
		    end
		    handle Get_Origin => NULL_ORIGIN

		(* Creates a structure node from the instance node found at
		 * position i and plugs it into the actual_subStrs array 
		 * for the current node.  If the structure is embedded, we 
		 * must recurse.
		 *)
		fun inst_to_struct (sym,i,_) =
		    case Array.sub(subs, i)
		      of (inst as (Final_top _)) =>
			   let val IP.IPATH path' = path
			       val str = 
			           instance_to_structure 
			             (IP.IPATH(sym::path'),make_stamp,inst)
			    in Array.update (actual_subStrs,i,str)
			   end
		       | (inst as (Final_embed 
				   {sign,subs,typs,origin_info,...})) =>
			   let val IP.IPATH path' = path
			       val orig = get_origin inst
			       val _ = app inst_to_struct (getSubSigs sign);
                               val str = INSTANCE{sign=sign,
						  origin=orig,
						  subStrs=actual_subStrs,
                                                  subFcts=fct_array,
						  path=IP.IPATH(sym::path'),
						  types=actual_types}
			    in Array.update(actual_subStrs,i,str)
			   end
		       | ERRORinstance => Array.update(actual_subStrs,i,ERROR_STR)
		       | _ => impossible "Instantiate:inst_to_struct"
	     in app inst_to_struct (getSubSigs sign);
		let val str =
		        INSTANCE {sign=sign,
				  subStrs=actual_subStrs,
				  subFcts=fct_array,
				  types=actual_types,
				  path=path,
				  origin=get_origin instance}
		 in final_struct := SOME str;
		    str
		end
	    end)
  | instance_to_structure(_,_,ERRORinstance) = ERROR_STR
  | instance_to_structure _ = impossible "instantiate:instance_to_structure"

(* debugging wrappers
val sig_to_instance = wrap "sig_to_instance" sig_to_instance
val instance_to_structure = wrap "instance_to_structure" instance_to_structure
val externalize_sharing = wrap "externalize_sharing" externalize_sharing
val update_structure = wrap "update_structure" update_structure
*)

fun instantiate_raw(ctx as (path, scope, error_fn),
		    argOption, sign) : Structure =
    let val _ = error_found := false
	val makeStamp = newStamp scope
	fun err sev msg = (error_found := true; error_fn sev msg)
	val instance_graph = 
	      sig_to_instance(sign,path,makeStamp,err,argOption)
	val result = instance_to_structure(path,makeStamp,instance_graph)
     in EqTypes.eqAnalyze(result,Stamps.isBound scope,err);
	result
    end


fun instantiate_partial(ctx: context,
			inParent: Stamps.stamp->bool,
			argOption: Symbol.symbol option,
			name: Symbol.symbol,
			parent: Structure,
			argument: Signature) =
    let val arg_x = externalize_sharing(name,parent,argument)
	val arg_str = instantiate_raw(ctx,argOption,arg_x)
     in update_structure(name,parent,arg_str);
	visit_functor(ctx,inParent,arg_str,arg_str);
        Fixup.fixupStr arg_str;
	arg_str
    end

and instantiate_functor(ctx,inParent,parent,ERROR_FSIG) = ERROR_FCT
  | instantiate_functor(ctx,inParent,parent,FULL_FSIG) = 
      impossible "instantiate: instantiate_functor called on FULL_FSIG"
  | instantiate_functor(ctx as (path, scope, error_fn),inParent,
			parent,FSIG{paramName,argument,body,...}) =
      let val argScope = Stamps.newBoundScope ()
          val bodyScope = Stamps.newBoundScope ()
          val inBody = Stamps.isBound bodyScope
	  fun inArg s = inParent s orelse Stamps.isBound argScope s
          fun newInParent s = inBody s orelse inArg s
	  val makeStamp = newStamp scope
          val arg_str = 
		instantiate_partial((IP.IPATH[],argScope,error_fn),inParent,
				    (SOME paramName),parentId,parent,argument)
          val body_str =
                instantiate_partial((path,bodyScope,error_fn),newInParent,
				    NONE,argumentId,arg_str,body)

          val bindParam = lookBindingSTR(arg_str,SP.SPATH[parameterId])
          val arglty = TransBinding.transStrLty bindParam
          val reslty = TransBinding.transStrLty body_str
          val lambdaty = LambdaType.injARROW(arglty,reslty)
           
          val body_abs = 
              AbstractFct.abstractBody (body_str,arg_str,inBody,inArg)
       in FCT{stamp=makeStamp(),paramName=paramName,parent=parent,
	      lambdaty=lambdaty,argument=argument,body=body_abs}
      end

and visit_functor(ctx,inParent,parent,
		  str as INSTANCE{sign=SIG{env,kind,...},subFcts,subStrs,...}) =
      let val parent' = 
	      case !kind 
		of TOP _ => str
		 | _ => parent
       in Env.app
	   (fn (name,STRbind(STRvar{binding=STR_FORMAL{pos,...},...})) => 
		  if Symbol.eq(name,parentId) then () 
		  else visit_functor(ctx,inParent,parent',Array.sub(subStrs, pos))
	     | (name,FCTbind(FCTvar{binding=FCT_FORMAL{pos,spec},...})) =>
		  Array.update(subFcts,pos,instantiate_functor(ctx,inParent,parent',spec))
	     | (name,bind) => ())
	   (!env)
      end
  | visit_functor _ = ()


(* debugging wrappers
val instantiate_raw = wrap "instantiate_raw" instantiate_raw
val instantiate_partial = wrap "instantiate_partial" instantiate_partial
val visit_functor = wrap "visit_functor" visit_functor
*)

fun instantiate (ctx as (path,scope,error_fn),sign) =
    let val result = instantiate_raw(ctx,NONE,sign)
     in visit_functor(ctx,(Stamps.isBound scope),result,result); 
        Fixup.fixupStr result;
        result
    end 

fun instantiate_argument(ctx as (_,scope,_),paramName,str,sign)  =
    instantiate_partial(ctx,(Stamps.isBound scope),(SOME paramName),parentId,
			str,sign)

(* debugging wrappers
val instantiate = wrap "instantiate" instantiate
val instantiate_argument = wrap "instantiate_argument" instantiate_argument
*)

end (* struct *)
