(* Copyright 1992 by AT&T Bell Laboratories *)

(***************************************************************************
 
  EXTERN.SML: externalize sharings on a structure already defined in a 
  signature. It is used by instantiate.sml when part of the signature is
  already instantiated (parent for an argument signature - argument for a
  functor body signature).

 ***************************************************************************)

signature EXTERN =
sig
  val argumentId: Symbol.symbol
  val parentId: Symbol.symbol
  val parameterId: Symbol.symbol
  val openId: Symbol.symbol
  val bodyId: Symbol.symbol
  val hidden: Symbol.symbol -> bool
  val externalize_sharing: 
	Symbol.symbol * Modules.Structure * Modules.Signature -> Modules.Signature
  val update_structure:
        Symbol.symbol * Modules.Structure * Modules.Structure -> unit
  val make_argument :
        {parent:Modules.Structure, parameter:Modules.Structure}
         -> Modules.Structure
  val debugging : bool ref
end

structure Extern: EXTERN =
struct

open Symbol Access Modules ModuleUtil ErrorMsg TypesUtil

val debugging = ref false  (* standard debugging flag *)

val argumentId = strSymbol "<Argument>"
val parentId = strSymbol "<Parent>"
val parameterId = strSymbol "<Parameter>"
val openId = strSymbol "<open>"
val bodyId = strSymbol "<body>"

fun hidden s =
    Symbol.eq(s,argumentId) orelse Symbol.eq(s,parentId) orelse
      Symbol.eq(s,openId)

fun externalize_sharing_str (name,str) {internal,external} =
    foldr 
      (fn (path as SymPath.SPATH(first::rest),{internal,external}) =>
	   if Symbol.eq(first, name) then 
	     let val new_ext = lookBindingSTR(str,SymPath.SPATH rest)
	      in case external
		   of NONE => {internal=internal,external=SOME new_ext}
		    | SOME old_ext => 
		        if eqOrigin (new_ext,old_ext)
			then {internal=internal,external=external}
			else impossible "Extern: extern_sharing_str"
	     end
	   else {internal=path::internal,external=external})
    {internal=[],external=external} internal


fun externalize_sharing_tyc (name,str) {internal,external} =
    foldr 
      (fn (path as SymPath.SPATH(first::rest),{internal,external})=>
	   if Symbol.eq(first, name) then 
	     let val new_ext =  lookBindingTYC (str, SymPath.SPATH rest)
	      in case external
		   of NONE => {internal=internal,external=SOME new_ext}
		    | SOME old_ext => 
		        if equalTycon (new_ext,old_ext)
			then {internal=internal,external=external}
			else impossible "Extern: extern_sharing_tyc"
	     end
	   else {internal=path::internal,external=external})
    {internal=[],external=external} internal


fun externalize_sharing(name,parent,
    (SIG{stamp,symbols,name=name',env, 
         kind=ref(TOP{strcount,fctcount,typecount,slotcount,
                      sConstraints,tConstraints,abbreviations})})) =
    SIG{stamp=stamp,env=env,symbols=symbols,name=name',
        kind=ref(TOP{strcount=strcount,fctcount=fctcount,
		     typecount=typecount,slotcount=slotcount,
		     abbreviations=abbreviations,
		     sConstraints= 
                       map (externalize_sharing_str(name,parent)) sConstraints,
                     tConstraints=
                       map (externalize_sharing_tyc(name,parent)) tConstraints})}
  | externalize_sharing(_,_,ERROR_SIG) = ERROR_SIG
  | externalize_sharing _ = impossible "Extern: externalize_sharing"


fun update_structure(name,str,arg) =
    case arg
      of INSTANCE{sign as SIG{env,...},subStrs,...} =>
	  ((case Env.look(!env,name)
	      of (STRbind (STRvar {binding=STR_FORMAL {pos, ...},...})) =>
	           Array.update(subStrs,pos,str)
	       | _ => impossible "Extern: update_structure 1")
	   handle Env.Unbound => 
	     if Symbol.eq(name,argumentId) then ()
	     else impossible "Extern: update_structure 3")
       | ERROR_STR => ()
       | INSTANCE{sign as ERROR_SIG,...} => ()
       | _ => impossible "Extern: update_structure 2"


fun make_argument {parent,parameter} =
    let val binding_X = 
	  STRbind(STRvar{name=parameterId,access=SLOT 1,
			 binding=STR_FORMAL{pos=1,spec=FULL_SIG}})
	val binding_P = 
	  STRbind(STRvar{name=parentId,access=SLOT 0,
			 binding=STR_FORMAL{pos=0,spec=FULL_SIG}})
	val env = 
	  Env.bind (parameterId, binding_X,
		    Env.bind (parentId, binding_P, Env.empty))
     in INSTANCE{sign=SIG{symbols = ref [parentId,parameterId],
			  name = NONE,
			  stamp = Stamps.newStamp Stamps.freeScope (),
			  env = ref env,
			  kind = ref(TOP{strcount=2,fctcount=0,
					 typecount=0,slotcount=0,
					 tConstraints=[],sConstraints=[],
					 abbreviations=[]})},
		 subStrs = Array.fromList [parent,parameter],
		 subFcts = Array.fromList [],
		 types = Array.fromList [],
		 origin = STAMP_ORIGIN(Stamps.newFree ()),
		 path = InvPath.IPATH [] (* bogus path *)}
    end

end (* structure Extern *)
