(* Copyright 1992 by AT&T Bell Laboratories *)

(***************************************************************************

   applyfct.sml: applying a functor to an argument
   exports 
     signature APPLYFUNCTOR
     structure ApplyFunctor: APPLYFUNCTOR

 ***************************************************************************)

signature APPLYFUNCTOR =
sig
  val applyFunctor : Modules.Functor 
		     * Modules.Structure * Stamps.scope
		     * InvPath.path
		     * ErrorMsg.complainer
		     * Modules.env
		     * ({abstract: bool,
			 err: ErrorMsg.complainer,
			 scope: Stamps.scope,
			 rpath: InvPath.path,
			 printEnv: Modules.env,
			 self: bool,
			 sign: Modules.Signature,
			 arg_option: Modules.structureVar option,
			 str: Modules.Structure} 
			 -> Modules.Structure * Modules.absWrapper
			    * Modules.thinning)
		      -> Modules.Structure
  val applyFunctorFull : Modules.Functor
			 * Modules.Structure * Stamps.scope
			 * InvPath.path
			 * ErrorMsg.complainer
			 * Modules.env
			 * ({abstract: bool,
			     err: ErrorMsg.complainer,
			     scope: Stamps.scope,
			     rpath: InvPath.path,
			     printEnv: Modules.env,
			     self: bool,
			     sign: Modules.Signature,
			     arg_option: Modules.structureVar option,
			     str: Modules.Structure} 
			     -> Modules.Structure * Modules.absWrapper
				* Modules.thinning)
			  -> Modules.Structure * Modules.thinning
			     * Modules.Structure
  val debugging : bool ref
end


structure ApplyFunctor : APPLYFUNCTOR =
struct
  open Symbol Access Modules Types ModuleUtil Extern Stamps ErrorMsg
  val say = Control.Print.say
  val debugging = ref false
  fun debugmsg(msg: string) = if !debugging then say msg else ()

(*****************************************************************************
   applyFunctor does not perform the argument signature matching because
   it is different if we apply the functor to the instantiation of its
   argument.
   This version is much more sequential than the original one. Structures
   must be processed in their definition order so that parent structures are
   sufficiently defined when they are provided to functors for functor 
   application.
   strinst or fctinst contain the result of instantiation and are initialized
   to ERROR_STR and ERROR_FCT
   strtemp and fcttemp contain the templates for building the corresponding
   elements of the inst version.
   strarrays and fctarrays provide as usual a memo mechanism for the arrays
   used by structures with EMBEDDED signature kind. 
   As in the original version  inststr provides the element by accessing it
   or calls inststr' if it is not ready. instfct and instfct' are the 
   counter-part for functors.
 ****************************************************************************)

fun applyFunctor
     (FCT{paramName: symbol,parent=parentFct,
          stamp,lambdaty,
	  argument: Signature,
	  body={strseq,fctseq,tyseq,str}},
      argstr: Structure,
      scope: scope,
      rpath: InvPath.path,
      err,
      parseEnv: Modules.env,
      sigmatch)
     : Structure =
    let	val makeStamp = Stamps.newStamp scope
        val strtemp = Array.fromList strseq
        val strinst = Array.array(length strseq,ERROR_STR)
        val tyctemp = Array.fromList tyseq
        val tycinst = Array.array(length tyseq,ERRORtyc)
	val fcttemp = Array.fromList fctseq
        val fctinst = Array.array(length fctseq,ERROR_FCT)
        val strarrays : (Structure array * Structure array) list ref = ref nil
        val tycarrays : (tycon array * tycon array) list ref = ref nil
        val fctarrays : (Functor array * Functor array) list ref = ref nil

	fun instOrigin NULL_ORIGIN = STAMP_ORIGIN(makeStamp())
	  | instOrigin (ABS_ORIGIN (SEQ pos)) = 
	     ((case Array.sub(strinst,pos)
		 of ERROR_STR => 
		     STAMP_ORIGIN(getStrStamp
		      (inststr' (fn str => Array.update(strinst,pos,str))
		        (Array.sub(strtemp,pos))))
		   | str => STAMP_ORIGIN(getStrStamp str))
	      handle General.Subscript => 
		impossible(concat ["Applyfct.inststr, pos = ", makestring pos]))
          | instOrigin (ABS_ORIGIN(SEQind(pos,path))) =
	     ((case Array.sub(strinst,pos)
		 of ERROR_STR => 
		     let val str =
			     inststr' (fn str => Array.update(strinst,pos,str))
			       (Array.sub(strtemp,pos))
		      in STAMP_ORIGIN(getStrStamp(transPosStr str path))
		     end
		  | str => STAMP_ORIGIN(getStrStamp(transPosStr str path)))
	      handle General.Subscript =>
		impossible(concat ["Applyfct.inststr, pos = ", makestring pos]))
          | instOrigin (ABS_ORIGIN(PARAM pos)) =
	      STAMP_ORIGIN(getStrStamp(transPosStr argstr pos))
	  | instOrigin origin = origin

        and inststr (STR_ABSFB (SEQ pos)) =
	     ((case Array.sub(strinst,pos)
		 of ERROR_STR => 
		     inststr' (fn str => Array.update(strinst,pos,str))
		       (Array.sub(strtemp,pos))
		   | str => str)
	      handle General.Subscript => 
		impossible(concat ["Applyfct.inststr, pos = ", makestring pos]))
          | inststr (STR_ABSFB (SEQind (pos,path))) =
	     ((case Array.sub(strinst,pos)
		 of ERROR_STR => 
		     let val str =
			     inststr' (fn str => Array.update(strinst,pos,str))
			       (Array.sub(strtemp,pos))
		      in transPosStr str path
		     end
		  | str => transPosStr str path)
	      handle General.Subscript =>
		impossible(concat ["Applyfct.inststr, pos = ", makestring pos]))
          | inststr (STR_ABSFB (PARAM pos)) = transPosStr argstr pos
	  | inststr str = str

        and instfct (FCT_ABSFB (SEQ pos)) =
	     ((case Array.sub(fctinst,pos)
		 of ERROR_FCT => 
		     instfct' (fn fct => Array.update(fctinst,pos,fct))
		       (Array.sub(fcttemp,pos))
		   | fct => fct)
	      handle General.Subscript => 
		impossible(concat ["Applyfct.instfct, pos = ", makestring pos]))
          | instfct (FCT_ABSFB (SEQind (pos,path))) =
	     ((case Array.sub(strinst,pos)
		 of ERROR_STR => 
		     let val str =
			     inststr' (fn str => Array.update(strinst,pos,str))
			       (Array.sub(strtemp,pos))
		      in transPosFct str path
		     end
		  | str => transPosFct str path)
	      handle General.Subscript =>
		impossible(concat["Applyfct.instfct, pos = ", makestring pos]))
          | instfct (FCT_ABSFB (PARAM pos)) = transPosFct argstr pos
	  | instfct (FCT_INSTANCE{fsig,parent,lambdaty,fct}) =
	     let val parent' = inststr parent
		 val fct' = instfct fct
	      in FCT_INSTANCE{fsig=fsig,fct=fct',lambdaty=lambdaty,
                              parent=parent'}
	     end
	  | instfct fct = fct

        and insttyc (ABSFBtyc (SEQ pos,_)) =
	     ((case Array.sub(tycinst,pos)
		 of ERRORtyc => 
		     insttyc' (fn tyc => Array.update(tycinst,pos,tyc))
		       (Array.sub(tyctemp,pos))
		   | tyc => tyc)
	      handle General.Subscript => 
		impossible(concat["inststr, pos = ", makestring pos]))
          | insttyc (ABSFBtyc (SEQind (pos,path),_)) =
	     ((case Array.sub(strinst,pos)
		 of ERROR_STR => 
		     let val str =
			     inststr' (fn str => Array.update(strinst,pos,str))
			       (Array.sub(strtemp,pos))
		      in transPosTycon str path
		     end
		  | str => transPosTycon str path)
	      handle General.Subscript =>
		impossible(concat["inststr, pos = ", makestring pos]))
          | insttyc (ABSFBtyc (PARAM pos, _)) = transPosTycon argstr pos
	  | insttyc tyc = tyc

        and inststr' update 
              (INSTANCE{sign=sign as SIG{env,...},path=rpath',
                        subStrs,types,subFcts,origin}) =
	     let fun find (elem,a,elt) =
		     let fun f ((key,data) :: t) =
			       if elem=key then data else f t
			   | f nil =
			       let val r=Array.array(Array.length elem,elt)
			        in a := (elem,r) :: (!a);
				   r
			       end
		      in f (!a)
		     end
		 val subStrs' = find(subStrs,strarrays,ERROR_STR)
		 val types' = find(types,tycarrays,ERRORtyc)
		 val subFcts' = find(subFcts,fctarrays,ERROR_FCT)
		 fun updtStr pos =
		      (case Array.sub(subStrs',pos)
			 of ERROR_STR =>
			     Array.update
			      (subStrs',pos,
			       inststr (Array.sub(subStrs,pos)))
			  | _ => ();
		       updtStr (pos+1))
		       handle General.Subscript => ()
		 fun updtFct pos =
		      (case Array.sub(subFcts',pos)
			 of ERROR_FCT =>
			     Array.update
			      (subFcts',pos,
			       instfct (Array.sub(subFcts,pos)))
			  | _ => ();
		       updtFct (pos+1))
		       handle General.Subscript => ()
		 fun updtTyc pos =
		      (case Array.sub(types',pos)
			 of ERRORtyc =>
			     Array.update
			      (types',pos,
			       insttyc (Array.sub(types,pos)))
			  | _ => ();
		       updtTyc (pos+1))
		       handle General.Subscript => ()
		 val origin' = instOrigin origin

		 val new_str = INSTANCE{sign=sign, subStrs=subStrs',
					types=types', subFcts=subFcts',
					path=InvPath.append(rpath,rpath'),
					origin=origin'}
              in update new_str;
		 updtStr 0; updtFct 0; updtTyc 0;
		 new_str
             end
          | inststr' update (APPLY{fct,arg,path,...}) =
	     let val fct' = instfct fct
		 val arg' = inststr arg
		 val (new_str,thinIn,_) = 
		     applyFunctorFull(fct',arg',scope,InvPath.append(rpath,path),
				      err,parseEnv,sigmatch)
	      in update new_str;
		 new_str
	     end
          | inststr' update _ = 
              impossible "ApplyFunctor.inststr'"

        and instfct' update 
                (fct as FCT{parent,lambdaty,argument,body,stamp,paramName}) =
              let val new_fct = 
		      FCT{parent=inststr parent, body=body, stamp=makeStamp(),
			  lambdaty=lambdaty, paramName=paramName, argument=argument}
               in update new_fct;
		  new_fct
	      end
          | instfct' update (FCT_INSTANCE{fsig,fct,lambdaty,parent}) =
              let val new_fct =
                      FCT_INSTANCE{fsig=fsig, fct = instfct fct,
				   lambdaty=lambdaty, parent=inststr parent}
               in update new_fct;
		  new_fct
	      end
          | instfct' update _ =
              impossible "instfct'"

        and insttyc' update (GENtyc{stamp,arity,eq,path=rpath',kind}) =
              let val kind' = case !kind
				of d as DATAtyc _ =>ref d
				 | _ => kind
	          val new_tyc =
                      GENtyc{stamp = makeStamp(), eq = eq, arity = arity,
			     path = InvPath.append(rpath,rpath'), kind = kind'}
               in update new_tyc;
		  case !kind'
		    of (DATAtyc dcons) =>
                       kind' := DATAtyc(
                          map (fn DATACON{name,const,typ,rep,
                                          sign,orig=u as SOME _} =>
                                  DATACON{name=name,const=const,
                                          rep=rep,sign=sign,
                                          typ=insttype typ,orig=u}
                                | DATACON{name,const,typ,rep,sign,orig} =>
                                    let val new = 
                                          (case rep of VARIABLE _ => NONE
                                                     | VARIABLEc _ => NONE
                                                     | _ => SOME typ)
                                   in DATACON{name=name,const=const,
                                              rep=rep,sign=sign,
                                              typ=insttype typ,orig=new}
                                    end
                              ) dcons)
		     | _ => ();
		  new_tyc
              end
	  | insttyc' update (DEFtyc{path=rpath',strict,tyfun=TYFUN{arity,body}}) =
	      let val new_tyc =
                      DEFtyc{path=InvPath.append(rpath,rpath'),strict=strict,
			     tyfun=TYFUN{arity=arity,body=insttype body}}
               in update new_tyc;
		  new_tyc
	      end
	  | insttyc' _ _ = impossible "applyFunctor.insttyc"

	and insttype ty =
	    case ty
	      of CONty(tycon,args) => 
                   TypesUtil.mkCONty(insttyc tycon,map insttype args)
	       | POLYty{sign,tyfun=TYFUN{arity,body},abs} =>
	           POLYty{sign=sign,abs=abs,
			  tyfun=TYFUN{arity=arity,body=insttype body}}
	       | _ => ty

     in inststr str
    end
  | applyFunctor _ =impossible "ApplyFunctor.strange functor"

and applyFunctorFull
      (fct_def, argstr, scope, rpath, err, parseEnv, sigmatch) =
    case fct_def
      of FCT{argument=arg_sig,body,parent=fct_parent,...} =>
	  let val _ = if !debugging then say "I\n" else ()
	      val argument = make_argument{parent=fct_parent,parameter=argstr}
	      val _ = if !debugging then say "II\n" else ()
	      val (arg_pair,_,thinInTotal) = 
		    sigmatch{abstract=false,err=err,scope=scope,rpath=rpath,
			     printEnv=parseEnv,self=false, arg_option=NONE,
			     str=argument, sign=arg_sig}
	      val _ = if !debugging then say "III\n" else ()
	      val thinIn = 
		    case thinInTotal
		      of SOME(_,[THINtrans(a,v,transl)]) => SOME (v,transl)
		       | _ => NONE
	      val res = 
		    applyFunctor
		      (fct_def,arg_pair,scope,rpath,err,parseEnv,sigmatch)
	      val _ = if !debugging then say "IV\n" else ()
	      val new_str = APPLY{fct=fct_def,arg=argstr,res=res,path=rpath}
	   in (new_str,thinIn,arg_pair)
	  end
       | FCT_INSTANCE{fsig = FSIG{argument=arg_sig,body,...},parent=fct_parent,
		      fct=fct', lambdaty=lambdaty'} =>
	  let val argument = make_argument{parent=fct_parent,parameter=argstr}
	      (* matching done for verification *)
	      val (realArg,_,thinInTotal) = 
		    sigmatch{abstract=false, err=err, scope=scope, rpath=rpath,
			     arg_option=NONE, printEnv=parseEnv, self=false,
			     str=argument, sign=arg_sig}
	      val thinIn =
		    case thinInTotal
		      of SOME(_,[THINtrans(a,v,transl)]) => SOME (v,transl)
		       | _ => NONE
	      val argumentVar = 
		    STRvar{name=argumentId, binding=argument, access=NO_ACCESS}
	      val res =
		    case applyFunctorFull
			   (fct',argstr,scope,rpath,err,parseEnv,sigmatch)
		      of (APPLY{res,...},_,_) => res
		       | _ => impossible "ApplyFunctor.strange result"
	      (* matches the result against the body specification of the functor
	       * signature *)
	      val (res_match,_,thinRes) = 
		    sigmatch {abstract=false, err=err, scope=scope, rpath=rpath,
			      arg_option=SOME argumentVar, printEnv=parseEnv,
			      sign=body, str=res, self=false}
	   in (APPLY{fct=fct_def,arg=argstr,res=res_match,path=rpath},
	       thinIn,realArg) 
	  end
       | FCT_INSTANCE _ => (ERROR_STR,NONE,ERROR_STR) (* The signature is wrong *)
       | ERROR_FCT => (ERROR_STR,NONE,ERROR_STR)
       | _ => impossible "ApplyFunctorFull"

end (* structure ApplyFunctor *)
