(* Copyright 1992 by AT&T Bell Laboratories *)
(***************************************************************************

  ABSTRACTFCT.SML takes a structure representing the application of a
  functor to an argument (also provided) and gives back the recipe to
  perform this application.

 ***************************************************************************)

structure AbstractFct: ABSTRACTFCT =
struct

open Types Modules ModuleUtil Variables
    

(* printing, errors, and debugging *)

val say = Control.Print.say

fun err s = ErrorMsg.impossible ("AbstractFct: "^s)

val debugging = ref false

fun debugmsg(msg: string) = if !debugging then say msg else ()


(* local utilities *)

exception NotFound

fun next (ri as ref i) = (ri := i+1; i)

(* naughty technique of terminating loop by handling Subscript exception *)
fun appArray f = 
    let fun app i = (f i; app (i+1)) handle General.Subscript => ()
     in app 0
    end

(*----------------------------------------------------------------------*)

(* Assumptions and invariants *)

(* ASSUMPTION 1: no SIMPLE structures will be found either in the
   dummy parameter nor in the functor body structure.  For the
   parameter structure, this is because it is constructed by 
   instantiating the parameter signature.  For the result structure,
   this is guaranteed by ElabStr/elaborate_functor (? case where
   body is a structure variable?). *)

(***** MAPS FOR MEMOIZING STRUCTURES AND FUNCTORS *****)

(* Maps are used to memoize the "positions" of structures
   functors and tycons that have already been encountered.
   The positions are represented by the ABSFB forms of structures,
   functors and types. *)

(* for a given structure stamp, a strInfo value records where INSTANCE
   and "original" versions of the associated structure are found in
   the construction tables. *)

type strInfo =
  {instances: (Stamps.stamp * Structure) list,
     (* association list of instance positions,
        indexed by corresponding signature
        stamps *)
   original: Structure option}
     (* position of an "original" version of
        the structure *)

type memoizeStrMap = strInfo Stamps.stampMap
 and memoizeFctMap = Functor Stamps.stampMap
 and memoizeTycMap = tycon Stamps.stampMap

exception StampMap
fun newMap() = Stamps.newMap StampMap


(* Add a new instance signature sign and "position" str to the map.
   The parameter str should always be an STR_ABSFB form. *)

fun addInstance(strStamp, signStamp, str: Structure, map: memoizeStrMap) =
    let val {instances,original} = Stamps.applyMap (map,strStamp)
     in Stamps.updateMap map (strStamp,{instances = (signStamp,str)::instances,
					 original = original})
    end
    handle StampMap =>  (* first time strStamp is encountered *)
      Stamps.updateMap map (strStamp,{instances=[(signStamp,str)],original=NONE})


(* add a "position" for a "original" structure to the map.
   The parameter str should always be an STR_ABSFB form. *)

fun addOriginal(strStamp, str: Structure, map: memoizeStrMap) =
    let val {instances,original} = Stamps.applyMap(map,strStamp)
     in (* We want to keep the first path we have found for a structure
	 * and we don't want to redefine it if we encounter the same
	 * structure again. *)
	 case original 
	  of NONE =>
	      Stamps.updateMap map
	        (strStamp,{instances = instances, original = SOME str}) 
	   | _ => ()
    end
    handle StampMap =>
      Stamps.updateMap map (strStamp,{instances=[],original= SOME str})


(* gets an instance from the map, given a structure stamp and signature stamp.
   The instance will be a STR_ABSFB form. *)

fun getInstance(strStamp,signStamp,map: memoizeStrMap): Structure =
    let val {instances,original} = Stamps.applyMap (map,strStamp)
        fun find [] =
	     (debugmsg "getInstance fail 1\n";
	      raise NotFound)
          | find ((s,str)::l) = 
             if Stamps.eq(s,signStamp) then str else find l
     in debugmsg("getInstance "^Stamps.stampToString strStamp ^","^
		 Stamps.stampToString signStamp ^ "\n");
        find instances
    end
    handle StampMap =>
      (debugmsg "getInstance fail 2\n";
       raise NotFound)


(* gets a original structure from the map, given a structure stamp *)

fun getOriginal(strStamp, map:memoizeStrMap): Structure =
    let val {instances,original} = Stamps.applyMap (map,strStamp)
     in case original 
	  of NONE => raise NotFound
	   | SOME str => str
    end
    handle StampMap => raise NotFound


(* produce an abstracted origin;
   strStamp assumed to belong to the parameter, hence there should be
   an associated INSTANCE in the map *)

fun getOrigin(strStamp: Stamps.stamp, map: memoizeStrMap) : origin =
    let val {instances,original} = Stamps.applyMap (map,strStamp)
        fun find [] =
	     (debugmsg "getOrigin fail 1\n";
	      raise NotFound)
          | find ((_,str)::_) = str
             (* any instance will do, since we will only need a
	        structure stamp for building an origin *)
     in debugmsg("getOrigin "^Stamps.stampToString strStamp ^"\n");
        case find instances
	  of STR_ABSFB absfbpos => ABS_ORIGIN absfbpos
	   | _ => ErrorMsg.impossible "AbstractFct.getOrigin - no instance"
    end
    handle StampMap =>
      ErrorMsg.impossible "AbstractFct.getOrigin - stamp not mapped"


(***************************************************************************
   enrichMap: create mappings from stamps satisfying inScope to locations
   in the structure they are stored using the function full_path that can
   be 
    - (fn x => PARAM x) when traversing the parameter
    - (fn x => SEQind(pos,x)) when traversing the result of a functor 
               application which will be stored in the pos slot

   Note that we make no assumption on the parameter str: it can contain any kind
   of structures.  No assumption can be made on Parents and result of functor
   application, because we use it on more general structures than signature 
   instantiation (as assumed in the original version).
 ***************************************************************************)

fun enrichMap (inScope: Stamps.stamp->bool, strStampMap: memoizeStrMap,
	       fctStampMap:memoizeFctMap, tycStampMap: memoizeTycMap,
	       full_path:int list -> absfbpos,
	       str: Structure) =
    let fun scanTyc (typ,loc) =
	  (case typ
	     of GENtyc{stamp,...} =>
		  if inScope stamp
		  then (Stamps.applyMap(tycStampMap,stamp); ())
			  handle StampMap =>
			    (Stamps.updateMap 
			       tycStampMap 
			       (stamp,ABSFBtyc (full_path (rev loc), typ)))
		  else ()
	      | _ => ())
	fun scanStr (ERROR_STR,_,strArray) = ()
	  | scanStr(s as INSTANCE{sign,subStrs,subFcts,types,origin,...},
		    loc,strArray) =
	      let val strStamp = getStrStamp s
		  val signStamp = getSignStamp sign
		  (* ugly but works: colapses paths if on the same array *)
		  val newLoc = 
		    case (loc,strArray)
		      of (posStr::endloc,SOME subStrs') => 
			   if subStrs=subStrs' then endloc else loc
		       | _ => loc
	       in if inScope strStamp
		  then
		    ((debugmsg("INSTANCE "^(Stamps.stampToString strStamp)^"/"^
			       (Stamps.stampToString signStamp)^"\n");
		      (* see if this instance is already in the map *)
		      getInstance(strStamp,signStamp,strStampMap);
		      ())
		     handle NotFound => (* hasn't been mapped *)
		      (addInstance(strStamp, signStamp,
				    STR_ABSFB(full_path(rev loc)), strStampMap);
		       case sign 
			 of SIG{kind=ref EMBEDDED,...} => ()
			  | _ => appArray (fn pos =>
					     scanStr(Array.sub(subStrs,pos),
						     pos::newLoc,SOME subStrs));
		       appArray (fn pos => scanFct(Array.sub(subFcts,pos),
						   pos::newLoc));
		       appArray (fn pos => scanTyc(Array.sub(types,pos),
						   pos::newLoc))))
		  else ()
	      end
	  | scanStr (APPLY{res,...},loc,_) = scanStr (res,loc,NONE)
	  | scanStr (STR_ABSFB _,_,_) = ()
	  | scanStr(s,loc,strArray) =
	      let val strStamp = getStrStamp s
	       in if inScope strStamp then
		   (debugmsg("SIMPLE "^(Stamps.stampToString strStamp)^"\n");
		    addOriginal(strStamp,STR_ABSFB(full_path(rev loc)),strStampMap))
	          else ()
	      end handle ErrorMsg.Error =>
		  let fun C f x y = f y x
		   in PrettyPrint.with_pp (ErrorMsg.defaultConsumer())
		        (C PPBasics.ppStructure (Env.empty,s,20));
		      err "scanStr"
		  end

	and scanFct (FCT{stamp,...},loc) =
	     if inScope stamp then
	       Stamps.updateMap fctStampMap (stamp,FCT_ABSFB(full_path(rev loc)))
	     else ()
	  | scanFct (FCT_INSTANCE{fct,fsig,lambdaty,parent},loc) = 
	       scanFct(fct,loc)
	  | scanFct (ERROR_FCT,_) = ()
	  | scanFct (FCT_FORMAL _,_) = err "scanFct: formal fct"
	  | scanFct (FCT_OPEN _,_) = err "scanFct: open fct"
	  | scanFct (FCT_ABSFB _,_) = ()
     in scanStr(str,[],NONE)
    end


val bogusTycSym = Symbol.tycSymbol "?.bogus"
val bogusPath = InvPath.IPATH[bogusTycSym]

(* transform a type expression, given a transformer for tycons *)
fun rewriteType(ty:ty, transTycon: tycon->tycon) : ty =
    let fun rewrite ty =
	case ty
	  of VARty(ref(INSTANTIATED ty')) => rewrite ty'
	   | POLYty{tyfun=TYFUN{body,arity},sign,abs} =>
		    POLYty{tyfun=TYFUN{body=rewrite body,arity=arity},
			   sign=sign,abs=abs}
	   | CONty(tyc,args) => 
	       CONty(transTycon tyc, map rewrite args)
	   | _ => ty
     in rewrite ty
    end


(* The main function: it looks at environment bindings in the order of their
   definition to ensure that maps are correctly defined while looking at
   them (problem of functor application).  Memoization and processing is
   done in a single function.
 *)

fun abstractBody (body,param,inBody: Stamps.stamp -> bool,
		  inParam: Stamps.stamp -> bool) =
let val tycCount = ref 0
    val strCount = ref 0
    val fctCount = ref 0
    val tycSeries : tycon list ref = ref nil
    val strSeries : Structure list ref = ref nil
    val fctSeries : Functor list ref = ref nil

    fun addTyc tyc = (tycSeries := tyc :: !tycSeries;
		      ABSFBtyc(SEQ (next tycCount), tyc))
    fun addStr str = (strSeries := str :: !strSeries;
		      STR_ABSFB(SEQ (next strCount)))
    fun addFct fct = (fctSeries := fct :: !fctSeries;
		      FCT_ABSFB(SEQ (next fctCount)))

    val strMap = newMap () : memoizeStrMap
    val fctMap = newMap () : memoizeFctMap

    (* parameter and body stamps mapped to transformed tycons *)
    val paramTycMap = newMap () : memoizeTycMap
    val bodyTycMap = newMap () : memoizeTycMap

    val _ = enrichMap(inParam,strMap,fctMap,paramTycMap,PARAM,param)


    (* Replace DEFtycs and GENtycs with parameter or body stamps
       with ABSFBtyc's.  This is applied to the elements of the types
       array in abstract_env below. *)
    fun abstractTyc (tyc: tycon) : tycon =
           (* In 0.97 and before, we used to memo-ize the DEFtycs.
	      But we no longer do this because it's too hard to
	      tell if two DEFtycs are the same, since tycon is no
	      longer an eqtype. *)
	let fun rewriteTycon tyc =
		case tyc
		  of DEFtyc {path,strict,tyfun as (TYFUN{arity,body})} =>
		       addTyc(DEFtyc{path=path,
				     strict=strict,
				     tyfun=TYFUN{arity=arity,
						 body=rewriteType(body,
								  rewriteTycon)}})
		   | GENtyc{stamp,path=spath,kind,...} =>
		       if inParam stamp then
			 (Stamps.applyMap(paramTycMap,stamp)
			  handle StampMap =>
			    err "rewriteTycon: param tycon not found")
		       else if inBody stamp then
			 (Stamps.applyMap(bodyTycMap,stamp)
			  handle StampMap =>
			    let val r = addTyc tyc
			     in Stamps.updateMap bodyTycMap (stamp,r); (* memoize *)
				case kind
				  of ref (DATAtyc dcons) =>  (* why? (dbm) *)
				       kind := DATAtyc
						(map (fn DATACON{name,const,
								 typ,rep,orig,sign} =>
						      DATACON
						        {name=name,
							 const=const,rep=rep,
							 sign=sign,orig=orig,
							 typ=rewriteType
							       (typ,rewriteTycon)})
						 dcons)
				   | _ => ();
				r
			    end)
		       else tyc
		   | tyc => tyc

	 in rewriteTycon tyc
	end

    fun relativizeSignEnv(env : StaticEnv.staticEnv, types: tycon array)
	 : StaticEnv.staticEnv * tycon array =
    (* When "new" tycons are discovered during the traversal of the
     * signature environment, they are stored in extratycs and will
     * later be added to an extension of the types instance array.  They
     * are mapped to RELtycs refering to their position in the extended
     * types array, and this mapping is memoized in tycMap.  The "new"
     * tycons may include tycons from the parameter or declared in the
     * body. *)
	let val extratycs = ref (nil : tycon list)
	    val tyccount = ref (Array.length types)
	    val tycMap = newMap()
	    fun addExtraTyc tyc =
		 let val r = RELtyc {path=bogusPath,pos=([],next tyccount),
				     spec=tyc}
		  in extratycs := tyc :: !extratycs;
		     r
		 end

	    fun relTyc (tyc: tycon) : tycon =
		let fun rewriteTycon tyc =
			case tyc
			  of DEFtyc {path,strict,tyfun as (TYFUN{arity,body})} =>
			      (* not right yet: trying to eliminate locally
			         bound type constructors (from param or body),
				 replacing them with RELtyc references.
				 Want to do nothing unless the body of the DEFtyc
				 contains locally (param, body) bound tycons.
				 If it does, want to put it into a type array
				 and then abstract the type array to put it into
				 the functor's tycSeries.
			         This adds it unconditionally.
				 (as in 105 & earlier) *)
			       addExtraTyc tyc
			   | GENtyc{stamp,path=spath,...} =>
			       if inBody stamp orelse inParam stamp then
				 (Stamps.applyMap(tycMap,stamp)
				  handle StampMap => 
				    let val r = addExtraTyc tyc
				     in Stamps.updateMap tycMap (stamp,r);
					r
				    end)
			       else tyc
			   | tyc => tyc
		 in rewriteTycon tyc
		end

	    fun relativeType ty = rewriteType(ty,relTyc)

	    fun abstractDcon(DATACON{name,const,typ,rep,sign,orig}) =
		DATACON{name=name,const=const,rep=rep,sign=sign,
			typ=relativeType typ,orig=orig}

	    fun relativizeBinding binding =
		case binding
		  of VARbind(VALvar{path,access,typ}) =>
		       VARbind(VALvar{path=path,access=access,
				      typ=ref(relativeType (!typ))})
		   | CONbind dcon => CONbind(abstractDcon dcon)
		   | _ => binding
	    
	    val newEnv = Env.map relativizeBinding env
	    val newTypes = 
		  case !extratycs
		    of nil => types
		     | l => Array.fromList(ArrayExt.listofarray types @ rev l)

	 in (newEnv, newTypes)
	end

    (* should also deal with type abbreviations in this function *)
    fun relativizeSign (arg as (SIG{symbols,env,kind,stamp,...},types)) =
	  if inBody stamp then
	    let val (env', types') = relativizeSignEnv(!env,types)
	     in (SIG{symbols=symbols,kind=kind,env=ref env',
		     stamp=Stamps.newStamp Stamps.freeScope (),
		     name=NONE},  (* anonymous *)
		 types')
	    end
	  else arg
      | relativizeSign arg = arg


    fun getPos(STR_ABSFB pos) = pos
      | getPos _ = ErrorMsg.impossible "AbstractFct.getPos"

    fun abstractStr (str as (INSTANCE{sign = (sign as SIG{env,...}),
				      origin,subStrs,subFcts,types,path})) =
	  let val strStamp = getStrStamp str
	      val sigStamp = getSignStamp sign
	   in if !debugging then
		(say(InvPath.makestring path);
		 app say [" get INSTANCE ",(Stamps.stampToString strStamp),"/",
			  (Stamps.stampToString sigStamp),"\n"];
		 if inBody strStamp orelse inParam strStamp 
		 then say " FUNC\n"
		 else say "\n")
	      else ();
	      getInstance(strStamp,sigStamp,strMap)
	      handle NotFound =>
		if inBody strStamp orelse inParam strStamp 
		   orelse inBody sigStamp
		then let val origin' =
			     case origin
			       of STAMP_ORIGIN s => 
				   (debugmsg("#AbstractFct 11: "
					     ^Stamps.stampToString s^"\n");
				    if inBody s then NULL_ORIGIN
					(* to be generated on application *)
				    else if inParam s 
				    then getOrigin(s,strMap)
				    else origin)
				| _ => origin
			 val (sign',types') = relativizeSign(sign,types)
			 val new_str =
			     INSTANCE{sign=sign',origin=origin',
				      subStrs=subStrs,types=types',
				      subFcts=subFcts, path=path}
			 val result = addStr new_str
		      in addInstance(strStamp, sigStamp, result, strMap);
			 (* perhaps there is a better place to put
			  the abstractStr etc. calls *)
			 case sign
			   of SIG{kind = ref EMBEDDED, ...} =>
			       (* HACKING AROUND BUG #729:
				  relativizeSign may have produced a new, extended
				  types array. *)
			       if types = types' then ()
			       else
				 appArray
				   (fn pos => Array.update (types',pos,
                                       abstractTyc (Array.sub(types',pos))))

			    | _ => (* we must follow the order of the definition 
				       while abstracting the environment *)
				  (appArray (fn pos => Array.update(subStrs,pos,
					  abstractStr(Array.sub(subStrs,pos))));
				   appArray (fn pos => Array.update(subFcts,pos,
					  abstractFct(Array.sub(subFcts,pos))));
				   appArray (fn pos => Array.update(types',pos,
					  abstractTyc(Array.sub(types',pos)))));
			 result
		     end
		else str
	  end
      | abstractStr (str as (APPLY{fct,arg,res,path})) =
	  let val strStamp = getStrStamp str
	   in getOriginal(strStamp,strMap)
	      handle NotFound =>
	       if inBody strStamp then
		 let val fct' = abstractFct fct
		     val arg' = abstractStr arg
		  (* we enrich our map with the structures produced by the
		     functor: we can only access them after the functor
		     has been applied. *)
		     val _ = enrichMap 
			      (inBody,strMap,fctMap,bodyTycMap,
			       (fn s => SEQind(!strCount,s)), res)
		     val result = addStr(APPLY{fct=fct',arg=arg',res=ERROR_STR,
					       path=path})
		  in addOriginal(strStamp, result, strMap);
		     result
		 end
	       else str
	  end
      | abstractStr (str as SIMPLE{stamp=strStamp,...}) =
	 (getOriginal(strStamp,strMap)
	  handle NotFound =>
	    if inBody strStamp 
	    then ErrorMsg.impossible "AbstractFct.abstractStr: SIMPLE in body"
	    else str)
      | abstractStr (str as STR_OPEN _) = str
      | abstractStr (str as STR_FORMAL _) = str
      | abstractStr (str as STR_ABSFB _) = str
      | abstractStr (str as ERROR_STR) = str
      | abstractStr (STR_EXTERN _) =
	  ErrorMsg.impossible "AbstractFct.abstractStr: STR_EXTERN"

    and abstractFct (FCT_INSTANCE{fsig,fct,lambdaty,parent}) = 
	  let val fct' = abstractFct fct
	      val parent' = abstractStr parent
	   in FCT_INSTANCE{fsig=fsig,parent=parent',
			   lambdaty=lambdaty,fct=fct'}
	  end
      | abstractFct (fct as (FCT{stamp=fctStamp,parent,lambdaty,
				 paramName,argument,body})) =
	 (debugmsg("get FCT"^(Stamps.stampToString fctStamp));
	  Stamps.applyMap(fctMap,fctStamp)
	  handle StampMap =>
	    let val parent' = abstractStr parent
		val fct' = 
		  addFct (FCT{stamp=fctStamp,parent=parent',lambdaty=lambdaty,
			      paramName=paramName,argument=argument,body=body})
	     in Stamps.updateMap fctMap (fctStamp,fct');
	        fct'
	    end)
      | abstractFct fct = fct


    val main = abstractStr body      

 in {tyseq=rev(!tycSeries),strseq=rev(!strSeries),
     fctseq=rev(!fctSeries),str=main}
end (* fun abstractBody *)

end (* structure AbstractFct *)
