(* Copyright 1989 by AT&T Bell Laboratories *)
functor M68CM(V : M68CODER) : CMACHINE = struct

structure V' : sig 
			exception BadReal of string
			datatype Register = DataReg of int
					  | AddrReg of int
					  | FloatReg of int
					  | PC
			
		        type Label sharing type Label = V.Label
			datatype Size = Byte | Word | Long
			
			datatype EA = Direct of Register
				    | PostInc of Register
				    | PreDec of Register
				    | Displace of Register * int
				    | Index of Register * int * Register * Size
				    | Immed of int
				    | Immed32 of Word32.word
				    | Immedlab of Label
				    | Abs of int
				    | Address of Label

		end = V
open V'

datatype condition = NEQ | EQL | LEQ | GEQ | LSS | GTR
  		   | GEU | GTU | LTU | LEU
(* +DEBUG
fun diag (s : string) f x =
	f x handle e =>
		(print "?exception "; print (SMLofNJ.exnName e);
		 print " in m68."; print s; print "\n";
		 raise e)
-DEBUG *)

fun defer(Direct r) = Displace(r,0)
  | defer(Immedlab lab) = Address lab
  | defer _ = ErrorMsg.impossible "defer in cpsm68"

(* DEBUG val defer = diag "defer" defer *) 

val usersp = AddrReg 7
val exnptr = Direct(DataReg 7)
val varptr = Direct(DataReg 3)
val varptr_indexable = false
val dataptr as Direct dataptr' = Direct(AddrReg 6)
val arithtemps = map (Direct o DataReg) [0,1]
val arithtemp2 = Direct(DataReg 2)
val arithtemp3 = Direct(DataReg 4)
val storeptr = Direct(DataReg 6) (*** STOREPTR ***)
val standardclosure = Direct(AddrReg 2)
val standardarg = Direct(AddrReg 0)
val standardcont = Direct(AddrReg 1)
val standardlink = Direct(AddrReg 3)

val nvregs = 8              (* must match M68.prim.asm and ml-state.h *)
val vreg_offset = 4         (* must match M68.prim.asm *)
fun mk_vregs n = 
    let fun aux 0 = []
	  | aux m = Displace(usersp,(n-m)*4+vreg_offset)::aux(m-1)
    in aux n
    end
val vregs = mk_vregs nvregs

val pseudo1 = Displace(usersp,vreg_offset+(nvregs*4))
val pseudo2 = Displace(usersp,vreg_offset+(nvregs*4)+4)

val miscregs = (map (Direct o AddrReg) [4]) @ vregs
val savedfpregs = map (Direct o FloatReg) [2,3,4,5,6]
val floatregs = [Direct(FloatReg 0), Direct(FloatReg 1)]
val fptempreg = Direct(FloatReg 7)
val datalimit as (Direct datalimitReg) = Direct(DataReg 5)

val threeaddress = false

val ptrtemp as Direct ptrtemp' = Direct(AddrReg 5)
local 
    val inuse = ref false
    val datatmps = ref [arithtemp2,arithtemp3]
    val addrtmps = ref [ptrtemp]
    fun get (ref []) = ErrorMsg.impossible "m68: no more temp registers"
      | get (l as (ref (r::rs))) = (l := rs; r)
    fun free l r = l := r :: (!l)
in
    datatype regtype = DATA | ADDR
    fun getTmp DATA = get datatmps
      | getTmp ADDR = get addrtmps
    fun freeTmp (t as Direct(DataReg _)) = free datatmps t
      | freeTmp (t as Direct(AddrReg _)) = free addrtmps t

    fun withTmp kind f = 
	let val r = getTmp kind
	in
	    f r;
	    freeTmp r
	end
    val withATmp = withTmp ADDR
    val withDTmp = withTmp DATA
end

fun reg(Direct r) = r

fun newlabel() = Immedlab(V.newlabel())
(* DEBUG val newlabel = diag "newlabel" newlabel *) 
fun emitlab(i,Immedlab lab) = V.emitlab(i,lab)
fun define (Immedlab lab) = V.define lab

(* stuff for simulating PC relative addressing (not used on M68) *)
fun beginStdFn _ = ()

val align = V.align
val mark = V.mark

(* (uses r1' r2) is true if r2 contains r1' *)
fun uses r1' r2 = 
    let fun aux (Direct r) = r1' = r
	  | aux (PostInc r) = r1' = r
	  | aux (PreDec r) = r1' = r
	  | aux (Displace(r,_)) = r1' = r
	  | aux (Index(r,_,r',_)) = (r1' = r) orelse (r1' = r')
	  | aux _ = false
    in aux r2
    end

fun move (src as Direct(FloatReg _),dst as Direct(FloatReg _)) = V.fmovex(src,dst)
  | move (src as Direct(FloatReg _), _) = ErrorMsg.impossible 
    "m68/m68/move: Destination not a floating point register"
  | move (_, dst as Direct(FloatReg _)) = ErrorMsg.impossible 
    "m68/m68/move: Source not a floating point register"
  | move (Immedlab l, dest as Direct(AddrReg x)) = V.lea(Address l, dest)
  | move (Immedlab l, dest) = 
    withATmp (fn r => (V.lea(Address l, r); move (r,dest)))
  | move (Displace(DataReg(d), i), dest) =
    withATmp (fn r => (V.movl(Direct(DataReg(d)), r);
		       move (Displace(reg(r), i), dest)))
  | move (src, Address l) =
    withATmp (fn r => (V.lea(Address l, r);
		       move (src, Displace(reg(r), 0))))
  | move x = V.movl x
(* DEBUG val move = diag "move" move *) 

fun jmp (x as Displace _) = (* needed for vregs *)
    withATmp (fn r => (move (x,r); 
		       jmp r))
  | jmp x = V.jra(defer x)
(* DEBUG val jmp = diag "jmp" jmp *) 

fun testLimit() = V.cmpl(dataptr,datalimit)
fun decLimit n = V.subl(Immed n,datalimit) (* for polling *)

(* checkLimit (n):
 * Generate code to check the heap limit to see if there is enough free space
 * to allocate n bytes.
 *)

fun checkLimit (maxAllocSize, lab, mask, rlab, fregs) =
   let val lab' = V.newlabel()
       val r = getTmp ADDR
       val _ = (r <> ptrtemp) andalso 
	           ErrorMsg.impossible "m68:checkLimit: need ptrtemp"
    in V.comment ("check limit, max alloc = "^(makestring maxAllocSize)^"\n");
       if maxAllocSize >= 4096
	   then withDTmp (fn t => (V.movl (dataptr, t);
				   V.addl(Immed(maxAllocSize-4096), t);
				   V.cmpl(t, datalimit)))
	   else ();
       V.jgt(Address lab');
       (case fregs of
	    [] => (move (mask, arithtemp3);  (* destination must be d4 *)
		   move (lab,ptrtemp);
		   V.rts())
	  | _ => let val len = length fregs
		     val floatSz = 8
		     val desc = System.Tags.make_desc(len * floatSz, 
						      System.Tags.tag_string)
		     val retlab = V.newlabel()
		     fun forall ([],_,_) = ()
		       | forall (freg::rest,i,f) = 
			 (f (freg,i);
			  forall (rest,i+floatSz,f))
		     fun deposit r (fr,i) = V.fmoved(fr,Displace(r,i))
		     fun restore r (fr,i) = V.fmoved(Displace(r,i),fr)
		 in
		     (* build fp record *)
		     withDTmp (fn t => (move(dataptr,t);  (* align *)
					V.orl(Immed 4,t);  
					move(t,dataptr)));
		     move(Immed desc,Displace(dataptr',0));
		     forall (fregs,4,deposit dataptr');
		     V.addl(Immed 4,dataptr);
		      
		     (* save it in pseudo1 *)
		     move(dataptr,pseudo1);
		      
		     V.addl(Immed (floatSz * len),dataptr);
		     move(mask,arithtemp3);
		     move(Immedlab retlab,ptrtemp);
		     V.rts ();
		      
		     V.define retlab;
		     move(pseudo1,ptrtemp);
		     forall (fregs,0,restore ptrtemp');
		     testLimit();
		     jmp rlab    (* don't know what rlab is *)
		 end);
       V.define lab';
       freeTmp r
   end

val emitlong = V.emitlong
val realconst = V.realconst
val emitstring = V.emitstring

fun ashl(s as Displace _, r, d) = (* needed for vregs *)
    withDTmp (fn t => (move (s,t); ashl(t,r,d)))
  | ashl(s as Immed k, r, d as Direct(DataReg _)) =
    (if r<>d then move (r,d) else (); 
     if k>8 then withDTmp (fn t => (move (s, t); V.asll(t, d)))
     else V.asll(s,d))
  | ashl(s as Direct(DataReg _),r,d as Direct(DataReg _)) =
     if r<>d then if s<>d then (move (r,d);
				V.asll(s,d))
		          else withDTmp (fn t => (move (r,t);
						  V.asll(s,t);
						  move (t,d)))
     else V.asll(s,d)
  | ashl(s as Direct(DataReg _),r,d) =
    withDTmp (fn t => (move (r,t); V.asll(s,t); move (t,d)))
  | ashl(s as Immed k,r,d) =
    withDTmp (fn t => let fun f(k) = if k=0 then () 
				     else if k>8 then (V.asll(Immed 8,t);
						       f(k-8))
				     else V.asll(Immed k, t)
		      in  move (r,t);
			  f k;
			  move (t,d)
		      end)

(* DEBUG val ashl = diag "ashl" ashl *) 

fun ashr(s as Displace _, r, d) = (* needed for vregs *)
    withDTmp (fn t => (move (s,t); ashr(t,r,d)))
  | ashr(s as Immed k, r, d as Direct(DataReg _)) =
    (if r<>d then move (r,d) else (); 
     if k>8 then withDTmp (fn t => (move (s, t); V.asrl(t, d)))
     else V.asrl(s,d))
  | ashr(s as Direct(DataReg _),r,d as Direct(DataReg _)) =
     if r<>d then if s<>d then (move (r,d);
				V.asrl(s,d))
	          else withDTmp (fn t => (move (r,t);
					  V.asrl(s,t);
					  move (t,d)))
     else V.asrl(s,d)
  | ashr(s as Direct(DataReg _),r,d) =
    withDTmp (fn t => (move (r,t); V.asrl(s,t); move (t,d)))
  | ashr(s as Immed k,r,d) =
    withDTmp (fn t => let fun f(k) = if k=0 then () 
				     else if k>8 then (V.asrl(Immed 8,t);
						       f(k-8))
				     else V.asrl(Immed k, t)
		      in move (r,t);
			  f k;
			  move (t,d)
		      end)

(* DEBUG val ashr = diag "ashr" ashr *) 

fun jmpindexb(lab,Direct r) = V.jra(Index(PC,2,r,Byte))
  | jmpindexb _ = ErrorMsg.impossible "bad args to jmpindexb in m68.sml"
(* DEBUG val jmpindexb = diag "jmpindexb" jmpindexb *) 

fun record(vl, z) =
    let open CPS
	val ptmp = getTmp ADDR
	fun f (Direct r, SELp(j,p)) = f(Displace(r,j*4),p)
	  | f (Immedlab l, p) = (V.lea(Address l, ptmp); f(ptmp,p))
	  | f (x,OFFp 0) = V.movl(x, PostInc dataptr')
	  | f (Direct r, OFFp j) = (V.lea(Displace(r,j*4),ptmp);
				    f(ptmp,OFFp 0))
	  | f (x,p) = (V.movl(x,ptmp); f(ptmp,p))
      in
	  app f vl;
	  (case z of
	       Displace _ => 
		   (V.lea(Displace(dataptr',~4*(List.length(vl)-1)),ptmp);
		    V.movl(ptmp,z))
	     | _ => V.lea(Displace(dataptr',~4*(List.length(vl)-1)),z));
	  freeTmp ptmp
    end

  (* recordStore(x, y, alwaysBoxed) records a store operation 
   * into mem[x+2*(z-1)].
   * The flag alwaysBoxed is true if the value stored is guaranteed 
   * to be boxed.
   *)
  (* NOTE: eventually we can optimize the case where alwaysBoxed = false *)
  fun recordStore arg = 
      let val ptmp = getTmp ADDR
	  fun aux (x, y, alwaysBoxed) = 
	      let fun storeVectorUpdate r = 
		  (V.movl (r,Displace(dataptr',0));
		   V.movl (storeptr,Displace(dataptr',4));
		   V.movl (dataptr,storeptr);
		   V.addl (Immed 8,dataptr))
	      in
		  (case (x, y)
		       of (_, Immed 1) => storeVectorUpdate x
		     | (Direct r, Immed i) => 
			   let val t = getTmp DATA
			   in V.lea(Displace(r, 2*(i-1)), ptmp);
			      V.movl(ptmp, t);
			      storeVectorUpdate t;
			      freeTmp t
			   end
		     | (Displace _,_) => 
			   (V.movl (x,ptmp);
			    aux (ptmp,y,alwaysBoxed))
		     | (_,Displace _) => 
			   withDTmp (fn t => (V.movl (y,t);
					      aux (x,t,alwaysBoxed)))
		     | (Direct r1, Direct r2) => 
			   withDTmp 
			     (fn t => (V.lea(Index(r1, ~2, r2, Word), ptmp);
				       V.movl(ptmp, t);
				       storeVectorUpdate t))
		     | _ => ErrorMsg.impossible "[M68CM.recordStore]")
	      end
      in
	  aux arg;
	  freeTmp ptmp
      end (* recordStore *)

fun fprecord(tag,vl,z) = 
    let open CPS
	val ptmp = getTmp ADDR
	val floatSz = 8
	fun f [] = ()
	  | f ((Direct r,SELp(j,OFFp 0))::rest) = 
	    (V.fmoved(Displace(r,j*floatSz),fptempreg);
	     V.fmoved(fptempreg,PostInc dataptr');
	     f rest)
	  | f ((Direct r,SELp(j,p))::rest) = f ((Displace(r,j*4),p)::rest)
	  | f ((fr as Direct(FloatReg _),OFFp 0)::rest) = 
	    (V.fmoved(fr,PostInc dataptr');
	     f rest)
	  | f ((ea,p)::rest) = 
	    (move (ea,ptmp);
	     f ((ptmp,p)::rest))
	  | f _ = ErrorMsg.impossible "invalid path in fprecord"
    in
	withDTmp (fn t => (move(dataptr,t);  (* align *)
			   V.orl(Immed 4,t);  
			   move(t,dataptr)));
	V.movl(tag,PostInc dataptr');
	f vl;
	(case z of
	     Displace _ =>
		 (V.lea(Displace(dataptr',~8*(List.length(vl))),ptmp);
		  V.movl(ptmp,z))
	   | _ => V.lea(Displace(dataptr',~8*(List.length(vl))),z));
	freeTmp ptmp
    end


fun recordcont _ = ErrorMsg.impossible "[M68CM: recordcont not implemented yet]"

fun select(i, Direct r, s) = move (Displace(r,i*4),s)
  | select(0, a as Immedlab _, s) = move (defer a,s)
  | select(i, a, s) = 
    withATmp (fn r => (move (a,r);
		       select(i,r,s)))
  | select _ = ErrorMsg.impossible "[M68CM.select]"

fun loadpseudo (x, i) = () (* ErrorMsg.impossible "loadpseudo not supported yet" *)
fun storepseudo (x, i) = () (* ErrorMsg.impossible "storepseudo not supported yet" *)

fun offset(i, r as Displace _,s) = (* needed for vregs *)
    withDTmp (fn t => (move (r,t);
		       offset(i,t,s)))
  | offset(i,r,s as Displace _) = (* needed for vregs *)
    withATmp (fn p => (offset(i,r,p); 
		       move (p,s)))
  | offset(i, Direct r, s) = V.lea(Displace(r,i*4),s)
(* DEBUG val select = diag "select" select *) 
(* DEBUG val offset = diag "offset" offset *) 

exception Three
fun three opcode (a as Direct(AddrReg _), b as Direct(AddrReg _),
		  c as Direct(AddrReg _)) =
    withDTmp (fn t => (three opcode(a,b,t); 
		       move (t,c)))
  | three opcode (a,b,c) = 
	    if b=c then opcode(a,b) 
	    else if a=c then 
		withDTmp (fn t => (move (a,t); 
				   three opcode(t,b,c)))
	    else (move (b,c); opcode(a,c))

fun threet opcode (a,b,c as Direct(AddrReg _)) =
    withDTmp (fn t => (threet opcode(a,b,t); 
		       move (t,c)))
  | threet opcode (a,b,c) = 
	    if b=c then (opcode(a,b); V.trapv())
	    else if a=c then 
		withDTmp (fn t => (move (a,t); 
				   threet opcode(t,b,c)))
	    else (move (b,c); opcode(a,c); V.trapv())

fun three' opcode (a as Immed _,b,c as Direct(DataReg _)) =
	    three opcode(a,b,c)
  | three' opcode (a as Direct(AddrReg _),b,c) =
    withDTmp (fn t1 => withDTmp (fn t2 => (move (b,t2); 
					   move (a,t1);
					   opcode(t1,t2); 
					   move (t2,c))))
  | three' opcode (a,b,c) =
    withDTmp (fn t => (move (b,t); 
		       opcode(a,t); 
		       move (t,c)))

(* abstract orb and xorb ?? *)
fun orb(a as Immed k,b,c as Direct(DataReg _)) =
    if k<65538
    then if k<=0
         then raise Match
	 else if b=c then V.orl(a,b) else (move (b,c); V.orl(a,c))
    else withDTmp (fn t => (move (a,t);
			    if b=c then V.orl(t,b) 
			    else (move (b,c); V.orl(t,c))))
  | orb(a as Direct(AddrReg _),b,c) =
    withDTmp (fn t1 => withDTmp (fn t2 => (move (b,t2); 
					   move (a,t1);
					   V.orl(t1,t2); 
					   move (t2,c))))
  | orb(a as Displace _,b,c) = 
    withDTmp (fn t1 => withDTmp (fn t2 => (move (b,t2); 
					   move (a,t1);
					   V.orl(t1,t2); 
					   move (t2,c))))
  | orb(a as Immed k,b,c) =
    if k<65536
    then if k<=0
         then raise Match
	 else withDTmp (fn t => (move (b,t); 
				 V.orl(a,t); 
				 move (t,c)))
    else withDTmp (fn t1 => withDTmp (fn t2 => (move (a,t1);
						move (b,t2);
						V.orl(t1,t2);
						move (t2,c))))
  | orb(a,b,c) = 
    withDTmp (fn t => (move (b,t); 
		       V.orl(a,t); 
		       move (t,c)))

fun xorb(a as Immed k,b,c as Direct(DataReg _)) =
    if k<65536
    then if k<=0
         then raise Match
	 else if b=c then V.eorl(a,b) else (move (b,c); V.eorl(a,c))
    else withDTmp (fn t => (move (a,t);
			    if b=c then V.eorl(t,b) 
			    else (move (b,c); V.eorl(t,c))))
  | xorb(a as Direct(AddrReg _),b,c) =
    withDTmp (fn t1 => withDTmp (fn t2 => (move (b,t2); 
					   move (a,t1);
					   V.eorl(t1,t2); 
					   move (t2,c))))
  | xorb(a as Displace _,b,c) = 
    withDTmp (fn t1 => withDTmp (fn t2 => (move (b,t2); 
					   move (a,t1);
					   V.eorl(t1,t2); 
					   move (t2,c))))
  | xorb(a as Immed k,b,c) =
    if k<65538
    then if k<=0
         then raise Match
	 else withDTmp (fn t => (move (b,t); 
				 V.eorl(a,t); 
				 move (t,c)))
    else withDTmp (fn t1 => withDTmp (fn t2 => (move (a,t1);
						move (b,t2);
						V.eorl(t1,t2);
						move (t2,c))))
  | xorb(a,b,c) = 
    withDTmp (fn t => (move (b,t); 
		       V.eorl(a,t); 
		       move (t,c)))
fun notb(a,b) = 
    withDTmp (fn t => (move (Immed ~1,t); 
		       V.subl(a,t);
		       move (t,b)))

val andb = three' V.andl

(* wrappers needed for vregs *)
fun wrap2 f (src,dst as Direct(DataReg _)) = f (src,dst)
  | wrap2 f (src,dst) = withDTmp (fn t => (V.movl(dst,t);
					   f (src,t);
					   V.movl (t,dst)))

fun wrap3 f (x,y,z as Displace _) = 
    withDTmp (fn t => (f (x,y,t);
		       V.movl (t,z)))
  | wrap3 f x = f x    

val add = wrap3 (three V.addl)
val addt = wrap3 (threet V.addl)
val op sub = wrap3 (three V.subl)
val subt = wrap3 (threet V.subl)
val mult = wrap2 (fn x => (V.mull x; V.trapv()))

fun divl (src,dst as Direct(DataReg _)) = V.divl (src,dst)
  | divl (src,dst) = 
    withDTmp (fn t => (move (dst,t);
		       V.divl (src,t);
		       move (t,dst)))

val divt = (wrap2 (fn x => (divl x; V.trapv())))  
(* bug?  test for overflow/divide-by-zero? *)

(* Needed for Word32  *)
val immed32 = V.Immed32
val mulu = mult
val divtu = divt
val lshr = ashr

exception Fetchindexb
(* fetchindexb(x,y,z) fetches a byte: y <- mem[x+z], *)
fun fetchindexb (x as Displace _, y, z) = (* needed for vregs *)
    withATmp (fn r => (move (x,r); 
		       fetchindexb(r,y,z)))
  | fetchindexb (Direct (x as AddrReg _), y, z) = 
    withDTmp (fn t => (move (Immed 0, t);
		       case z of 
			   (Immed indx) => V.movb(Displace(x, indx), t)
			 | (Direct indx) => V.movb(Index(x, 0, indx, Byte), t);
		       move (t,y)))

(* storeindexb(x,y,z) stores a byte: mem[y+z] <- x. *)
fun storeindexb (x as Displace _, y, i) = (* needed for vregs *)
    withDTmp (fn t => (move (x,t); 
		       storeindexb(t,y,i)))
  | storeindexb (x as Direct(AddrReg _), y, i) = (* needed for vregs *)
    withDTmp (fn t => (move (x,t); 
		       storeindexb(t, y, i)))
  | storeindexb (x, y as Displace _, i) = (* needed for vregs *)
    withATmp (fn r => (move (y,r); 
		       storeindexb(x,r,i)))
  | storeindexb (x, Direct (y as AddrReg _), Direct indx) = 
    V.movb(x, Index(y, 0, indx, Byte))
  | storeindexb (x, Direct y, Immed indx) = V.movb(x, Displace(y, indx))
    
(* DEBUG val fetchindexb = diag "fetchindexb" fetchindexb *)
(* DEBUG val storeindexb = diag "storeindexb" storeindexb *)

fun fetchindexl(x,y,z as Displace _) =   (* needed for vregs *)
    withDTmp (fn t => (move (z,t); 
		       fetchindexl(x,y,t)))
  | fetchindexl(x as Displace _,y,z) =   (* needed for vregs *)
    withATmp (fn r => (move (x,r); 
		       fetchindexl(r,y,z)))
  | fetchindexl(Direct x,y,Immed k) = move (Displace(x,k+k-2),y)
  | fetchindexl(Direct (x as AddrReg _),y,Direct z) = 
    move (Index(x,~2,z,Word),y)
  | fetchindexl(Immedlab lab, y, Direct z) = 
	    (* this is a hack, since it depends on lab being PC+6 *)
		    move (Index(PC,4,z,Word), y);
(* DEBUG val fetchindexl = diag "fetchindexl" fetchindexl *) 

fun storeindexl(x, y as Displace _,Immed 1) =   (* needed for vregs *)
    withDTmp (fn t => (move (y,t); 
		       move (x,defer t)))
  | storeindexl(x, y as Displace _,Immed k) =   (* needed for vregs *)
    let val t as Direct t' = getTmp DATA
    in move (y,t); 
       move (x,Displace(t',k+k-2));
       freeTmp t
    end
  | storeindexl(x, y as Displace _,z) =         (* needed for vregs *)
    withATmp (fn r => (move (y,r); 
		       storeindexl(x,r,z)))
  | storeindexl(x, y, z as Displace _) = (* needed for vregs *)
    withDTmp (fn t => (move (z,t); 
		       storeindexl(x,y,t)))
  | storeindexl(x, y, Immed 1) =  move (x, defer y)
  | storeindexl(x, Direct y, Immed k) = move (x, Displace(y,k+k-2))
  | storeindexl(x, Direct (y as AddrReg _), Direct z) = 
    move (x,Index(y,~2,z,Word))
(* DEBUG val storeindexl = diag "storeindexl" storeindexl *) 

fun float f (fp1,fp2,fp3) = 
    if fp2 <> fp3 then 
	(V.fmovex(fp1,fp3); f(fp2,fp3))
    else (V.fmovex(fp1,fptempreg); f(fp2,fptempreg); V.fmovex(fptempreg,fp3))

fun loadfloat(src as Displace _,dst) = (* needed for vregs *)
    withATmp (fn r => (move (src,r);
		       loadfloat(r,dst)))
  | loadfloat(src,dst) = case dst 
			   of Direct(FloatReg fp) => V.fmoved(defer src, dst)
			    | _ => ErrorMsg.impossible
				   "m68/m68/loadfloat: Bad destination register"
fun storefloat (src as Direct(FloatReg fp), dst) = (
      V.movl(Immed(System.Tags.desc_reald), PostInc dataptr');
      V.movl(dataptr, dst);
      V.fmoved(src, PostInc dataptr'))
  |storefloat  _ = ErrorMsg.impossible "m68/m68/storefloat: Bad source register"
	  

val fmuld = float V.fmulx
val fdivd = float V.fdivx
val faddd = float V.faddx
val fsubd = float V.fsubx

fun fnegd (src as Direct(FloatReg _), dst as Direct(FloatReg _)) = V.fnegx(src, dst)
  | fnegd _ = ErrorMsg.impossible "m68/m68/fnegd"

fun fabsd (src as Direct(FloatReg _), dst as Direct(FloatReg _)) = V.fabsx(src, dst)
  | fabsd _ = ErrorMsg.impossible "m68/m68/fabsd"

fun cvti2d (src, dst as Direct(FloatReg _)) = V.fmovel(src, dst)
  | cvti2d _ = ErrorMsg.impossible "m68/m68/cvti2d"

					(* y <- mem[x+4*(z-1) *)
fun fetchindexd(x as Displace _,y,z) = (* needed for vregs *)
    withATmp (fn r => (move (x,r); 
		       fetchindexd(r,y,z)))
  | fetchindexd (x,y,z as Displace _) = (* needed for vregs *)
    withDTmp (fn t => (move (z,t); 
		       fetchindexd(x,y,t)))
  | fetchindexd(Direct x', y as Direct(FloatReg fp), z) = 
    (case z 
      of Immed i => V.fmoved(Displace(x',4*(i-1)), y)
       | Direct z' => V.fmoved(Index(x',~4,z',Long), y)
       | _ => ErrorMsg.impossible "m68/m68/fetchindexd: bad index")
  | fetchindexd _ = ErrorMsg.impossible "m68/m68/fetchfloat: bad dest. reg"

					(* mem[y+4*(z-1)] <- x *)
fun storeindexd(x, y as Displace _, z) = (* needed for vregs *)
    withATmp (fn r => (move (y,r); 
		       storeindexd(x,r,z)))
  | storeindexd(x, y, z as Displace _) = (* needed for vregs *)
    withDTmp (fn t => (move (z,t); 
		       storeindexd(x,y,t)))
  | storeindexd(x as Direct(FloatReg fp), Direct (y' as AddrReg _), z) = 
    (case z 
      of Immed i => V.fmoved(x,Displace(y',4*(i-1)))
       | Direct z' => V.fmoved(x,Index(y',~4,z',Long))
       | _ => ErrorMsg.impossible "m68/m68/storeindexd: bad index")
  | storeindexd _ = ErrorMsg.impossible "m68/m68/storeindexd: bad src reg"

fun cbranch NEQ = V.jne
  | cbranch EQL = V.jeq
  | cbranch LEQ = V.jle
  | cbranch GEQ = V.jge
  | cbranch LSS = V.jlt
  | cbranch GTR = V.jgt
(* use signed jmps for Word32 for now; this is wrong *)
  | cbranch LEU = V.jle
  | cbranch GEU = V.jge
  | cbranch LTU = V.jlt
  | cbranch GTU = V.jgt
(*  | cbranch _ = fn _ => ()		(* support for Word32 *)
*)

fun fbranch NEQ = V.fjne
  | fbranch EQL = V.fjeq
  | fbranch LEQ = V.fjle
  | fbranch GEQ = V.fjge
  | fbranch LSS = V.fjlt
  | fbranch GTR = V.fjgt

fun rev LEQ = GEQ
  | rev GEQ = LEQ
  | rev LSS = GTR
  | rev GTR = LSS
  | rev NEQ = NEQ
  | rev EQL = EQL


fun cmpl (op1,op2 as Displace(AddrReg _,_)) = (* needed for vregs *)
    withDTmp (fn t => (V.movl (op2,t);
		       V.cmpl (op1,t)))
  | cmpl x = V.cmpl x

fun ibranch (cond, op1 as Immed _, op2, label) =
	(cmpl(op1, op2); cbranch (rev cond) (defer label))
  | ibranch (cond, op1, op2, label) =
	(cmpl(op2, op1); cbranch cond (defer label))

(* rangeChk (a, b, lab):  pc <- lab if ((a < 0) or (b <= a)) *)
fun rangeChk (op1 as Immed _, op2, label) = (cmpl(op1, op2); V.jls (defer label))
  | rangeChk (op1, op2, label) = (cmpl(op2, op1); V.jcc (defer label))
(* DEBUG val rangeChk = diag "rangeChk" rangeChk *)

fun fbranchd (cond, op1, op2, label) = (V.fcmpx(op2,op1); 
				       fbranch cond (defer label))


fun bbs (x,dest as Direct(AddrReg _) ,l) = 
    withDTmp (fn t => (move (dest,t);
		       bbs(x,t,l)))
  | bbs (x,dest as Displace _,l) = 
    (* note the goofy semantics of btst: if dest is an address,
     * a _byte_ is feteched and the x'th bit tested; if dest is
     * a data reg then any bit in the dest word can be tested.
     *) 
    (* possible optimization: if |x| <= 8 bits then btst(x,Displace(r,i+3)) *)
    withDTmp (fn t => (move (dest,t);
		       bbs(x,t,l)))
  | bbs (x,y,l) = (V.btst(x,y); V.jne(defer l))
(* DEBUG val bbs = diag "bbs" bbs *) 

val immed = Immed

val comment = V.comment

end
