(* hppaAsm.sml
 *
 * COPYRIGHT (c) 1996 Bell Laboratories.
 *)

structure HppaAsmEmitter : EMITTER_NEW = struct
  structure I = HppaInstruction
  structure T = System.Tags

  fun ms n = if n<0 then ("-"^ Int.toString (~n)) else Int.toString n

  fun error msg = ErrorMsg.impossible ("HppaAsEmitter." ^ msg)

  fun emit s = output(!AsmStream.asmOutStream,s)

  val loc = ref 0
 
  fun incLoc n = loc := (!loc) + n

  fun emitLong n = (emit (".long\t" ^ ms n ^ "\n"); incLoc 4)

  fun align () = 
    case Word.andb(Word.fromInt(!loc), 0w7) 
     of 0w0 => ()
      | 0w4 => emitLong 0
      | _ => error "align"

  fun mark () =
    emitLong (T.make_desc((!loc + 4) quot 4,T.tag_backptr))

  fun emitString (lab, sz, s) = (
	mark();
	emitLong(T.make_desc(sz,T.tag_string));
	emit (concat[
	  Label.nameOf lab, ":\t.ascii ", AsmUtil.printableString s, "\n"]);
	incLoc (size s))

  fun emitReal (lab,f) = (
	align();
	mark ();
	emitLong(T.make_desc(size f,T.tag_reald));
	emit(Label.nameOf lab ^ ":" ^ "\t.double " ^ f ^ "\n");
	incLoc 8)

  fun emitJmpTable(base,targets) = 
    (emit(".jumpTable\n" ^ Label.nameOf base ^ ":\n");
     app (fn lab => emit("\t\t" ^ Label.nameOf lab ^ "\n")) targets;
     incLoc(4*length targets))

  fun defineLabel(lab as Label.Label{comment,...}) = 
        (emit(Label.nameOf lab ^ ":\t\t/* at " ^ AsmUtil.itoa(!loc));
	 emit(if comment = "" then "*/\n"
	      else  " = lvar: " ^ comment ^ "*/\n"))

  fun comment msg = emit ("/*" ^ msg ^ "*/\n")

  fun init size = (comment ("Code Size = " ^ ms size);
		   loc := 0)

    
  fun emitInstr(instr,regmaps) = let
    val regmap		= hd regmaps
    val fregmap		= hd(tl regmaps)
    fun emitReg r	= emit ("r" ^ ms(Array.sub(regmap, r)))
    fun emitFreg f	= emit ("f" ^ ms(Array.sub(fregmap, f)))
    fun emitCreg c      = emit ("f" ^ ms c)
    fun emitLabel lab	= emit (Label.nameOf lab)
    fun comma() 	= emit ", "
    fun tab()           = emit "\t"
    fun newline ()      = emit "\n"
    fun paren f         = (emit "("; f(); emit ")")
    fun cond I.EQ  = "="
      | cond I.LT  = "<"
      | cond I.LE  = "<="
      | cond I.LTU = "<<"
      | cond I.LEU = "<<="
      | cond I.NE  = "<>"
      | cond I.GE  = ">="
      | cond I.GT  = ">"
      | cond I.GTU = ">>"
      | cond I.GEU = ">>="
    fun store(st, b, d, r) = 
      (emit st; emitReg r; comma(); emit(ms d); paren(fn () => emitReg b))
    fun fstore(fst, b, d, r) = 
      (emit fst; emitFreg r; comma(); emit(ms d); paren(fn () => emitReg b))
    fun fstorex(fstx, b, x, r) = 
      (emit fstx; emitFreg r; emitReg x; paren(fn () => emitReg b))
    fun loadx(ld, r1, r2, t) = 
      (emit ld; emitReg r2; paren(fn () => emitReg r1); comma(); emitReg t)
    fun floadx(flx, b, x, t) = 
      (emit flx; emitReg x; paren(fn () => emitReg b); comma(); emitFreg t)
    fun arith(a, r1, r2, t) = 
      (emit a; emitReg r1; comma(); emitReg r2; comma(); emitReg t)
    fun loadi(ld, i, r, t) = 
      (emit ld; emit(ms i); paren(fn () => emitReg r); comma(); emitReg t)
    fun fload(fl, b, d, t) = 
      (emit fl; emit(ms d); paren(fn () => emitReg b); comma(); emitFreg t)
    fun arithi(ai, i, r ,t) =
      (emit ai; emit(ms i); comma(); emitReg r; comma(); emitReg t)
    fun cmp{cc, r1, r2, t} = arith("comclr," ^ cond cc ^ "\t", r1, r2, t)
    fun shiftv(sv, r, len, t) = 
      (emit sv; emitReg r; comma(); emit(ms len); comma(); emitReg t)
    fun shift(s, r, p, len, t) = 
      (emit s; emitReg r; comma(); emitReg p; comma(); emit(ms len); emitReg t)
    fun emitCmp I.COMBT  = emit "combt,"
      | emitCmp I.COMBF  = emit "combf,"
    fun emitCmpi I.COMIBT = emit "comibt,"  
      | emitCmpi I.COMIBF = emit "comibf," 
    fun bcond{cmp, bc, r1, r2, t, f} = 
      (emitCmp cmp;  emit (cond bc ^ ",n\t");
       emitReg r1; comma(); emitReg r2; comma(); emitLabel t)
    fun bcondi{cmpi, bc, i, r2, t, f} = 
      (emitCmpi cmpi; emit (cond bc ^ ",n\t");
       emit(ms i); comma(); emitReg r2; comma(); emitLabel t)
    fun emitLExp(I.POSLAB(lab,k)) =
	  (emit"("; emit(ms k); emit"+"; emitLabel lab; emit")" )
      | emitLExp(I.NEGLAB(lab,k)) =
	  (emit"("; emit(ms k); emit"-"; emitLabel lab; emit")" )
    fun emitOperand(I.HIMaskop rl) = (emit "HI("; app emitReg rl; emit ")")
      | emitOperand(I.LOMaskop rl) = (emit "LO("; app emitReg rl; emit ")")
      | emitOperand(I.HILabExp lexp) = (emit "HI("; emitLExp lexp; emit ")")
      | emitOperand(I.LOLabExp lexp) = (emit "LO("; emitLExp lexp; emit ")")
      | emitOperand(I.LabExp lexp) = emitLExp lexp
      | emitOperand(I.IMMED  i) = emit (ms i)

    fun farith{fa, r1, r2, t} = let
      val oper = case fa of I.FADD => "fadd\t" | I.FSUB => "fsub\t"
			  | I.FDIV => "fdiv\t" | I.FMPY => "fmpy\t"
			  | I.XMPYU => "xmpyu\t"
    in
      emit oper; emitFreg r1; comma(); emitFreg r2; comma(); emitFreg t
    end

    fun funary(fu, f, t) = (emit fu; emitFreg f; comma(); emitFreg t)
  in
    case instr
     of I.STORE{st=I.STW, b, d, r}	=> store("stw\t", b, d, r)
      | I.STORE{st=I.STH, b, d, r}	=> store("sth\t", b, d, r)
      | I.STORE{st=I.STB, b, d, r}	=> store("stb\t", b, d, r)

      | I.ARITH{a=I.LDWX, r1, r2, t}	=> loadx("ldwx\t", r1, r2, t)
      | I.ARITH{a=I.LDHX, r1, r2, t}	=> loadx("ldhx\t", r1, r2, t)
      | I.ARITH{a=I.LDBX, r1, r2, t}	=> loadx("ldbx\t", r1, r2, t)
      | I.ARITH{a=I.ADD,  r1, r2, t}	=> arith("add\t", r1, r2, t)
      | I.ARITH{a=I.ADDO,  r1, r2, t}	=> arith("addo\t", r1, r2, t)
      | I.ARITH{a=I.SH1ADD,  r1, r2, t}	=> arith("sh1add\t", r1, r2, t)
      | I.ARITH{a=I.SH1ADDO, r1, r2, t} => arith("sh1addo\t", r1, r2, t)
      | I.ARITH{a=I.SUB,  r1, r2, t}	=> arith("sub\t", r1, r2, t)
      | I.ARITH{a=I.SUBO,  r1, r2, t}	=> arith("subo\t", r1, r2, t)
      | I.ARITH{a=I.OR,  r1, r2, t}	=> arith("or\t", r1, r2, t)
      | I.ARITH{a=I.XOR,  r1, r2, t}	=> arith("xor\t", r1, r2, t)
      | I.ARITH{a=I.AND,  r1, r2, t}	=> arith("and\t", r1, r2, t)

      | I.ARITHI{ai=I.LDB, i, r, t}	=> loadi("ldb\t", i, r, t)
      | I.ARITHI{ai=I.LDH,  i, r, t}	=> loadi("ldh\t", i, r, t)
      | I.ARITHI{ai=I.LDW, i, r, t}	=> loadi("ldw\t", i, r, t)
      | I.ARITHI{ai=I.ADDI, i, r, t}	=> arithi("addi\t", i, r, t)
      | I.ARITHI{ai=I.ADDIO, i, r, t}	=> arithi("addio\t", i, r, t)
      | I.ARITHI{ai=I.SUBI, i, r, t}	=> arithi("subi\t", i, r, t)
      | I.ARITHI{ai=I.SUBIO, i, r, t}	=> arithi("subio\t", i, r, t)

      | I.COMCLR arg			=> cmp arg
 
      | I.SHIFTV{sv=I.VEXTRU, r, len, t}=> shiftv("vextru\t",  r, len, t)
      | I.SHIFTV{sv=I.VEXTRS, r, len, t}=> shiftv("vextrs\t",  r, len, t)
      | I.SHIFTV{sv=I.ZVDEP, r, len, t}	=> shiftv("zvdep\t", r, len, t)

      | I.SHIFT{s=I.EXTRU, r, p, len, t}=> shift("extru\t", r, p, len, t)
      | I.SHIFT{s=I.EXTRS, r, p, len, t}=> shift("extrs\t", r, p, len, t)
      | I.SHIFT{s=I.ZDEP, r, p, len, t}	=> shift("zdep\t", r, p, len, t)

      | I.BCOND arg			=> bcond arg
      | I.BCONDI arg			=> bcondi arg
      | I.B(lab)			=> (emit "b,n\t"; emitLabel lab)
      | I.FBCC{t, ...}			=> (emit "fbcc,n\t"; emitLabel t)

      | I.CALL{t, defs, uses} => let
          fun prList([]) = ()
	    | prList(n::ns) = (emit(ms n); emit " "; prList ns)
        in
          (emit "ble,n\t"; 
	   emit(ms 0); paren(fn () => (emit "%sr5,"; emitReg t));
	   emit "\tdefs="; prList (#1 defs); emit " uses="; prList (#1 uses))
	end
      | I.BV{x, b, ...} => 
	  (emit "bv,n\t"; emitReg x; paren(fn () => emitReg b))
      | I.LDIL{i, t} => (emit "ldil\t"; emitOperand i; comma(); emitReg t)
      | I.LDO{i,b,t} => 
	  (emit "ldo\t";  emitOperand i; paren(fn () => emitReg b); 
	   comma(); emitReg t)
      | I.LADDR{i, b, t} => 
	  (emit "laddr\t"; emitOperand i; comma(); emitReg t)
      | I.MTCTL{r,t} => (emit "mtctl\t"; emitReg r; emitCreg t)
      | I.FSTORE{fst, b, d, r} => 
	  (case fst 
	    of I.FSTDS => fstore("fstds\t", b, d, r)
	     | I.FSTWS => fstore("fstws\t", b, d, r)
	  (*esac*))
      | I.FSTOREX{fstx, b, x, r} => 
	  (case fstx
	    of I.FSTDX => fstorex("fstdx\t", b, x, r)
	     | I.FSTWX => fstorex("fstwx\t", b, x, r)
	  (*esac*))
      | I.FLOAD{fl=I.FLDDS, b, d, t} => fload("fldds\t", b, d, t)
      | I.FLOAD{fl=I.FLDWS, b, d, t} => fload("fldws\t", b, d, t)
      | I.FLOADX{flx=I.FLDDX, b, x, t} => floadx("flddx\t", b, x, t)
      | I.FLOADX{flx=I.FLDWX, b, x, t} => floadx("fldwx\t", b, x, t)
      | I.FARITH arg => farith arg
      | I.FUNARY{fu=I.FCPY, f, t} => funary("fcpy\t", f, t)
      | I.FUNARY{fu=I.FABS, f, t} => funary("fabs\t", f, t)
      | I.FUNARY{fu=I.FCNVXF, f, t} => funary("fcnvxf\t", f, t)
      | I.FCMP(cc,f1,f2) => 
	  (emit ("fcmp," ^ cond cc ^"\t"); emitFreg f1; comma(); emitFreg f2)
      | I.FTEST => emit "ftest"
      | I.BREAK(i, j) => emit ("break(" ^ ms i ^ ", " ^ ms j ^ ")")
      | I.NOP => emit "nop"
    (*esac*);
    emit "\n";
    incLoc 4
  end

end



