(* scstatenv.sml
 *
 * COPYRIGHT (c) 1996 Bell Laboratories.
 *)

structure SCStaticEnv : SCSTATICENV =
struct

  structure M = Modules and I = ModuleId and V = Variables and T = Types
      and U = ModuleUtil

   datatype bdg = STRb of M.Structure
	        | SIGb of M.Signature
 	        | FCTb of M.Functor
		| FSIGb of M.FctSignature
		| TYCb of Types.tycon

  type modmap = (I.modId * bdg) list Stamps.stampMap

  type staticEnv = StaticEnv.staticEnv * (modmap list)
  type environment = {static: staticEnv, dynamic: DynamicEnv.dynenv}

  fun unSC(x,y) = x

  fun atop((e1,m1),(e2,m2)) = (StaticEnv.atop(e1,e2), m1 @ m2)

  fun layerEnv({static=a,dynamic=b},{static=c,dynamic=d}) =
      {static=atop(a,c), dynamic=DynamicEnv.atop(b,d)}

   fun getstamp m =
            case m of I.SIMPLEid s => s
		    | I.INSTANCEid {origin,...} => origin
		    | I.SIGid s => s
		    | I.FCTid s => s
		    | I.FCT_INSTANCEid{fct,...} => getstamp fct
		    | I.FSIGid{body,...} => body
		    | I.TYCid s => s

   val isEx = Stamps.isExternal

   fun isExternal m =
       case m of I.SIMPLEid s => isEx s
               | I.INSTANCEid {origin,sign} => isEx origin andalso isEx sign
	       | I.SIGid s => isEx s
	       | I.FCTid s => isEx s
	       | I.FCT_INSTANCEid{fct,fsig} => 
		       isExternal fct andalso isExternal fsig
	       | I.FSIGid{argument,body} => isEx argument andalso isEx body
	       | I.TYCid s => isEx s

  exception SCStatEnv

  fun genlook get (_,modmaps) id =
      let 
          fun look((id',b)::rest) = if I.equalId(id,id')
				     then SOME(get b)
			             else look rest
            | look nil = raise SCStatEnv
	  fun lookall(m1::rest) = (look(Stamps.applyMap(m1,getstamp id))
	                           handle SCStatEnv => lookall rest)
            | lookall nil = NONE
      in lookall modmaps
     end

  val lookSTR = genlook(fn STRb x =>x)
  val lookSIG = genlook(fn SIGb x =>x)
  val lookFCT = genlook(fn FCTb x =>x)
  val lookFSIG = genlook(fn FSIGb x =>x)
  val lookTYC = genlook(fn TYCb x =>x)

  val debugging = ref false

  fun ifdebug f x= if !debugging then f x else ()


  val printst = print o Stamps.stampToString

  fun printid(I.SIMPLEid st) = (print "SIMPLEid("; printst st; print ")")
    | printid(I.INSTANCEid{origin,sign}) =
        (print "INSTANCEid{origin="; printst origin; print ",sign=";
	 printst sign; print ")")
    | printid(I.SIGid st) = (print "SIGid("; printst st; print ")")
    | printid(I.FCTid st) = (print "FCTid("; printst st; print ")")
    | printid(I.FCT_INSTANCEid{fct,fsig}) =
        (print "FCT_INSTANCEid{fct="; printid fct; print ",fsig=";
	 printid fsig; print "}")
    | printid(I.FSIGid{argument,body}) =
        (print "FSIGid{argument="; printst argument; print ",body=";
         printst body; print ")")
    | printid(I.TYCid st) = (print "TYCid("; printst st; print ")")

  fun SC env0 =
  let 

  (* unique processing of arrays in INSTANCE structures *)
    val sarray = ref (nil: (M.Structure array * int) list)
    val farray = ref (nil: (M.Functor array * int) list)
    val tarray = ref (nil: (Types.tycon array * int) list)

    fun for(i,j) f = if i<=j then (f i; for(i+1,j) f) else ()

    fun array amap transform from =
	if not (List.exists (fn(a,b)=>a=from) (!amap))
	    then let
		      val new = length(!amap)
		  in amap := (from,new) :: !amap;
		      for(0,Array.length from - 1) 
		          (fn i => transform(Array.sub(from,i)))
		  end
	    else ()

    val table = Stamps.newMap SCStatEnv : modmap
    val btable = Stamps.newMap SCStatEnv : modmap

    fun enter(getid, thing, b : bdg) =
     let val id = getid thing
	 val st = getstamp id
	 fun enter' tab =
	      let val bucket = Stamps.applyMap(tab, st) 
	                       handle SCStatEnv => nil
	      in if not (List.exists (fn (id',_) => I.equalId(id,id')) bucket)
		     then (Stamps.updateMap tab (st, (id,b)::bucket); 
			   true)
		 else false
	      end    
     in if isExternal id 
	    then enter' table andalso     (* free stamp *)
		   (ifdebug (fn()=>(printid id; print "\n")) ();
		    true)
	 else if Stamps.eq(st,st) then enter' btable  (* bound stamp *)
         else true       (* null stamp *)
    end  handle U.Id => true (* hack! wish I understood this. AWA *)
       
    fun binding(M.VARbind v) =  var v
      | binding(M.CONbind v) =  datacon v
      | binding(M.TYCbind v) =  tycon v
      | binding(M.SIGbind v) =  signatureVar v
      | binding(M.STRbind v) =  structureVar v
      | binding(M.FSIGbind v) = funsigVar v
      | binding(M.FCTbind v) =  functorVar v
      | binding(M.FIXbind v) =  ()

    and var(V.VALvar{typ=ref t,...}) = ty t
      | var(V.OVLDvar{name=n,options=ref p,scheme=s}) = 
	             (app var_option p; tyfun s)
      | var(V.ERRORvar) = ()

    and var_option{indicator,variant} = (ty indicator; var variant)

    and tyfun(T.TYFUN{arity,body}) = ty body

    and tyvar(T.INSTANTIATED t) = ty t
      | tyvar(T.OPEN{depth,weakness,eq,kind}) = ()

    and ty(T.VARty t) = tyvar(!t)
      | ty(T.CONty(tyc,tyl)) = (tycon tyc; app ty tyl)
      | ty(T.IBOUND i) = ()
      | ty(T.WILDCARDty) = ()
      | ty(T.POLYty{sign=s,tyfun=t,abs}) = tyfun t
      | ty(T.UNDEFty) = ()


    and tycon(t as T.GENtyc{stamp=s,arity,eq=ref p,path,kind=ref k}) = 
	      (enter(I.TYCid, s,TYCb t); ())
      | tycon(T.DEFtyc{path,strict,tyfun=t}) = tyfun t
      | tycon(T.RECORDtyc labs) = ()
      | tycon(T.FORMtyc{pos,spec,name}) = tycon spec
      | tycon(T.OPENFORMtyc{pos=(il,i),spec,path}) = tycon spec
      | tycon(T.RELtyc{pos=(il,i),path,...}) = ()
      | tycon(T.ABSFBtyc(p,t)) = tycon t
      | tycon(T.ERRORtyc) = ()
      | tycon(T.FULLtyc) = ()
      | tycon(T.EXTERNtyc _) = ErrorMsg.impossible "EXTERNtyc in scstatenv"

    and datacon(T.DATACON{name,const,typ,orig,rep,sign}) = ty typ

    and signatureVar(M.SIGvar{name,binding}) = Signature binding

    and Signature(s as M.SIG{env=ref e,stamp=st,...}) =
         if enter(I.SIGid, st, SIGb s) then env e else ()
      | Signature(M.EXTERN_SIG(I.SIGid st)) = 
	   ErrorMsg.impossible ("EXTERN_SIG in scstatenv: " ^
	                        Stamps.stampToString st ^ "\n")
      | Signature(M.FULL_SIG) = ()
      | Signature(M.ERROR_SIG) = ()

    and env e = StaticEnv.app (fn (s,b) => binding b) e

    and structureVar(M.STRvar{name,access=a,binding=b}) = Structure b

    and Structure(s as M.SIMPLE{stamp=st,env=e,path,id}) =
          if enter(I.SIMPLEid, st, STRb s) then env e else ()
      | Structure(s as M.INSTANCE{sign,subStrs,subFcts,types,origin,path}) =
	  if enter(U.strId, s, STRb s)
	   then (Signature sign;
                 array sarray Structure subStrs;
	         array farray Functor subFcts;
	         array tarray tycon types)
           else ()
      | Structure(M.STR_OPEN{pos,spec,path}) = Signature spec
      | Structure(M.STR_FORMAL{pos,spec}) = Signature spec
      | Structure(M.STR_ABSFB p) = ()
      | Structure(M.APPLY{fct,arg,res,path}) = (Functor fct; Structure arg; Structure res)
      | Structure(M.STR_EXTERN _) = ErrorMsg.impossible "STR_EXTERN in scstatenv"
      | Structure(M.ERROR_STR) = ()

    and funsigVar(M.FSIGvar{name,binding}) = FctSignature binding

    and FctSignature(f as M.FSIG{name,paramName,argument,body}) =
		if enter(U.fsigId, f, FSIGb f)
		 then (Signature argument; Signature body)
		 else ()
      | FctSignature(M.EXTERN_FSIG _) = ErrorMsg.impossible "EXTERN_FSIG in scstatenv"
      | FctSignature(M.FULL_FSIG) = ()
      | FctSignature(M.ERROR_FSIG) = ()

    and functorVar(M.FCTvar{name,access,binding}) = Functor binding
    and Functor(f as M.FCT{stamp=st,parent,paramName=_,lambdaty=_,argument,
		    body={strseq,fctseq,tyseq,str}}) =
	    if enter(U.fctId, f, FCTb f)
            then (Structure parent;
                  Signature argument; app Structure strseq;
		  app Functor fctseq; app tycon tyseq; Structure str)
            else ()
      | Functor(M.FCT_FORMAL{pos,spec}) = FctSignature spec
      | Functor(M.FCT_OPEN{pos,spec,path}) = FctSignature spec
      | Functor(M.FCT_INSTANCE{fsig,fct,parent,...}) =
		  (FctSignature fsig; Functor fct; Structure parent)
      | Functor(M.FCT_ABSFB p) = ()
      | Functor(M.FCT_EXTERN _) = ErrorMsg.impossible "FCT_EXTERN in scstatenv"
      | Functor(M.ERROR_FCT) = ()

 in ifdebug print "[Starting SCStatenv...";
    env env0;
    ifdebug print "Finishing SCStatenv...]";
    (env0,[table])
end

  fun SCe{static=x,dynamic=y} = {static=SC x,dynamic=y}
  fun unSCe{static=x,dynamic=y} = {static=unSC x, dynamic=y}

end
