(* hashenv.sml
 *
 * COPYRIGHT (c) 1996 Bell Laboratories.
 *)

(* hashenv.sml *)

signature HASH_ENV =
sig
  val hashEnv : StaticEnv.staticEnv -> string
  val debugging : bool ref
end

structure HashEnv : HASH_ENV =
struct

 open Modules Access Variables Types Fixity ModuleId

 val debugging = CRCUtil.debugging
     
fun hashEnv (env0: StaticEnv.staticEnv) : string = 
let val crc = CRC.new()
 
    val _ = CRCUtil.debugmsg "\nCRC: "

    val { app = c, for, int, bool, string, list, option,
	  access, conrep, ... } =
	CRCUtil.mkUtils (crc, false)

    local exception HashEnv
	  val m : (Stamps.stamp * int) Intmap.intmap = Intmap.new(32,HashEnv)
          val map = Intmap.map m and add = Intmap.add m
          val alphacount = ref 0
      in fun enter(st,c) = #2(map c)
	      handle HashEnv =>
		  let val a = !alphacount
		   in alphacount := a+1;
		      add(c,(st,a));
		      a
		  end
         fun setScope scope =
	     Intmap.app (fn(c,(st,a))=>Stamps.setFreeStamp(st,scope,a)) m
     end	      

    fun stamp st = Stamps.hashStamp(c,int,string,enter,st)

    fun spath (SymPath.SPATH s) = list (string o Symbol.name) s

    fun ipath (InvPath.IPATH s) = list (string o Symbol.name) s

    fun Sharing f {internal,external} =
	(list spath internal; option f external)

    fun moduleId(SIMPLEid st) = (c 0; stamp st)
      | moduleId(INSTANCEid{origin,sign}) = (c 1; stamp origin; stamp sign)
      | moduleId(SIGid st) = (c 2; stamp st)
      | moduleId(FCTid st) = (c 3; stamp st)
      | moduleId(FCT_INSTANCEid{fct,fsig}) = (c 4; moduleId fct; moduleId fsig)
      | moduleId(FSIGid{argument,body}) = (c 5; stamp argument; stamp body)
      | moduleId(TYCid st) = (c 6; stamp st)

  (* unique processing of arrays in INSTANCE structures *)
    val sarray = ref (nil: (Structure array * int) list)
    val farray = ref (nil: (Functor array * int) list)
    val tarray = ref (nil: (tycon array * int) list)

    fun array amap transform from =
	  let fun look((a,b)::rest) = if a=from then (c 1; int b) else look rest
		| look nil = 
		  let open Array infix 9 sub
		      val new = List.length(!amap)
		  in amap := (from,new) :: !amap;
		      c 0;
		      for(0,length from - 1) (fn i => transform(from sub i))
		  end
	  in look (!amap)
	  end

    val table = ref(ModuleTable.empty : unit ModuleTable.table)

    fun uniq(modid,body) =
      case ModuleTable.look(modid,!table)
       of NONE => (table := ModuleTable.bind(modid,(),!table);
		   body())
	| SOME () => moduleId modid

    fun binding(VARbind v) = (c 0; var v)
      | binding(CONbind v) = (c 1; datacon v)
      | binding(TYCbind v) = (c 2; tycon v)
      | binding(SIGbind v) = (c 3; signatureVar v)
      | binding(STRbind v) = (c 4; structureVar v)
      | binding(FSIGbind v) = (c 5; funsigVar v)
      | binding(FCTbind v) = (c 6; functorVar v)
      | binding(FIXbind v) = (c 7; fixityVar v)

    and var(VALvar{access=a,path=p,typ=ref t}) =
	  (c 0; access a; spath p; ty t)
      | var(OVLDvar{name=n,options=ref p,scheme=s}) =
	  (c 1; symbol n; list var_option p; tyfun s)
      | var(ERRORvar) = c 2

    and var_option{indicator,variant} = (ty indicator; var variant)

    and tyfun(TYFUN{arity,body}) = (int arity; ty body)

    and tyvar(INSTANTIATED t) = (c 0; ty t)
      | tyvar(OPEN{depth,weakness,eq,kind}) =
	  (c 1; int depth; int weakness; bool eq; tvkind kind)

    and tvkind(META) = (c 0)
      | tvkind(UBOUND s) = (c 1; symbol s)
      | tvkind(FLEX _) = (c 2)  (* shouldn't occur *)

    and ty(VARty t) = (c 0; tyvar(!t))
      | ty(CONty(tyc,tyl)) = (c 1; tycon tyc; list ty tyl)
      | ty(IBOUND i) = (c 2; int i)
      | ty(WILDCARDty) = c 3
      | ty(POLYty{sign=s,tyfun=t,abs}) = (
	  c 4;
	  list (fn{weakness,eq} => (int weakness; bool eq)) s;
	  tyfun t;
	  int abs)
      | ty(UNDEFty) = c 5


    and tycon(GENtyc{stamp=s,arity,eq=ref p,path,kind=ref k}) = 
	  (c 0; stamp s; int arity; eqprop p; ipath path; tyckind k)
      | tycon(DEFtyc{path,strict,tyfun=t}) = 
	  (c 1; ipath path; list bool strict; tyfun t)
      | tycon(RECORDtyc labs) = (c 2; list (string o Symbol.name) labs)  (* nice pun, no? *)
      | tycon(FORMtyc{pos,spec,name}) = (c 3; int pos; tycon spec; symbol name)
      | tycon(OPENFORMtyc{pos=(il,i),spec,path}) = 
	  (c 4; list int il; int i; tycon spec; spath path)
      | tycon(RELtyc{pos=(il,i),spec,path}) = 
	  (c 5; list int il; int i; (*tycon spec;*)
	   ipath path)
      | tycon(ABSFBtyc(p,t)) = (c 6; absfbpos p; tycon t)
      | tycon(ERRORtyc) = c 7
      | tycon(FULLtyc) = c 8
      | tycon(EXTERNtyc _) = ErrorMsg.impossible "EXTERNtyc in hashenv"

    and eqprop YES = c 0 | eqprop NO = c 1 | eqprop IND = c 2 
      | eqprop OBJ = c 3 | eqprop DATA = c 4 | eqprop UNDEF = c 5

    and absfbpos(PARAM il) = (c 0; list int il)
      | absfbpos(SEQ i) = (c 1; int i)
      | absfbpos(SEQind(i,il)) = (c 2; int i; list int il)

    and tyckind(PRIMtyc) = c 0
      | tyckind(ABStyc _) = c 1
      | tyckind(DATAtyc _) = c 2
      | tyckind(FORMtyck) = c 3
      | tyckind(FORMDEFtyc _) = c 3

    and tyoption (SOME t) = ty t
      | tyoption NONE = ()

    and datacon(DATACON{name,const,typ,rep,sign,orig}) =
	(string(Symbol.name name); bool const; ty typ; 
         tyoption orig; conrep rep (* omit sign! *))

    and signatureVar(SIGvar{name,binding}) = (string(Symbol.name name);
					      Signature binding)

    and Signature(SIG{symbols=ref s,env=ref e,stamp=st,name=n,kind=ref k}) =
	if Stamps.isExternal st
	    then (c 4; stamp st)
	    else uniq(SIGid st,
		      fn()=>(c 0; list (string o Symbol.name) s; env e;
			     stamp st; option symbol n; sigkind k))
      | Signature(EXTERN_SIG _) = ErrorMsg.impossible "EXTERN_SIG in hashenv"
      | Signature(FULL_SIG) = c 2
      | Signature(ERROR_SIG) = c 3

    and sigkind(TOP{strcount,fctcount,typecount,slotcount,     (* FIX *)
		    sConstraints,tConstraints,abbreviations}) =
	(c 0; int strcount; int fctcount; int typecount; int slotcount;
	 list (Sharing Structure) sConstraints; 
	 list (Sharing tycon) tConstraints;
	 list (fn{internal,external}=>(int internal; tycon external)) 
	    abbreviations)
      | sigkind EMBEDDED = c 1
      | sigkind IRRELEVANT = c 2


    and env e = list (fn (s,b) => (symbol s; binding b)) (ModuleUtil.sortEnvBindings e)
    and symbol s = (string(Symbol.name s); int(Symbol.number s))

    and structureVar(STRvar{name,access=a,binding=b}) =
	(symbol name; access a; Structure b)

    and Structure(SIMPLE{stamp=st,env=e,path=s,id}) =
	  if Stamps.isExternal st
	    then (c 0; stamp st)
	    else uniq(id, fn()=> (c 1; stamp st; env e; ipath s))
      | Structure(INSTANCE{sign,subStrs,subFcts,types,origin,path}) =
	  (c 2; Signature sign; ipath path;
	   case origin
	     of STAMP_ORIGIN st =>
		 (c 0; stamp st;
		  if Stamps.isExternal st
		  then c 0
		  else (c 1;
			array sarray Structure subStrs;
			array farray Functor subFcts;
			array tarray tycon types))
	      | ABS_ORIGIN pos =>
		 (c 1; absfbpos pos;
		  array sarray Structure subStrs;
		  array farray Functor subFcts;
		  array tarray tycon types)
	      | NULL_ORIGIN =>
		 (c 3;
		  array sarray Structure subStrs;
		  array farray Functor subFcts;
		  array tarray tycon types))
      | Structure(STR_OPEN{pos,spec,path}) =
	  (c 3; list int pos; Signature spec; spath path)
      | Structure(STR_FORMAL{pos,spec}) =
	  (c 4; int pos; Signature spec) 
      | Structure(STR_ABSFB p) = (c 5; absfbpos p)
      | Structure(APPLY{fct,arg,res,path}) =
	  (c 7; Functor fct; Structure arg; Structure res; ipath path)
      | Structure(STR_EXTERN _) = ErrorMsg.impossible "STR_EXTERN in hashenv"
      | Structure(ERROR_STR) = c 9


    and funsigVar(FSIGvar{name,binding}) = (symbol name; FctSignature binding)

    and FctSignature(FSIG{name,paramName,argument,body}) =
	  (c 0; option symbol name; symbol paramName;
	   Signature argument; Signature body)
      | FctSignature(EXTERN_FSIG _) = ErrorMsg.impossible "EXTERN_FSIG in hashenv"
      | FctSignature(FULL_FSIG) = c 1
      | FctSignature(ERROR_FSIG) = c 2

    and functorVar(FCTvar{name,access=a,binding}) =
	  (symbol name; access a; Functor binding)

    and Functor(FCT{stamp=st,parent,paramName=_,lambdaty=_,argument,
		    body={strseq,fctseq,tyseq,str}}) =
	    if Stamps.isExternal st then (c 0; stamp st)
	    else uniq(FCTid st,
		      fn()=> (c 1; stamp st; Structure parent;
			      Signature argument; list Structure strseq;
			      list Functor fctseq; list tycon tyseq; Structure str))
      | Functor(FCT_FORMAL{pos,spec}) =
		  (c 2; int pos; FctSignature spec)
      | Functor(FCT_OPEN{pos,spec,path}) =
	  (c 3; list int pos; FctSignature spec; spath path)
      | Functor(FCT_INSTANCE{fsig,fct,parent,...}) =
	  (c 4; FctSignature fsig; Functor fct; Structure parent)
      | Functor(FCT_ABSFB p) = (c 5; absfbpos p)
      | Functor(FCT_EXTERN _) = ErrorMsg.impossible "FCT_EXTERN in hashenv"
      | Functor(ERROR_FCT) = c 6

    and fixityVar(FIXvar{name,binding}) = (symbol name; fixity binding)
    and fixity(NONfix) = c 0
      | fixity(INfix(i,j)) = (c 1; int i; int j)

   val envHash = (env env0; 
		  CRCUtil.debugmsg "\n";
		  CRC.extract crc)

 in setScope envHash; envHash
end

val hashenv = Stats.doPhase(Stats.makePhase "Compiler 042 HashEnv") hashEnv

end (* structure HashEnv *)
