(* Copyright 1989 by AT&T Bell Laboratories *)
(* environ.sml *)

structure Environment: ENVIRONMENT = struct

  type symbol = Symbol.symbol
  type staticEnv = StaticEnv.staticEnv
  type dynenv  = DynamicEnv.dynenv
  type symenv = SymbolicEnv.symenv

  type environment = { static: staticEnv, dynamic: dynenv, symbolic: symenv }

  fun staticPart (e: environment) = #static e
  fun dynamicPart (e: environment) = #dynamic e
  fun symbolicPart (e: environment) = #symbolic e
      
  fun mkenv (e as { static, dynamic, symbolic }) = e

  val emptyEnv = { static   = StaticEnv.empty,
		   dynamic  = DynamicEnv.empty,
		   symbolic = SymbolicEnv.empty }

  fun layerEnv({ static, dynamic, symbolic },
	       { static = sta, dynamic = dy, symbolic = sy }) =
      { static =  StaticEnv.atop (static, sta),
        dynamic = DynamicEnv.atop (dynamic, dy),
	symbolic = SymbolicEnv.atop (symbolic, sy) }
  
  val layerStatic =  StaticEnv.atop
  val layerSymbolic = SymbolicEnv.atop
  
  fun consolidateEnv ({ static, dynamic, symbolic }) =
      { static = StaticEnv.consolidate static,
        dynamic = DynamicEnv.consolidate dynamic,
        symbolic = SymbolicEnv.consolidate symbolic }

  val consolidateStatic = StaticEnv.consolidate
  val consolidateSymbolic = SymbolicEnv.consolidate

  fun reff a = let val r = ref a
               in {get=(fn()=> !r),set=(fn x=> r:=x)}
	      end

  val topLevelEnvRef = reff emptyEnv
  val pervasiveEnvRef = reff emptyEnv
  val coreEnvRef = reff StaticEnv.empty

 (* functions to collect stale lvars for unbinding in concatEnv *)

  open ErrorMsg Access Variables Types Modules

  exception NotStale

  (* staleLvars: takes a new environment and a base environment to which
     it is to be added and returns a list of lvars that are unreachable 
     when the new environment is added to the base environment*)

(*
  ** -- this doesn't work anymore since we have changed the interface
  **    to compilation units.
  **      M.Blume

  fun stalePids(deltaEnv,baseEnv) : PersStamps.persstamp list =
      let val lvarset = ref([] : PersStamps.persstamp list)

	  (* collect: for each symbol bound in the new environment,
	     check if there was a binding in the base environment.
	     If there was a binding and it has an lvar associated
             with it, add it to the list of stale lvars. *)

	  fun collect (s,_) =   (* should use Env.fold *)
	    let val v = case Env.look(baseEnv,s)
			 of VARbind(VALvar{access=EXTERN v,...}) => v
			  | STRbind(STRvar{access=EXTERN v,...}) => v
			  | FCTbind(FCTvar{access=EXTERN v,...}) => v
			  | CONbind(DATACON{rep=VARIABLE(EXTERN v),...}) => v
			  | CONbind(DATACON{rep=VARIABLEc(EXTERN v),...}) => v
			  | _ => raise NotStale
	     in lvarset := v :: !lvarset
	    end handle NotStale => ()
		     | Env.Unbound => ()
       in Env.app collect deltaEnv;
	  !lvarset
      end

  *)

  (* what we do instead:
   *  - count the number of occurences for each pid in baseEnv bindings
   *    that is going to be shadowed by deltaEnv
   *  - count the total number of total occurences for each such
   *    pids in baseEnv
   *  - the ones where the counts coincide are stale
   *
   * This code is ok, because deltaEnv is the output of `export'.  `export'
   * calls consolidateStatic, therefore we don't have duplicate bindings
   * of the same symbol.
   *)
  fun stalePids (deltaEnv, baseEnv) = let
      (* getting the stamp from an access *)
      fun acc (EXTERN v) = SOME v
	| acc (PATH (_, a)) = acc a
	| acc _ = NONE
      (* getting the stamp from a binding *)
      fun stampOf (VARbind (VALvar { access = a, ... })) = acc a
	| stampOf (CONbind (DATACON { rep = VARIABLE a, ... })) = acc a
	| stampOf (CONbind (DATACON { rep = VARIABLEc a, ... })) = acc a
	| stampOf (STRbind (STRvar { access = a, ... })) = acc a
	| stampOf (FCTbind (FCTvar { access = a, ... })) = acc a
	| stampOf _ = NONE
      (* any rebindings? *)
      val anyrebound = ref false
      (* counting map *)
      val countM = ref (PersMap.empty: int PersMap.map)
      val look = PersMap.lookup (!countM)
      fun add (s, c) = countM := PersMap.add (!countM, s, c)
      (* initialize the counter map: for each new binding with stamp
       * check if the same symbol was bound in the old env and enter
       * the old stamp into the map *)
      fun initOne s = add (s, (look s handle PersMap.MapF => 0) - 1)
      fun initC (sy, _) =
	  (case stampOf (Env.look (baseEnv, sy)) of
	       NONE => ()
	     | SOME s => (initOne s; anyrebound := true))
	       handle Env.Unbound => ()
      (* increment counter for a given stamp *)
      fun incr NONE = ()
	| incr (SOME s) = (add (s, (look s) + 1)) handle PersMap.MapF => ()
      fun incC (_, b) = incr (stampOf b)
      (* select the 0s *)
      fun selZero ((s, 0), zeros) = s :: zeros
	| selZero (_, zeros) = zeros
  in
      Env.app initC deltaEnv;		(* init counter map *)
      if !anyrebound then let		(* shortcut if no rebindings *)
	  (* count the pids *)
	  val _ = Env.app incC baseEnv
	  (* pick out the stale ones *)
	  val stalepids = foldl selZero [] (PersMap.members (!countM))
      in
	  stalepids
      end
      else []
  end

  fun exports (deltaStat: staticEnv, basePid) = let

      val lvars = ref (nil: Access.lvar list)
      val pos = ref 0

      fun newpos () = let val p = !pos in pos := p + 1; p end

      fun rebind v = let
	  val p = newpos ()
      in
	  lvars := v :: !lvars;
	  PATH (p, EXTERN basePid)
      end

      fun h (VARbind (VALvar {access = LVAR v, path, typ })) =
	  VARbind (VALvar { access = rebind v, path = path, typ = typ })
	| h (CONbind (DATACON { name, const, typ, rep = VARIABLE (LVAR v),
			        sign, orig })) =
	  CONbind (DATACON { name = name, const = const, typ = typ,
			     orig = orig, rep = VARIABLE (rebind v),
			     sign = sign })
	| h (CONbind (DATACON { name, const, typ, rep = VARIABLEc (LVAR v),
			        sign, orig })) =
	  CONbind (DATACON { name = name, const = const, typ = typ,
			     orig = orig,
			     rep = VARIABLEc (rebind v), sign = sign })
	| h (STRbind (STRvar { name, access = LVAR v, binding })) =
	  STRbind (STRvar { name = name, binding = binding,
			    access = rebind v })
	| h (FCTbind (FCTvar { name, access = LVAR v, binding })) =
	  FCTbind (FCTvar { name = name, binding = binding,
			    access = rebind v })
	| h b = b

      val bindings = 
	  map (fn (s, b) => (s, h b)) 
	      (ModuleUtil.sortEnvBindings (StaticEnv.consolidate deltaStat))
      (* sortEnvBindings should only be applied to static environments
       * with no duplicate bindings *)
      fun bind ((s, b), e) = StaticEnv.bind (s, b, e)

      fun wrap (v, l) =
	  Lambda.WRAP (LambdaType.injBOXED, Lambda.VAR v) :: l

      fun var (v, l) = (Lambda.VAR v) :: l

      val envSlice =
	  StaticEnv.consolidate (foldr bind StaticEnv.empty bindings)
      (* The double application of consolidate, before and after
       * sorting the bindings, is unfortunate.  This could probably
       * be streamlined.  This function perhaps belongs in StaticEnv
       * or perhaps there should be a new operation in Env to support
       * it. *)

      val (exportLexp, exportPid) =
	  case foldl wrap [] (!lvars) of
	      [] => (Lambda.WRAP (LambdaType.injBOXED, Lambda.INT 0), NONE)
	    | l => (Lambda.WRAP (LambdaType.injBOXED, Lambda.SRECORD l),
		    SOME basePid)

  in
      (envSlice, exportLexp, exportPid)
  end
      
  fun concatEnv ({ static = newstat, dynamic = newdyn, symbolic = newsym },
		 { static = oldstat, dynamic = olddyn, symbolic = oldsym }) =
      let
	  val hidden_pids = stalePids (newstat, oldstat)
	  val slimdyn = DynamicEnv.remove (hidden_pids, olddyn)
	  val slimsym = SymbolicEnv.remove (hidden_pids, oldsym)
      in
	  { static =StaticEnv.consolidate (StaticEnv.atop (newstat, oldstat)),
	    dynamic = DynamicEnv.atop (newdyn, slimdyn),
	    symbolic = SymbolicEnv.atop (newsym, slimsym) }
      end

  fun getbindings(static: staticEnv, symbols: Symbol.symbol list) :
        (Symbol.symbol * Modules.binding) list =
      let fun loop([], bindings) = bindings
	    | loop(s::rest, bindings) =
	      let val bindings' = (s,Env.look(static,s)) :: bindings
				  handle Env.Unbound => bindings
	      in loop (rest, bindings') end
      in  loop (symbols,[])
      end

  fun copystat ([], senv) = senv
    | copystat ((s, b) :: l, senv) = copystat (l, Env.bind (s, b, senv))

  fun filterStaticEnv(static: staticEnv,
		      symbols: Symbol.symbol list) : staticEnv =
      copystat(getbindings(static, symbols), Env.empty)

  fun root(EXTERN pid) = SOME pid 
    | root(PATH(_,p)) = root p
    | root _ = NONE

  fun pidOfBinding(VARbind(Variables.VALvar{access,...})) = root access
    | pidOfBinding(CONbind(DATACON{rep=VARIABLE access,...})) = root access
    | pidOfBinding(CONbind(DATACON{rep=VARIABLEc access,...})) = root access
    | pidOfBinding(STRbind(STRvar{name,access,...})) = root access
    | pidOfBinding(FCTbind(FCTvar{name,access,...})) = root access
    | pidOfBinding _ = NONE

  fun filterEnv ({ static, dynamic, symbolic }, symbols) = let
      val sbindings = getbindings (static, symbols)
      fun copydynsym ([], denv, syenv) = (denv, syenv)
	| copydynsym ((_, b) :: l, denv, syenv) =
	  (case pidOfBinding b of
	       NONE => copydynsym (l, denv, syenv)
	     | SOME lv => let
		   val dy = DynamicEnv.look dynamic lv
		   val denv = DynamicEnv.bind (lv, dy, denv)
		   val sy = SymbolicEnv.look symbolic lv
		   val syenv = case sy of
		       NONE => syenv
		     | SOME sy => SymbolicEnv.bind (lv, sy, syenv)
	       in
		   copydynsym (l, denv, syenv)
	       end)
      val senv = copystat (sbindings, Env.empty) 
      val (denv, syenv) =
	  copydynsym (sbindings, DynamicEnv.empty, SymbolicEnv.empty)
  in
      { static =senv, dynamic = denv, symbolic = syenv }
  end

  fun catalogEnv static : Symbol.symbol list =
      map #1 (ModuleUtil.sortEnvBindings static)

  datatype cmEnv =
      CM_NONE
    | CM_ENV of Symbol.symbol -> cmEnv

  (* CM-style environment lookup *)
  exception CmEnvOfModule
  fun cmEnvOfModule env sym = let

      fun complainer _ _ _ = raise CmEnvOfModule

      fun s2cm senv = CM_ENV (cmEnvOfModule senv)

      fun sigenv (SIG { env = ref env, ... }) = s2cm env
	| sigenv _ = CM_NONE

      fun strenv (SIMPLE { env, ... }) = s2cm env
	| strenv (INSTANCE { sign, ... }) = sigenv sign
	| strenv (STR_OPEN { spec, ... }) = sigenv spec
	| strenv (STR_FORMAL { spec, ... }) = sigenv spec
	| strenv (APPLY { res, ... }) = strenv res
	| strenv _ = CM_NONE

      fun fsigenv (FSIG { body, ... }) = sigenv body
	| fsigenv _ = CM_NONE

      fun fctenv (FCT { parent, argument, paramName,
		        body = { strseq, fctseq, str, ... }, ... }) =
	  let
	      fun sub (l, i) = List.nth (l, i)
	      fun sub_str i = sub (strseq, i)
	      fun sub_fct i = sub (fctseq, i)
	      fun fetch_str (s, il) = ModuleUtil.transPosStr s il
	      fun fetch_fct (s, il) = ModuleUtil.transPosFct s il
	      fun fake () = let
		  val ipath = InvPath.IPATH []
		  val scope = Stamps.newBoundScope ()
		  val contxt = (ipath, scope, complainer)
		  val insta = Instantiate.instantiate_argument
	      in
		  insta (contxt, paramName, parent, argument)
	      end

	      val SP = SymPath.SPATH

	      fun ctxt (subStrs, subFcts) = let
	  
		  fun look env sym =
		      (case Symbol.nameSpace sym of
			   Symbol.SIGspace => let
			       val sign =
				   ModuleUtil.lookSIG (env, sym, complainer)
			   in
			       sige sign
			   end
			 | Symbol.STRspace => let
			       val STRvar { binding, ... } =
				   ModuleUtil.lookSTR (env, SP [sym],
						       complainer)
			   in
			       stre binding
			   end
			 | Symbol.FSIGspace => let
			       val fsign =
				   ModuleUtil.lookFSIG (env, sym, complainer)
			   in
			       fsige fsign
			   end
			 | Symbol.FCTspace => let
			       val FCTvar { binding, ... } =
				   ModuleUtil.lookFCT (env, SP [sym],
						       complainer)
			   in
			       fcte binding
			   end
			 | _ => CM_NONE)
			   handle _ => CM_NONE

		  and sige (SIG { env = ref e, ... }) = CM_ENV (look e)
		    | sige _ = CM_NONE

		  and stre (SIMPLE { env, ... }) = CM_ENV (look env)
		    | stre (INSTANCE { sign, subStrs, subFcts, ... }) =
		      ctxt (subStrs, subFcts) sign
		    | stre (STR_OPEN { spec, ... }) = sige spec
		    | stre (STR_FORMAL { spec = FULL_SIG, pos }) =
		      stre (Array.sub (subStrs, pos))
		    | stre (STR_FORMAL { spec, ... }) = sige spec
		    | stre (APPLY { res, ... }) = stre res
		    | stre (STR_ABSFB (Types.PARAM il)) =
		      stre (fetch_str (fake (), il))
		    | stre (STR_ABSFB (Types.SEQ i)) =
		      stre (sub_str i)
		    | stre (STR_ABSFB (Types.SEQind (i, il))) =
		      stre (fetch_str (sub_str i, il))
		    | stre _ = CM_NONE

		  and fsige (FSIG { body, ... }) = sige body
		    | fsige _ = CM_NONE

		  and fcte (f as FCT _) = fctenv f
		    | fcte (FCT_FORMAL { spec = FULL_FSIG, pos }) =
		      fcte (Array.sub (subFcts, pos))
		    | fcte (FCT_OPEN { spec, ... }) = fsige spec
		    | fcte (FCT_INSTANCE { fsig, ... }) = fsige fsig
		    | fcte (FCT_ABSFB (Types.PARAM il)) =
		      fcte (fetch_fct (fake (), il))
		    | fcte (FCT_ABSFB (Types.SEQ i)) =
		      fcte (sub_fct i)
		    | fcte (FCT_ABSFB (Types.SEQind (i, il))) =
		      fcte (fetch_fct (sub_str i, il))
		    | fcte _ = CM_NONE

	      in
		  sige
	      end

	      fun inst (INSTANCE { sign, subStrs, subFcts, ... }) =
		  ctxt (subStrs, subFcts) sign
		| inst (STR_ABSFB (Types.PARAM il)) =
		  inst (fetch_str (fake () , il))
		| inst (STR_ABSFB (Types.SEQ i)) =
		  inst (sub_str i)
		| inst (STR_ABSFB (Types.SEQind (i, il))) =
		  inst (fetch_str (sub_str i, il))
		| inst str = strenv str
	  in
	      inst str
	  end
	| fctenv (FCT_OPEN { spec, ... }) = fsigenv spec
	| fctenv (FCT_FORMAL { spec, ... }) = fsigenv spec
	| fctenv (FCT_INSTANCE { fsig, ... }) = fsigenv fsig
	| fctenv _ = CM_NONE

  in
      case Symbol.nameSpace sym of
	  Symbol.SIGspace =>
	      sigenv (ModuleUtil.lookSIG (env, sym, complainer))
	| Symbol.STRspace => let
	      val STRvar { binding, ... } =
		  ModuleUtil.lookSTR (env, SymPath.SPATH [sym], complainer)
	  in
	      strenv binding
	  end
	| Symbol.FSIGspace =>
	      fsigenv (ModuleUtil.lookFSIG (env, sym, complainer))
	| Symbol.FCTspace => let
	      val FCTvar { binding, ... } =
		  ModuleUtil.lookFCT (env, SymPath.SPATH [sym], complainer)
	  in
	      fctenv binding
	  end
	| _ => CM_NONE
  end handle _ => CM_NONE

  (* for backward compatibility to SC *)
  exception EnvOfStructure
  fun envOfStructure (env, sym) = let

      fun complainer _ _ _ = raise EnvOfStructure

      val STRvar { binding, ... } =
	  ModuleUtil.lookSTR (env, SymPath.SPATH [sym], complainer)
	  
      fun sigenv (SIG { env = ref env, ... }) = env
	| sigenv _ = raise EnvOfStructure

      fun strenv (SIMPLE { env, ... }) = env
	| strenv (INSTANCE { sign, ... }) = sigenv sign
	| strenv (STR_OPEN { spec, ... }) = sigenv spec
	| strenv (STR_FORMAL { spec, ... }) = sigenv spec
	| strenv (APPLY { res, ... }) = strenv res
	| strenv _ = raise EnvOfStructure

  in
      strenv binding
  end

  fun describe static (s: Symbol.symbol) : unit =
      let open PrettyPrint
       in with_pp (ErrorMsg.defaultConsumer())
	   (fn ppstrm =>
	    (begin_block ppstrm CONSISTENT 0;
	     PPBasics.ppBinding ppstrm (static,StaticEnv.look(static,s),
					!Control.Print.printDepth);
	     add_newline ppstrm;
	       end_block ppstrm))
      end
      handle Env.Unbound => print (Symbol.name s ^ " not found\n")

  val primEnv = Prim.primEnv

end (* structure Environment *)
